﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/fs.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/nn_TimeSpan.h>

#include <nn/xcd/xcd.h>
#include <nnt/nntest.h>

namespace {
    struct XcdDeviceBlock
    {
        nn::xcd::DeviceHandle handle;
        nn::os::SystemEventType event;
        nn::os::MultiWaitHolderType holder;
        nn::xcd::DeviceInfo info;
        nn::xcd::PadState pad;
        nn::xcd::DeviceStatus status;
        nn::xcd::PadButtonSet previousButton;
        bool connected;
    };
    XcdDeviceBlock g_Devices[nn::xcd::DeviceCountMax];

    nn::os::MultiWaitType g_MultiWait;
    nn::os::SystemEventType g_LinkEvent;
    nn::os::MultiWaitHolderType g_LinkHolder;
    nn::os::EventType g_TerminateEvent;
    nn::os::MultiWaitHolderType g_TerminateHolder;
    nn::os::TimerEventType g_TimerEvent;
    nn::os::MultiWaitHolderType g_TimerHolder;
    nn::os::SystemEventType g_FirmwareUpdateEvent;
    nn::os::MultiWaitHolderType g_FirmwareUpdateHolder;

    const uint8_t LedPattern[] = {0x01, 0x03, 0x07, 0x0f, 0x09, 0x05, 0x0d, 0x06};

    nn::fs::FileHandle g_File;
    int g_FirmwareUpdatingDeviceIndex;
} // namespace

void PrintOtaStatus(nn::xcd::FirmwareUpdateStage updateStage, int progress)
{
    // 描画
    switch (updateStage)
    {
    case nn::xcd::FirmwareUpdateStage_Download:
        NN_LOG("Download : ");
        break;
    case nn::xcd::FirmwareUpdateStage_Verify:
        NN_LOG("Verify   : ");
        break;
    case nn::xcd::FirmwareUpdateStage_Commit:
        NN_LOG("Commit   : ");
        break;
    case nn::xcd::FirmwareUpdateStage_Completed:
        NN_LOG("Completed: ");
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    NN_LOG("(%3d) ", progress);
    for (int i = 0; i < progress / 2; i += 2)
    {
        NN_LOG("|");
    }
    NN_LOG("\n");
}

void InitializeEvents()
{
    nn::os::InitializeMultiWait(&g_MultiWait);

    for(int i = 0; i < nn::xcd::DeviceCountMax; i++)
    {
        nn::os::CreateSystemEvent(&g_Devices[i].event, nn::os::EventClearMode_ManualClear, false);
        nn::os::InitializeMultiWaitHolder(&g_Devices[i].holder, &g_Devices[i].event);
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_Devices[i].holder);
    }

    nn::os::CreateSystemEvent(&g_LinkEvent, nn::os::EventClearMode_ManualClear, false);
    nn::os::InitializeMultiWaitHolder(&g_LinkHolder, &g_LinkEvent);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_LinkHolder);

    nn::os::CreateSystemEvent(&g_FirmwareUpdateEvent, nn::os::EventClearMode_ManualClear, false);
    nn::os::InitializeMultiWaitHolder(&g_FirmwareUpdateHolder, &g_FirmwareUpdateEvent);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_FirmwareUpdateHolder);

    nn::os::InitializeEvent(&g_TerminateEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&g_TerminateHolder, &g_TerminateEvent);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_TerminateHolder);

    nn::os::InitializeTimerEvent(&g_TimerEvent, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&g_TimerHolder, &g_TimerEvent);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_TimerHolder);
    nn::os::StartPeriodicTimerEvent(&g_TimerEvent, 0, nn::TimeSpan::FromMilliSeconds(200));
}

void OpenOtaFile()
{
    nn::Result result;

    // リソースデータを読み込み
    result = nn::fs::OpenFile(
        &g_File,
        "rom:/fwUpdate/ukyosakyo_ep2_0253_ota.bin",
        nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());
}

void CloseOtaFile()
{
    // リソースデータを閉じる
    nn::fs::CloseFile(g_File);
}

/**
 * XCD の基礎機能をテストする
 */
TEST( XcdOtaUpdate, Full )
{
    nn::Result result;

    InitializeEvents();

    for(int i = 0; i < nn::xcd::DeviceCountMax; i++)
    {
        g_Devices[i].connected = false;
    }

    // XCDの初期化
    nn::xcd::Initialize(&g_MultiWait);
    nn::xcd::BindLinkUpdateEvent(&g_LinkEvent);
    nn::xcd::StartButtonPairing();

    // ファイルシステムをマウント
    static const size_t MountRomCacheBufferSize = 4 * 1024;
    static char s_MountRomCacheBuffer[MountRomCacheBufferSize];
    size_t mountRomCacheUseSize = 0;
    result = nn::fs::QueryMountRomCacheSize(&mountRomCacheUseSize);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT_LESS_EQUAL(mountRomCacheUseSize, MountRomCacheBufferSize);

    result = nn::fs::MountRom("rom", s_MountRomCacheBuffer, MountRomCacheBufferSize);
    NN_ASSERT(result.IsSuccess());

    while(NN_STATIC_CONDITION(true))
    {
        // デバイス情報の更新を待機
        auto pHolder = nn::os::WaitAny(&g_MultiWait);

        if(nn::xcd::Proceed(pHolder) == true)
        {
            // 何もしない
        }
        else if(pHolder == &g_LinkHolder)
        {
            nn::os::ClearSystemEvent(&g_LinkEvent);

            // nn::hid::GetDebugPadStates による状態取得
            nn::xcd::DeviceList list;
            if(nn::xcd::ListDevices(&list).IsFailure())
            {
                NN_LOG("No device Found\n");
            }

            // 接続されたデバイスの探索
            for(int i = 0; i < list.deviceCount; i++)
            {
                NN_LOG("New XCD Device Found\n");
                g_Devices[i].handle = list.handleList[i];
                nn::xcd::SetSamplingEvent(&g_Devices[i].event, g_Devices[i].handle);
                nn::xcd::SleepSensor(false, g_Devices[i].handle);
                nn::xcd::SetPlayerIndicatorPattern(LedPattern[i], g_Devices[i].handle);
                g_Devices[i].connected = true;
                nn::xcd::GetDeviceInfo(&g_Devices[i].info, g_Devices[i].handle);
            }
        }
        else if (pHolder == &g_FirmwareUpdateHolder)
        {
            nn::os::ClearSystemEvent(&g_FirmwareUpdateEvent);

            nn::xcd::FirmwareUpdateStage updateStage;
            int updateProgress;
            if (nn::xcd::GetBtFirmwareUpdateProgress(g_Devices[g_FirmwareUpdatingDeviceIndex].handle, &updateStage, &updateProgress).IsSuccess())
            {
                if (updateStage == nn::xcd::FirmwareUpdateStage_Completed)
                {
                    NN_LOG("Start Rebooting Device\n");
                    nn::xcd::Reboot(g_Devices[g_FirmwareUpdatingDeviceIndex].handle, true);
                }
            }

            CloseOtaFile();
        }
        else if (pHolder == &g_TimerHolder)
        {
            nn::os::ClearTimerEvent(&g_TimerEvent);

            if (g_Devices[g_FirmwareUpdatingDeviceIndex].connected == true)
            {
                nn::xcd::FirmwareUpdateStage updateStage;
                int updateProgress;
                if (nn::xcd::GetBtFirmwareUpdateProgress(g_Devices[g_FirmwareUpdatingDeviceIndex].handle, &updateStage, &updateProgress).IsSuccess())
                {
                    PrintOtaStatus(updateStage, updateProgress);
                }
            }
        }
        else
        {
            for (int i = 0; i < nn::xcd::DeviceCountMax; i++)
            {
                auto& device = g_Devices[i];
                if(pHolder == &device.holder && device.connected == true)
                {
                    nn::os::ClearSystemEvent(&device.event);

                    // 入力状態の取得
                    nn::xcd::GetPadState(&device.pad, device.handle);
                    nn::xcd::GetDeviceStatus(&device.status, device.handle);

                    // 入力がある場合は状態をプリント
                    if (device.pad.buttons.Test<nn::xcd::PadButton::SR>() == false && device.previousButton.Test<nn::xcd::PadButton::SR>())
                    {
                        OpenOtaFile();
                        if (nn::xcd::StartBtFirmwareUpdate(
                            g_Devices[g_FirmwareUpdatingDeviceIndex].handle,
                            g_File,
                            &g_FirmwareUpdateEvent).IsSuccess())
                        {
                            g_FirmwareUpdatingDeviceIndex = i;
                            CloseOtaFile();
                        }
                    }

                    device.previousButton = device.pad.buttons;

                }
            }
        }
    }

    // ファイルシステムをアンマウント
    nn::fs::Unmount("rom");
} // NOLINT(impl/function_size)
