﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Async.h"

#include <nn/os/os_ThreadApi.h>
#include <nn/socket/socket_Api.h>
#include <map>

#ifdef __INTELLISENSE__  // IntelliSense が #include_next を理解しないので仕方なく
#include <sys/poll.h>
#endif


namespace
{

struct AsyncRecvEntry
{
    int socket;
    void* userptr;
    void(*callback)(int, int, void*);
};


NN_OS_ALIGNAS_THREAD_STACK uint8_t g_PollThreadStack[32768];
nn::os::ThreadType g_PollThread;

int g_MessageSocket = -1;
sockaddr_in MassageSocketAddr = {
    sizeof(sockaddr_in),                        // sin_len
    AF_INET,                                    // sin_family
    nn::socket::InetHtons(26543),               // sin_port
    { nn::socket::InetHtonl(INADDR_LOOPBACK) }  // sin_addr.s_addr
};

std::map<int, AsyncRecvEntry> g_Waits;
nn::os::Mutex g_WaitsMutex(false);


// 非同期 Recv 管理スレッド
void AsyncRecvThread(void*)
{
    pollfd fds[32] = {};

    for (;;)
    {
        fds[0].fd = g_MessageSocket;
        fds[0].events = POLLIN;

        int i = 1;
        for (auto& pair : g_Waits)
        {
            if (i > sizeof(fds) / sizeof(*fds))
            {
                break;
            }

            fds[i].fd = pair.second.socket;
            fds[i].events = POLLIN;
        }

        auto numSignaled = nn::socket::Poll(fds, g_Waits.size() + 1, INFTIM);

        if (fds[0].revents & POLLNVAL)
        {
            break;
        }
        else if (fds[0].revents & POLLIN)
        {
            char buf[1];
            nn::socket::Recv(g_MessageSocket, buf, sizeof(buf), 0);
            if (!buf[0])
            {
                break;
            }

            --numSignaled;
        }

        for (int i = 1; i < sizeof(fds) / sizeof(*fds) && numSignaled; ++i)
        {
            if (!fds[i].revents)
            {
                continue;
            }

            auto iter = g_Waits.find(fds[i].fd);
            g_Waits.erase(fds[i].fd);

            if (iter == g_Waits.end())
            {
                continue;
            }

            iter->second.callback(fds[i].fd, fds[i].revents, iter->second.userptr);
            --numSignaled;
        }
    }
}

}


namespace Async
{

void Initialize()
{
    g_MessageSocket = nn::socket::Socket(AF_INET, SOCK_DGRAM, 0);

    nn::socket::Bind(g_MessageSocket, reinterpret_cast<sockaddr*>(&MassageSocketAddr), sizeof(MassageSocketAddr));

    nn::os::CreateThread(&g_PollThread, AsyncRecvThread, nullptr, g_PollThreadStack, sizeof(g_PollThreadStack), nn::os::DefaultThreadPriority);
    nn::os::StartThread(&g_PollThread);
}


void Finalize()
{
    char buf[1] = { 0 };
    nn::socket::SendTo(g_MessageSocket, buf, sizeof(buf), 0, reinterpret_cast<sockaddr*>(&MassageSocketAddr), sizeof(MassageSocketAddr));

    nn::os::DestroyThread(&g_PollThread);
    nn::socket::Close(g_MessageSocket);
}


void WaitRecv(int socket, void(*callback)(int, int, void*), void* userptr)
{
    g_WaitsMutex.Lock();

    g_Waits[socket] = { socket, userptr, callback };

    char buf[1] = { 1 };
    nn::socket::SendTo(g_MessageSocket, buf, sizeof(buf), 0, reinterpret_cast<sockaddr*>(&MassageSocketAddr), sizeof(MassageSocketAddr));

    g_WaitsMutex.Unlock();
}

}
