﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/



#include "TcpClass.h"

namespace {
    size_t g_logSize = 0;
    std::unique_ptr<char[]> g_logBuffer;
    HANDLE g_logMutex = INVALID_HANDLE_VALUE;
    FILE* g_LogFile = nullptr;
}

bool TcpSjisToUtf8(char strSjis[], int len) NN_NOEXCEPT
{
    std::unique_ptr<wchar_t[]> bufUnicode(new wchar_t[len]);
    std::unique_ptr<char[]> bufUtf8(new char[len]);
    int calcLen     = 0;
    int iLenUnicode = 0;
    bool isResult   = false;

    if (bufUnicode == nullptr || bufUtf8 == nullptr)
    {
        return false;
    }

    // SJISからUniocde変換サイズ算出
    calcLen = static_cast<int>(sizeof(wchar_t) * len);
    iLenUnicode = MultiByteToWideChar(CP_ACP, 0, strSjis, nn::util::Strnlen(strSjis, len) + 1,
                     nullptr, 0);
    if (iLenUnicode <= calcLen)
    {
        // SJISからUniocde変換
        MultiByteToWideChar(CP_ACP, 0, strSjis, nn::util::Strnlen(strSjis, len) + 1,
            bufUnicode.get(), len);

        // UniocdeからUTF8変換サイズ算出
        calcLen = static_cast<int>(sizeof(char) * len);
        int iLenUtf8 = WideCharToMultiByte(CP_UTF8, 0, bufUnicode.get(), iLenUnicode,
            nullptr, 0, nullptr, nullptr);
        if (iLenUtf8 <= calcLen)
        {
            // UniocdeからUTF8変換
            WideCharToMultiByte(CP_UTF8, 0, bufUnicode.get(), iLenUnicode, bufUtf8.get(),
                sizeof(char) * len,
                nullptr, nullptr);

            nn::util::Strlcpy(strSjis, bufUtf8.get(), len);
            isResult = true;
        }
    }

    return isResult;
}


bool TcpLogInitialize(size_t logSize) NN_NOEXCEPT
{
    g_logBuffer.reset(new char[logSize]);
    if (g_logBuffer == nullptr)
    {
        return false;
    }

    g_logSize = logSize;

    CTcpAutoLock::Initialize(g_logMutex);

    return true;
}

void TcpTraceLogFileOpen(LPCSTR lpPath) NN_NOEXCEPT
{
    CTcpAutoLock lock(g_logMutex);
    char drv[_MAX_DRIVE + 1];
    char dir[MAX_PATH + 1];
    char fname[_MAX_FNAME + 1];
    char ext[_MAX_EXT + 1];
    char path[MAX_PATH + 1];
    char setFileName[_MAX_FNAME + 1];
    uint32_t fcnt = 0;

    if (nn::util::Strncmp(lpPath, "", 1) == 0)
    {
        return;
    }

    // ログパスを分割
    _splitpath(lpPath, drv, dir, fname, ext);
    if (nn::util::Strncmp(drv, "", 1) != 0 &&
        nn::util::Strncmp(dir, "", 1) != 0)
    {
        // ログディレクトリが正しいか判定
        _makepath(path, drv, dir, nullptr, nullptr);
        if (GetFileAttributesA(path) == 0xFFFFFFFF)
        {
            TcpTraceLog("TCP LogFolder Failler : %s\n", path);
            return;
        }
    }

    do
    {
        // ログファイル名を初回以外は番号を付与しインクリメント。新規作成ファイルになるまでループ
        if (fcnt > 0)
        {
            nn::util::SNPrintf(setFileName, sizeof(setFileName), "%s_%04d", fname, fcnt);
        }
        else
        {
            nn::util::SNPrintf(setFileName, sizeof(setFileName), "%s", fname);
        }
        fcnt++;

        // ログパス結合生成
        _makepath(path, drv, dir, setFileName, ext);
    } while (GetFileAttributesA(path) != 0xFFFFFFFF);

    g_LogFile = _fsopen(path, "at", _SH_DENYWR);
}

void TcpLogFinalize() NN_NOEXCEPT
{
    CTcpAutoLock::Finalize(g_logMutex);

    g_logBuffer.reset();
}

void TcpTraceLog(LPCSTR lpFormat, ...) NN_NOEXCEPT
{
    va_list vlist;
    SYSTEMTIME st;
    CTcpAutoLock lock(g_logMutex);

    if (g_logMutex == INVALID_HANDLE_VALUE)
    {
        return;
    }

    GetSystemTime(&st);
    va_start(vlist, lpFormat);

    vsprintf_s(g_logBuffer.get(), g_logSize, lpFormat, vlist);

    NN_LOG("[%04d/%02d/%02d %02d:%02d:%02d.%03d]: %s\n", st.wYear, st.wMonth, st.wDay,
        st.wHour + 9, st.wMinute, st.wSecond, st.wMilliseconds, g_logBuffer.get());

    if (g_LogFile != nullptr)
    {
        fprintf_s(g_LogFile, "[%04d/%02d/%02d %02d:%02d:%02d.%03d]: %s\n", st.wYear, st.wMonth,
            st.wDay, st.wHour + 9, st.wMinute, st.wSecond, st.wMilliseconds, g_logBuffer.get());
    }

    va_end(vlist);
}

void TcpTraceLogFileClose() NN_NOEXCEPT
{
    CTcpAutoLock lock(g_logMutex);

    if (g_LogFile == nullptr)
    {
        return;
    }

    if (g_LogFile != NULL) {
        fclose(g_LogFile);
        g_LogFile = nullptr;
    }
}

CTcpAutoLock::CTcpAutoLock(HANDLE hMutex) NN_NOEXCEPT
{
    if (hMutex == INVALID_HANDLE_VALUE) return;

    m_hMutex = hMutex;
    WaitForSingleObject(m_hMutex, INFINITE);
};

CTcpAutoLock::~CTcpAutoLock() NN_NOEXCEPT
{
    if (m_hMutex == INVALID_HANDLE_VALUE) return;

    ReleaseMutex(m_hMutex);
};

bool CTcpAutoLock::Initialize(HANDLE& hMutex) NN_NOEXCEPT
{
    if (hMutex != INVALID_HANDLE_VALUE)
    {
        return false;
    }

    hMutex = CreateMutex(nullptr, FALSE, nullptr);
    return (hMutex != INVALID_HANDLE_VALUE);
}

void CTcpAutoLock::Finalize(HANDLE& hMutex) NN_NOEXCEPT
{
    BOOL isResult = FALSE;
    if (hMutex == INVALID_HANDLE_VALUE)
    {
        return;
    }

    isResult = CloseHandle(hMutex);
    if (isResult == TRUE)
    {
        hMutex = INVALID_HANDLE_VALUE;
    }
}

CTcpClass::CTcpClass() NN_NOEXCEPT
{
    WSAStartup(MAKEWORD(2, 0), &m_wsaData);

    InitializeCriticalSection(&m_crtPool);
    InitializeCriticalSection(&m_crtTcp);

    m_isStart                   = false;
    m_pRecvBuffer               = static_cast<char*>(malloc(TcpBufferSize));
    m_recvPoolMng.bufferSize    = TcpBufferSize;
    m_tcpInfo.srvSock           = static_cast<SOCK>(-1);
    m_tcpInfo.port              = 0;

    m_logoutTime                = 30000;
    m_winSize                   = 0;

    for (int i = 0; i < TcpServerConnectMax; i++)
    {
        m_tcpInfo.clientInfo[i].isStopThread  = false;
        m_tcpInfo.clientInfo[i].isSetAccept   = false;
        m_tcpInfo.clientInfo[i].hStartEvent   = CreateEvent(nullptr, TRUE, FALSE, nullptr);
        m_tcpInfo.clientInfo[i].hServerThread = INVALID_HANDLE_VALUE;
        m_tcpInfo.clientInfo[i].hClientThread = INVALID_HANDLE_VALUE;
    }
}

CTcpClass::~CTcpClass() NN_NOEXCEPT
{
    TearDownTcpServer();
    TearDownTcpClient();

    for (int acnt = 0; acnt < TcpServerConnectMax; acnt++)
    {
        if (m_tcpInfo.clientInfo[acnt].hServerThread != INVALID_HANDLE_VALUE)
        {
            CloseHandle(m_tcpInfo.clientInfo[acnt].hStartEvent);
            m_tcpInfo.clientInfo[acnt].hStartEvent = INVALID_HANDLE_VALUE;
        }
    }

    free(m_pRecvBuffer);

    DeleteCriticalSection(&m_crtPool);
    DeleteCriticalSection(&m_crtTcp);

    WSACleanup();
}

DWORD CTcpClass::SetupTcpServer(char* pIpAddress, USHORT port, size_t sendSize,
    DWORD sendInterval) NN_NOEXCEPT
{
    if (m_isStart == true)
    {
        return static_cast<DWORD>(-1);
    }

    const char DefaultIpAddress[] = "000.000.000.000";
    TearDownTcpServer();
    int result = 0;

    SetPoolMngSetting(&m_recvPoolMng, TcpRecvSize);

    m_tcpInfo.srvSock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (m_tcpInfo.srvSock == -1) {
        return GetLastError();
    }

    std::memset(&m_tcpInfo.srvAddr, 0x00, sizeof(struct sockaddr_in));
    m_tcpInfo.srvAddr.sin_family            = AF_INET;
    m_tcpInfo.srvAddr.sin_port              = htons(port);

    if (nn::util::Strncmp(pIpAddress, DefaultIpAddress, sizeof(DefaultIpAddress)) == 0)
    {
        m_tcpInfo.srvAddr.sin_addr.S_un.S_addr = INADDR_ANY;
    }
    else
    {
        InetPtonA(AF_INET, pIpAddress, &m_tcpInfo.srvAddr.sin_addr);
    }

    if (m_winSize > 0)
    {
        result = setsockopt(m_tcpInfo.srvSock, SOL_SOCKET, SO_SNDBUF,
            reinterpret_cast<char*>(&m_winSize), sizeof(m_winSize));
        if (result != 0)
        {
            m_tcpInfo.srvSock = closesocket(m_tcpInfo.srvSock);
            return GetLastError();
        }

        result = setsockopt(m_tcpInfo.srvSock, SOL_SOCKET, SO_RCVBUF,
            reinterpret_cast<char*>(&m_winSize), sizeof(m_winSize));
        if (result != 0)
        {
            m_tcpInfo.srvSock = closesocket(m_tcpInfo.srvSock);
            return GetLastError();
        }
    }

    result = bind(m_tcpInfo.srvSock, reinterpret_cast<struct sockaddr *>(&m_tcpInfo.srvAddr),
        sizeof(struct sockaddr_in));
    if (result != 0)
    {
        m_tcpInfo.srvSock = closesocket(m_tcpInfo.srvSock);
        return GetLastError();
    }

    result = listen(m_tcpInfo.srvSock, TcpServerConnectMax);
    if (result != 0)
    {
        m_tcpInfo.srvSock = closesocket(m_tcpInfo.srvSock);
        return GetLastError();
    }

    TcpTraceLog("TCP Server Listen\n");

    CreateServerThread();

    m_isStart = true;
    while (m_isStart)
    {
        int pos = -1;
        for (int acnt = 0; acnt < TcpServerConnectMax; acnt++)
        {
            if (m_tcpInfo.clientInfo[acnt].isSetAccept != true)
            {
                pos = acnt;
                break;
            }
        }

        if (pos != -1)
        {
            int nsaddrLen = 0;

            nsaddrLen = sizeof(struct sockaddr_in);
            std::memset(&m_tcpInfo.clientInfo[pos].acceptAddr, 0x00, nsaddrLen);

            m_tcpInfo.clientInfo[pos].acceptSock = accept(m_tcpInfo.srvSock,
                reinterpret_cast<struct sockaddr *>(&m_tcpInfo.clientInfo[pos].acceptAddr),
                &nsaddrLen);
            if (m_tcpInfo.clientInfo[pos].acceptSock == -1)
            {
                continue;
            }

            std::memset(&m_tcpInfo.clientInfo[pos].ipAddress, 0x00, INET_ADDRSTRLEN);

            inet_ntop(AF_INET, &m_tcpInfo.clientInfo[pos].acceptAddr.sin_addr,
                m_tcpInfo.clientInfo[pos].ipAddress, INET_ADDRSTRLEN);

            m_tcpInfo.clientInfo[pos].sendSize      = sendSize;
            m_tcpInfo.clientInfo[pos].sendInterval  = sendInterval;
            m_tcpInfo.clientInfo[pos].isSetAccept   = true;

            TcpTraceLog("TCP Server Client Connect : %s(%d)\n",
                m_tcpInfo.clientInfo[pos].ipAddress,
                ntohs(m_tcpInfo.clientInfo[pos].acceptAddr.sin_port));
            SetEvent(m_tcpInfo.clientInfo[pos].hStartEvent);
        }
        else
        {
            Sleep(1000);
        }
    }

    return 0;
}

void CTcpClass::TearDownTcpServer() NN_NOEXCEPT
{
    if (m_isStart != true)
    {
        return;
    }

    m_isStart = false;

    ReleaseServerThread();
    if (m_tcpInfo.srvSock != -1)
    {
        closesocket(m_tcpInfo.srvSock);
        m_tcpInfo.srvSock = static_cast<SOCK>(-1);
    }
}

DWORD CTcpClass::SetupTcpClient(char* pIpAddress, USHORT port, size_t sendSize,
    DWORD sendInterval) NN_NOEXCEPT
{
    if (m_isStart == true)
    {
        return static_cast<DWORD>(-1);
    }

    TearDownTcpClient();

    int result;

    SetPoolMngSetting(&m_recvPoolMng, TcpRecvSize);

    std::memset(&m_tcpInfo.srvAddr, 0x00, sizeof(struct sockaddr_in));

    m_tcpInfo.srvSock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (m_tcpInfo.srvSock == -1) {
        return GetLastError();
    }

    bool bBroad = true;
    setsockopt(m_tcpInfo.srvSock,
        SOL_SOCKET, SO_BROADCAST, (char *)&bBroad, sizeof(bBroad));

    if (m_winSize > 0)
    {
        result = setsockopt(m_tcpInfo.srvSock, SOL_SOCKET, SO_SNDBUF,
            (char*)&m_winSize, sizeof(m_winSize));
        if (result != 0)
        {
            m_tcpInfo.srvSock = closesocket(m_tcpInfo.srvSock);
            return GetLastError();
        }

        result = setsockopt(m_tcpInfo.srvSock, SOL_SOCKET, SO_RCVBUF, (char*)&m_winSize,
            sizeof(m_winSize));
        if (result != 0)
        {
            m_tcpInfo.srvSock = closesocket(m_tcpInfo.srvSock);
            return GetLastError();
        }
    }

    m_tcpInfo.srvAddr.sin_family = AF_INET;
    m_tcpInfo.srvAddr.sin_port = htons(port);
    m_tcpInfo.srvAddr.sin_addr.s_addr = htonl(INADDR_ANY);
    InetPtonA(AF_INET, pIpAddress, (PVOID)&m_tcpInfo.srvAddr.sin_addr.S_un.S_addr);

    result = connect(m_tcpInfo.srvSock, reinterpret_cast<struct sockaddr *>(&m_tcpInfo.srvAddr),
        sizeof(m_tcpInfo.srvAddr));
    if (result != 0)
    {
        closesocket(m_tcpInfo.srvSock);
        m_tcpInfo.srvSock = static_cast<SOCK>(-1);;
        return GetLastError();
    }

    TcpTraceLog("TCP Client Connect\n");

    CreateClientThread();

    m_isStart = true;
    while (m_isStart)
    {
        int pos = -1;
        for (int acnt = 0; acnt < TcpServerConnectMax; acnt++)
        {
            if (m_tcpInfo.clientInfo[acnt].isSetAccept != true)
            {
                pos = acnt;
                break;
            }
        }

        if (pos != -1)
        {
            int sockaddLen = sizeof(struct sockaddr_in);

            memcpy_s(&m_tcpInfo.clientInfo[pos].acceptAddr, sockaddLen,
                &m_tcpInfo.srvAddr, sockaddLen);

            std::memset(&m_tcpInfo.clientInfo[pos].ipAddress, 0x00, INET_ADDRSTRLEN);
            inet_ntop(AF_INET, &m_tcpInfo.srvAddr.sin_addr,
                m_tcpInfo.clientInfo[pos].ipAddress, INET_ADDRSTRLEN);

            m_tcpInfo.clientInfo[pos].acceptSock    = m_tcpInfo.srvSock;
            m_tcpInfo.clientInfo[pos].sendSize      = sendSize;
            m_tcpInfo.clientInfo[pos].sendInterval  = sendInterval;
            m_tcpInfo.clientInfo[pos].isSetAccept   = true;
            m_tcpInfo.srvSock                       = static_cast<SOCK>(-1);;

            SetEvent(m_tcpInfo.clientInfo[pos].hStartEvent);
        }
        else
        {
            Sleep(1000);
        }
    }

    return 0;
}


void CTcpClass::TearDownTcpClient() NN_NOEXCEPT
{
    if (m_isStart != true)
    {
        return;
    }

    m_isStart = false;
    ReleaseClientThread();
}


void CTcpClass::CreateServerThread() NN_NOEXCEPT
{
    for (int acnt = 0; acnt < TcpServerConnectMax; acnt++)
    {
        if (m_tcpInfo.clientInfo[acnt].isSetAccept != true)
        {
            m_tcpInfo.clientInfo[acnt].isStopThread     = false;
            m_tcpInfo.clientInfo[acnt].isSetAccept      = false;
            m_tcpInfo.clientInfo[acnt].pThis            = this;
            m_tcpInfo.clientInfo[acnt].hServerThread    = CreateThread(nullptr,
                TcpAcceptStackSize, reinterpret_cast<LPTHREAD_START_ROUTINE>(TcpServerThread),
                &m_tcpInfo.clientInfo[acnt], 0, nullptr);
        }
    }
}

void CTcpClass::CreateClientThread() NN_NOEXCEPT
{
    EnterCriticalSection(&m_crtTcp);

    for (int acnt = 0; acnt < TcpServerConnectMax; acnt++)
    {
        if (m_tcpInfo.clientInfo[acnt].isSetAccept != true)
        {
            m_tcpInfo.clientInfo[acnt].isStopThread = false;
            m_tcpInfo.clientInfo[acnt].isSetAccept  = false;
            m_tcpInfo.clientInfo[acnt].pThis        = this;
            ResetEvent(m_tcpInfo.clientInfo[acnt].hStartEvent);
            m_tcpInfo.clientInfo[acnt].hClientThread = CreateThread(nullptr,
                    TcpAcceptStackSize, reinterpret_cast<LPTHREAD_START_ROUTINE>(TcpClientThread),
                    &m_tcpInfo.clientInfo[acnt], 0, nullptr);
            break;
        }
    }

    LeaveCriticalSection(&m_crtTcp);
}

void CTcpClass::ReleaseServerThread() NN_NOEXCEPT
{
    EnterCriticalSection(&m_crtTcp);
    for (int acnt = 0; acnt < TcpServerConnectMax; acnt++)
    {
        m_tcpInfo.clientInfo[acnt].isStopThread = true;

        if (m_tcpInfo.clientInfo[acnt].isSetAccept == true)
        {
            m_tcpInfo.clientInfo[acnt].isSetAccept = false;
        }

        if (m_tcpInfo.clientInfo[acnt].acceptSock != -1)
        {
            closesocket(m_tcpInfo.clientInfo[acnt].acceptSock);
            m_tcpInfo.clientInfo[acnt].acceptSock = static_cast<SOCK>(-1);;
        }

        if (m_tcpInfo.clientInfo[acnt].hServerThread != INVALID_HANDLE_VALUE)
        {
            SetEvent(m_tcpInfo.clientInfo[acnt].hStartEvent);
            WaitForSingleObject(m_tcpInfo.clientInfo[acnt].hServerThread, INFINITE);
            CloseHandle(m_tcpInfo.clientInfo[acnt].hServerThread);
            m_tcpInfo.clientInfo[acnt].hServerThread = INVALID_HANDLE_VALUE;
        }
    }
    LeaveCriticalSection(&m_crtTcp);
}

void CTcpClass::ReleaseClientThread() NN_NOEXCEPT
{
    for (int acnt = 0; acnt < TcpServerConnectMax; acnt++)
    {
        if (m_tcpInfo.clientInfo[acnt].acceptSock != -1)
        {
            closesocket(m_tcpInfo.clientInfo[acnt].acceptSock);
            m_tcpInfo.clientInfo[acnt].acceptSock = static_cast<SOCK>(-1);;
        }

        if (m_tcpInfo.clientInfo[acnt].isSetAccept == true)
        {
            m_tcpInfo.clientInfo[acnt].isStopThread = true;
            m_tcpInfo.clientInfo[acnt].isSetAccept = false;
            SetEvent(m_tcpInfo.clientInfo[acnt].hStartEvent);
        }

        if (m_tcpInfo.clientInfo[acnt].hClientThread != INVALID_HANDLE_VALUE)
        {
            WaitForSingleObject(m_tcpInfo.clientInfo[acnt].hClientThread, INFINITE);
            CloseHandle(m_tcpInfo.clientInfo[acnt].hClientThread);
            m_tcpInfo.clientInfo[acnt].hClientThread = INVALID_HANDLE_VALUE;
        }
    }
}

void CTcpClass::TcpServerThread(LPVOID pArg) NN_NOEXCEPT
{
    TcpDatail* pClient              = static_cast<TcpDatail*>(pArg);
    CTcpClass* pThis                = static_cast<CTcpClass*>(pClient->pThis);
    TcpPoolMngDetail* pPoolDatail   = nullptr;
    size_t recvSize                 = 0;
    DWORD checkTime                 = 0;
    DWORD logTime                   = 0;
    int rd_bytes;

    TcpTraceLog("Tcp Recv Thread Start");
    while (pClient->isStopThread != true)
    {
        WaitForSingleObject(pClient->hStartEvent, INFINITE);
        ResetEvent(pClient->hStartEvent);

        if (pClient->isSetAccept == true)
        {
            checkTime = GetTickCount();
            while (NN_STATIC_CONDITION(1))
            {
                pPoolDatail = pThis->GetNextWritePool(&pThis->m_recvPoolMng);
                rd_bytes = recv(pClient->acceptSock, pPoolDatail->pData,
                    (int)pPoolDatail->bufferSize, 0);
                if (rd_bytes <= 0) {
                    closesocket(pClient->acceptSock);
                    pClient->acceptSock = static_cast<SOCK>(-1);;
                    pClient->isSetAccept = false;
                    break;
                }
                recvSize += rd_bytes;

                if (pThis->GetCalcElapsedTick(logTime) > pThis->m_logoutTime)
                {
                    TcpTraceLog("IP Address  : %s (%d)", pClient->ipAddress,
                        ntohs(pClient->acceptAddr.sin_port));
                    TcpTraceLog("Recv Size   : %.1lf %s (%zu Byte Time %lu msec)",
                        GetByteCalc(recvSize), GetUnitByteCalc(recvSize).c_str(), recvSize,
                        pThis->GetCalcElapsedTick(checkTime));

                    double time = 0;
                    double rate = 0;
                    if (pThis->GetCalcElapsedTick(checkTime) > 1000)
                    {
                        time = (pThis->GetCalcElapsedTick(checkTime) / 1000);
                        rate = ((static_cast<double>(recvSize) / time) * 8.0) / (1024 * 1024);
                    }
                    TcpTraceLog("Recv Rate   : %f Mbps (Time %lu msec)\n", rate,
                        pThis->GetCalcElapsedTick(checkTime));
                    logTime = GetTickCount();
                }
            }
            break;
        }
    }

    TcpTraceLog("Tcp Recv Thread End");
}

void CTcpClass::TcpClientThread(LPVOID pArg) NN_NOEXCEPT
{
    TcpDatail* pClient              = static_cast<TcpDatail*>(pArg);
    CTcpClass* pThis                = static_cast<CTcpClass*>(pClient->pThis);
    size_t sendSize                 = pClient->sendSize;
    DWORD checkTime                 = 0;
    DWORD sendTime                  = 0;
    DWORD logTime                   = 0;
    int sd_bytes;

    TcpTraceLog("Client Send Thread Start");
    while (NN_STATIC_CONDITION(1))
    {
        WaitForSingleObject(pClient->hStartEvent, INFINITE);
        ResetEvent(pClient->hStartEvent);

        if (pClient->isStopThread == true)
        {
            break;
        }

        if (pClient->isSetAccept == true)
        {
            int bufferSize = static_cast<int>(pClient->sendSize);
            std::unique_ptr<char[]> sendBuffer(new char[bufferSize]);
            sendSize = 0;
            checkTime = GetTickCount();
            sendTime = GetTickCount();
            while (NN_STATIC_CONDITION(1))
            {
                if (pThis->GetCalcElapsedTick(sendTime) > pClient->sendInterval)
                {
                    sendTime = GetTickCount();
                    sd_bytes = send(pClient->acceptSock, sendBuffer.get(), bufferSize, 0);
                    if (sd_bytes < 0) {
                        break;
                    }
                    sendSize += sd_bytes;
                }

                if (pThis->GetCalcElapsedTick(logTime) > pThis->m_logoutTime)
                {
                    TcpTraceLog("IP Address  : %s (%u)", pClient->ipAddress,
                        ntohs(pClient->acceptAddr.sin_port));
                    TcpTraceLog("Send Size   : %.1lf %s (%zu Byte Time %lu msec)",
                        GetByteCalc(sendSize), GetUnitByteCalc(sendSize).c_str(), sendSize,
                        pThis->GetCalcElapsedTick(checkTime));

                    double time = 0;
                    double rate = 0;
                    if (pThis->GetCalcElapsedTick(checkTime) > pThis->m_logoutTime)
                    {
                        time = (pThis->GetCalcElapsedTick(checkTime) / 1000);
                        rate = ((static_cast<double>(sendSize) / time) * 8.0) / (1024 * 1024);
                    }

                    TcpTraceLog("Send Rate   : %f Mbps (Time %lu msec)\n", rate,
                        pThis->GetCalcElapsedTick(checkTime));
                    logTime = GetTickCount();
                }
                if (pClient->sendInterval > 0)
                {
                    Sleep(2);
                }
            }
            break;
        }
    }

    TcpTraceLog("Client Send Thread End");

    pThis->TcpStop(pClient);
}

bool CTcpClass::SetPoolMngSetting(TcpPoolMng* pPoolMng, const size_t packetSize) NN_NOEXCEPT
{
    if (packetSize > pPoolMng->bufferSize)
    {
        return false;
    }

    EnterCriticalSection(&m_crtPool);

    pPoolMng->maxCount  = 0;
    pPoolMng->readIdx   = 0;
    pPoolMng->writeIdx  = 0;
    char* pBufferPos    = m_pRecvBuffer;
    size_t actSize      = TcpBufferSize;

    while (actSize > packetSize)
    {
        pPoolMng->info[pPoolMng->maxCount].pData = pBufferPos;
        pPoolMng->info[pPoolMng->maxCount].bufferSize = packetSize;
        pPoolMng->info[pPoolMng->maxCount].dataSize = 0;
        pPoolMng->maxCount++;
        actSize -= packetSize;
        pBufferPos += packetSize;

        if (pPoolMng->maxCount >= TcpBufferPoolCnt)
        {
            break;
        }
    }

    LeaveCriticalSection(&m_crtPool);

    return false;
}


TcpPoolMngDetail* CTcpClass::GetNextWritePool(TcpPoolMng* pPoolMng) NN_NOEXCEPT
{
    TcpPoolMngDetail* pData = &pPoolMng->info[pPoolMng->writeIdx];
    pData->dataSize = 0;
    ++pPoolMng->writeIdx %= pPoolMng->maxCount;

    return pData;
}

void CTcpClass::TcpStop(TcpDatail* pClient) NN_NOEXCEPT
{
    EnterCriticalSection(&m_crtTcp);
    if (pClient->isSetAccept == true)
    {
        pClient->isSetAccept = false;
        pClient->isStopThread = true;
        SetEvent(pClient->hStartEvent);
    }
    LeaveCriticalSection(&m_crtTcp);
}

void CTcpClass::LogOutIpAddress(USHORT port) NN_NOEXCEPT
{
    std::unique_ptr<BYTE[]> pAdapterAddressArray;
    PIP_ADAPTER_ADDRESSES pAdapterAddresses, pAdapterView;
    DWORD dwRet, dwSize;

    dwRet = GetAdaptersAddresses(AF_UNSPEC, GAA_FLAG_INCLUDE_PREFIX, nullptr, nullptr, &dwSize);
    if (dwRet != ERROR_BUFFER_OVERFLOW) {
        exit(1);
    }

    pAdapterAddressArray.reset(new BYTE[dwSize]);
    pAdapterAddresses = reinterpret_cast<PIP_ADAPTER_ADDRESSES>(pAdapterAddressArray.get());
    if (pAdapterAddresses == nullptr) {
        exit(1);
    }

    dwRet = GetAdaptersAddresses(AF_UNSPEC, GAA_FLAG_INCLUDE_PREFIX,
                                    nullptr, pAdapterAddresses, &dwSize);
    if (dwRet != ERROR_SUCCESS) {
        exit(1);
    }

    TcpTraceLog("**** Server Ether Adapter List ****");
    for (pAdapterView = pAdapterAddresses; pAdapterView; pAdapterView = pAdapterView->Next)
    {
        char szAdapterName[BUFSIZ];
        int len;

        len = WideCharToMultiByte(CP_ACP, 0,
            pAdapterView->FriendlyName, static_cast<int>(wcslen(pAdapterView->FriendlyName)),
            szAdapterName, sizeof(szAdapterName), nullptr, nullptr);
        if (len == 0) {
            exit(1);
        }

        szAdapterName[len] = '\0';
        if (TcpSjisToUtf8(szAdapterName, BUFSIZ) != true)
        {
            // UTF8に変換できないアダプタ名
            TcpTraceLog("    Adapter Name Change UTF8 String\n");
        }

        for (auto uni = pAdapterView->FirstUnicastAddress; uni; uni = uni->Next)
        {
            SOCKET_ADDRESS addr = uni->Address;
            char ipAddress[INET_ADDRSTRLEN];
            struct sockaddr_in *sa;

            if (!(uni->Flags & IP_ADAPTER_ADDRESS_DNS_ELIGIBLE)) {
                continue;
            }

            TcpTraceLog("    Adapter Name : %s", szAdapterName);
            TcpTraceLog("    IfIndex      : %u", pAdapterView->IfIndex);

            sa = reinterpret_cast<struct sockaddr_in*>(addr.lpSockaddr);
            InetNtopA(AF_INET, &sa->sin_addr, ipAddress, INET_ADDRSTRLEN);
            if (uni->Next != nullptr)
            {
                TcpTraceLog("    IP Address   : %s(%u)\n", ipAddress, port);
            }
            else
            {
                TcpTraceLog("    IP Address   : %s(%u)", ipAddress, port);
            }
        }
    }

    TcpTraceLog("**** Server Ether Adapter List ****\n");
}

DWORD CTcpClass::GetCalcElapsedTick(DWORD dwStartTick) NN_NOEXCEPT
{
    DWORD dwNowTick = GetTickCount();
    DWORD dwElapsedTick = 0;

    if (dwStartTick > dwNowTick)
    {
        dwElapsedTick = 0xFFFFFF - dwNowTick;
        dwElapsedTick += dwStartTick;
    }
    else
    {
        dwElapsedTick = dwNowTick - dwStartTick;
    }

    return dwElapsedTick;
}

void CTcpClass::SetWindowsSize(int winSize) NN_NOEXCEPT
{
    m_winSize = winSize;
}

void CTcpClass::SetLogoutTime(DWORD dwSetLogTime) NN_NOEXCEPT
{
    if (dwSetLogTime > 0)
    {
        m_logoutTime = dwSetLogTime;
    }
}

double CTcpClass::GetByteCalc(size_t size) NN_NOEXCEPT
{
    double dbRet = static_cast<double>(size);
    int32_t unitType = 0;

    while (dbRet > 1024)
    {
        dbRet /= 1024;
    }

    GetUnitByteCalc(size, &unitType);
    if (unitType == 0)
    {
        dbRet = static_cast<double>(size);
    }

    return dbRet;
}

std::string CTcpClass::GetUnitByteCalc(size_t size, int32_t* pOutType) NN_NOEXCEPT
{
    double dbRet = static_cast<double>(size);
    int32_t unitType = 0;
    std::string lpUnitStr = "";

    while (dbRet > 1024)
    {
        dbRet /= 1024;
        unitType++;
    }

    switch (unitType)
    {
    case 1:
        lpUnitStr = "KByte\0";
        break;
    case 2:
        lpUnitStr = "MByte";
        break;
    case 3:
        lpUnitStr = "GByte";
        break;
    case 4:
        lpUnitStr = "TByte";
        break;
    default:
        lpUnitStr = "Byte\0";
        unitType = 0;
        break;
    }

    if (pOutType != nullptr)
    {
        *pOutType = unitType;
    }

    return lpUnitStr;
}
