﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Include/testWlan_LocalInfraStressTest.h"

namespace {
    NN_ALIGNAS(4096) uint8_t SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

    const uint16_t EtherTypes[1] = { 0x88b7 };

    const size_t threadStackSize = nn::os::ThreadStackAlignment * 4;
};

namespace WlanTest {
    class LocalMasterStressTest : public ::testing::Test
    {
    protected:

        LocalApiClass localMaster;

        std::unique_ptr<uint8_t[]> getFrameBuffer;

        nn::wlan::MacAddress macAddress;
        nn::wlan::MacAddress connectMacArray[ClientMaxConnect];

        uint32_t rxId;
        uint32_t actionRxId;
        bool isCreBss;

        nn::os::ThreadType actionframeThread;
        nn::os::EventType actionframeEvent;

        nn::os::SystemEventType connectionEvent;

        nn::os::Tick sysTick;
        nn::os::Tick logOutTime;

        LocalConnectsInfo connectsInfo;

        int32_t sockOpen;

        nn::os::ThreadType scanThread;
        nn::wlan::ScanParameters unsyncScanParam;
        uint64_t intervalScanTime;
        bool isStartScan;
        const bool isEnabled = true;

        nn::wlan::MacAddress testPutMac;
        int16_t testChannel;

        int64_t agingTime;
        int32_t waitClientCount;

    protected:

        LocalMasterStressTest() NN_NOEXCEPT
        {
            rxId = static_cast<uint32_t>(-1);
            actionRxId = static_cast<uint32_t>(-1);
            isCreBss = false;

            isStartScan = false;
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_STATE_SET(WlanTestState_Local);

            agingTime       = GetAgintTime();
            waitClientCount = GetClientsCount();

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::settings::fwdbg::SetSettingsItemValue("nifm",
                    "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));
            nn::nifm::SetWirelessCommunicationEnabledForTest(false);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::socket::Initialize(reinterpret_cast<void*>(SocketMemoryPoolBuffer),
                nn::socket::DefaultSocketMemoryPoolSize,
                nn::socket::MinSocketAllocatorSize,
                kLocalConcurrencyLimit));

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());

            localMaster.InitializeBtm();

            // デフォルトのジョイコン設定は０台
            WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNodeNone));
#ifdef WLAN_TEST_STATE_ASSER_STOP
            localMaster.StateThreadStart(Time100msec, TraceState);
#else
            NN_UNUSED(Time100msec);
#endif
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenMasterMode());
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionEvent(&connectionEvent));
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            localMaster.LocalRelease();

            if (connectionEvent._state != nn::os::SystemEventType::State::State_NotInitialized)
            {
                nn::os::DestroySystemEvent(&connectionEvent);
            }

            if (rxId != static_cast<uint32_t>(-1))
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
            }

            if (actionRxId != static_cast<uint32_t>(-1))
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actionRxId));
            }

            if (isCreBss == true)
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DestroyBss());
                isCreBss = false;
            }

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CloseMasterMode());

#ifdef WLAN_TEST_STATE_ASSER_STOP
            localMaster.StateThreadStop();
#endif

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::socket::Finalize());
        }

        nn::Result SetupBss(const nn::wlan::MasterBssParameters & bssParamer) NN_NOEXCEPT
        {
            nn::Result result;

            nn::wlan::Local::GetMacAddress(&macAddress);
            result = nn::wlan::Local::CreateBss(bssParamer);
            if (result.IsSuccess() != true)
            {
                return result;
            }

            isCreBss = true;

            return nn::ResultSuccess();
        }

        nn::Result SetupBssAndEntry(const nn::wlan::MasterBssParameters & bssParamer,
                const nn::wlan::ReceivedDataMatchInfo matchInfo[], uint32_t matchCount) NN_NOEXCEPT
        {
            nn::Result result;

            result = SetupBss(bssParamer);
            if (result.IsSuccess() != true)
            {
                return result;
            }

            result = nn::wlan::Local::CreateRxEntry(&rxId, EtherTypes, sizeof(EtherTypes) / sizeof(EtherTypes), 30);
            if (result.IsSuccess() != true)
            {
                return result;
            }

            for (int32_t i = 0; i < matchCount; i++)
            {
                result = nn::wlan::Local::AddMatchingDataToRxEntry(rxId, matchInfo[i]);
                if (result.IsSuccess() != true)
                {
                    return result;
                }
            }

            return nn::ResultSuccess();
        }

        nn::Result SetupBssAndActionEntry(const nn::wlan::MasterBssParameters & bssParamer) NN_NOEXCEPT
        {
            nn::Result result;

            result = SetupBss(bssParamer);
            if (result.IsSuccess() != true)
            {
                return result;
            }

            result = nn::wlan::Local::CreateRxEntryForActionFrame(&actionRxId, EtherTypes,
                        sizeof(EtherTypes) / sizeof(EtherTypes), 10);
            if (result.IsSuccess() != true)
            {
                return result;
            }

            return nn::ResultSuccess();
        }

        void AllScan() NN_NOEXCEPT
        {
            std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
            NN_ABORT_UNLESS_NOT_NULL(pTestBuffer.get());
            nn::wlan::ScanIeMatchInfo info;

            const nn::wlan::ScanParameters StartScanTestParams = {
                nn::wlan::ScanType_Active,{ 1 }, 0, 40, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
            };

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(),
                    TestScanMaxSize, StartScanTestParams, &info));
        }

        int GetRandom(int min, int max) NN_NOEXCEPT
        {
            static time_t check;
            int rand_ret;
            if (check != time(nullptr))
            {
                check = time(nullptr);
                srand(static_cast<unsigned int>(time(nullptr)));
            }
            rand_ret = min + static_cast<int>(rand() * (max - min + 1.0) / (1.0 + RAND_MAX));
            return rand_ret;
        }

        bool CompareEtherType(uint16_t dscType, uint8_t typeArray[2])
        {
            uint16_t etherType = static_cast<uint16_t>((typeArray[0] << 8 & 0xFF00) | typeArray[1]);
            return (dscType == etherType);
        }

        bool CompareScanSsid(uint8_t pScanBuffer[], size_t bufferSize, nn::wlan::Ssid compareSsid) NN_NOEXCEPT
        {
            nn::wlan::BeaconScanResultReader resultReader(pScanBuffer);
            uint32_t bssCount = resultReader.GetCount();
            NN_UNUSED(bufferSize);

            for (uint32_t i = 0; i < bssCount; i++)
            {
                nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();
                if (compareSsid == beacon.GetSsid())
                {
                    return true;
                }
            }

            return false;
        }

        void StartUnsyncScanThread(nn::wlan::ScanParameters TestScanParam, uint64_t intervalTime) NN_NOEXCEPT
        {
            if (isStartScan == true)
            {
                return;
            }

            static NN_ALIGNAS(4096) uint8_t scanStack[threadStackSize];
            unsyncScanParam = TestScanParam;
            intervalScanTime = intervalTime;

            isStartScan = true;
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&scanThread, UnSyncScanThread, this,
                    scanStack, threadStackSize, nn::os::DefaultThreadPriority));
            nn::os::StartThread(&scanThread);
        }

        void StopUnsyncScanThread() NN_NOEXCEPT
        {
            if (isStartScan != true)
            {
                return;
            }

            isStartScan = false;

            nn::os::WaitThread(&scanThread);
            nn::os::DestroyThread(&scanThread);
        }

        static void SendActionFrameThread(void* arg) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);
            LocalMasterStressTest* pThis = static_cast<LocalMasterStressTest*>(arg);

            size_t putSize = ActionFrameBufferSize;
            std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
            nn::wlan::MacAddress fromMacAddress;

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&fromMacAddress));

            for (int i = 0; i < 1000; i++)
            {
                WLANTEST_EXPECT_TRUE(LocalApiClass::LocalMakeActionFrame(putBuffer.get(), putSize,
                        pThis->testPutMac, fromMacAddress, EtherTypes[0], TestMatchInfo[0],
                        nn::wlan::ActionFrameType_Local, i));
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutActionFrameOneShot(pThis->testPutMac,
                        putBuffer.get(), putSize, pThis->testChannel, 30));
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time300msec));
            }
        }

        bool SetActionFrameWithBeaconTestWait(uint32_t stopConnectCnt) NN_NOEXCEPT
        {
            size_t putSize = 1360;
            std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[putSize]);
            nn::wlan::MacAddress myMac;
            nn::wlan::MacAddress putMac = nn::wlan::MacAddress::CreateBroadcastMacAddress();

            if (pTestBuffer == nullptr)
            {
                return false;
            }

            if (localMaster.LocalMakeActionFrame(pTestBuffer.get(), putSize, putMac, localMaster.m_macAddress,
                EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Beacon) != true)
            {
                return false;
            }

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::SetActionFrameWithBeacon(pTestBuffer.get(), putSize));

            sysTick = nn::os::GetSystemTick();
            while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time3s)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time100msec));
                // CLIENTが切断状態の場合は常にTick更新。試験CLIENTの終了時に接続待機させループ終了タイムまでTickは更新しない
                if (localMaster.GetStateClientCount(nn::wlan::ConnectionState_Connected) < stopConnectCnt)
                {
                    sysTick = nn::os::GetSystemTick();
                }
            }

            return true;
        }

        static void TraceState(nn::wlan::WlanState oldState, nn::wlan::WlanState newState) NN_NOEXCEPT
        {
            NN_LOG("\n             **** WLAN STATE CHANGE EVENT *****\n");
            NN_LOG("             Previous State : %d\n", oldState);
            NN_LOG("             Current State  : %d\n", newState);
            NN_LOG("             **** WLAN STATE CHANGE EVENT *****\n\n");
        }

        static void TraceState(LocalApiClass* plocal) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(plocal);

            nn::wlan::WlanState state;
            nn::Result result;

#ifdef WLAN_TEST_STATE_ASSER_STOP
            result = plocal->GetState(&state);
#else
            NN_UNUSED(plocal);
            result = nn::wlan::Local::GetState(&state);
#endif
            if (result.IsSuccess() == true)
            {
                NN_LOG("\n             **** WLAN STATE *****\n");
                NN_LOG("             Current State  : %d\n", state);
                NN_LOG("             **** WLAN STATE *****\n\n");
            }
            else
            {
                NN_LOG("             GetState Result failed : 0x%08x\n", result.GetInnerValueForDebug());
            }
        }

        static void UnSyncScanThread(void* arg) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);
            LocalMasterStressTest* pThis = static_cast<LocalMasterStressTest*>(arg);

            std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
            NN_ABORT_UNLESS_NOT_NULL(pTestBuffer.get());
            nn::wlan::ScanIeMatchInfo info;

            NN_LOG("             Master UnSyncScanThread -> StartScan Start\n");
            while (pThis->isStartScan == true)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pThis->intervalScanTime));
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(),
                        TestScanMaxSize, pThis->unsyncScanParam, &info));
            }
            NN_LOG("             Master UnSyncScanThread -> StartScan End\n\n");
        }
    };

    // 同一Masterに1時間連続で接続と切断を繰り返し
    TEST_F(LocalMasterStressTest, L103)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            1, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo,
                sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(WlanTestConnectTimeOut, waitClientCount) > 0);

        sysTick = nn::os::GetSystemTick();
        while (NN_STATIC_CONDITION(1))
        {
            // CLIENTからの接続または、切断イベントが３０秒以内になければ終了
            if (nn::os::TimedWaitSystemEvent(&localMaster.m_connectEvent, nn::TimeSpan::FromMilliSeconds(Time30s)) != true)
            {
                break;
            }
        }
    }

    // Masterが指定台のClientと接続後、16msec毎に512Bytesの通信を行い、1秒間隔で1,6,11chの既接続スキャンを指定時間連続通信 MASTER
    // Masterが指定台のClientと接続後、16msec毎に512Bytesの通信を行い、1秒間隔で1,6,11chの既接続スキャンを指定時間連続通信とClientのリンク維持 MASTER
    TEST_F(LocalMasterStressTest, L109_L110)
    {
        // 通信評価定義
        const uint64_t frameByteNormal = 60 * WlanTestPacketBufferSize * (agingTime / 1000);
        const uint64_t frameByteMin = frameByteNormal - (frameByteNormal * 0.1);
        const uint64_t frameByteMax = frameByteNormal + (frameByteNormal * 0.1);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key },
            nn::wlan::Ssid(LocalMasterTest3SSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss,
                TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));
        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k,
                WlanTestPacketBufferSize, waitClientCount, waitClientCount, WlanTestAgingGetFrameCount));
        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut));
        // MASTER送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize,
                agingTime, Interval_16m, false));

        logOutTime = nn::os::GetSystemTick();
        // 全CLIENTが切断されるまでループに変更
        while (localMaster.LocalMasterDisConnectWait(Time1s) > 0)
        {
            if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time180s)
            {
                localMaster.LocalTraceConnectsInfo(true);
                logOutTime = nn::os::GetSystemTick();
            }
        }

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();
        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount >= waitClientCount);

        localMaster.LocalGetConnectInfo(&connectsInfo);

        // フレーム送信サイズ範囲評価（1.max(最小, 送信サイズ)、2.min(1.結果, 最大）⇒ 範囲内なら送信サイズが戻り値。以外は最大／最小が戻り値）
        NN_LOG("             L109_L110 Master Frame Send Total Size:%lld(Range:%lld - %lld)\n",
                connectsInfo.totalSendSize, frameByteMin, frameByteMax);
        WLANTEST_EXPECT_TRUE(std::min(std::max(frameByteMin, connectsInfo.totalSendSize),
                frameByteMax) == connectsInfo.totalSendSize);
    }

};
