﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/mem.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/eth/eth_EthClient.h>
#include <nn/socket.h>
#include <nn/socket/socket_TypesPrivate.h>
#include <nn/socket/socket_SystemConfig.h>

#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>

#include "wlan_MemoryInit.h"
#include "IfInit.h"
#include "DhdInit.h"

const int myPort = 23;
const int messageBufferSize = 64;

namespace
{
    const int TcpSocketLimit    = 16;
    const int UdpSocketLimit    = 0;
    const int ConcurrencyLimit  = 11;

    nn::socket::SystemConfigDefaultWithMemory
        <
        TcpSocketLimit, // Tcp Sockets
        UdpSocketLimit   // UDP Sockets
        > g_SocketConfigWithMemory(ConcurrencyLimit);
}

/* Wait DHCP reply and display my IP address */
void DisplayMyIp()
{
    int ipdisp_fg = 0;

    NN_LOG("DHCP reply waiting");

    while(ipdisp_fg == 0)
    {
        NN_LOG(".");
        /* Try to get IP */
        do
        {
            int mibsock = -1;

            struct ifreq req;
            nn::socket::SockAddrIn* pSa = (nn::socket::SockAddrIn *)(void *)&req.ifr_addr;
            memset(&req, 0, sizeof(req));
            nn::util::Strlcpy(req.ifr_name, "usb0", sizeof(req.ifr_name));
            if ((mibsock = nn::socket::Socket(nn::socket::Family::Pf_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp)) == -1) break;
            if (nn::socket::Ioctl(mibsock, static_cast<nn::socket::IoctlCommand>(nn::socket::IoctlCommandPrivate::SiocGIfAddr), (caddr_t)&req, sizeof(req)) == -1) break;
            NN_LOG("My IP address is %s.\n", nn::socket::InetNtoa(pSa->sin_addr));
            ipdisp_fg = 1;
        }while(false);

        /* wait 200msec */
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(200));
    }
    NN_LOG("\n");
}

void ServerFunction()
{
    nn::socket::SockAddrIn saServer = {0};
    nn::socket::FdSet      activeSocketDescriptors;
    nn::socket::FdSet      readSocketDescriptors;
    char        clientMessage[messageBufferSize];
    int         activeSocketDescriptor;
    int         serverSocketDescriptor;
    int         clientSocketDescriptor;
    int         highestActiveDescriptor;
    ssize_t     readByteCount;
    int         selectResult;
    nn::socket::TimeVal     timeout = {60, 0};

    NN_LOG("%s start\n", __FUNCTION__);

    DisplayMyIp();

    if ((highestActiveDescriptor = serverSocketDescriptor = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp)) < 0)
    {
        NN_LOG("server:  socket failed (error %d)\n", nn::socket::GetLastError());
        goto out;
    }

    saServer.sin_addr.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Any);
    saServer.sin_port        = nn::socket::InetHtons(myPort);
    saServer.sin_family      = nn::socket::Family::Af_Inet;

    if (nn::socket::Bind(serverSocketDescriptor, reinterpret_cast<nn::socket::SockAddr *>(&saServer), sizeof(saServer)) < 0)
    {
        NN_LOG("server:  bind failed (error %d)\n", nn::socket::GetLastError());
        goto out;
    }

    nn::socket::Listen(serverSocketDescriptor, 1);

    nn::socket::FdSetZero(&activeSocketDescriptors);
    nn::socket::FdSetSet(serverSocketDescriptor, &activeSocketDescriptors);

    while (NN_STATIC_CONDITION(true))
    {
//        NN_LOG("server:  waiting for incoming connections or data...\n");]

        readSocketDescriptors = activeSocketDescriptors;

        if ((selectResult = nn::socket::Select(highestActiveDescriptor + 1, &readSocketDescriptors, NULL, NULL, &timeout)) < 0)
        {
            NN_LOG("server:  select failed (error %d)\n", nn::socket::GetLastError());
            goto out;
        }
        else if (selectResult == 0)
        {
            NN_LOG("server:  select timeout\n");
            goto out;
        }

        for (activeSocketDescriptor = 0; activeSocketDescriptor < highestActiveDescriptor + 1; activeSocketDescriptor++)
        {
            if (nn::socket::FdSetIsSet(activeSocketDescriptor, &readSocketDescriptors))
            {
                if (activeSocketDescriptor == serverSocketDescriptor)
                {
                    if ((clientSocketDescriptor = nn::socket::Accept(activeSocketDescriptor, static_cast<nn::socket::SockAddr *>(nullptr), 0)) < 0)
                    {
                        NN_LOG("server:  accept failed (error %d)\n", nn::socket::GetLastError());
                        goto out;
                    }

                    NN_LOG("server:  connection accepted on socket %d\n", clientSocketDescriptor);

                    if (clientSocketDescriptor > highestActiveDescriptor)
                    {
                        highestActiveDescriptor = clientSocketDescriptor;
                    }

                    nn::socket::FdSetSet(clientSocketDescriptor, &activeSocketDescriptors);
                }
                else
                {
                    if ((readByteCount = nn::socket::Recv(activeSocketDescriptor, clientMessage, sizeof(clientMessage) - 1, nn::socket::MsgFlag::Msg_None)) > 0)
                    {
                        clientMessage[readByteCount] = '\0';

                        if (nn::socket::Send(activeSocketDescriptor, clientMessage, readByteCount, nn::socket::MsgFlag::Msg_None) < 0)
                        {
                            NN_LOG("server: send failed (error %d)\n", nn::socket::GetLastError());
                            goto out;
                        }
                    }
                    else if (readByteCount == 0)
                    {
                        NN_LOG("server:  client disconnect on socket %d\n", activeSocketDescriptor);
                        nn::socket::FdSetClr(activeSocketDescriptor, &activeSocketDescriptors);
                        nn::socket::Close(activeSocketDescriptor);
                        goto out;
                    }
                    else
                    {
                        NN_LOG("server:  recv failed on socket %d (error %d)\n", activeSocketDescriptor, nn::socket::GetLastError());
                    }
                }
            }
        }
    }
out:

    nn::socket::Close(serverSocketDescriptor);

    NN_LOG("%s end\n", __FUNCTION__);
}

extern "C" int nn_cmain(int argc, const char **argv, void *wl);

void EchoServer(void *wl)
{
    nn::Result result;

    result = nn::socket::Initialize(g_SocketConfigWithMemory);
    NN_ASSERT(result.IsSuccess());

    NN_LOG("[%s] wl is %p\n", __FUNCTION__, wl);
    //ServerFunction();

#if 1
    // Have to cook up the cmd line args here
    {
        int err;
        int argc = 1;
        const char *argv[2];

        argv[0] = "remote_wl";
        argv[1] = NULL;
        err = nn_cmain(argc, argv, wl);
    NN_LOG("%s nn_cmain returned %d\n", __FUNCTION__, err);
    }
#endif

    nn::socket::Finalize();
}


extern "C"
void nnMain()
{
    void *wl;
    NN_LOG("Remote WL tool\n");

    nn::wlan::InitializeMemory();

    wl = InitializeWlan();
    NN_LOG("dhd initialize done\n");

    IfInit ifctrl;
    ifctrl.InitializeEtherInterface();

    EchoServer(wl);

    ifctrl.FinalizeEtherInterface();

    FinalizeWlan();

    NN_LOG("End of Remote WL\n");

    return;
}
