﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/tsc.h>
#include <nn/eth/eth_EthClient.h>
#include <nn/socket/socket_Api.h>
#include <nn/bsdsocket/cfg/cfg_Types.h>
#include <nn/bsdsocket/cfg/cfg_ClientApi.h>

#include "IfInit.h"

namespace {

const uint32_t g_MtuValue                 = 1500;
const uint32_t g_ApplyTimeoutMsec         = 1000 * 10;
const char*    g_ipStringInterfaceAddress = "192.168.0.202";
const char*    g_ipStringSubnetMask       = "255.255.255.0";
const char*    g_pInterfaceName = nullptr;
const char*    g_ipStringDefaultGateway   = "192.168.0.1";
const char*    g_ipStringPreferredDns     = "192.168.0.1";
const char*    g_ipStringAlternativeDns   = "192.168.0.2";

}

static nn::Result ApplyIpv4Config(nn::tsc::Ipv4ConfigContext* context)
{
    nn::Result result = context->ApplyConfig(nn::tsc::IpApplyModeMask_DuplicateIpCheck);

    if( result.IsSuccess() )
    {
        nn::os::Event* pEvent = context->GetEventPointer();
        NN_SDK_ASSERT_NOT_NULL(pEvent);

        if( !pEvent->TimedWait(nn::TimeSpan::FromMilliSeconds(g_ApplyTimeoutMsec)) )
        {
            NN_LOG("Socket config has timed out.\n");
            context->CancelApplyConfig();
            // キャンセルに成功した場合は GetApplyResult() が失敗になり、
            // キャンセルが間に合わなかった場合は GetApplyResult() が成功になるので
            // キャンセル自身の結果はハンドリングしない
        }
        result = context->GetApplyResult();
        if( result.IsSuccess() )
        {
            NN_LOG("ApplyConfig success!\n");
            return nn::ResultSuccess();
        }
    }

    return result;
}

IfInit::IfInit()
{
    m_Running = 1;
}

IfInit::~IfInit()
{
}

// Handle adapter insertion/removal
void IfInit::HandleInterfaceGroupEvent()
NN_NOEXCEPT
{
    nn::Result result;
    nn::eth::InterfaceList removedList;
    nn::eth::InterfaceList addedList;
    nn::eth::InterfaceList newList;

    m_EthernetInterfaceGroup->ClearEvent();

    result = m_EthernetInterfaceGroup->GetInterfaceList(&newList);
    NN_ABORT_UNLESS(result.IsSuccess());

    // parse and update local interface list
    m_EthernetInterfaceGroup->ParseInterfaceList(&newList, &m_CurrentInterfaceList, &addedList, &removedList);
    m_CurrentInterfaceList = newList;

    // handle removals
    for (uint32_t i = 0; i < removedList.adapterCount; i++)
    {
        NN_LOG("Removed:\n"
               "[%s]model:    %08x\n"
               "[%s]revision: %08x\n"
               "[%s]mac:      %02x:%02x:%02x:%02x:%02x:%02x\n",
               removedList.adapterInfo[i].interfaceName,
               removedList.adapterInfo[i].interfaceModel,
               removedList.adapterInfo[i].interfaceName,
               removedList.adapterInfo[i].interfaceRevision,
               removedList.adapterInfo[i].interfaceName,
               removedList.adapterInfo[i].interfaceMacAddress[0],
               removedList.adapterInfo[i].interfaceMacAddress[1],
               removedList.adapterInfo[i].interfaceMacAddress[2],
               removedList.adapterInfo[i].interfaceMacAddress[3],
               removedList.adapterInfo[i].interfaceMacAddress[4],
               removedList.adapterInfo[i].interfaceMacAddress[5]
               );

        // deallocations will be done inside link state handler on
        // an event containing ResultInterfaceRemoved() result
    }

    // handle additions
    for (uint32_t i = 0; i < addedList.adapterCount; i++)
    {
        nn::eth::client::InterfaceHandler* ethernetInterface;
        nn::os::MultiWaitHolderType*          ethernetInterfaceHolder;

        strncpy(m_pStringStorage, addedList.adapterInfo[i].interfaceName, 16);
        g_pInterfaceName = m_pStringStorage;

        NN_LOG("Added:\n"
               "[%s]model:    %08x\n"
               "[%s]revision: %08x\n"
               "[%s]mac:      %02x:%02x:%02x:%02x:%02x:%02x\n",
               addedList.adapterInfo[i].interfaceName,
               addedList.adapterInfo[i].interfaceModel,
               addedList.adapterInfo[i].interfaceName,
               addedList.adapterInfo[i].interfaceRevision,
               addedList.adapterInfo[i].interfaceName,
               addedList.adapterInfo[i].interfaceMacAddress[0],
               addedList.adapterInfo[i].interfaceMacAddress[1],
               addedList.adapterInfo[i].interfaceMacAddress[2],
               addedList.adapterInfo[i].interfaceMacAddress[3],
               addedList.adapterInfo[i].interfaceMacAddress[4],
               addedList.adapterInfo[i].interfaceMacAddress[5]
               );

        // create interface handler for new interface,
        // add its event to multi wait

        ethernetInterface = new nn::eth::client::InterfaceHandler;
        NN_ABORT_UNLESS(ethernetInterface != nullptr);

        result = ethernetInterface->Initialize(m_EthernetInterfaceGroup, &addedList.adapterInfo[i], nn::os::EventClearMode_ManualClear);
        NN_ABORT_UNLESS(result.IsSuccess());

        ethernetInterfaceHolder = new nn::os::MultiWaitHolderType;
        NN_ABORT_UNLESS(ethernetInterfaceHolder != nullptr);

        SetupWaitHolder<nn::eth::client::InterfaceHandler>(ethernetInterfaceHolder, ethernetInterface);

        // request auto negotiation
        result = ethernetInterface->SetMediaType(nn::eth::MediaType_AUTO);
        NN_ABORT_UNLESS(result.IsSuccess());

    }
}

// Configure IP
void IfInit::ConfigureInterface(const char* interfaceName, bool enable, bool useDhcp)
{
    nn::Result result;
    nn::tsc::Ipv4ConfigContext ipv4ConfigContext(g_pInterfaceName, nn::os::EventClearMode_AutoClear);

    {
        nn::tsc::Ipv4Config ipv4Config;

        if( useDhcp == true )
        {
            ipv4Config.interfaceAddress.storage = 0;
            ipv4Config.subnetMask.storage       = 0;
            ipv4Config.defaultGateway.storage   = 0;

            // DNS setting auto
            {
                ipv4Config.method = nn::tsc::Ipv4ConfigMethod_Dhcp;

                ipv4Config.preferredDns.storage     = 0;
                ipv4Config.alternativeDns.storage   = 0;
            }
        }
        else
        {
            NN_LOG("Static IP setup\n");
            ipv4Config.method = nn::tsc::Ipv4ConfigMethod_Static;

            nn::socket::InetAton(g_ipStringInterfaceAddress, reinterpret_cast<nn::socket::InAddr *>(&ipv4Config.interfaceAddress));
            nn::socket::InetAton(g_ipStringSubnetMask,       reinterpret_cast<nn::socket::InAddr *>(&ipv4Config.subnetMask));
            nn::socket::InetAton(g_ipStringDefaultGateway,   reinterpret_cast<nn::socket::InAddr *>(&ipv4Config.defaultGateway));
            nn::socket::InetAton(g_ipStringPreferredDns,     reinterpret_cast<nn::socket::InAddr *>(&ipv4Config.preferredDns));
            nn::socket::InetAton(g_ipStringAlternativeDns,   reinterpret_cast<nn::socket::InAddr *>(&ipv4Config.alternativeDns));
        }

        ipv4ConfigContext.SetConfig(&ipv4Config);
    }

    ipv4ConfigContext.SetMtu(g_MtuValue);

    for (;;)
    {
        result = ApplyIpv4Config(&ipv4ConfigContext);
        if ( result.IsSuccess() )
        {
            break;
        }
        else if ( !nn::tsc::ResultRetryRequired::Includes(result) )
        {
            NN_LOG("result=%08x\n", result.GetInnerValueForDebug());
            NN_SDK_ASSERT(false, "ApplyConfig failed");
        }

        // 成功するまで2分近くかかることもある
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5000));
        NN_LOG("[RemoteWl] Retrying ApplyConfig...(result=%08x)\n", result.GetInnerValueForDebug());
    }
}

// Handle link state changes
void IfInit::HandleInterfaceMediaEvent(nn::os::MultiWaitHolderType* holder)
NN_NOEXCEPT
{
    nn::eth::MediaType current;
    nn::eth::MediaType requested;
    nn::Result result;
    uint32_t   eventCounter;

    nn::eth::client::InterfaceHandler* ethernetInterface =
    reinterpret_cast<nn::eth::client::InterfaceHandler*>
    (nn::os::GetMultiWaitHolderUserData(holder));

    // clear this before processing so that if
    // event happens while processing is in progress
    // we would not miss it.
    ethernetInterface->ClearEvent();

    if (ethernetInterface->GetResult() <= ResultInterfaceRemoved())
    {
        NN_LOG("[%s] Adapter removed\n", ethernetInterface->GetInterfaceName());
        nn::os::UnlinkMultiWaitHolder(holder);
        delete holder;
        delete ethernetInterface;
    }
    else
    {
        // obtain interface state
        result = ethernetInterface->GetMediaType(&requested, &current, &eventCounter);
        NN_ABORT_UNLESS(result.IsSuccess());

        if (!EthLinkUp(current))
        {
            NN_LOG("[%s] Lost link\n", ethernetInterface->GetInterfaceName());
            ConfigureInterface(ethernetInterface->GetInterfaceName(), false, false);
        } else {
            NN_LOG("[%s] Link: %s, Speed: %d, Fullduplex: %s, Halfduplex: %s, Flow: %s, %s\n",
                   ethernetInterface->GetInterfaceName(),
                   EthLinkUp(current)     ? "Up"   : "Down",
                   EthSpeed(current),
                   EthFullDuplex(current) ? "On"   : "Off",
                   EthHalfDuplex(current) ? "On"   : "Off",
                   EthFlow(current)       ? "On"   : "Off",
                   EthAuto(current)       ? "Auto" : "Manual");
            ConfigureInterface(ethernetInterface->GetInterfaceName(), true, false);

            m_Running = 0;
        }
    }
}

void IfInit::InitializeEtherInterface()
{
    nn::Result result;
    nn::tsc::Initialize();

    m_CurrentInterfaceList.adapterCount = 0;

    m_EthernetInterfaceGroup = new nn::eth::client::InterfaceGroupHandler;
    NN_ABORT_UNLESS(m_EthernetInterfaceGroup != nullptr);

    // open session to the server
    result = m_EthernetInterfaceGroup->Initialize(nn::os::EventClearMode_ManualClear);
    NN_ABORT_UNLESS(result.IsSuccess());

    // add group interface handler to multi wait
    nn::os::InitializeMultiWait(&m_MultiWait);
    SetupWaitHolder<nn::eth::client::InterfaceGroupHandler>(&m_InterfaceGroupHolder, m_EthernetInterfaceGroup);

    NN_LOG("%s wait\n", __FUNCTION__);
    while (m_Running)
    {
        // wait for group change or link state change
        nn::os::MultiWaitHolderType* pendingEventHolder = nn::os::WaitAny(&m_MultiWait);

        if (pendingEventHolder == &m_InterfaceGroupHolder)
        {
            // interface was added or removed
            HandleInterfaceGroupEvent();
        }
        else
        {
            // link state has changed
            HandleInterfaceMediaEvent(pendingEventHolder);
        }
    }
}

void IfInit::FinalizeEtherInterface()
{
    nn::tsc::Finalize();
    delete m_EthernetInterfaceGroup;
}

