﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nnt/nntest.h>

#include <nn/util/util_Arithmetic.h>
#include <nn/util/util_Constant.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Quaternion.h>
#include <nn/util/util_Vector.h>

#include "testUtil_Vector.h"

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
// do{}while(0) 文を使うと、C4127（定数条件式に対する警告）が発生するため、これを無効化
#pragma warning( push )
#pragma warning( disable:4127 )
#endif

namespace
{
    const float FloatError = 0.000001f;
}

//
// ベクトル
//
// 2次元ベクトル
class MathVector2ClassTest : public ::testing::Test
{
};

TEST_F(MathVector2ClassTest, Constant)
{
    NNT_UTIL_VECTOR2_EXPECT_EQ(nn::util::Vector2f::Zero(), 0.f, 0.f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(nn::util::Vector2f::One(), 1.f, 1.f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(nn::util::Vector2f::UnitX(), 1.f, 0.f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(nn::util::Vector2f::UnitY(), 0.f, 1.f);
}

TEST_F(MathVector2ClassTest, Constractor)
{
    {
        nn::util::Vector2f vector(-3.f, 1.f);

        NNT_UTIL_VECTOR2_EXPECT_EQ(vector, -3.f, 1.f);
    }

    {
        nn::util::Float2 floatValue = NN_UTIL_FLOAT_2_INITIALIZER(-3.f, 1.f);
        nn::util::Vector2f vector(floatValue);

        NNT_UTIL_VECTOR2_EXPECT_EQ(vector, -3.f, 1.f);
    }

    {
        nn::util::Vector2fType vectorTypeValue = NN_UTIL_VECTOR_2F_INITIALIZER(-3.f, 1.f);
        nn::util::Vector2f vector(vectorTypeValue);

        NNT_UTIL_VECTOR2_EXPECT_EQ(vector, -3.f, 1.f);
    }
}


TEST_F(MathVector2ClassTest, GetXyz)
{
    nn::util::Vector2f vector(-3.f, 1.f);

    EXPECT_EQ(-3.f, vector.GetX());
    EXPECT_EQ(1.f, vector.GetY());
}

TEST_F(MathVector2ClassTest, Get)
{
    nn::util::Vector2f vector(-3.f, 1.f);

    float x, y;
    vector.Get(&x, &y);

    EXPECT_EQ(-3.f, x);
    EXPECT_EQ(1.f, y);
}

TEST_F(MathVector2ClassTest, SetXyz)
{
    nn::util::Vector2f vector;
    vector.Set(0.f, 0.f);

    vector.SetX(-3.f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector, -3.f, 0.f);

    vector.SetY(1.f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector, -3.f, 1.f);
}

TEST_F(MathVector2ClassTest, Set)
{
    nn::util::Vector2f vector;

    vector.Set(-3.f, 1.f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector, -3.f, 1.f);
}

TEST_F(MathVector2ClassTest, Substitution)
{
    nn::util::Vector2f vector1(-3.f, 1.f);
    nn::util::Vector2f vector2(vector1);

    vector1.Zero();

    vector1 = vector2;

    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, -3.f, 1.f);
}

TEST_F(MathVector2ClassTest, PlusSign)
{
    nn::util::Vector2f vector1(-3.f, 1.f);

    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, -3.f, 1.f);
}

TEST_F(MathVector2ClassTest, MinusSign)
{
    nn::util::Vector2f vector1(-3.f, 1.f);

    NNT_UTIL_VECTOR2_EXPECT_EQ(-vector1, 3.f, -1.f);
}

TEST_F(MathVector2ClassTest, Add)
{
    {
        nn::util::Vector2f vector1(-3.f, 1.f);
        nn::util::Vector2f vector2(-6.f, 2.f);

        vector1 = vector1 + vector2;
        NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, -9.f, 3.f);
    }

    {
        nn::util::Vector2f vector1(-3.f, 1.f);
        nn::util::Vector2f vector2(-6.f, 2.f);

        vector1 += vector2;
        NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, -9.f, 3.f);
    }
}

TEST_F(MathVector2ClassTest, Subtract)
{
    {
        nn::util::Vector2f vector1(-3.f, 1.f);
        nn::util::Vector2f vector2(-6.f, 2.f);

        vector1 = vector1 - vector2;
        NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 3.f, -1.f);
    }

    {
        nn::util::Vector2f vector1(-3.f, 1.f);
        nn::util::Vector2f vector2(-6.f, 2.f);

        vector1 -= vector2;
        NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 3.f, -1.f);
    }
}

TEST_F(MathVector2ClassTest, Multiply)
{
    {
        nn::util::Vector2f vector1(-3.f, 1.f);

        vector1 = vector1 * (-1.f / 2.f);
        NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 1.5f, -0.5f);
    }

    {
        nn::util::Vector2f vector1(-3.f, 1.f);

        vector1 = (-1.f / 2.f) * vector1;
        NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 1.5f, -0.5f);
    }

    {
        nn::util::Vector2f vector1(-3.f, 1.f);

        vector1 *= (-1.f / 2.f);
        NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 1.5f, -0.5f);
    }
}

TEST_F(MathVector2ClassTest, Dot)
{
    {
        nn::util::Vector2f vector1(-3.f, 1.f);
        nn::util::Vector2f vector2(-6.f, 2.f);

        EXPECT_EQ(20.f, vector1.Dot(vector2));
    }

    {
        nn::util::Vector2f vector1(-3.f, 1.f);
        nn::util::Vector2f vector2(-6.f, 2.f);

        EXPECT_EQ(20.f, nn::util::Vector2f::Dot(vector1, vector2));
    }
}

TEST_F(MathVector2ClassTest, Cross)
{
    {
        nn::util::Vector2f vector1(-3.f, 1.f);
        nn::util::Vector2f vector2(6.f, 2.f);

        EXPECT_EQ(-12.f, vector1.Cross(vector2));
    }

    {
        nn::util::Vector2f vector1(-3.f, 1.f);
        nn::util::Vector2f vector2(6.f, 2.f);

        EXPECT_EQ(-12.f, nn::util::Vector2f::Cross(vector1, vector2));
    }
}

TEST_F(MathVector2ClassTest, Length)
{
    nn::util::Vector2f vector1(-3.f, 1.f);

    EXPECT_FLOAT_EQ(3.16227766017f, vector1.Length());
    EXPECT_EQ(10.f, vector1.LengthSquared());

    nn::util::Vector2f vectorZero(0.f, 0.f);

    EXPECT_FLOAT_EQ(0.f, vectorZero.Length());
    EXPECT_EQ(0.f, vectorZero.LengthSquared());
}

TEST_F(MathVector2ClassTest, Distance)
{
    nn::util::Vector2f vector1(-3.f, 1.f);
    nn::util::Vector2f vector2(6.f, 2.f);

    EXPECT_FLOAT_EQ(9.05538513814f, nn::util::Vector2f::Distance(vector1, vector2));
    EXPECT_EQ(82.f, nn::util::Vector2f::DistanceSquared(vector1, vector2));

    EXPECT_FLOAT_EQ(0.f, nn::util::Vector2f::Distance(vector1, vector1));
    EXPECT_EQ(0.f, nn::util::Vector2f::DistanceSquared(vector1, vector1));
}

TEST_F(MathVector2ClassTest, Normalize)
{
    {
        nn::util::Vector2f vector1(-3.f, 1.f);

        bool result = vector1.Normalize();
        NNT_UTIL_VECTOR2_EXPECT_NEARLY_EQ(vector1, -0.94868329805f, 0.31622776601f, FloatError);
        EXPECT_TRUE(result);
    }

    {
        nn::util::Vector2f vector1(-3.f, 1.f);

        vector1 = nn::util::Vector2f::Normalize(vector1);
        NNT_UTIL_VECTOR2_EXPECT_NEARLY_EQ(vector1, -0.94868329805f, 0.31622776601f, FloatError);
    }

    {
        nn::util::Vector2f vector1(0.f, 0.f);

        bool result = vector1.Normalize();
        NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 0.f, 0.f);
        EXPECT_FALSE(result);
    }

    {
        nn::util::Vector2f vector1(0.f, 0.f);

        vector1 = nn::util::Vector2f::Normalize(vector1);
        NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 0.f, 0.f);
    }
}

TEST_F(MathVector2ClassTest, Maximize)
{
    nn::util::Vector2f vector1(-3.f, 1.f);
    nn::util::Vector2f vector2(6.f, -2.f);

    vector1 = nn::util::Vector2f::Maximize(vector1, vector2);

    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 6.f, 1.f);
}

TEST_F(MathVector2ClassTest, Minimize)
{
    nn::util::Vector2f vector1(-3.f, 1.f);
    nn::util::Vector2f vector2(6.f, -2.f);

    vector1 = nn::util::Vector2f::Minimize(vector1, vector2);

    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, -3.f, -2.f);
}

TEST_F(MathVector2ClassTest, Lerp)
{
    nn::util::Vector2f vector1(-3.f, 1.f);
    nn::util::Vector2f vector2(6.f, -2.f);

    nn::util::Vector2f vector3;

    vector3 = nn::util::Vector2f::Lerp(vector1, vector2, 0.f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector3, -3.f, 1.f);

    vector3 = nn::util::Vector2f::Lerp(vector1, vector2, 1.f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector3, 6.f, -2.f);

    vector3 = nn::util::Vector2f::Lerp(vector1, vector2, 0.5f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector3, 1.5f, -0.5f);
}

TEST_F(MathVector2ClassTest, Transform)
{
    nn::util::Vector2f vector1(2.f, -4.f);

    nn::util::MatrixRowMajor3x2f matrix1(
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    vector1 = nn::util::Vector2f::Transform(vector1, matrix1);

    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 19.f, -26.f);
}

TEST_F(MathVector2ClassTest, TransformNormal)
{
    nn::util::Vector2f vector1(2.f, -4.f);

    nn::util::MatrixRowMajor3x2f matrix1(
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    vector1 = nn::util::Vector2f::TransformNormal(vector1, matrix1);

    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 14.f, -20.f);
}

// 3次元ベクトル
class MathVector3ClassTest : public ::testing::Test
{
};

TEST_F(MathVector3ClassTest, Constant)
{
    NNT_UTIL_VECTOR3_EXPECT_EQ(nn::util::Vector3f::Zero(), 0.f, 0.f, 0.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(nn::util::Vector3f::One(), 1.f, 1.f, 1.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(nn::util::Vector3f::UnitX(), 1.f, 0.f, 0.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(nn::util::Vector3f::UnitY(), 0.f, 1.f, 0.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(nn::util::Vector3f::UnitZ(), 0.f, 0.f, 1.f);
}

TEST_F(MathVector3ClassTest, Constractor)
{
    {
        nn::util::Vector3f vector(-3.f, 1.f, -2.f);

        NNT_UTIL_VECTOR3_EXPECT_EQ(vector, -3.f, 1.f, -2.f);
    }

    {
        nn::util::Float3 floatValue = NN_UTIL_FLOAT_3_INITIALIZER(-3.f, 1.f, -2.f);
        nn::util::Vector3f vector(floatValue);

        NNT_UTIL_VECTOR3_EXPECT_EQ(vector, -3.f, 1.f, -2.f);
    }

    {
        nn::util::Vector3fType vectorTypeValue = NN_UTIL_VECTOR_3F_INITIALIZER(-3.f, 1.f, -2.f);
        nn::util::Vector3f vector(vectorTypeValue);

        NNT_UTIL_VECTOR3_EXPECT_EQ(vector, -3.f, 1.f, -2.f);
    }
}


TEST_F(MathVector3ClassTest, GetXyz)
{
    nn::util::Vector3f vector(-3.f, 1.f, -2.f);

    EXPECT_EQ(-3.f, vector.GetX());
    EXPECT_EQ(1.f, vector.GetY());
    EXPECT_EQ(-2.f, vector.GetZ());
}

TEST_F(MathVector3ClassTest, Get)
{
    nn::util::Vector3f vector(-3.f, 1.f, -2.f);

    float x, y, z;
    vector.Get(&x, &y, &z);

    EXPECT_EQ(-3.f, x);
    EXPECT_EQ(1.f, y);
    EXPECT_EQ(-2.f, z);
}

TEST_F(MathVector3ClassTest, SetXyz)
{
    nn::util::Vector3f vector;
    vector.Set(0.f, 0.f, 0.f);

    vector.SetX(-3.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector, -3.f, 0.f, 0.f);

    vector.SetY(1.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector, -3.f, 1.f, 0.f);

    vector.SetZ(-2.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector, -3.f, 1.f, -2.f);
}

TEST_F(MathVector3ClassTest, Set)
{
    nn::util::Vector3f vector;

    vector.Set(-3.f, 1.f, -2.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector, -3.f, 1.f, -2.f);
}

TEST_F(MathVector3ClassTest, Substitution)
{
    nn::util::Vector3f vector1(-3.f, 1.f, -2.f);
    nn::util::Vector3f vector2(vector1);

    vector1.Zero();

    vector1 = vector2;

    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, -3.f, 1.f, -2.f);
}

TEST_F(MathVector3ClassTest, PlusSign)
{
    nn::util::Vector3f vector1(-3.f, 1.f, -2.f);

    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, -3.f, 1.f, -2.f);
}

TEST_F(MathVector3ClassTest, MinusSign)
{
    nn::util::Vector3f vector1(-3.f, 1.f, -2.f);

    NNT_UTIL_VECTOR3_EXPECT_EQ(-vector1, 3.f, -1.f, 2.f);
}

TEST_F(MathVector3ClassTest, Add)
{
    {
        nn::util::Vector3f vector1(-3.f, 1.f, -2.f);
        nn::util::Vector3f vector2(-6.f, 2.f, 4.f);

        vector1 = vector1 + vector2;
        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, -9.f, 3.f, 2.f);
    }

    {
        nn::util::Vector3f vector1(-3.f, 1.f, -2.f);
        nn::util::Vector3f vector2(-6.f, 2.f, 4.f);

        vector1 += vector2;
        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, -9.f, 3.f, 2.f);
    }
}

TEST_F(MathVector3ClassTest, Subtract)
{
    {
        nn::util::Vector3f vector1(-3.f, 1.f, -2.f);
        nn::util::Vector3f vector2(-6.f, 2.f, 4.f);

        vector1 = vector1 - vector2;
        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 3.f, -1.f, -6.f);
    }

    {
        nn::util::Vector3f vector1(-3.f, 1.f, -2.f);
        nn::util::Vector3f vector2(-6.f, 2.f, 4.f);

        vector1 -= vector2;
        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 3.f, -1.f, -6.f);
    }
}

TEST_F(MathVector3ClassTest, Multiply)
{
    {
        nn::util::Vector3f vector1(-3.f, 1.f, -2.f);

        vector1 = vector1 * (-1.f / 2.f);
        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 1.5f, -0.5f, 1.f);
    }

    {
        nn::util::Vector3f vector1(-3.f, 1.f, -2.f);

        vector1 = (-1.f / 2.f) * vector1;
        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 1.5f, -0.5f, 1.f);
    }

    {
        nn::util::Vector3f vector1(-3.f, 1.f, -2.f);

        vector1 *= (-1.f / 2.f);
        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 1.5f, -0.5f, 1.f);
    }
}

TEST_F(MathVector3ClassTest, Dot)
{
    {
        nn::util::Vector3f vector1(-3.f, 1.f, -2.f);
        nn::util::Vector3f vector2(-6.f, 2.f, 4.f);

        EXPECT_EQ(12.f, vector1.Dot(vector2));
    }

    {
        nn::util::Vector3f vector1(-3.f, 1.f, -2.f);
        nn::util::Vector3f vector2(-6.f, 2.f, 4.f);

        EXPECT_EQ(12.f, nn::util::Vector3f::Dot(vector1, vector2));
    }
}

TEST_F(MathVector3ClassTest, Cross)
{
    {
        nn::util::Vector3f vector1(-3.f, 1.f, -2.f);
        nn::util::Vector3f vector2(-6.f, 2.f, 4.f);

        vector1 = vector1.Cross(vector2);

        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 8.f, 24.f, 0.f);
    }

    {
        nn::util::Vector3f vector1(-3.f, 1.f, -2.f);
        nn::util::Vector3f vector2(-6.f, 2.f, 4.f);

        vector1 = nn::util::Vector3f::Cross(vector1, vector2);

        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 8.f, 24.f, 0.f);
    }
}

TEST_F(MathVector3ClassTest, Length)
{
    nn::util::Vector3f vector1(-3.f, 1.f, -2.f);

    EXPECT_FLOAT_EQ(3.74165738677f, vector1.Length());
    EXPECT_EQ(14.f, vector1.LengthSquared());

    nn::util::Vector3f vectorZero(0.f, 0.f, 0.f);

    EXPECT_FLOAT_EQ(0.f, vectorZero.Length());
    EXPECT_EQ(0.f, vectorZero.LengthSquared());
}

TEST_F(MathVector3ClassTest, Distance)
{
    nn::util::Vector3f vector1(-3.f, 1.f, -2.f);
    nn::util::Vector3f vector2(-6.f, 2.f, 4.f);

    EXPECT_FLOAT_EQ(6.78232998312f, nn::util::Vector3f::Distance(vector1, vector2));
    EXPECT_EQ(46.f, nn::util::Vector3f::DistanceSquared(vector1, vector2));

    EXPECT_FLOAT_EQ(0.f, nn::util::Vector3f::Distance(vector1, vector1));
    EXPECT_EQ(0.f, nn::util::Vector3f::DistanceSquared(vector1, vector1));
}

TEST_F(MathVector3ClassTest, Normalize)
{
    {
        nn::util::Vector3f vector1(-3.f, 1.f, -2.f);

        bool result = vector1.Normalize();
        NNT_UTIL_VECTOR3_EXPECT_NEARLY_EQ(vector1, -0.80178372574f, 0.267261241912705f, -0.53452248382541f, FloatError);
        EXPECT_TRUE(result);
    }

    {
        nn::util::Vector3f vector1(-3.f, 1.f, -2.f);

        vector1 = nn::util::Vector3f::Normalize(vector1);
        NNT_UTIL_VECTOR3_EXPECT_NEARLY_EQ(vector1, -0.80178372574f, 0.267261241912705f, -0.53452248382541f, FloatError);
    }

    {
        nn::util::Vector3f vector1(0.f, 0.f, 0.f);

        bool result = vector1.Normalize();
        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 0.f, 0.f, 0.f);
        EXPECT_FALSE(result);
    }

    {
        nn::util::Vector3f vector1(0.f, 0.f, 0.f);

        vector1 = nn::util::Vector3f::Normalize(vector1);
        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 0.f, 0.f, 0.f);
    }
}

TEST_F(MathVector3ClassTest, Maximize)
{
    nn::util::Vector3f vector1(-3.f, 1.f, -2.f);
    nn::util::Vector3f vector2(6.f, -2.f, 4.f);

    vector1 = nn::util::Vector3f::Maximize(vector1, vector2);

    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 6.f, 1.f, 4.f);
}

TEST_F(MathVector3ClassTest, Minimize)
{
    nn::util::Vector3f vector1(-3.f, 1.f, -2.f);
    nn::util::Vector3f vector2(6.f, -2.f, 4.f);

    vector1 = nn::util::Vector3f::Minimize(vector1, vector2);

    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, -3.f, -2.f, -2.f);
}

TEST_F(MathVector3ClassTest, Lerp)
{
    nn::util::Vector3f vector1(-3.f, 1.f, -2.f);
    nn::util::Vector3f vector2(6.f, -2.f, 4.f);

    nn::util::Vector3f vector3;

    vector3 = nn::util::Vector3f::Lerp(vector1, vector2, 0.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector3, -3.f, 1.f, -2.f);

    vector3 = nn::util::Vector3f::Lerp(vector1, vector2, 1.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector3, 6.f, -2.f, 4.f);

    vector3 = nn::util::Vector3f::Lerp(vector1, vector2, 0.5f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector3, 1.5f, -0.5f, 1.f);
}

TEST_F(MathVector3ClassTest, Rotate)
{
    {
        nn::util::Quaternion quaternion1(3.f, 4.f, -5.f, -6.f);

        nn::util::Vector3f vector1(2.f, -4.f, 6.f);
        vector1 = nn::util::Vector3f::Rotate(vector1, quaternion1);

        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, -316.f, 72.f, 556.f);
    }

    {
        // 軸(1.f, 2.f, -3.f) を中心に 60度 回転させる、正規化されたクォータニオンです。
        nn::util::Quaternion quaternion1(0.1336306184530258f, 0.2672612369060516f, -0.4008918404579163f, 0.8660253882408142f);

        nn::util::Vector3f vector1(2.f, -4.f, 6.f);
        vector1 = nn::util::Vector3f::Rotate(vector1, quaternion1);

        NNT_UTIL_VECTOR3_EXPECT_NEARLY_EQ(vector1, 0.1428570747375488f, -6.4917459487915039f, 3.7197880744934082f, FloatError);
    }
}

TEST_F(MathVector3ClassTest, Transform)
{
    {
        nn::util::Vector3f vector1(2.f, -4.f, 6.f);

        nn::util::MatrixRowMajor4x3f matrix1(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        vector1 = nn::util::Vector3f::Transform(vector1, matrix1);

        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 50.f, -61.f, 72.f);
    }
}

TEST_F(MathVector3ClassTest, TransformNormal)
{
    {
        nn::util::Vector3f vector1(2.f, -4.f, 6.f);

        nn::util::MatrixRowMajor4x3f matrix1(
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        vector1 = nn::util::Vector3f::TransformNormal(vector1, matrix1);

        NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 60.f, -72.f, 84.f);
    }
}

TEST_F(MathVector3ClassTest, TransformCoord)
{
    {
        nn::util::Vector3f vector1(2.f, -4.f, 6.f);

        nn::util::MatrixRowMajor4x4f matrix1(
            1.f, -2.f, 3.f, -4.f,
            -5.f, 6.f, -7.f, 8.f,
            9.f, -10.f, 11.f, -12.f,
            -13.f, 14.f, -15.f, 16.f);

        vector1 = nn::util::Vector3f::TransformCoord(vector1, matrix1);

        NNT_UTIL_VECTOR3_EXPECT_NEARLY_EQ(vector1, -0.65625f, 0.7708333333333333f, -0.8854166666666666f, FloatError);
    }
}

// 4次元ベクトル
class MathVector4ClassTest : public ::testing::Test
{
};

TEST_F(MathVector4ClassTest, Constant)
{
    NNT_UTIL_VECTOR4_EXPECT_EQ(nn::util::Vector4f::Zero(), 0.f, 0.f, 0.f, 0.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(nn::util::Vector4f::One(), 1.f, 1.f, 1.f, 1.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(nn::util::Vector4f::UnitX(), 1.f, 0.f, 0.f, 0.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(nn::util::Vector4f::UnitY(), 0.f, 1.f, 0.f, 0.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(nn::util::Vector4f::UnitZ(), 0.f, 0.f, 1.f, 0.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(nn::util::Vector4f::UnitW(), 0.f, 0.f, 0.f, 1.f);
}

TEST_F(MathVector4ClassTest, Constractor)
{
    {
        nn::util::Vector4f vector(-3.f, 1.f, -2.f, 4.f);

        NNT_UTIL_VECTOR4_EXPECT_EQ(vector, -3.f, 1.f, -2.f, 4.f);
    }

    {
        nn::util::Float4 floatValue = NN_UTIL_FLOAT_4_INITIALIZER(-3.f, 1.f, -2.f, 4.f);
        nn::util::Vector4f vector(floatValue);

        NNT_UTIL_VECTOR4_EXPECT_EQ(vector, -3.f, 1.f, -2.f, 4.f);
    }

    {
        nn::util::Vector4fType vectorTypeValue = NN_UTIL_VECTOR_4F_INITIALIZER(-3.f, 1.f, -2.f, 4.f);
        nn::util::Vector4f vector(vectorTypeValue);

        NNT_UTIL_VECTOR4_EXPECT_EQ(vector, -3.f, 1.f, -2.f, 4.f);
    }
}

TEST_F(MathVector4ClassTest, GetXyzw)
{
    nn::util::Vector4f vector(-3.f, 1.f, -2.f, 4.f);

    EXPECT_EQ(-3.f, vector.GetX());
    EXPECT_EQ(1.f, vector.GetY());
    EXPECT_EQ(-2.f, vector.GetZ());
    EXPECT_EQ(4.f, vector.GetW());
}

TEST_F(MathVector4ClassTest, Get)
{
    nn::util::Vector4f vector(-3.f, 1.f, -2.f, 4.f);

    float x, y, z, w;
    vector.Get(&x, &y, &z, &w);

    EXPECT_EQ(-3.f, x);
    EXPECT_EQ(1.f, y);
    EXPECT_EQ(-2.f, z);
    EXPECT_EQ(4.f, w);
}

TEST_F(MathVector4ClassTest, SetXyzw)
{
    nn::util::Vector4f vector(0.f, 0.f, 0.f, 0.f);

    vector.SetX(-3.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector, -3.f, 0.f, 0.f, 0.f);

    vector.SetY(1.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector, -3.f, 1.f, 0.f, 0.f);

    vector.SetZ(-2.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector, -3.f, 1.f, -2.f, 0.f);

    vector.SetW(4.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector, -3.f, 1.f, -2.f, 4.f);
}

TEST_F(MathVector4ClassTest, Set)
{
    nn::util::Vector4f vector;
    vector.Set(-3.f, 1.f, -2.f, 4.f);

    NNT_UTIL_VECTOR4_EXPECT_EQ(vector, -3.f, 1.f, -2.f, 4.f);
}

TEST_F(MathVector4ClassTest, Substitution)
{
    nn::util::Vector4f vector1(-3.f, 1.f, -2.f, 4.f);
    nn::util::Vector4f vector2(vector1);

    vector1 = nn::util::Vector4f::Zero();

    vector1 = vector2;

    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, -3.f, 1.f, -2.f, 4.f);
}

TEST_F(MathVector4ClassTest, PlusSign)
{
    nn::util::Vector4f vector1(-3.f, 1.f, -2.f, 4.f);

    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, -3.f, 1.f, -2.f, 4.f);
}

TEST_F(MathVector4ClassTest, MinusSign)
{
    nn::util::Vector4f vector1(-3.f, 1.f, -2.f, 4.f);

    NNT_UTIL_VECTOR4_EXPECT_EQ(-vector1, 3.f, -1.f, 2.f, -4.f);
}

TEST_F(MathVector4ClassTest, Add)
{
    {
        nn::util::Vector4f vector1(-3.f, 1.f, -2.f, 4.f);
        nn::util::Vector4f vector2(-6.f, 2.f, 4.f, -8.f);

        vector1 = vector1 + vector2;
        NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, -9.f, 3.f, 2.f, -4.f);
    }

    {
        nn::util::Vector4f vector1(-3.f, 1.f, -2.f, 4.f);
        nn::util::Vector4f vector2(-6.f, 2.f, 4.f, -8.f);

        vector1 += vector2;
        NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, -9.f, 3.f, 2.f, -4.f);
    }
}

TEST_F(MathVector4ClassTest, Subtract)
{
    {
        nn::util::Vector4f vector1(-3.f, 1.f, -2.f, 4.f);
        nn::util::Vector4f vector2(-6.f, 2.f, 4.f, -8.f);

        vector1 = vector1 - vector2;
        NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 3.f, -1.f, -6.f, 12.f);
    }

    {
        nn::util::Vector4f vector1(-3.f, 1.f, -2.f, 4.f);
        nn::util::Vector4f vector2(-6.f, 2.f, 4.f, -8.f);

        vector1 -= vector2;
        NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 3.f, -1.f, -6.f, 12.f);
    }
}

TEST_F(MathVector4ClassTest, Multiply)
{
    {
        nn::util::Vector4f vector1(-3.f, 1.f, -2.f, 4.f);

        vector1 = vector1 * (-1.f / 2.f);
        NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 1.5f, -0.5f, 1.f, -2.f);
    }

    {
        nn::util::Vector4f vector1(-3.f, 1.f, -2.f, 4.f);

        vector1 *= (-1.f / 2.f);
        NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 1.5f, -0.5f, 1.f, -2.f);
    }
}

TEST_F(MathVector4ClassTest, Dot)
{
    {
        nn::util::Vector4f vector1(-3.f, 1.f, -2.f, 4.f);
        nn::util::Vector4f vector2(-6.f, 2.f, 4.f, -8.f);

        EXPECT_EQ(-20.f, vector1.Dot(vector2));
    }

    {
        nn::util::Vector4f vector1(-3.f, 1.f, -2.f, 4.f);
        nn::util::Vector4f vector2(-6.f, 2.f, 4.f, -8.f);

        EXPECT_EQ(-20.f, nn::util::Vector4f::Dot(vector1, vector2));
    }
}

TEST_F(MathVector4ClassTest, Length)
{
    nn::util::Vector4f vector1(-3.f, 1.f, -2.f, 4.f);

    EXPECT_FLOAT_EQ(5.47722557505f, vector1.Length());
    EXPECT_EQ(30.f, vector1.LengthSquared());

    nn::util::Vector3f vectorZero(0.f, 0.f, 0.f);

    EXPECT_FLOAT_EQ(0.f, vectorZero.Length());
    EXPECT_EQ(0.f, vectorZero.LengthSquared());
}

TEST_F(MathVector4ClassTest, Distance)
{
    nn::util::Vector4f vector1(-3.f, 1.f, -2.f, 4.f);
    nn::util::Vector4f vector2(-6.f, 2.f, 4.f, -8.f);

    EXPECT_FLOAT_EQ(13.7840487520f, nn::util::Vector4f::Distance(vector1, vector2));
    EXPECT_EQ(190.f, nn::util::Vector4f::DistanceSquared(vector1, vector2));

    EXPECT_FLOAT_EQ(0.f, nn::util::Vector4f::Distance(vector1, vector1));
    EXPECT_EQ(0.f, nn::util::Vector4f::DistanceSquared(vector1, vector1));
}

TEST_F(MathVector4ClassTest, Normalize)
{
    {
        nn::util::Vector4f vector1(-3.f, 1.f, -2.f, 4.f);

        bool result = vector1.Normalize();
        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(vector1, -0.54772255750f, 0.182574185835055f, -0.36514837167011f, 0.73029674334022f, FloatError);
        EXPECT_TRUE(result);
    }

    {
        nn::util::Vector4f vector1(-3.f, 1.f, -2.f, 4.f);

        vector1 = nn::util::Vector4f::Normalize(vector1);
        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(vector1, -0.54772255750f, 0.182574185835055f, -0.36514837167011f, 0.73029674334022f, FloatError);
    }

    {
        nn::util::Vector4f vector1(0.f, 0.f, 0.f, 0.f);

        bool result = vector1.Normalize();
        NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 0.f, 0.f, 0.f, 0.f);
        EXPECT_FALSE(result);
    }

    {
        nn::util::Vector4f vector1(0.f, 0.f, 0.f, 0.f);

        vector1 = nn::util::Vector4f::Normalize(vector1);
        NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 0.f, 0.f, 0.f, 0.f);
    }
}

TEST_F(MathVector4ClassTest, Maximize)
{
    nn::util::Vector4f vector1(-3.f, 1.f, -2.f, 4.f);
    nn::util::Vector4f vector2(6.f, -2.f, 4.f, -8.f);

    vector1 = nn::util::Vector4f::Maximize(vector1, vector2);

    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 6.f, 1.f, 4.f, 4.f);
}

TEST_F(MathVector4ClassTest, Minimize)
{
    nn::util::Vector4f vector1(-3.f, 1.f, -2.f, 4.f);
    nn::util::Vector4f vector2(6.f, -2.f, 4.f, -8.f);

    vector1 = nn::util::Vector4f::Minimize(vector1, vector2);

    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, -3.f, -2.f, -2.f, -8.f);
}

TEST_F(MathVector4ClassTest, Lerp)
{
    nn::util::Vector4f vector1(-3.f, 1.f, -2.f, 4.f);
    nn::util::Vector4f vector2(6.f, -2.f, 4.f, -8.f);

    nn::util::Vector4f vector3;

    vector3 = nn::util::Vector4f::Lerp(vector1, vector2, 0.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector3, -3.f, 1.f, -2.f, 4.f);

    vector3 = nn::util::Vector4f::Lerp(vector1, vector2, 1.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector3, 6.f, -2.f, 4.f, -8.f);

    vector3 = nn::util::Vector4f::Lerp(vector1, vector2, 0.5f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector3, 1.5f, -0.5f, 1.f, -2.f);
}

TEST_F(MathVector4ClassTest, Transform)
{
    {
        nn::util::Vector4f vector1;
        nn::util::VectorSet(&vector1, 2.f, -4.f, 6.f, -8.f);

        nn::util::MatrixRowMajor4x4f matrix1(
            1.f, -2.f, 3.f, -4.f,
            -5.f, 6.f, -7.f, 8.f,
            9.f, -10.f, 11.f, -12.f,
            -13.f, 14.f, -15.f, 16.f);

        vector1 = nn::util::Vector4f::Transform(vector1, matrix1);

        NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 180.f, -200.f, 220.f, -240.f);
    }
}

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning( pop )
#endif
