﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nnt/nntest.h>

#include <nn/util/util_Arithmetic.h>
#include <nn/util/util_Constant.h>
#include <nn/util/util_MatrixApi.h>
#include <nn/util/util_VectorApi.h>
#include <nn/util/util_QuaternionApi.h>

#include "testUtil_Matrix.h"
#include "testUtil_Vector.h"

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
// do{}while(0) 文を使うと、C4127（定数条件式に対する警告）が発生するため、これを無効化
#pragma warning( push )
#pragma warning( disable:4127 )
#endif

namespace
{
    const float FloatError = 0.001f;
}

//
// Float
//

class MathFloatMatrixTest : public ::testing::Test
{
};

TEST_F(MathFloatMatrixTest, Initializer)
{
    {
        nn::util::FloatRowMajor3x2 f = NN_UTIL_FLOAT_ROW_MAJOR_3X2_INITIALIZER(
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);

        NNT_UTIL_FLOAT_ROW_MAJOR_3X2_EXPECT_EQ(f,
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);
    }

    {
        nn::util::FloatColumnMajor3x2 f = NN_UTIL_FLOAT_COLUMN_MAJOR_3X2_INITIALIZER(
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);

        NNT_UTIL_FLOAT_COLUMN_MAJOR_3X2_EXPECT_EQ(f,
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);
    }

    {
        nn::util::FloatRowMajor4x3 f = NN_UTIL_FLOAT_ROW_MAJOR_4X3_INITIALIZER(
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);

        NNT_UTIL_FLOAT_ROW_MAJOR_4X3_EXPECT_EQ(f,
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);
    }

    {
        nn::util::FloatColumnMajor4x3 f = NN_UTIL_FLOAT_COLUMN_MAJOR_4X3_INITIALIZER(
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);

        NNT_UTIL_FLOAT_COLUMN_MAJOR_4X3_EXPECT_EQ(f,
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);
    }

    {
        nn::util::FloatRowMajor4x4 f = NN_UTIL_FLOAT_ROW_MAJOR_4X4_INITIALIZER(
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);

        NNT_UTIL_FLOAT_ROW_MAJOR_4X4_EXPECT_EQ(f,
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);
    }

    {
        nn::util::FloatColumnMajor4x4 f = NN_UTIL_FLOAT_COLUMN_MAJOR_4X4_INITIALIZER(
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);

        NNT_UTIL_FLOAT_COLUMN_MAJOR_4X4_EXPECT_EQ(f,
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);
    }

    {
        nn::util::Float3x2 f = NN_UTIL_FLOAT_3X2_INITIALIZER(
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);

        NNT_UTIL_FLOAT_3X2_EXPECT_EQ(f,
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);
    }

    {
        nn::util::FloatT3x2 f = NN_UTIL_FLOAT_T3X2_INITIALIZER(
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);

        NNT_UTIL_FLOAT_T3X2_EXPECT_EQ(f,
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);
    }

    {
        nn::util::Float4x3 f = NN_UTIL_FLOAT_4X3_INITIALIZER(
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);

        NNT_UTIL_FLOAT_4X3_EXPECT_EQ(f,
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);
    }

    {
        nn::util::FloatT4x3 f = NN_UTIL_FLOAT_T4X3_INITIALIZER(
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);

        NNT_UTIL_FLOAT_T4X3_EXPECT_EQ(f,
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);
    }

    {
        nn::util::Float4x4 f = NN_UTIL_FLOAT_4X4_INITIALIZER(
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);

        NNT_UTIL_FLOAT_4X4_EXPECT_EQ(f,
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);
    }

    {
        nn::util::FloatT4x4 f = NN_UTIL_FLOAT_T4X4_INITIALIZER(
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);

        NNT_UTIL_FLOAT_T4X4_EXPECT_EQ(f,
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);
    }
} // NOLINT(impl/function_size)

//
// 行列
//

// 3x2 行列
template <typename T>
class MathMatrix3x2Test : public ::testing::Test
{
};

typedef ::testing::Types<nn::util::MatrixRowMajor3x2fType, nn::util::MatrixColumnMajor3x2fType> Matrix3x2Types;
TYPED_TEST_CASE(MathMatrix3x2Test, Matrix3x2Types);

TYPED_TEST(MathMatrix3x2Test, Set)
{
    TypeParam matrix;
    nn::util::MatrixSet(&matrix,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);
}

TYPED_TEST(MathMatrix3x2Test, SetRow)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    nn::util::Vector2fType vector1;

    nn::util::VectorSet(&vector1, -7.f, -8.f);
    nn::util::MatrixSetRow(&matrix1, 0, vector1);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -7.f, -8.f,
        3.f, 4.f,
        5.f, 6.f);

    nn::util::VectorSet(&vector1, -9.f, -10.f);
    nn::util::MatrixSetRow(&matrix1, 1, vector1);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -7.f, -8.f,
        -9.f, -10.f,
        5.f, 6.f);

    nn::util::VectorSet(&vector1, -11.f, -12.f);
    nn::util::MatrixSetRow(&matrix1, 2, vector1);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -7.f, -8.f,
        -9.f, -10.f,
        -11.f, -12.f);
}

TYPED_TEST(MathMatrix3x2Test, SetRows)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    nn::util::Vector2fType row0, row1, row2;

    nn::util::VectorSet(&row0, -7.f, -8.f);
    nn::util::VectorSet(&row1, -9.f, -10.f);
    nn::util::VectorSet(&row2, -11.f, -12.f);

    nn::util::MatrixSetRows(&matrix1, row0, row1, row2);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -7.f, -8.f,
        -9.f, -10.f,
        -11.f, -12.f);
}

TYPED_TEST(MathMatrix3x2Test, GetRow)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    nn::util::Vector2fType vector1;

    nn::util::MatrixGetRow(&vector1, matrix1, 0);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 1.f, 2.f);

    nn::util::MatrixGetRow(&vector1, matrix1, 1);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 3.f, 4.f);

    nn::util::MatrixGetRow(&vector1, matrix1, 2);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 5.f, 6.f);
}

TYPED_TEST(MathMatrix3x2Test, GetRows)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    nn::util::Vector2fType row0, row1, row2;
    nn::util::MatrixGetRows(&row0, &row1, &row2, matrix1);

    NNT_UTIL_VECTOR2_EXPECT_EQ(row0, 1.f, 2.f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(row1, 3.f, 4.f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(row2, 5.f, 6.f);
}

TYPED_TEST(MathMatrix3x2Test, SetColumn)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    nn::util::Vector3fType vector1;

    nn::util::VectorSet(&vector1, -7.f, -9.f, -11.f);
    nn::util::MatrixSetColumn(&matrix1, 0, vector1);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -7.f, 2.f,
        -9.f, 4.f,
        -11.f, 6.f);

    nn::util::VectorSet(&vector1, -8.f, -10.f, -12.f);
    nn::util::MatrixSetColumn(&matrix1, 1, vector1);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -7.f, -8.f,
        -9.f, -10.f,
        -11.f, -12.f);
}

TYPED_TEST(MathMatrix3x2Test, SetColumns)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    nn::util::Vector3fType column0, column1;

    nn::util::VectorSet(&column0, -7.f, -9.f, -11.f);
    nn::util::VectorSet(&column1, -8.f, -10.f, -12.f);

    nn::util::MatrixSetColumns(&matrix1, column0, column1);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -7.f, -8.f,
        -9.f, -10.f,
        -11.f, -12.f);
}

TYPED_TEST(MathMatrix3x2Test, GetColumn)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    nn::util::Vector3fType vector1;

    nn::util::MatrixGetColumn(&vector1, matrix1, 0);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 1.f, 3.f, 5.f);

    nn::util::MatrixGetColumn(&vector1, matrix1, 1);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 2.f, 4.f, 6.f);
}

TYPED_TEST(MathMatrix3x2Test, GetColumns)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    nn::util::Vector3fType column0, column1;
    nn::util::MatrixGetColumns(&column0, &column1, matrix1);

    NNT_UTIL_VECTOR3_EXPECT_EQ(column0, 1.f, 3.f, 5.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(column1, 2.f, 4.f, 6.f);
}

TYPED_TEST(MathMatrix3x2Test, SetAxis)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    nn::util::Vector2fType vector1;

    nn::util::VectorSet(&vector1, -7.f, -8.f);
    nn::util::MatrixSetAxisX(&matrix1, vector1);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -7.f, -8.f,
        3.f, 4.f,
        5.f, 6.f);

    nn::util::VectorSet(&vector1, -9.f, -10.f);
    nn::util::MatrixSetAxisY(&matrix1, vector1);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -7.f, -8.f,
        -9.f, -10.f,
        5.f, 6.f);

    nn::util::VectorSet(&vector1, -11.f, -12.f);
    nn::util::MatrixSetAxisZ(&matrix1, vector1);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -7.f, -8.f,
        -9.f, -10.f,
        -11.f, -12.f);
}

TYPED_TEST(MathMatrix3x2Test, SetAxes)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    nn::util::Vector2fType axisX, axisY, axisZ;

    nn::util::VectorSet(&axisX, -7.f, -8.f);
    nn::util::VectorSet(&axisY, -9.f, -10.f);
    nn::util::VectorSet(&axisZ, -11.f, -12.f);

    nn::util::MatrixSetAxes(&matrix1, axisX, axisY, axisZ);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -7.f, -8.f,
        -9.f, -10.f,
        -11.f, -12.f);
}

TYPED_TEST(MathMatrix3x2Test, GetAxis)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    nn::util::Vector2fType vector1;

    nn::util::MatrixGetAxisX(&vector1, matrix1);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 1.f, 2.f);

    nn::util::MatrixGetAxisY(&vector1, matrix1);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 3.f, 4.f);

    nn::util::MatrixGetAxisZ(&vector1, matrix1);
    NNT_UTIL_VECTOR2_EXPECT_EQ(vector1, 5.f, 6.f);
}

TYPED_TEST(MathMatrix3x2Test, GetAxes)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    nn::util::Vector2fType axisX, axisY, axisZ;
    nn::util::MatrixGetAxes(&axisX, &axisY, &axisZ, matrix1);

    NNT_UTIL_VECTOR2_EXPECT_EQ(axisX, 1.f, 2.f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(axisY, 3.f, 4.f);
    NNT_UTIL_VECTOR2_EXPECT_EQ(axisZ, 5.f, 6.f);
}

TYPED_TEST(MathMatrix3x2Test, Load)
{
    {
        TypeParam matrix;
        nn::util::FloatRowMajor3x2 value = NN_UTIL_FLOAT_ROW_MAJOR_3X2_INITIALIZER
        (
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f
        );

        nn::util::MatrixLoad(&matrix, value);

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix,
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);
    }

    {
        TypeParam matrix;
        nn::util::FloatColumnMajor3x2 value = NN_UTIL_FLOAT_COLUMN_MAJOR_3X2_INITIALIZER
        (
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f
        );

        nn::util::MatrixLoad(&matrix, value);

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix,
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);
    }
}

TYPED_TEST(MathMatrix3x2Test, Store)
{
    TypeParam matrix;
    nn::util::MatrixSet(&matrix,
        1.f, 2.f,
        3.f, 4.f,
        5.f, 6.f);

    {
        nn::util::FloatRowMajor3x2 value;
        nn::util::MatrixStore(&value, matrix);

        EXPECT_EQ(1.f, value.m[0][0]);
        EXPECT_EQ(2.f, value.m[0][1]);

        EXPECT_EQ(3.f, value.m[1][0]);
        EXPECT_EQ(4.f, value.m[1][1]);

        EXPECT_EQ(5.f, value.m[2][0]);
        EXPECT_EQ(6.f, value.m[2][1]);
    }

    {
        nn::util::FloatColumnMajor3x2 value;
        nn::util::MatrixStore(&value, matrix);

        EXPECT_EQ(1.f, value.m[0][0]);
        EXPECT_EQ(2.f, value.m[1][0]);

        EXPECT_EQ(3.f, value.m[0][1]);
        EXPECT_EQ(4.f, value.m[1][1]);

        EXPECT_EQ(5.f, value.m[0][2]);
        EXPECT_EQ(6.f, value.m[1][2]);
    }
}

TYPED_TEST(MathMatrix3x2Test, Zero)
{
    TypeParam matrix;
    nn::util::MatrixSet(&matrix,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    nn::util::MatrixZero(&matrix);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix,
        0.f, 0.f,
        0.f, 0.f,
        0.f, 0.f);
}

TYPED_TEST(MathMatrix3x2Test, Identity)
{
    TypeParam matrix;
    nn::util::MatrixSet(&matrix,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    nn::util::MatrixIdentity(&matrix);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix,
        1.f, 0.f,
        0.f, 1.f,
        0.f, 0.f);
}

TYPED_TEST(MathMatrix3x2Test, Add)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    TypeParam matrix2;
    nn::util::MatrixSet(&matrix2,
        2.f, -4.f,
        6.f, 8.f,
        -10.f, 12.f);

    nn::util::MatrixAdd(&matrix1, matrix1, matrix2);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        3.f, -6.f,
        3.f, 12.f,
        -5.f, 6.f);
}

TYPED_TEST(MathMatrix3x2Test, Subtract)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    TypeParam matrix2;
    nn::util::MatrixSet(&matrix2,
        2.f, -4.f,
        6.f, 8.f,
        -10.f, 12.f);

    nn::util::MatrixSubtract(&matrix1, matrix1, matrix2);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -1.f, 2.f,
        -9.f, -4.f,
        15.f, -18.f);
}

TYPED_TEST(MathMatrix3x2Test, Multiply)
{
    {
        TypeParam matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        nn::util::MatrixMultiply(&matrix1, matrix1, -1.f / 2.f);

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            -0.5f, 1.f,
            1.5f, -2.f,
            -2.5f, 3.f);
    }

    {
        TypeParam matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        TypeParam matrix2;
        nn::util::MatrixSet(&matrix2,
            2.f, -4.f,
            6.f, 8.f,
            -10.f, 12.f);

        nn::util::MatrixMultiply(&matrix1, matrix1, matrix2);

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            -10.f, -20.f,
            18.f, 44.f,
            -36.f, -56.f);
    }
}

TYPED_TEST(MathMatrix3x2Test, MultiplyAdd)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    TypeParam matrix2;
    nn::util::MatrixSet(&matrix2,
        2.f, -4.f,
        6.f, 8.f,
        -10.f, 12.f);

    nn::util::MatrixMultiplyAdd(&matrix1, 3.f, matrix1, matrix2);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        5.f, -10.f,
        -3.f, 20.f,
        5.f, -6.f);
}

TYPED_TEST(MathMatrix3x2Test, Divide)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    nn::util::MatrixDivide(&matrix1, matrix1, -2.f);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -0.5f, 1.f,
        1.5f, -2.f,
        -2.5f, 3.f);
}

TYPED_TEST(MathMatrix3x2Test, Inverse)
{
    {
        TypeParam matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f,
            -4.f, 8.f,
            5.f, -6.f);

        EXPECT_FALSE(nn::util::MatrixInverse(&matrix1, matrix1));

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            0.f, 0.f,
            0.f, 0.f,
            0.f, 0.f);
    }

    {
        TypeParam matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        EXPECT_TRUE(nn::util::MatrixInverse(&matrix1, matrix1));

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            -2.f, -1.f,
            -1.5f, -0.5f,
            1.f, 2.f);
    }
}

TYPED_TEST(MathMatrix3x2Test, Transpose)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    nn::util::MatrixTranspose(&matrix1, matrix1);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        1.f, -3.f,
        -2.f, 4.f,
        0.f, 0.f);
}

TYPED_TEST(MathMatrix3x2Test, InverseTranspose)
{
    {
        TypeParam matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f,
            -4.f, 8.f,
            5.f, -6.f);

        EXPECT_FALSE(nn::util::MatrixInverseTranspose(&matrix1, matrix1));

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            0.f, 0.f,
            0.f, 0.f,
            0.f, 0.f);
    }

    {
        TypeParam matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f,
            -3.f, 4.f,
            5.f, -6.f);

        EXPECT_TRUE(nn::util::MatrixInverseTranspose(&matrix1, matrix1));

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
            -2.f, -1.5f,
            -1.f, -0.5f,
            0.f, 0.f);
    }
}

TYPED_TEST(MathMatrix3x2Test, SetTranslate)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    nn::util::Vector2fType translate;
    nn::util::VectorSet(&translate, -3.f, 1.f);

    nn::util::MatrixSetTranslate(&matrix1, translate);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        -3.f, 1.f);
}

TYPED_TEST(MathMatrix3x2Test, SetScale)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    nn::util::Vector2fType scale;
    nn::util::VectorSet(&scale, -3.f, 1.f);

    nn::util::MatrixSetScale(&matrix1, scale);

    NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix1,
        -3.f, 0.f,
        0.f, 1.f,
        5.f, -6.f);
}

TYPED_TEST(MathMatrix3x2Test, MatrixSetRotate)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    float rotation = 40.f / 180.f * nn::util::FloatPi;

    nn::util::MatrixSetRotate(&matrix1, rotation);

    NNT_UTIL_MATRIX_3X2_EXPECT_NEARLY_EQ(matrix1,
        0.76604444311f, 0.64278760968f,
        -0.64278760968f, 0.76604444311f,
        5.f, -6.f,
        FloatError);
}

TYPED_TEST(MathMatrix3x2Test, MatrixSetScaleRotate)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f,
        -3.f, 4.f,
        5.f, -6.f);

    nn::util::Vector2fType scale;
    nn::util::VectorSet(&scale, -3.f, 2.f);

    float rotation = 40.f / 180.f * nn::util::FloatPi;

    nn::util::MatrixSetScaleRotate(&matrix1, scale, rotation);

    NNT_UTIL_MATRIX_3X2_EXPECT_NEARLY_EQ(matrix1,
        -2.298133329f, -1.928362829f,
        -1.285575219f, 1.532088886f,
        5.f, -6.f,
        FloatError);
}

// 4x3 行列
template <typename T>
class MathMatrix4x3Test : public ::testing::Test
{
};

typedef ::testing::Types<nn::util::MatrixRowMajor4x3fType, nn::util::MatrixColumnMajor4x3fType> Matrix4x3Types;
TYPED_TEST_CASE(MathMatrix4x3Test, Matrix4x3Types);

TYPED_TEST(MathMatrix4x3Test, Set)
{
    TypeParam matrix;
    nn::util::MatrixSet(&matrix,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);
}

TYPED_TEST(MathMatrix4x3Test, SetRow)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    nn::util::Vector3fType vector1;

    nn::util::VectorSet(&vector1, -13.f, -14.f, -15.f);
    nn::util::MatrixSetRow(&matrix1, 0, vector1);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
        -13.f, -14.f, -15.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    nn::util::VectorSet(&vector1, -16.f, -17.f, -18.f);
    nn::util::MatrixSetRow(&matrix1, 1, vector1);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
        -13.f, -14.f, -15.f,
        -16.f, -17.f, -18.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    nn::util::VectorSet(&vector1, -19.f, -20.f, -21.f);
    nn::util::MatrixSetRow(&matrix1, 2, vector1);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
        -13.f, -14.f, -15.f,
        -16.f, -17.f, -18.f,
        -19.f, -20.f, -21.f,
        10.f, 11.f, 12.f);

    nn::util::VectorSet(&vector1, -22.f, -23.f, -24.f);
    nn::util::MatrixSetRow(&matrix1, 3, vector1);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
        -13.f, -14.f, -15.f,
        -16.f, -17.f, -18.f,
        -19.f, -20.f, -21.f,
        -22.f, -23.f, -24.f);
}

TYPED_TEST(MathMatrix4x3Test, SetRows)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    nn::util::Vector3fType row0, row1, row2, row3;

    nn::util::VectorSet(&row0, -13.f, -14.f, -15.f);
    nn::util::VectorSet(&row1, -16.f, -17.f, -18.f);
    nn::util::VectorSet(&row2, -19.f, -20.f, -21.f);
    nn::util::VectorSet(&row3, -22.f, -23.f, -24.f);

    nn::util::MatrixSetRows(&matrix1, row0, row1, row2, row3);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
        -13.f, -14.f, -15.f,
        -16.f, -17.f, -18.f,
        -19.f, -20.f, -21.f,
        -22.f, -23.f, -24.f);
}

TYPED_TEST(MathMatrix4x3Test, GetRow)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    nn::util::Vector3fType vector1;

    nn::util::MatrixGetRow(&vector1, matrix1, 0);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 1.f, 2.f, 3.f);

    nn::util::MatrixGetRow(&vector1, matrix1, 1);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 4.f, 5.f, 6.f);

    nn::util::MatrixGetRow(&vector1, matrix1, 2);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 7.f, 8.f, 9.f);

    nn::util::MatrixGetRow(&vector1, matrix1, 3);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 10.f, 11.f, 12.f);
}

TYPED_TEST(MathMatrix4x3Test, GetRows)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    nn::util::Vector3fType row0, row1, row2, row3;
    nn::util::MatrixGetRows(&row0, &row1, &row2, &row3, matrix1);

    NNT_UTIL_VECTOR3_EXPECT_EQ(row0, 1.f, 2.f, 3.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(row1, 4.f, 5.f, 6.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(row2, 7.f, 8.f, 9.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(row3, 10.f, 11.f, 12.f);
}

TYPED_TEST(MathMatrix4x3Test, SetColumn)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    nn::util::Vector4fType vector1;

    nn::util::VectorSet(&vector1, -13.f, -16.f, -19.f, -22.f);
    nn::util::MatrixSetColumn(&matrix1, 0, vector1);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
        -13.f, 2.f, 3.f,
        -16.f, 5.f, 6.f,
        -19.f, 8.f, 9.f,
        -22.f, 11.f, 12.f);

    nn::util::VectorSet(&vector1, -14.f, -17.f, -20.f, -23.f);
    nn::util::MatrixSetColumn(&matrix1, 1, vector1);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
        -13.f, -14.f, 3.f,
        -16.f, -17.f, 6.f,
        -19.f, -20.f, 9.f,
        -22.f, -23.f, 12.f);

    nn::util::VectorSet(&vector1, -15.f, -18.f, -21.f, -24.f);
    nn::util::MatrixSetColumn(&matrix1, 2, vector1);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
        -13.f, -14.f, -15.f,
        -16.f, -17.f, -18.f,
        -19.f, -20.f, -21.f,
        -22.f, -23.f, -24.f);
}

TYPED_TEST(MathMatrix4x3Test, SetColumns)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    nn::util::Vector4fType column0, column1, column2;

    nn::util::VectorSet(&column0, -13.f, -16.f, -19.f, -22.f);
    nn::util::VectorSet(&column1, -14.f, -17.f, -20.f, -23.f);
    nn::util::VectorSet(&column2, -15.f, -18.f, -21.f, -24.f);

    nn::util::MatrixSetColumns(&matrix1, column0, column1, column2);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
        -13.f, -14.f, -15.f,
        -16.f, -17.f, -18.f,
        -19.f, -20.f, -21.f,
        -22.f, -23.f, -24.f);
}

TYPED_TEST(MathMatrix4x3Test, GetColumn)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    nn::util::Vector4fType vector1;

    nn::util::MatrixGetColumn(&vector1, matrix1, 0);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 1.f, 4.f, 7.f, 10.f);

    nn::util::MatrixGetColumn(&vector1, matrix1, 1);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 2.f, 5.f, 8.f, 11.f);

    nn::util::MatrixGetColumn(&vector1, matrix1, 2);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 3.f, 6.f, 9.f, 12.f);
}

TYPED_TEST(MathMatrix4x3Test, GetColumns)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    nn::util::Vector4fType column0, column1, column2;
    nn::util::MatrixGetColumns(&column0, &column1, &column2, matrix1);

    NNT_UTIL_VECTOR4_EXPECT_EQ(column0, 1.f, 4.f, 7.f, 10.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(column1, 2.f, 5.f, 8.f, 11.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(column2, 3.f, 6.f, 9.f, 12.f);
}

TYPED_TEST(MathMatrix4x3Test, SetAxis)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    nn::util::Vector3fType vector1;

    nn::util::VectorSet(&vector1, -13.f, -14.f, -15.f);
    nn::util::MatrixSetAxisX(&matrix1, vector1);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -13.f, -14.f, -15.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);

    nn::util::VectorSet(&vector1, -16.f, -17.f, -18.f);
    nn::util::MatrixSetAxisY(&matrix1, vector1);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -13.f, -14.f, -15.f,
            -16.f, -17.f, -18.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);

    nn::util::VectorSet(&vector1, -19.f, -20.f, -21.f);
    nn::util::MatrixSetAxisZ(&matrix1, vector1);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -13.f, -14.f, -15.f,
            -16.f, -17.f, -18.f,
            -19.f, -20.f, -21.f,
            10.f, 11.f, 12.f);

    nn::util::VectorSet(&vector1, -22.f, -23.f, -24.f);
    nn::util::MatrixSetAxisW(&matrix1, vector1);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -13.f, -14.f, -15.f,
            -16.f, -17.f, -18.f,
            -19.f, -20.f, -21.f,
            -22.f, -23.f, -24.f);
}

TYPED_TEST(MathMatrix4x3Test, SetAxes)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    nn::util::Vector3fType axisX, axisY, axisZ, axisW;

    nn::util::VectorSet(&axisX, -13.f, -14.f, -15.f);
    nn::util::VectorSet(&axisY, -16.f, -17.f, -18.f);
    nn::util::VectorSet(&axisZ, -19.f, -20.f, -21.f);
    nn::util::VectorSet(&axisW, -22.f, -23.f, -24.f);

    nn::util::MatrixSetAxes(&matrix1, axisX, axisY, axisZ, axisW);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -13.f, -14.f, -15.f,
            -16.f, -17.f, -18.f,
            -19.f, -20.f, -21.f,
            -22.f, -23.f, -24.f);
}

TYPED_TEST(MathMatrix4x3Test, GetAxis)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    nn::util::Vector3fType vector1;

    nn::util::MatrixGetAxisX(&vector1, matrix1);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 1.f, 2.f, 3.f);

    nn::util::MatrixGetAxisY(&vector1, matrix1);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 4.f, 5.f, 6.f);

    nn::util::MatrixGetAxisZ(&vector1, matrix1);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 7.f, 8.f, 9.f);

    nn::util::MatrixGetAxisW(&vector1, matrix1);
    NNT_UTIL_VECTOR3_EXPECT_EQ(vector1, 10.f, 11.f, 12.f);
}

TYPED_TEST(MathMatrix4x3Test, GetAxes)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    nn::util::Vector3fType axisX, axisY, axisZ, axisW;
    nn::util::MatrixGetAxes(&axisX, &axisY, &axisZ, &axisW, matrix1);

    NNT_UTIL_VECTOR3_EXPECT_EQ(axisX, 1.f, 2.f, 3.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(axisY, 4.f, 5.f, 6.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(axisZ, 7.f, 8.f, 9.f);
    NNT_UTIL_VECTOR3_EXPECT_EQ(axisW, 10.f, 11.f, 12.f);
}

TYPED_TEST(MathMatrix4x3Test, Load)
{
    {
        TypeParam matrix;
        nn::util::FloatRowMajor4x3 value = NN_UTIL_FLOAT_ROW_MAJOR_4X3_INITIALIZER
        (
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f
        );

        nn::util::MatrixLoad(&matrix, value);

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);
    }

    {
        TypeParam matrix;
        nn::util::FloatColumnMajor4x3 value = NN_UTIL_FLOAT_COLUMN_MAJOR_4X3_INITIALIZER
        (
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f
        );

        nn::util::MatrixLoad(&matrix, value);

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);
    }
}

TYPED_TEST(MathMatrix4x3Test, Store)
{
    TypeParam matrix;
    nn::util::MatrixSet(&matrix,
        1.f, 2.f, 3.f,
        4.f, 5.f, 6.f,
        7.f, 8.f, 9.f,
        10.f, 11.f, 12.f);

    {
        nn::util::FloatRowMajor4x3 value;
        nn::util::MatrixStore(&value, matrix);

        EXPECT_EQ(1.f, value.m[0][0]);
        EXPECT_EQ(2.f, value.m[0][1]);
        EXPECT_EQ(3.f, value.m[0][2]);

        EXPECT_EQ(4.f, value.m[1][0]);
        EXPECT_EQ(5.f, value.m[1][1]);
        EXPECT_EQ(6.f, value.m[1][2]);

        EXPECT_EQ(7.f, value.m[2][0]);
        EXPECT_EQ(8.f, value.m[2][1]);
        EXPECT_EQ(9.f, value.m[2][2]);

        EXPECT_EQ(10.f, value.m[3][0]);
        EXPECT_EQ(11.f, value.m[3][1]);
        EXPECT_EQ(12.f, value.m[3][2]);
    }

    {
        nn::util::FloatColumnMajor4x3 value;
        nn::util::MatrixStore(&value, matrix);

        EXPECT_EQ(1.f, value.m[0][0]);
        EXPECT_EQ(2.f, value.m[1][0]);
        EXPECT_EQ(3.f, value.m[2][0]);

        EXPECT_EQ(4.f, value.m[0][1]);
        EXPECT_EQ(5.f, value.m[1][1]);
        EXPECT_EQ(6.f, value.m[2][1]);

        EXPECT_EQ(7.f, value.m[0][2]);
        EXPECT_EQ(8.f, value.m[1][2]);
        EXPECT_EQ(9.f, value.m[2][2]);

        EXPECT_EQ(10.f, value.m[0][3]);
        EXPECT_EQ(11.f, value.m[1][3]);
        EXPECT_EQ(12.f, value.m[2][3]);
    }
}

TYPED_TEST(MathMatrix4x3Test, Zero)
{
    TypeParam matrix;
    nn::util::MatrixSet(&matrix,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    nn::util::MatrixZero(&matrix);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix,
        0.f, 0.f, 0.f,
        0.f, 0.f, 0.f,
        0.f, 0.f, 0.f,
        0.f, 0.f, 0.f);
}

TYPED_TEST(MathMatrix4x3Test, Identity)
{
    TypeParam matrix;
    nn::util::MatrixSet(&matrix,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    nn::util::MatrixIdentity(&matrix);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix,
        1.f, 0.f, 0.f,
        0.f, 1.f, 0.f,
        0.f, 0.f, 1.f,
        0.f, 0.f, 0.f);
}

TYPED_TEST(MathMatrix4x3Test, Add)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    TypeParam matrix2;
    nn::util::MatrixSet(&matrix2,
        2.f, -4.f, 6.f,
        8.f, -10.f, 12.f,
        -14.f, 16.f, -18.f,
        -20.f, 22.f, -24.f);

    nn::util::MatrixAdd(&matrix1, matrix1, matrix2);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
        3.f, -6.f, 9.f,
        4.f, -5.f, 6.f,
        -7.f, 8.f, -9.f,
        -30.f, 33.f, -36.f);
}

TYPED_TEST(MathMatrix4x3Test, Subtract)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    TypeParam matrix2;
    nn::util::MatrixSet(&matrix2,
        2.f, -4.f, 6.f,
        8.f, -10.f, 12.f,
        -14.f, 16.f, -18.f,
        -20.f, 22.f, -24.f);

    nn::util::MatrixSubtract(&matrix1, matrix1, matrix2);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
        -1.f, 2.f, -3.f,
        -12.f, 15.f, -18.f,
        21.f, -24.f, 27.f,
        10.f, -11.f, 12.f);
}

TYPED_TEST(MathMatrix4x3Test, Multiply)
{
    {
        TypeParam matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        nn::util::MatrixMultiply(&matrix1, matrix1, -1.f / 2.f);

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -0.5f, 1.f, -1.5f,
            2.f, -2.5f, 3.f,
            -3.5f, 4.f, -4.5f,
            5.f, -5.5f, 6.f);
    }

    {
        TypeParam matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        TypeParam matrix2;
        nn::util::MatrixSet(&matrix2,
            2.f, -4.f, 6.f,
            8.f, -10.f, 12.f,
            -14.f, 16.f, -18.f,
            -20.f, 22.f, -24.f);

        nn::util::MatrixMultiply(&matrix1, matrix1, matrix2);

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            -56.f, 64.f, -72.f,
            116.f, -130, 144.f,
            -176.f, 196.f, -216.f,
            216.f, -240.f, 264.f);
    }
}

TYPED_TEST(MathMatrix4x3Test, MultiplyAdd)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    TypeParam matrix2;
    nn::util::MatrixSet(&matrix2,
        2.f, -4.f, 6.f,
        8.f, -10.f, 12.f,
        -14.f, 16.f, -18.f,
        -20.f, 22.f, -24.f);

    nn::util::MatrixMultiplyAdd(&matrix1, 3.f, matrix1, matrix2);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
        5.f, -10.f, 15.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -50.f, 55.f, -60.f);
}

TYPED_TEST(MathMatrix4x3Test, Divide)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    nn::util::MatrixDivide(&matrix1, matrix1, -2.f);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
        -0.5f, 1.f, -1.5f,
        2.f, -2.5f, 3.f,
        -3.5f, 4.f, -4.5f,
        5.f, -5.5f, 6.f);
}

TYPED_TEST(MathMatrix4x3Test, Inverse)
{
    {
        TypeParam matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        EXPECT_FALSE(nn::util::MatrixInverse(&matrix1, matrix1));

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            0.f, 0.f, 0.f,
            0.f, 0.f, 0.f,
            0.f, 0.f, 0.f,
            0.f, 0.f, 0.f);
    }

    {
        TypeParam matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, 0.f, 1.f,
            -1.f, 1.f, 1.f,
            1.f, -1.f, 0.f,
            1.f, 1.f, -1.f);

        EXPECT_TRUE(nn::util::MatrixInverse(&matrix1, matrix1));

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            1.f, -1.f, -1.f,
            1.f, -1.f, -2.f,
            0.f, 1.f, 1.f,
            -2.f, 3.f, 4.f);
    }
}

TYPED_TEST(MathMatrix4x3Test, Transpose)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    nn::util::MatrixTranspose(&matrix1, matrix1);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            1.f, -4.f, 7.f,
            -2.f, 5.f, -8.f,
            3.f, -6.f, 9.f,
            0.f, 0.f, 0.f);
}

TYPED_TEST(MathMatrix4x3Test, InverseTranspose)
{
    {
        TypeParam matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        EXPECT_FALSE(nn::util::MatrixInverseTranspose(&matrix1, matrix1));

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            0.f, 0.f, 0.f,
            0.f, 0.f, 0.f,
            0.f, 0.f, 0.f,
            0.f, 0.f, 0.f);
    }

    {
        TypeParam matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, 0.f, 1.f,
            -1.f, 1.f, 1.f,
            1.f, -1.f, 0.f,
            1.f, 1.f, -1.f);

        EXPECT_TRUE(nn::util::MatrixInverseTranspose(&matrix1, matrix1));

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            1.f, 1.f, 0.f,
            -1.f, -1.f, 1.f,
            -1.f, -2.f, 1.f,
            0.f, 0.f, 0.f);
    }
}

TYPED_TEST(MathMatrix4x3Test, LookAtRightHanded)
{
    TypeParam matrix1;

    nn::util::Vector3fType position, up, target;
    nn::util::VectorSet(&position, 2.f, -2.f, 4.f);
    nn::util::VectorSet(&up, 0.f, 1.f, 0.f);
    nn::util::VectorSet(&target, 0.f, 0.f, 0.f);

    nn::util::MatrixLookAtRightHanded(&matrix1, position, target, up);

    NNT_UTIL_MATRIX_4X3_EXPECT_NEARLY_EQ(matrix1,
        0.8944271802902222f, 0.1825741827487946f, 0.4082482755184174f,
        0.0000000000000000f, 0.9128708839416504f, -0.4082482755184174f,
        -0.4472135901451111f, 0.3651483654975891f, 0.8164965510368347f,
        -0.0000000000000000f, -0.0000000000000000f, -4.8989791870117187f,
        FloatError);
}

TYPED_TEST(MathMatrix4x3Test, LookAtRightHandedWithTwist)
{
    {
        TypeParam matrix1;

        nn::util::Vector3fType position, target;
        nn::util::VectorSet(&position, 2.f, -2.f, 4.f);
        nn::util::VectorSet(&target, 0.f, 0.f, 0.f);

        nn::util::MatrixLookAtRightHanded(&matrix1, position, target, 60.f / 180.f * nn::util::FloatPi);

        NNT_UTIL_MATRIX_4X3_EXPECT_NEARLY_EQ(matrix1,
            0.6052868366241455f, -0.6832639575004578f, 0.4082483053207398f,
            0.7905166149139404f, 0.4564047753810883f, -0.4082483053207398f,
            0.0926148891448975f, 0.5698343515396118f, 0.8164966106414795f,
            -0.0000000000000000f, -0.0000000000000000f, -4.8989796638488770f,
            FloatError);
    }

    {
        TypeParam matrix1;

        nn::util::Vector3fType position, target;
        nn::util::VectorSet(&position, 2.f, -2.f, 4.f);
        nn::util::VectorSet(&target, 2.f, 0.f, 4.f);

        nn::util::MatrixLookAtRightHanded(&matrix1, position, target, 60.f / 180.f * nn::util::FloatPi);

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            1.f, 0.f, 0.f,
            0.f, 0.f, -1.f,
            0.f, 1.f, 0.f,
            -2.f, -4.f, -2.f);
    }

    {
        TypeParam matrix1;

        nn::util::Vector3fType position, target;
        nn::util::VectorSet(&position, 2.f, -2.f, 4.f);
        nn::util::VectorSet(&target, 2.f, -3.f, 4.f);

        nn::util::MatrixLookAtRightHanded(&matrix1, position, target, 60.f / 180.f * nn::util::FloatPi);

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
            1.f, 0.f, 0.f,
            0.f, 0.f, 1.f,
            0.f, -1.f, 0.f,
            -2.f, 4.f, 2.f);
    }
}

TYPED_TEST(MathMatrix4x3Test, CameraRotate)
{
    TypeParam matrix1;

    nn::util::Vector3fType position, rotation;
    nn::util::VectorSet(&position, -3.f, 1.f, -2.f);
    nn::util::VectorSet(&rotation, 30.f / 180.f * nn::util::FloatPi, 40.f / 180.f * nn::util::FloatPi, 50.f / 180.f * nn::util::FloatPi);

    nn::util::MatrixCameraRotateZxy(&matrix1, position, rotation);

    NNT_UTIL_MATRIX_4X3_EXPECT_NEARLY_EQ(matrix1,
        0.7384793758392334f, -0.3801935613155365f, 0.5565917491912842f,
        0.6633203029632568f, 0.5565916895866394f, -0.4999663829803467f,
        -0.1197656691074372f, 0.7384793758392334f, 0.6633203029632568f,
        1.3125865459442138f, -0.2202136516571045f, 3.4963822364807129f,
        FloatError);
}

TYPED_TEST(MathMatrix4x3Test, SetTranslate)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    nn::util::Vector3fType translate;
    nn::util::VectorSet(&translate, -3.f, 1.f, -2.f);

    nn::util::MatrixSetTranslate(&matrix1, translate);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -3.f, 1.f, -2.f);
}

TYPED_TEST(MathMatrix4x3Test, SetScale)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    nn::util::Vector3fType scale;
    nn::util::VectorSet(&scale, -3.f, 1.f, -2.f);

    nn::util::MatrixSetScale(&matrix1, scale);

    NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix1,
        -3.f, 0.f, 0.f,
        0.f, 1.f, 0.f,
        0.f, 0.f, -2.f,
        -10.f, 11.f, -12.f);
}

TYPED_TEST(MathMatrix4x3Test, MatrixSetRotate)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    nn::util::Vector4fType quaternion1;
    nn::util::VectorSet(&quaternion1, -3.f, 1.f, -2.f, 4.f);
    nn::util::QuaternionNormalize(&quaternion1, quaternion1);

    nn::util::MatrixSetRotate(&matrix1, quaternion1);

    NNT_UTIL_MATRIX_4X3_EXPECT_NEARLY_EQ(matrix1,
        0.6666666269302368f, -0.7333333492279053f, 0.1333333551883698f,
        0.3333333432674408f, 0.1333333253860474f, -0.9333333969116210f,
        0.6666667461395264f, 0.6666667461395264f, 0.3333333134651184f,
        -10.f, 11.f, -12.f,
        FloatError);
}

TYPED_TEST(MathMatrix4x3Test, MatrixSetRotateXyz)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    nn::util::Vector3fType rotation;
    nn::util::VectorSet(&rotation,
        30.f / 180.f * nn::util::FloatPi,
        40.f / 180.f * nn::util::FloatPi,
        50.f / 180.f * nn::util::FloatPi);

    nn::util::MatrixSetRotateXyz(&matrix1, rotation);

    NNT_UTIL_MATRIX_4X3_EXPECT_NEARLY_EQ(matrix1,
        0.4923306107521058f, 0.5867368578910828f, -0.6427397727966309f,
        -0.4567770063877106f, 0.8027404546737671f, 0.3829680085182190f,
        0.7407116293907166f, 0.1049940884113312f, 0.6633203029632568f,
        -10.f, 11.f, -12.f,
        FloatError);
}

TYPED_TEST(MathMatrix4x3Test, MatrixSetScaleRotate)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    nn::util::Vector3fType scale;
    nn::util::VectorSet(&scale, -3.f, 1.f, -2.f);

    nn::util::Vector4fType quaternion1;
    nn::util::VectorSet(&quaternion1, -3.f, 1.f, -2.f, 4.f);
    nn::util::QuaternionNormalize(&quaternion1, quaternion1);

    nn::util::MatrixSetScaleRotate(&matrix1, scale, quaternion1);

    NNT_UTIL_MATRIX_4X3_EXPECT_NEARLY_EQ(matrix1,
        0.6666666269302368f * -3.f, -0.7333333492279053f * -3.f, 0.1333333551883698f * -3.f,
        0.3333333432674408f, 0.1333333253860474f, -0.9333333969116210f,
        0.6666667461395264f * -2.f, 0.6666667461395264f * -2.f, 0.3333333134651184f * -2.f,
        -10.f, 11.f, -12.f,
        FloatError);
}

TYPED_TEST(MathMatrix4x3Test, MatrixSetScaleRotateXyz)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f, 3.f,
        -4.f, 5.f, -6.f,
        7.f, -8.f, 9.f,
        -10.f, 11.f, -12.f);

    nn::util::Vector3fType scale;
    nn::util::VectorSet(&scale, -3.f, 1.f, -2.f);

    nn::util::Vector3fType rotation;
    nn::util::VectorSet(&rotation,
        30.f / 180.f * nn::util::FloatPi,
        40.f / 180.f * nn::util::FloatPi,
        50.f / 180.f * nn::util::FloatPi);

    nn::util::MatrixSetScaleRotateXyz(&matrix1, scale, rotation);

    NNT_UTIL_MATRIX_4X3_EXPECT_NEARLY_EQ(matrix1,
        0.4923306107521058f * -3.f, 0.5867368578910828f * -3.f, -0.6427397727966309f * -3.f,
        -0.4567770063877106f, 0.8027404546737671f, 0.3829680085182190f,
        0.7407116293907166f * -2.f, 0.1049940884113312f * -2.f, 0.6633203029632568f * -2.f,
        -10.f, 11.f, -12.f,
        FloatError);
}

TYPED_TEST(MathMatrix4x3Test, TextureProjectionPerspectiveOffCenterRightHanded)
{
    TypeParam matrix1;

    nn::util::MatrixTextureProjectionPerspectiveOffCenterRightHanded(&matrix1, -2.f, 4.f, -1.f, 3.f, 10.f);

    NNT_UTIL_MATRIX_4X3_EXPECT_NEARLY_EQ(matrix1,
        1.6666667461395264f, 0.0000000000000000f, 0.0000000000000000f,
        0.0000000000000000f, 2.5000000000000000f, 0.0000000000000000f,
        -0.3333333134651184f, -0.2500000000000000f, -1.0000000000000000f,
        0.0000000000000000f, 0.0000000000000000f, 0.0000000000000000f,
        FloatError);
}

TYPED_TEST(MathMatrix4x3Test, TextureProjectionPerspectiveFieldOfViewRightHanded)
{
    TypeParam matrix1;

    nn::util::MatrixTextureProjectionPerspectiveFieldOfViewRightHanded(&matrix1, 60.f / 180.f * nn::util::FloatPi, 16.0f / 9.0f);

    NNT_UTIL_MATRIX_4X3_EXPECT_NEARLY_EQ(matrix1,
        0.4871395230293274f, 0.0000000000000000f, 0.0000000000000000f,
        0.0000000000000000f, 0.8660258054733276f, 0.0000000000000000f,
        -0.5000000000000000f, -0.5000000000000000f, -1.0000000000000000f,
        0.0000000000000000f, 0.0000000000000000f, 0.0000000000000000f,
        FloatError);
}

TYPED_TEST(MathMatrix4x3Test, TextureProjectionOrthographicOffCenterRightHanded)
{
    TypeParam matrix1;

    nn::util::MatrixTextureProjectionOrthographicOffCenterRightHanded(&matrix1, -2.f, 4.f, -1.f, 3.f);

    NNT_UTIL_MATRIX_4X3_EXPECT_NEARLY_EQ(matrix1,
        0.1666666716337204f, 0.0000000000000000f, 0.0000000000000000f,
        0.0000000000000000f, 0.2500000000000000f, 0.0000000000000000f,
        0.0000000000000000f, 0.0000000000000000f, 0.0000000000000000f,
        0.3333333134651184f, 0.2500000000000000f, 1.0000000000000000f,
        FloatError);
}

TYPED_TEST(MathMatrix4x3Test, MatrixFromQuaternion)
{
    // 以下のロール、ピッチ、ヨーによる回転クォータニオン
    // roll = 40.f / 180.f * PI;
    // pitch = 60.f / 180.f * PI;
    // yaw = 30.f / 180.f * PI;
    nn::util::Vector4fType quaternion1;
    nn::util::VectorSet(&quaternion1, 0.5304983258247376f, 0.0454432815313339f, 0.1645002365112305f, 0.8303288221359253f);

    TypeParam matrix1;

    nn::util::MatrixFromQuaternion(&matrix1, quaternion1);

    NNT_UTIL_MATRIX_4X3_EXPECT_NEARLY_EQ(matrix1,
        0.9417491555213928f, 0.3213937878608704f, 0.0990685001015663f,
        -0.2249634265899658f, 0.3830222487449646f, 0.8959270715713501f,
        0.2499999701976776f, -0.8660253286361694f, 0.4330127239227295f,
        0.0000000000000000f, 0.0000000000000000f, 0.0000000000000000f,
        FloatError);
}

// 4x4 行列
template <typename T>
class MathMatrix4x4Test : public ::testing::Test
{
};

typedef ::testing::Types<nn::util::MatrixRowMajor4x4fType, nn::util::MatrixColumnMajor4x4fType> Matrix4x4Types;
TYPED_TEST_CASE(MathMatrix4x4Test, Matrix4x4Types);

TYPED_TEST(MathMatrix4x4Test, Set)
{
    TypeParam matrix;
    nn::util::MatrixSet(&matrix,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);
}

TYPED_TEST(MathMatrix4x4Test, Load)
{
    {
        TypeParam matrix;
        nn::util::FloatRowMajor4x4 value = NN_UTIL_FLOAT_ROW_MAJOR_4X4_INITIALIZER
        (
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f
        );

        nn::util::MatrixLoad(&matrix, value);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);
    }

    {
        TypeParam matrix;
        nn::util::FloatColumnMajor4x4 value = NN_UTIL_FLOAT_COLUMN_MAJOR_4X4_INITIALIZER
        (
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f
        );

        nn::util::MatrixLoad(&matrix, value);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);
    }
}

TYPED_TEST(MathMatrix4x4Test, Store)
{
    TypeParam matrix;
    nn::util::MatrixSet(&matrix,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    {
        nn::util::FloatRowMajor4x4 value;
        nn::util::MatrixStore(&value, matrix);

        EXPECT_EQ(1.f, value.m[0][0]);
        EXPECT_EQ(2.f, value.m[0][1]);
        EXPECT_EQ(3.f, value.m[0][2]);
        EXPECT_EQ(4.f, value.m[0][3]);

        EXPECT_EQ(5.f, value.m[1][0]);
        EXPECT_EQ(6.f, value.m[1][1]);
        EXPECT_EQ(7.f, value.m[1][2]);
        EXPECT_EQ(8.f, value.m[1][3]);

        EXPECT_EQ(9.f,  value.m[2][0]);
        EXPECT_EQ(10.f, value.m[2][1]);
        EXPECT_EQ(11.f, value.m[2][2]);
        EXPECT_EQ(12.f, value.m[2][3]);

        EXPECT_EQ(13.f, value.m[3][0]);
        EXPECT_EQ(14.f, value.m[3][1]);
        EXPECT_EQ(15.f, value.m[3][2]);
        EXPECT_EQ(16.f, value.m[3][3]);
    }

    {
        nn::util::FloatColumnMajor4x4 value;
        nn::util::MatrixStore(&value, matrix);

        EXPECT_EQ(1.f, value.m[0][0]);
        EXPECT_EQ(2.f, value.m[1][0]);
        EXPECT_EQ(3.f, value.m[2][0]);
        EXPECT_EQ(4.f, value.m[3][0]);

        EXPECT_EQ(5.f, value.m[0][1]);
        EXPECT_EQ(6.f, value.m[1][1]);
        EXPECT_EQ(7.f, value.m[2][1]);
        EXPECT_EQ(8.f, value.m[3][1]);

        EXPECT_EQ(9.f,  value.m[0][2]);
        EXPECT_EQ(10.f, value.m[1][2]);
        EXPECT_EQ(11.f, value.m[2][2]);
        EXPECT_EQ(12.f, value.m[3][2]);

        EXPECT_EQ(13.f, value.m[0][3]);
        EXPECT_EQ(14.f, value.m[1][3]);
        EXPECT_EQ(15.f, value.m[2][3]);
        EXPECT_EQ(16.f, value.m[3][3]);
    }
}


TYPED_TEST(MathMatrix4x4Test, SetRow)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::Vector4fType vector1;

    nn::util::VectorSet(&vector1, -17.f, -18.f, -19.f, -20.f);
    nn::util::MatrixSetRow(&matrix1, 0, vector1);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        -17.f, -18.f, -19.f, -20.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::VectorSet(&vector1, -21.f, -22.f, -23.f, -24.f);
    nn::util::MatrixSetRow(&matrix1, 1, vector1);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        -17.f, -18.f, -19.f, -20.f,
        -21.f, -22.f, -23.f, -24.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::VectorSet(&vector1, -25.f, -26.f, -27.f, -28.f);
    nn::util::MatrixSetRow(&matrix1, 2, vector1);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        -17.f, -18.f, -19.f, -20.f,
        -21.f, -22.f, -23.f, -24.f,
        -25.f, -26.f, -27.f, -28.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::VectorSet(&vector1, -29.f, -30.f, -31.f, -32.f);
    nn::util::MatrixSetRow(&matrix1, 3, vector1);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        -17.f, -18.f, -19.f, -20.f,
        -21.f, -22.f, -23.f, -24.f,
        -25.f, -26.f, -27.f, -28.f,
        -29.f, -30.f, -31.f, -32.f);
}

TYPED_TEST(MathMatrix4x4Test, SetRows)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::Vector4fType row0, row1, row2, row3;

    nn::util::VectorSet(&row0, -17.f, -18.f, -19.f, -20.f);
    nn::util::VectorSet(&row1, -21.f, -22.f, -23.f, -24.f);
    nn::util::VectorSet(&row2, -25.f, -26.f, -27.f, -28.f);
    nn::util::VectorSet(&row3, -29.f, -30.f, -31.f, -32.f);

    nn::util::MatrixSetRows(&matrix1, row0, row1, row2, row3);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        -17.f, -18.f, -19.f, -20.f,
        -21.f, -22.f, -23.f, -24.f,
        -25.f, -26.f, -27.f, -28.f,
        -29.f, -30.f, -31.f, -32.f);
}

TYPED_TEST(MathMatrix4x4Test, GetRow)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::Vector4fType vector1;

    nn::util::MatrixGetRow(&vector1, matrix1, 0);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 1.f, 2.f, 3.f, 4.f);

    nn::util::MatrixGetRow(&vector1, matrix1, 1);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 5.f, 6.f, 7.f, 8.f);

    nn::util::MatrixGetRow(&vector1, matrix1, 2);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 9.f, 10.f, 11.f, 12.f);

    nn::util::MatrixGetRow(&vector1, matrix1, 3);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 13.f, 14.f, 15.f, 16.f);
}

TYPED_TEST(MathMatrix4x4Test, GetRows)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::Vector4fType row0, row1, row2, row3;
    nn::util::MatrixGetRows(&row0, &row1, &row2, &row3, matrix1);

    NNT_UTIL_VECTOR4_EXPECT_EQ(row0, 1.f, 2.f, 3.f, 4.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(row1, 5.f, 6.f, 7.f, 8.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(row2, 9.f, 10.f, 11.f, 12.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(row3, 13.f, 14.f, 15.f, 16.f);
}

TYPED_TEST(MathMatrix4x4Test, SetColumn)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::Vector4fType vector1;

    nn::util::VectorSet(&vector1, -17.f, -21.f, -25.f, -29.f);
    nn::util::MatrixSetColumn(&matrix1, 0, vector1);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        -17.f, 2.f, 3.f, 4.f,
        -21.f, 6.f, 7.f, 8.f,
        -25.f, 10.f, 11.f, 12.f,
        -29.f, 14.f, 15.f, 16.f);

    nn::util::VectorSet(&vector1, -18.f, -22.f, -26.f, -30.f);
    nn::util::MatrixSetColumn(&matrix1, 1, vector1);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        -17.f, -18.f, 3.f, 4.f,
        -21.f, -22.f, 7.f, 8.f,
        -25.f, -26.f, 11.f, 12.f,
        -29.f, -30.f, 15.f, 16.f);

    nn::util::VectorSet(&vector1, -19.f, -23.f, -27.f, -31.f);
    nn::util::MatrixSetColumn(&matrix1, 2, vector1);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        -17.f, -18.f, -19.f, 4.f,
        -21.f, -22.f, -23.f, 8.f,
        -25.f, -26.f, -27.f, 12.f,
        -29.f, -30.f, -31.f, 16.f);

    nn::util::VectorSet(&vector1, -20.f, -24.f, -28.f, -32.f);
    nn::util::MatrixSetColumn(&matrix1, 3, vector1);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        -17.f, -18.f, -19.f, -20.f,
        -21.f, -22.f, -23.f, -24.f,
        -25.f, -26.f, -27.f, -28.f,
        -29.f, -30.f, -31.f, -32.f);
}

TYPED_TEST(MathMatrix4x4Test, SetColumns)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::Vector4fType column0, column1, column2, column3;

    nn::util::VectorSet(&column0, -17.f, -21.f, -25.f, -29.f);
    nn::util::VectorSet(&column1, -18.f, -22.f, -26.f, -30.f);
    nn::util::VectorSet(&column2, -19.f, -23.f, -27.f, -31.f);
    nn::util::VectorSet(&column3, -20.f, -24.f, -28.f, -32.f);

    nn::util::MatrixSetColumns(&matrix1, column0, column1, column2, column3);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        -17.f, -18.f, -19.f, -20.f,
        -21.f, -22.f, -23.f, -24.f,
        -25.f, -26.f, -27.f, -28.f,
        -29.f, -30.f, -31.f, -32.f);
}

TYPED_TEST(MathMatrix4x4Test, GetColumn)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::Vector4fType vector1;

    nn::util::MatrixGetColumn(&vector1, matrix1, 0);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 1.f, 5.f, 9.f, 13.f);

    nn::util::MatrixGetColumn(&vector1, matrix1, 1);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 2.f, 6.f, 10.f, 14.f);

    nn::util::MatrixGetColumn(&vector1, matrix1, 2);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 3.f, 7.f, 11.f, 15.f);

    nn::util::MatrixGetColumn(&vector1, matrix1, 3);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 4.f, 8.f, 12.f, 16.f);
}

TYPED_TEST(MathMatrix4x4Test, GetColumns)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::Vector4fType column0, column1, column2, column3;
    nn::util::MatrixGetColumns(&column0, &column1, &column2, &column3, matrix1);

    NNT_UTIL_VECTOR4_EXPECT_EQ(column0, 1.f, 5.f, 9.f, 13.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(column1, 2.f, 6.f, 10.f, 14.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(column2, 3.f, 7.f, 11.f, 15.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(column3, 4.f, 8.f, 12.f, 16.f);
}

TYPED_TEST(MathMatrix4x4Test, SetAxis)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::Vector4fType vector1;

    nn::util::VectorSet(&vector1, -17.f, -18.f, -19.f, -20.f);
    nn::util::MatrixSetAxisX(&matrix1, vector1);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        -17.f, -18.f, -19.f, -20.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::VectorSet(&vector1, -21.f, -22.f, -23.f, -24.f);
    nn::util::MatrixSetAxisY(&matrix1, vector1);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        -17.f, -18.f, -19.f, -20.f,
        -21.f, -22.f, -23.f, -24.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::VectorSet(&vector1, -25.f, -26.f, -27.f, -28.f);
    nn::util::MatrixSetAxisZ(&matrix1, vector1);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        -17.f, -18.f, -19.f, -20.f,
        -21.f, -22.f, -23.f, -24.f,
        -25.f, -26.f, -27.f, -28.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::VectorSet(&vector1, -29.f, -30.f, -31.f, -32.f);
    nn::util::MatrixSetAxisW(&matrix1, vector1);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        -17.f, -18.f, -19.f, -20.f,
        -21.f, -22.f, -23.f, -24.f,
        -25.f, -26.f, -27.f, -28.f,
        -29.f, -30.f, -31.f, -32.f);
}

TYPED_TEST(MathMatrix4x4Test, SetAxes)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::Vector4fType axisX, axisY, axisZ, axisW;

    nn::util::VectorSet(&axisX, -17.f, -18.f, -19.f, -20.f);
    nn::util::VectorSet(&axisY, -21.f, -22.f, -23.f, -24.f);
    nn::util::VectorSet(&axisZ, -25.f, -26.f, -27.f, -28.f);
    nn::util::VectorSet(&axisW, -29.f, -30.f, -31.f, -32.f);

    nn::util::MatrixSetRows(&matrix1, axisX, axisY, axisZ, axisW);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        -17.f, -18.f, -19.f, -20.f,
        -21.f, -22.f, -23.f, -24.f,
        -25.f, -26.f, -27.f, -28.f,
        -29.f, -30.f, -31.f, -32.f);
}

TYPED_TEST(MathMatrix4x4Test, GetAxis)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::Vector4fType vector1;

    nn::util::MatrixGetAxisX(&vector1, matrix1);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 1.f, 2.f, 3.f, 4.f);

    nn::util::MatrixGetAxisY(&vector1, matrix1);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 5.f, 6.f, 7.f, 8.f);

    nn::util::MatrixGetAxisZ(&vector1, matrix1);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 9.f, 10.f, 11.f, 12.f);

    nn::util::MatrixGetAxisW(&vector1, matrix1);
    NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 13.f, 14.f, 15.f, 16.f);
}

TYPED_TEST(MathMatrix4x4Test, GetAxes)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, 2.f, 3.f, 4.f,
        5.f, 6.f, 7.f, 8.f,
        9.f, 10.f, 11.f, 12.f,
        13.f, 14.f, 15.f, 16.f);

    nn::util::Vector4fType axisX, axisY, axisZ, axisW;
    nn::util::MatrixGetAxes(&axisX, &axisY, &axisZ, &axisW, matrix1);

    NNT_UTIL_VECTOR4_EXPECT_EQ(axisX, 1.f, 2.f, 3.f, 4.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(axisY, 5.f, 6.f, 7.f, 8.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(axisZ, 9.f, 10.f, 11.f, 12.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(axisW, 13.f, 14.f, 15.f, 16.f);
}

TYPED_TEST(MathMatrix4x4Test, Zero)
{
    TypeParam matrix;
    nn::util::MatrixSet(&matrix,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    nn::util::MatrixZero(&matrix);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix,
        0.f, 0.f, 0.f, 0.f,
        0.f, 0.f, 0.f, 0.f,
        0.f, 0.f, 0.f, 0.f,
        0.f, 0.f, 0.f, 0.f);
}

TYPED_TEST(MathMatrix4x4Test, Identity)
{
    TypeParam matrix;
    nn::util::MatrixSet(&matrix,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    nn::util::MatrixIdentity(&matrix);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix,
        1.f, 0.f, 0.f, 0.f,
        0.f, 1.f, 0.f, 0.f,
        0.f, 0.f, 1.f, 0.f,
        0.f, 0.f, 0.f, 1.f);
}

TYPED_TEST(MathMatrix4x4Test, Add)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    TypeParam matrix2;
    nn::util::MatrixSet(&matrix2,
        2.f, -4.f, 6.f, -8.f,
        10.f, -12.f, 14.f, -16.f,
        -18.f, 20.f, -22.f, 24.f,
        -26.f, 28.f, -30.f, 32.f);

    nn::util::MatrixAdd(&matrix1, matrix1, matrix2);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        3.f, -6.f, 9.f, -12.f,
        5.f, -6.f, 7.f, -8.f,
        -9.f, 10.f, -11.f, 12.f,
        -39.f, 42.f, -45.f, 48.f);
}

TYPED_TEST(MathMatrix4x4Test, Subtract)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    TypeParam matrix2;
    nn::util::MatrixSet(&matrix2,
        2.f, -4.f, 6.f, -8.f,
        10.f, -12.f, 14.f, -16.f,
        -18.f, 20.f, -22.f, 24.f,
        -26.f, 28.f, -30.f, 32.f);

    nn::util::MatrixSubtract(&matrix1, matrix1, matrix2);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        -1.f, 2.f, -3.f, 4.f,
        -15.f, 18.f, -21.f, 24.f,
        27.f, -30.f, 33.f, -36.f,
        13.f, -14.f, 15.f, -16.f);
}

TYPED_TEST(MathMatrix4x4Test, Multiply)
{
    {
        TypeParam matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f, -4.f,
            -5.f, 6.f, -7.f, 8.f,
            9.f, -10.f, 11.f, -12.f,
            -13.f, 14.f, -15.f, 16.f);

        nn::util::MatrixMultiply(&matrix1, matrix1, -1.f / 2.f);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            -0.5f, 1.f, -1.5f, 2.f,
            2.5f, -3.f, 3.5f, -4.f,
            -4.5f, 5.f, -5.5f, 6.f,
             6.5f, -7.f, 7.5f, -8.f);
    }

    {
        TypeParam matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f, -4.f,
            -5.f, 6.f, -7.f, 8.f,
            9.f, -10.f, 11.f, -12.f,
            -13.f, 14.f, -15.f, 16.f);

        TypeParam matrix2;
        nn::util::MatrixSet(&matrix2,
            2.f, -4.f, 6.f, -8.f,
            -10.f, -12.f, -14.f, -16.f,
            18.f, 20.f, 22.f, 24.f,
            -26.f, -28.f, -30.f, -32.f);

        nn::util::MatrixMultiply(&matrix1, matrix1, matrix2);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            180.f, 192.f, 220.f, 224.f,
            -404.f, -416.f, -508.f, -480.f,
            628.f, 640.f, 796.f, 736.f,
            -852.f, -864.f, -1084.f, -992.f);
    }
}

TYPED_TEST(MathMatrix4x4Test, Divide)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    nn::util::MatrixDivide(&matrix1, matrix1, -2.f);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
        -0.5f, 1.f, -1.5f, 2.f,
        2.5f, -3.f, 3.5f, -4.f,
        -4.5f, 5.f, -5.5f, 6.f,
        6.5f, -7.f, 7.5f, -8.f);
}

TYPED_TEST(MathMatrix4x4Test, Inverse)
{
    {
        TypeParam matrix1;
        nn::util::MatrixSet(&matrix1,
            0.f, 0.f, 0.f, 0.f,
            -5.f, 6.f, -7.f, 8.f,
            9.f, -10.f, 11.f, -12.f,
            -13.f, 14.f, -15.f, 16.f);

        EXPECT_FALSE(nn::util::MatrixInverse(&matrix1, matrix1));

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            0.f, 0.f, 0.f, 0.f,
            0.f, 0.f, 0.f, 0.f,
            0.f, 0.f, 0.f, 0.f,
            0.f, 0.f, 0.f, 0.f);
    }

    {
        TypeParam matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, 0.f, 1.f, 2.f,
            -1.f, 1.f, 1.f, 1.f,
            1.f, -1.f, 0.f, 1.f,
            1.f, 1.f, -1.f, -2.f);

        EXPECT_TRUE(nn::util::MatrixInverse(&matrix1, matrix1));

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            1.f, -1.f, -1.f, 0.f,
            -1.f, 2.f, 2.f, 1.f,
            4.f, -5.f, -7.f, -2.f,
            -2.f, 3.f, 4.f, 1.f);
    }
}

TYPED_TEST(MathMatrix4x4Test, Transpose)
{
    TypeParam matrix1;
    nn::util::MatrixSet(&matrix1,
        1.f, -2.f, 3.f, -4.f,
        -5.f, 6.f, -7.f, 8.f,
        9.f, -10.f, 11.f, -12.f,
        -13.f, 14.f, -15.f, 16.f);

    nn::util::MatrixTranspose(&matrix1, matrix1);

    NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            1.f, -5.f, 9.f, -13.f,
            -2.f, 6.f, -10.f, 14.f,
            3.f, -7.f, 11.f, -15.f,
            -4.f, 8.f, -12.f, 16.f);
}

TYPED_TEST(MathMatrix4x4Test, PerspectiveOffCenterRightHanded)
{
    TypeParam matrix1;

    nn::util::MatrixPerspectiveOffCenterRightHanded(&matrix1, -2.f, 4.f, -1.f, 3.f, 10.f, 100.f);

    NNT_UTIL_MATRIX_4X4_EXPECT_NEARLY_EQ(matrix1,
        3.3333334922790527f, 0.0000000000000000f, 0.0000000000000000f, 0.0000000000000000f,
        0.0000000000000000f, 5.0000000000000000f, 0.0000000000000000f, 0.0000000000000000f,
        0.3333333432674408f, 0.5000000000000000f, -1.1111111640930176f, -1.0000000000000000f,
        0.0000000000000000f, 0.0000000000000000f, -11.1111116409301760f, 0.0000000000000000f,
        FloatError);
}

TYPED_TEST(MathMatrix4x4Test, PerspectiveFieldOfViewRightHanded)
{
    TypeParam matrix1;

    nn::util::MatrixPerspectiveFieldOfViewRightHanded(&matrix1, 60.f / 180.f * nn::util::FloatPi, 16.0f / 9.0f, 0.1f, 100.0f);

    NNT_UTIL_MATRIX_4X4_EXPECT_NEARLY_EQ(matrix1,
        0.9742785692214966f, 0.0000000000000000f, 0.0000000000000000f, 0.0000000000000000f,
        0.0000000000000000f, 1.7320507764816284f, 0.0000000000000000f, 0.0000000000000000f,
        0.0000000000000000f, 0.0000000000000000f, -1.0010010004043579f, -1.0000000000000000f,
        0.0000000000000000f, 0.0000000000000000f, -0.1001001000404358f, 0.0000000000000000f,
        FloatError);
}

TYPED_TEST(MathMatrix4x4Test, OrthographicOffCenterRightHanded)
{
    TypeParam matrix1;

    nn::util::MatrixOrthographicOffCenterRightHanded(&matrix1, -2.f, 4.f, -1.f, 3.f, 10.f, 100.f);

    NNT_UTIL_MATRIX_4X4_EXPECT_NEARLY_EQ(matrix1,
        0.3333333432674408f, 0.0000000000000000f, 0.0000000000000000f, 0.0000000000000000f,
        0.0000000000000000f, 0.5000000000000000f, 0.0000000000000000f, 0.0000000000000000f,
        0.0000000000000000f, 0.0000000000000000f, -0.0111111113801599f, 0.0000000000000000f,
        -0.3333333432674408f, -0.5000000000000000f, -0.1111111119389534f, 1.0000000000000000f,
        FloatError);
}

// 3x2, 4x3, 4x4 行列を同時に扱うテスト
class MathMatrixTest : public ::testing::Test
{
};

TEST_F(MathMatrixTest, Initializer)
{
    {
        nn::util::MatrixRowMajor3x2fType matrix = NN_UTIL_MATRIX_ROW_MAJOR_3X2F_INITIALIZER(
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix,
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);
    }

    {
        nn::util::MatrixColumnMajor3x2fType matrix = NN_UTIL_MATRIX_COLUMN_MAJOR_3X2F_INITIALIZER(
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix,
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);
    }

    {
        nn::util::MatrixRowMajor4x3fType matrix = NN_UTIL_MATRIX_ROW_MAJOR_4X3F_INITIALIZER(
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);
    }

    {
        nn::util::MatrixColumnMajor4x3fType matrix = NN_UTIL_MATRIX_COLUMN_MAJOR_4X3F_INITIALIZER(
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);
    }

    {
        nn::util::MatrixRowMajor4x4fType matrix = NN_UTIL_MATRIX_ROW_MAJOR_4X4F_INITIALIZER(
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);
    }

    {
        nn::util::MatrixColumnMajor4x4fType matrix = NN_UTIL_MATRIX_COLUMN_MAJOR_4X4F_INITIALIZER(
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);
    }

    {
        nn::util::Matrix3x2fType matrix = NN_UTIL_MATRIX_3X2F_INITIALIZER(
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix,
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);
    }

    {
        nn::util::MatrixT3x2fType matrix = NN_UTIL_MATRIX_T3X2F_INITIALIZER(
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);

        NNT_UTIL_MATRIX_3X2_EXPECT_EQ(matrix,
            1.f, 2.f,
            3.f, 4.f,
            5.f, 6.f);
    }


    {
        nn::util::Matrix4x3fType matrix = NN_UTIL_MATRIX_4X3F_INITIALIZER(
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);
    }

    {
        nn::util::MatrixT4x3fType matrix = NN_UTIL_MATRIX_T4X3F_INITIALIZER(
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);
    }

    {
        nn::util::Matrix4x4fType matrix = NN_UTIL_MATRIX_4X4F_INITIALIZER(
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);
    }

    {
        nn::util::MatrixT4x4fType matrix = NN_UTIL_MATRIX_T4X4F_INITIALIZER(
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix,
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);
    }
} // NOLINT(impl/function_size)

TEST_F(MathMatrixTest, Convert)
{
    {
        nn::util::MatrixRowMajor4x3fType source;
        nn::util::MatrixSet(&source,
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);

        nn::util::MatrixRowMajor4x4fType dest;
        nn::util::MatrixConvert(&dest, source);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(dest,
            1.f, 2.f, 3.f, 0.f,
            4.f, 5.f, 6.f, 0.f,
            7.f, 8.f, 9.f, 0.f,
            10.f, 11.f, 12.f, 1.f);
    }

    {
        nn::util::MatrixRowMajor4x4fType source;
        nn::util::MatrixSet(&source,
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);

        nn::util::MatrixRowMajor4x3fType dest;
        nn::util::MatrixConvert(&dest, source);

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(dest,
            1.f, 2.f, 3.f,
            5.f, 6.f, 7.f,
            9.f, 10.f, 11.f,
            13.f, 14.f, 15.f);
    }

    {
        nn::util::MatrixColumnMajor4x3fType source;
        nn::util::MatrixSet(&source,
            1.f, 2.f, 3.f,
            4.f, 5.f, 6.f,
            7.f, 8.f, 9.f,
            10.f, 11.f, 12.f);

        nn::util::MatrixColumnMajor4x4fType dest;
        nn::util::MatrixConvert(&dest, source);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(dest,
            1.f, 2.f, 3.f, 0.f,
            4.f, 5.f, 6.f, 0.f,
            7.f, 8.f, 9.f, 0.f,
            10.f, 11.f, 12.f, 1.f);
    }

    {
        nn::util::MatrixColumnMajor4x4fType source;
        nn::util::MatrixSet(&source,
            1.f, 2.f, 3.f, 4.f,
            5.f, 6.f, 7.f, 8.f,
            9.f, 10.f, 11.f, 12.f,
            13.f, 14.f, 15.f, 16.f);

        nn::util::MatrixColumnMajor4x3fType dest;
        nn::util::MatrixConvert(&dest, source);

        NNT_UTIL_MATRIX_4X3_EXPECT_EQ(dest,
            1.f, 2.f, 3.f,
            5.f, 6.f, 7.f,
            9.f, 10.f, 11.f,
            13.f, 14.f, 15.f);
    }
}

TEST_F(MathMatrixTest, Multiply)
{
    {
        nn::util::MatrixRowMajor4x3fType matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        nn::util::MatrixRowMajor4x4fType matrix2;
        nn::util::MatrixSet(&matrix2,
            2.f, -4.f, 6.f, -8.f,
            -10.f, -12.f, -14.f, -16.f,
            18.f, 20.f, 22.f, 24.f,
            -26.f, -28.f, -30.f, -32.f);

        nn::util::MatrixMultiply(&matrix2, matrix1, matrix2);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix2,
            76.f, 80.f, 100.f, 96.f,
            -166.f, -164.f, -226.f, -192.f,
            256.f, 248.f, 352.f, 288.f,
            -372.f, -360.f, -508.f, -416.f);
    }

    {
        nn::util::MatrixColumnMajor4x3fType matrix1;
        nn::util::MatrixSet(&matrix1,
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        nn::util::MatrixColumnMajor4x4fType matrix2;
        nn::util::MatrixSet(&matrix2,
            2.f, -4.f, 6.f, -8.f,
            -10.f, -12.f, -14.f, -16.f,
            18.f, 20.f, 22.f, 24.f,
            -26.f, -28.f, -30.f, -32.f);

        nn::util::MatrixMultiply(&matrix2, matrix1, matrix2);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix2,
            76.f, 80.f, 100.f, 96.f,
            -166.f, -164.f, -226.f, -192.f,
            256.f, 248.f, 352.f, 288.f,
            -372.f, -360.f, -508.f, -416.f);
    }

    {
        nn::util::MatrixRowMajor4x4fType matrix1;
        nn::util::MatrixSet(&matrix1,
            2.f, -4.f, 6.f, -8.f,
            -10.f, -12.f, -14.f, -16.f,
            18.f, 20.f, 22.f, 24.f,
            -26.f, -28.f, -30.f, -32.f);

        nn::util::MatrixRowMajor4x3fType matrix2;
        nn::util::MatrixSet(&matrix2,
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        nn::util::MatrixMultiply(&matrix1, matrix1, matrix2);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            140.f, -160.f, 180.f, -8.f,
            100.f, -104.f, 108.f, -16.f,
            -148.f, 152.f, -156.f, 24.f,
            196.f, -200.f, 204.f, -32.f);
    }

    {
        nn::util::MatrixColumnMajor4x4fType matrix1;
        nn::util::MatrixSet(&matrix1,
            2.f, -4.f, 6.f, -8.f,
            -10.f, -12.f, -14.f, -16.f,
            18.f, 20.f, 22.f, 24.f,
            -26.f, -28.f, -30.f, -32.f);

        nn::util::MatrixColumnMajor4x3fType matrix2;
        nn::util::MatrixSet(&matrix2,
            1.f, -2.f, 3.f,
            -4.f, 5.f, -6.f,
            7.f, -8.f, 9.f,
            -10.f, 11.f, -12.f);

        nn::util::MatrixMultiply(&matrix1, matrix1, matrix2);

        NNT_UTIL_MATRIX_4X4_EXPECT_EQ(matrix1,
            140.f, -160.f, 180.f, -8.f,
            100.f, -104.f, 108.f, -16.f,
            -148.f, 152.f, -156.f, 24.f,
            196.f, -200.f, 204.f, -32.f);
    }
}

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning( pop )
#endif
