﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <limits>
#include <cstring>
#include <nnt/nntest.h>
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_TFormatString.h>
#include <nn/util/util_FormatString.h>

// TORIAEZU: SigloGcc の sprintf は浮動小数点数のフォーマット時に malloc するので、GCC では無効にする
#if defined(NN_BUILD_CONFIG_COMPILER_VC) || defined(NN_BUILD_CONFIG_COMPILER_GHS) || defined(NN_BUILD_CONFIG_COMPILER_CLANG)
// 浮動小数点数の変換を有効にする
#define NNT_DETAIL_UTIL_FORMATSTRING_ENABLE_FP
#endif

#if defined(NNT_DETAIL_UTIL_FORMATSTRING_ENABLE_FP)
#include "testUtil_ScopedFpSetRound.h"
#endif

namespace {

const size_t BufferLength = 256;

// FormatString の結果をバッファに格納する
class OutputSink
{
public:
    OutputSink(char* buffer, size_t length) NN_NOEXCEPT
        : m_pBuffer(buffer), m_pCurrent(buffer), m_pEnd(buffer + length)
    {
        if (m_pCurrent != m_pEnd)
        {
            *m_pCurrent = '\0';
        }
    }

    static void Output(uintptr_t arg, const char* pCharacters, int count) NN_NOEXCEPT
    {
        reinterpret_cast<OutputSink*>(arg)->OutputCore(pCharacters, count);
    }

    const char* Get() NN_NOEXCEPT
    {
        return m_pBuffer;
    }
private:
    void OutputCore(const char* pCharacters, int count) NN_NOEXCEPT
    {
        NN_ASSERT(m_pCurrent <= m_pEnd);
        NN_ASSERT(count >= 0);

        if (m_pCurrent == m_pEnd)
        {
            return;
        }

        const ptrdiff_t copyCount = std::min(m_pEnd - m_pCurrent - 1, static_cast<ptrdiff_t>(count));
        std::memcpy(m_pCurrent, pCharacters, copyCount * sizeof(char));
        m_pCurrent += copyCount;
        *m_pCurrent = '\0';
    }

    char* m_pBuffer;
    char* m_pCurrent;
    char* m_pEnd;
};

// va_list を与えて呼び出し直す関数
void CallVFormatString(
    nn::util::FormatStringOutputFunction pOutputFunction,
    uintptr_t outputFunctionArg,
    const char* pFormat,
    ...) NN_NOEXCEPT
{
    std::va_list va;
    va_start(va, pFormat);
    nn::util::VFormatString(pOutputFunction, outputFunctionArg, pFormat, va);
    va_end(va);
}

void CallTVFormatString(
    nn::util::FormatStringOutputFunction pOutputFunction,
    uintptr_t outputFunctionArg,
    const char* pFormat,
    ...) NN_NOEXCEPT
{
    std::va_list va;
    va_start(va, pFormat);
    nn::util::TVFormatString(pOutputFunction, outputFunctionArg, pFormat, va);
    va_end(va);
}

void CallVSNPrintf(char* buffer, size_t bufferLength, const char* pFormat, ...) NN_NOEXCEPT
{
    std::va_list va;
    va_start(va, pFormat);
    nn::util::VSNPrintf(buffer, bufferLength, pFormat, va);
    va_end(va);
}

void CallTVSNPrintf(char* buffer, size_t bufferLength, const char* pFormat, ...) NN_NOEXCEPT
{
    std::va_list va;
    va_start(va, pFormat);
    nn::util::VSNPrintf(buffer, bufferLength, pFormat, va);
    va_end(va);
}


// T 系以外をテスト: FormatString, VFormatString, SNPrintf, VSNPrintf
#define NNT_UTIL_FORMATSTRINGTEST_NONTINY(expected, format, ...) \
    do \
    { \
        char buffer[BufferLength]; \
        { \
            OutputSink sink(buffer, BufferLength); \
            nn::util::FormatString(OutputSink::Output, reinterpret_cast<uintptr_t>(&sink), format, __VA_ARGS__); \
            EXPECT_STREQ(expected, buffer); \
        } \
        { \
            OutputSink sink(buffer, BufferLength); \
            CallVFormatString(OutputSink::Output, reinterpret_cast<uintptr_t>(&sink), format, __VA_ARGS__); \
            EXPECT_STREQ(expected, buffer); \
        } \
        \
        nn::util::SNPrintf(buffer, BufferLength, format, __VA_ARGS__); \
        EXPECT_STREQ(expected, buffer); \
        \
        CallVSNPrintf(buffer, BufferLength, format, __VA_ARGS__); \
        EXPECT_STREQ(expected, buffer); \
    } while (NN_STATIC_CONDITION(false))

// T 系をテスト: TFormatString, TVFormatString, TSNPrintf, TVSNPrintf
#define NNT_UTIL_FORMATSTRINGTEST_TINY(expected, format, ...) \
    do \
    { \
        char buffer[BufferLength]; \
        { \
            OutputSink sink(buffer, BufferLength); \
            nn::util::TFormatString(OutputSink::Output, reinterpret_cast<uintptr_t>(&sink), format, __VA_ARGS__); \
            EXPECT_STREQ(expected, buffer); \
        } \
        { \
            OutputSink sink(buffer, BufferLength); \
            CallTVFormatString(OutputSink::Output, reinterpret_cast<uintptr_t>(&sink), format, __VA_ARGS__); \
            EXPECT_STREQ(expected, buffer); \
        } \
        \
        nn::util::TSNPrintf(buffer, BufferLength, format, __VA_ARGS__); \
        EXPECT_STREQ(expected, buffer); \
        \
        CallTVSNPrintf(buffer, BufferLength, format, __VA_ARGS__); \
        EXPECT_STREQ(expected, buffer); \
    } while (NN_STATIC_CONDITION(false))

#define NNT_UTIL_FORMATSTRINGTEST(expected, format, ...) \
    NNT_UTIL_FORMATSTRINGTEST_NONTINY(expected, format, __VA_ARGS__); \
    NNT_UTIL_FORMATSTRINGTEST_TINY(expected, format, __VA_ARGS__)

} // anonymous namespace

#if defined(NNT_DETAIL_UTIL_FORMATSTRING_ENABLE_FP)
class FormatStringTest : public testing::Test
{
protected:
    virtual void SetUp()
    {
        // テストの期待値が丸めモード round to nearest を仮定した値であるため、
        // 丸めモードを round to nearest に設定します
        m_ScopedFpSetRoundToNearest.emplace();
    }
    virtual void TearDown()
    {
        m_ScopedFpSetRoundToNearest = nn::util::nullopt;
    }
private:
    nn::util::optional<ScopedFpSetRoundToNearest> m_ScopedFpSetRoundToNearest;
};
#else // defined(NNT_DETAIL_UTIL_FORMATSTRING_ENABLE_FP)
class FormatStringTest : public testing::Test
{
};
#endif // defined(NNT_DETAIL_UTIL_FORMATSTRING_ENABLE_FP)

TEST_F(FormatStringTest, InvalidFormat)
{
    // 不正なフォーマット
    NNT_UTIL_FORMATSTRINGTEST("%1", "%1", 1);
}

TEST_F(FormatStringTest, SNPrintfTruncation)
{
    char buffer[BufferLength];
    int formattedStringLength;

    formattedStringLength = nn::util::SNPrintf(buffer, 0, "0123456789");
    EXPECT_EQ(10, formattedStringLength);
    // バッファ長 == 0 の場合バッファには書き込まれない

    formattedStringLength = nn::util::SNPrintf(buffer, 10, "0123456789");
    EXPECT_EQ(10, formattedStringLength);
    EXPECT_STREQ("012345678", buffer);

    formattedStringLength = nn::util::SNPrintf(buffer, 11, "0123456789");
    EXPECT_EQ(10, formattedStringLength);
    EXPECT_STREQ("0123456789", buffer);
}

TEST_F(FormatStringTest, IntegerBasic)
{
    // 基本
    NNT_UTIL_FORMATSTRINGTEST("1", "%d", 1);
    // フィールド幅の解釈
    NNT_UTIL_FORMATSTRINGTEST("  1", "%3d", 1);
    // フィールド幅を * で指定する
    NNT_UTIL_FORMATSTRINGTEST("  1", "%*d", 3, 1);
    // '-' フラグがある場合、右にスペースが詰められる
    NNT_UTIL_FORMATSTRINGTEST("1  ", "%-3d", 1);
    // マイナスのフィールド幅を * で指定する
    NNT_UTIL_FORMATSTRINGTEST("1  ", "%*d", -3, 1);
    // 精度の解釈
    NNT_UTIL_FORMATSTRINGTEST("001", "%.3d", 1);
    // 数字を指定しない場合、0 を指定したことになる
    NNT_UTIL_FORMATSTRINGTEST("", "%.d", 0);
    // 精度を * で指定する
    NNT_UTIL_FORMATSTRINGTEST("000", "%.*d", 3, 0);
    // 負の数は無視される（デフォルトになる）
    NNT_UTIL_FORMATSTRINGTEST("0", "%.*d", -3, 0);
}

TEST_F(FormatStringTest, IntegerD)
{
    // %d
    // 符号 + フィールド幅
    NNT_UTIL_FORMATSTRINGTEST("  1", "%3d", 1);
    NNT_UTIL_FORMATSTRINGTEST(" +1", "%+3d", 1);
    NNT_UTIL_FORMATSTRINGTEST("-1 ", "%-3d", -1);
    NNT_UTIL_FORMATSTRINGTEST("  1", "% 3d", 1);
    NNT_UTIL_FORMATSTRINGTEST("100", "%3d", 100);
    // フィールド幅より長い
    NNT_UTIL_FORMATSTRINGTEST("+100", "%+3d", 100);
    NNT_UTIL_FORMATSTRINGTEST("-100", "%-3d", -100);
    NNT_UTIL_FORMATSTRINGTEST("1000", "%3d", 1000);
    // ゼロ
    NNT_UTIL_FORMATSTRINGTEST("0", "%d", 0);
    // 精度0の時、0は表示しない
    NNT_UTIL_FORMATSTRINGTEST("", "%.0d", 0);
    NNT_UTIL_FORMATSTRINGTEST("1", "%.0d", 1);
    // 0フラグ
    NNT_UTIL_FORMATSTRINGTEST("-01", "%03d", -1);
    // 精度が指定された場合は、0フラグを無視する
    NNT_UTIL_FORMATSTRINGTEST(" -01", "%04.2d", -1);
    // 長さ修飾子
    NNT_UTIL_FORMATSTRINGTEST("4294967296", "%lld", 4294967296LL);
    NNT_UTIL_FORMATSTRINGTEST("-4294967296", "%lld", -4294967296LL);
    NNT_UTIL_FORMATSTRINGTEST("65793", "%ld", 0x10101L);
    NNT_UTIL_FORMATSTRINGTEST("65793", "%d", 0x10101);
    NNT_UTIL_FORMATSTRINGTEST("257", "%hd", 65793);
    NNT_UTIL_FORMATSTRINGTEST("1", "%hhd", 65793);
    NNT_UTIL_FORMATSTRINGTEST("4294967296", "%jd", (intmax_t)4294967296LL);
    NNT_UTIL_FORMATSTRINGTEST("65793", "%zd", 65793);
    NNT_UTIL_FORMATSTRINGTEST("65793", "%td", 65793);
}

TEST_F(FormatStringTest, IntegerI)
{
    // %i
    NNT_UTIL_FORMATSTRINGTEST("  015", "%5.3i", 15);
}

TEST_F(FormatStringTest, IntegerO)
{
    // %o
    NNT_UTIL_FORMATSTRINGTEST("  017", "%5.3o", 15);
    NNT_UTIL_FORMATSTRINGTEST("017  ", "%-5.3o", 15);
    // ゼロ
    NNT_UTIL_FORMATSTRINGTEST("0", "%o", 0);
    // 精度0の時、0は表示しない
    NNT_UTIL_FORMATSTRINGTEST("", "%.0o", 0);
    NNT_UTIL_FORMATSTRINGTEST("1", "%.0o", 1);
    // 引数は unsigned
    NNT_UTIL_FORMATSTRINGTEST("37777777777", "%o", ~0u);
}

TEST_F(FormatStringTest, IntegerU)
{
    // %u
    NNT_UTIL_FORMATSTRINGTEST("  015", "%5.3u", 15);
    // ゼロ
    NNT_UTIL_FORMATSTRINGTEST("0", "%u", 0);
    // 精度0の時、0は表示しない
    NNT_UTIL_FORMATSTRINGTEST("", "%.0u", 0);
    NNT_UTIL_FORMATSTRINGTEST("1", "%.0u", 1);
    // 引数は unsigned
    NNT_UTIL_FORMATSTRINGTEST("4294967295", "%u", ~0u);
}

TEST_F(FormatStringTest, IntegerX)
{
    // %x
    NNT_UTIL_FORMATSTRINGTEST("  00f", "%5.3x", 15);
    // ゼロ
    NNT_UTIL_FORMATSTRINGTEST("0", "%x", 0);
    // 精度0の時、0は表示しない
    NNT_UTIL_FORMATSTRINGTEST("", "%.0x", 0);
    NNT_UTIL_FORMATSTRINGTEST("1", "%.0x", 1);
    // 引数は unsigned
    NNT_UTIL_FORMATSTRINGTEST("ffffffff", "%x", ~0u);

    // %X
    NNT_UTIL_FORMATSTRINGTEST("  00F", "%5.3X", 15);
    NNT_UTIL_FORMATSTRINGTEST("1ABCDEF", "%X", 0x1abcdef);
}

TEST_F(FormatStringTest, IntegerSharpO)
{
    // %#o
    // #o: 1桁目が0になるようにする。必要なら精度を増加させる。
    NNT_UTIL_FORMATSTRINGTEST("0", "%#o", 0);
    NNT_UTIL_FORMATSTRINGTEST("01", "%#o", 1);
    NNT_UTIL_FORMATSTRINGTEST("000", "%#03o", 0);
    NNT_UTIL_FORMATSTRINGTEST("001", "%#03o", 1);
    NNT_UTIL_FORMATSTRINGTEST("0100", "%#03o", 0100);
}

TEST_F(FormatStringTest, IntegerSharpX)
{
    // %#x
    // #x: 0 でない場合 0x を prefix する
    NNT_UTIL_FORMATSTRINGTEST("0", "%#x", 0);
    NNT_UTIL_FORMATSTRINGTEST("0x1", "%#x", 1);
    NNT_UTIL_FORMATSTRINGTEST("0000", "%#04x", 0);
    NNT_UTIL_FORMATSTRINGTEST("0x01", "%#04x", 1);
    NNT_UTIL_FORMATSTRINGTEST(" 0x1", "%#4x", 1);
}

TEST_F(FormatStringTest, IntegerSharpC)
{
    // %c
    NNT_UTIL_FORMATSTRINGTEST("a", "%c", 'a');
    NNT_UTIL_FORMATSTRINGTEST("a", "%lc", L'a');
}

TEST_F(FormatStringTest, StringS)
{
    // %s
    NNT_UTIL_FORMATSTRINGTEST("abcde", "%s", "abcde");
    NNT_UTIL_FORMATSTRINGTEST("abcde", "%ls", L"abcde");
    NNT_UTIL_FORMATSTRINGTEST("   abcde", "%8s", "abcde");
    // 精度は、表示する最大のバイト数
    NNT_UTIL_FORMATSTRINGTEST("  abc", "%5.3s", "abcde");
    NNT_UTIL_FORMATSTRINGTEST("abc", "%.3s", "abcde");
    NNT_UTIL_FORMATSTRINGTEST("abc", "%.3ls", L"abcde");
    NNT_UTIL_FORMATSTRINGTEST("a", "%.3s", "a");
    // NULLが入力された場合
    NNT_UTIL_FORMATSTRINGTEST("(null)", "%s", NULL);
    NNT_UTIL_FORMATSTRINGTEST("(null)", "%ls", NULL);
}

TEST_F(FormatStringTest, PointerP)
{
    // %p の表示は処理系定義。ARMCC は %08x
#if UINTPTR_MAX == 0xffffffff
    NNT_UTIL_FORMATSTRINGTEST("ffffffff", "%p", reinterpret_cast<void*>(0xffffffff));
    NNT_UTIL_FORMATSTRINGTEST("  ffffffff", "%10p", reinterpret_cast<void*>(0xffffffff));
#elif  UINTPTR_MAX == 0xffffffffffffffff
    NNT_UTIL_FORMATSTRINGTEST("ffffffffffffffff", "%p", reinterpret_cast<void*>(0xffffffffffffffff));
    NNT_UTIL_FORMATSTRINGTEST("  ffffffffffffffff", "%18p", reinterpret_cast<void*>(0xffffffffffffffff));
#else
#error "サポートされていないポインタのサイズです。"
#endif
}

#if defined(NNT_DETAIL_UTIL_FORMATSTRING_ENABLE_FP)

TEST_F(FormatStringTest, FloatingPointF)
{
    // 負の数は無視される（デフォルトになる）
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("1.000000", "%.*f", -3, 1.0);

    // %f
    // 精度は、小数点以下の桁数。デフォルトは6
    // 下位の桁は処理系定義の方法で丸められる。四捨五入する。
    NNT_UTIL_FORMATSTRINGTEST_NONTINY(" 1.234568", "%9f", 1.2345678);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("+1.234568", "%+9f", 1.2345678);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("    1.235", "%9.3f", 1.2345678);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("-1.234568", "%9f", -1.2345678);
    // 整数部が大きい場合
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("-1234560000000000.000000", "%f", -123456e10);
    // # 明示的に 精度 0 を指定し、# フラグが指定されていないなら、小数点を表示しない
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("1", "%.0f", 1.2345678);
    // 精度0でも0を表示する
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("0", "%.0f", 0.1);
    // long double
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("    1.235", "%9.3Lf", 1.2345678L);
}

TEST_F(FormatStringTest, FloatingPointE)
{
    // %e
    // 引数が0の場合、指数は0
    // 指数部は、最低でも2桁
    // 精度は、小数点以下の桁数。デフォルトは 6。
    // 下位の桁は処理系定義の方法で丸められる。四捨五入する。
    NNT_UTIL_FORMATSTRINGTEST_NONTINY(" 1.234568e+00", "%13e", 1.2345678);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("+1.234568e+00", "%+13e", 1.2345678);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("    1.235e+00", "%13.3e", 1.2345678);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("-1.234568e+00", "%13e", -1.2345678);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY(" 1.234568e+12", "%13e", 1.2345678e12);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("1.234568e+123", "%13e", 1.2345678e123);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY(" 1.234568e-12", "%13e", 1.2345678e-12);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("1.234568e-123", "%13e", 1.2345678e-123);
    // %E
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("    1.235E+00", "%13.3E", 1.2345678);
    // long double
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("    1.235e+00", "%13.3Le", 1.2345678L);
}

TEST_F(FormatStringTest, FloatingPointG)
{
    // %g
    // 精度は、有効数字の桁数。デフォルトは 6。
    // 指数部が -4 未満は e 形式で表示
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("  1.23457e-05", "%13g", 1.2345678e-5);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("  0.000123457", "%13g", 1.2345678e-4);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("      1.23457", "%13g", 1.2345678);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("       123457", "%13g", 1.2345678e5);
    // 指数部が精度以上（f 形式で表示すると整数部の桁数が有効数字の桁数を超える）なら e 形式で表示
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("  1.23457e+06", "%13g", 1.2345678e6);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY(" +1.23457e+06", "%+13g", 1.2345678e6);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("      1234568", "%13.7g", 1.2345678e6);
    // 小数部の末尾の0は取り除かれる
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("      1.2", "%9g", 1.2);
    // 小数部が0なら小数点も取り除かれる
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("       12", "%9g", 12.0);
    // long double
    NNT_UTIL_FORMATSTRINGTEST_NONTINY(" +1.23457e+06", "%+13Lg", 1.2345678e6L);
    // %G
    NNT_UTIL_FORMATSTRINGTEST_NONTINY(" +1.23457E+06", "%+13G", 1.2345678e6);
}

TEST_F(FormatStringTest, FloatingPointSharpF)
{
    // %#FP
    // #: 小数部が存在しなくても小数点を含める
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("1.", "%#.0f", 1.2345678);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("1.e+00", "%#.0e", 1.2345678);
}

TEST_F(FormatStringTest, FloatingPointSharpG)
{
    // %#g
    // #: 末尾の0を取り除かない
    // 小数部の末尾の0は取り除かれる
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("  1.20000", "%#9g", 1.2);
}

TEST_F(FormatStringTest, FloatingPointInfinity)
{
    // INF, NAN ("INF" の 表示は処理系定義。例えば "INFINITY" でもよい。)
    const double nan = std::numeric_limits<double>::quiet_NaN();
    const double infinity = std::numeric_limits<double>::infinity();
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("   inf,  -inf,   nan", "%6.3f,%6.3f,%6.3f", infinity, -infinity, nan);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("   INF,  -INF,   NAN", "%6.3F,%6.3F,%6.3F", infinity, -infinity, nan);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("   inf,  -inf,   nan", "%6.3e,%6.3e,%6.3e", infinity, -infinity, nan);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("   INF,  -INF,   NAN", "%6.3E,%6.3E,%6.3E", infinity, -infinity, nan);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("   inf,  -inf,   nan", "%6.3g,%6.3g,%6.3g", infinity, -infinity, nan);
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("   INF,  -INF,   NAN", "%6.3G,%6.3G,%6.3G", infinity, -infinity, nan);
}

TEST_F(FormatStringTest, FloatingPointComplex)
{
    // 複合
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("1.234e+03 abcde  +001", "%-10.3e%s%+*.*d", 1234.0, "abcde", 6, 3, 1);
}

#else // #if defined(NNT_DETAIL_UTIL_FORMATSTRING_ENABLE_FP)

TEST_F(FormatStringTest, FloatingPointNotSupported)
{
    // 浮動小数点数のフォーマットがサポートされていないことを確認する
    // （サポートされるようになった際、テストを有効化しないとテストが失敗するようにする）
    NNT_UTIL_FORMATSTRINGTEST_NONTINY("%f", "%f", 1.0);
}

#endif  // #if defined(NNT_DETAIL_UTIL_FORMATSTRING_ENABLE_FP)
