﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

////////////////////////////////////////////////////////////////
/**
 * 圧縮/伸長ライブラリのテスト
 */
////////////////////////////////////////////////////////////////

#include <algorithm>
#include <cstring>
#include <cstdlib>

#include <nnt/nntest.h>

#include <nn/os.h>

#include <nn/util/util_Compression.h>
#include <nn/util/util_Decompression.h>
#include <nn/util/util_StreamingCompression.h>
#include <nn/util/util_StreamingDecompression.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>

// エラーパターンのテストであることを明示するために使用するログ出力
// Release ビルド時はエラーログが表示されないためこちらも非表示に
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
#define NNT_UTIL_COMPRESSION_ERROR_PATTERN_LOG      NN_LOG
#elif defined(NN_SDK_BUILD_RELEASE)
#define NNT_UTIL_COMPRESSION_ERROR_PATTERN_LOG(...) static_cast<void>(0)
#else
#error "ビルドタイプマクロが定義されていません。"
#endif

namespace
{

// テスト用のランダムなデータ
const uint8_t TestData[] ={0x08, 0x01, 0xDF, 0x02, 0x7F, 0x03, 0x01, 0x22, 0x00,
     0x02, 0x11, 0x01, 0x03, 0x11, 0x01, 0xFF, 0xC4, 0x00, 0x1F, 0x00, 0x00, 0x01,
     0x05, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B,
     0xFF, 0xC4, 0x00, 0xB5, 0x10, 0x00, 0x02, 0x01, 0x03, 0x03, 0x02, 0x04, 0x03,
     0x05, 0x05, 0x04, 0x04, 0x00, 0x00, 0x01, 0x7D, 0x01, 0x02, 0x03, 0x00, 0x04,
     0x11, 0x05, 0x12, 0x21, 0x31, 0x41, 0x06, 0x13, 0x51, 0x61, 0x07, 0x22, 0x71,
     0x14, 0x32, 0x81, 0x91, 0xA1, 0x08, 0x23, 0x42, 0xB1, 0xC1, 0x15, 0x52, 0xD1,
     0xF0, 0x24, 0x33, 0x62, 0x72, 0x82, 0x09, 0x0A, 0x16, 0x17, 0x18, 0x19, 0x1A,
     0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A,
     0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4A, 0x53, 0x54, 0x55, 0x56, 0x57,
     0x58, 0x59, 0x5A, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6A, 0x73, 0x74,
     0x75, 0x76, 0x77, 0x78, 0x79, 0x7A, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89,
     0x8A, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9A, 0xA2, 0xA3, 0xA4,
     0xA5, 0xA6, 0xA7, 0xA8, 0xA9, 0xAA, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7, 0xB8,
     0xB9, 0xBA, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8, 0xC9, 0xCA, 0xD2, 0xD3,
     0xD4, 0xD5, 0xD6, 0xD7, 0xD8, 0xD9, 0xDA, 0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6,
     0xE7, 0xE8, 0xE9, 0xEA, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8, 0xF9,
     0xFA, 0xFF, 0xC4, 0x00, 0x1F, 0x01, 0x00, 0x03, 0x01, 0x01, 0x01, 0x01, 0x01,
     0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x03,
     0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0xFF, 0xC4, 0x00, 0xB5, 0x11,
     0x00, 0x02, 0x01, 0x02, 0x04, 0x04, 0x03, 0x04, 0x07, 0x05, 0x04, 0x04, 0x00,
     0x01, 0x02, 0x77, 0x00, 0x01, 0x02, 0x03, 0x11, 0x04, 0x05, 0x21, 0x31, 0x06,
     0x12, 0x41, 0x51, 0x07, 0x61, 0x71, 0x13, 0x22, 0x32, 0x81, 0x08, 0x14, 0x42,
     0x91, 0xA1, 0xB1, 0xC1, 0x09, 0x23, 0x33, 0x52, 0xF0, 0x15, 0x62, 0x72, 0xD1,
     0x0A, 0x16, 0x24, 0x34, 0xE1, 0x25, 0xF1, 0x17, 0x18, 0x19, 0x1A, 0x26, 0x27,
     0x28, 0x29, 0x2A, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x43, 0x44, 0x45, 0x46,
     0x47, 0x48, 0x49, 0x4A, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5A, 0x63,
     0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6A, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
     0x79, 0x7A, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8A, 0x92, 0x93,
     0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9A, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7,
     0xA8, 0xA9, 0xAA, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7, 0xB8, 0xB9, 0xBA, 0xC2,
     0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8, 0xC9, 0xCA, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6,
     0xD7, 0xD8, 0xD9, 0xDA, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7, 0xE8, 0xE9, 0xEA,
     0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8, 0xF9, 0xFA, 0xFF, 0xDD, 0x00, 0x04,
     0x00, 0x28, 0xFF, 0xDA, 0x00, 0x0C, 0x03, 0x01, 0x00, 0x02, 0x11, 0x03, 0x11,
     0x00, 0x3F, 0x00, 0xFC, 0xB3, 0x86, 0x2C, 0x7D, 0xE1, 0xEB, 0x9F, 0xF1, 0xFE,
     0x58, 0xFE, 0x55, 0x65, 0x63, 0x03, 0x1C, 0x7E, 0x1F, 0xE7, 0xA9, 0xFE, 0x75,
     0x6D, 0x21, 0x04, 0xF5, 0xE3, 0x38, 0xC7, 0xF8, 0xFB, 0x7F, 0x9C, 0x1A, 0x9C,
     0x44, 0x01, 0xC0, 0x00, 0xF1, 0xC9, 0x3D, 0xBD, 0x07, 0x4A, 0xF0, 0x65, 0x3B,
     0xAD, 0x74, 0xB6, 0xFF, 0x00, 0x87, 0xF4, 0x8E, 0xD8, 0x6C, 0xDF, 0x9F, 0xE5,
     0xFF, 0x00, 0x0E, 0x54, 0x8E, 0x30, 0x73, 0xC6, 0xDF, 0x4E, 0xDC, 0xE7, 0xDF,
     0x27, 0xA7, 0x4A, 0xB2, 0x90, 0xF1, 0xDF, 0xFF, 0x00, 0xAF, 0xE9, 0xDE, 0xA7,
     0x54, 0xCF, 0x18, 0x39, 0xCF, 0x6F, 0xE2, 0xFD, 0x2A, 0xC4, 0x71, 0xE7, 0x23,
     0x1D, 0x06, 0x71, 0xCF, 0xFF, 0x00, 0xAE, 0xA5, 0x3D, 0x6E, 0xB5, 0xB3, 0xE8,
     0x1C, 0x8A, 0xF7, 0xF3, 0xDB, 0xEE, 0xFF, 0x00, 0x82, 0x40, 0xB1, 0x0F, 0xC3,
     0xD8, 0x7F, 0x80, 0x3F, 0xE7, 0xD6, 0xA6, 0x58, 0xB9, 0xC6, 0x30, 0x7F, 0x3F,
     0xEB, 0x56, 0x44, 0x6A, 0xA7, 0x23, 0x3F, 0x8D, 0x58, 0x54, 0xC8, 0xCF, 0x3F,
     0x87, 0x6A, 0xD7, 0x9D, 0x79, 0xA3, 0x4B, 0x3D, 0x34, 0xDD, 0xD9, 0x15, 0x92,
     0x1E, 0xFC, 0xF4, 0xFF, 0x00, 0x3D, 0xBA, 0xFB, 0x54, 0xBE, 0x57, 0x27, 0x83,
     0xD7, 0xBF, 0xF9, 0x5C, 0xD5, 0xC8, 0xA2, 0xFA, 0x67, 0x1D, 0xFF, 0x00, 0x0F,
     0xF3, 0x8F, 0xEB, 0x56, 0x52, 0x2C, 0xF0, 0x70, 0x47, 0xA7, 0x6F, 0xF3, 0xED,
     0x47, 0x3A, 0xF3, 0x2E, 0x31, 0x69, 0xDF, 0x6F, 0xCF, 0x62, 0x8A, 0xC3, 0xD3,};

// 圧縮されたデータを置いておくバッファ
char           g_ZlibCompressedBuffer[sizeof(TestData) / sizeof(TestData[0])];
size_t         g_ZlibCompressedDataSize;
char           g_GzipCompressedBuffer[sizeof(TestData) / sizeof(TestData[0])];
size_t         g_GzipCompressedDataSize;
char           g_DeflateCompressedBuffer[sizeof(TestData) / sizeof(TestData[0])];
size_t         g_DeflateCompressedDataSize;

// テスト用の各種定数
const size_t  HeapSize                   = nn::os::MemoryBlockUnitSize; // Heap のサイズ
const size_t  CompressedBufferOffsetSize = 64;                          // ヘッダとフッタを考慮したオフセットのサイズ
const size_t  CompressedBufferSize       = sizeof(TestData) / sizeof(TestData[0]) + CompressedBufferOffsetSize; // 圧縮されたデータを入れるバッファのサイズ（無圧縮テスト用にオフセットを付加）
const size_t  DecompressedBufferSize     = sizeof(TestData) / sizeof(TestData[0]);                              // 伸長されたデータを入れるバッファのサイズ
const size_t  OptionalHeaderSize         = 5;  // 無圧縮時にヘッダに付加される情報のサイズ
const size_t  ZlibHeaderSize             = 2;  // zlib 形式のヘッダサイズ
const size_t  GzipHeaderSize             = 10; // gzip 形式のヘッダサイズ

enum DataType
{
    DataType_Zlib = 0,
    DataType_Gzip,
    DataType_Deflate
};


/**
 * @brief   BasicCompressTest で利用するテストフィクスチャです。
 */
class BasicCompressTest : public testing::TestWithParam<int>
{
protected:

    // Heap 関連のメンバ変数
    uintptr_t            m_Heap;
    nn::Result           m_HeapResult;
    nn::Result           m_Result;

    // 入出力に使用するバッファへのポインタ
    void* m_pCompressedBuffer;
    void* m_pDecompressedBuffer;

    /**
     * @brief テスト開始時に毎回呼び出される関数です。
     */
    virtual void SetUp()
    {
        // 各バッファの動的確保
        m_pCompressedBuffer = std::malloc(CompressedBufferSize);
        ASSERT_TRUE(m_pCompressedBuffer != nullptr);

        m_pDecompressedBuffer = std::malloc(DecompressedBufferSize);
        ASSERT_TRUE(m_pCompressedBuffer != nullptr);
    }

    /**
     * @brief テスト終了時に毎回呼び出される関数です。
     */
    virtual void TearDown()
    {
        // メモリの解放
        std::free(m_pCompressedBuffer);
        std::free(m_pDecompressedBuffer);
    }
};

// Src 側をストリーミング伸長する関数
bool TestSrcStreaming(DataType dataType, char* pOutputBuf, size_t outputBufSize, const char* pSrcData, size_t srcDataSize, nn::util::StreamingDecompressZlibContext* pContext)
{
    char*          pCompressedBufferPtr     = const_cast<char*>(pSrcData);   // 元の圧縮されたデータを差すポインタ
    size_t         compressedDataRemainSize = srcDataSize;                   // 圧縮データの残りサイズ

    char*          pDstCurrent   = pOutputBuf;                               // 最終的に展開されたデータを置くバッファへのポインタ
    size_t         remainDstSize = outputBufSize;                            // 展開されたデータを置く場所の残りサイズ

    const size_t   SrcBufSize = 100;   // ストリーミング展開する入力データを一旦置くバッファのサイズ
    char           srcBuf[SrcBufSize]; // ストリーミング展開する入力データを一旦置くバッファ(小さいのでスタックから確保)

    size_t         dstConsumedSize;    // 出力されたデータ量
    size_t         srcConsumedSize;    // 消費されたデータ量

    bool           isSuccess;          // 伸長結果の確認用

    // ストリーミング伸長用コンテキストの初期化
    // ストリーミング伸長用コンテキストの初期化
    switch(dataType)
    {
    case DataType_Zlib:
        nn::util::InitializeStreamingDecompressZlibContext(pContext);
        break;

    case DataType_Gzip:
        nn::util::InitializeStreamingDecompressGzipContext(pContext);
        break;

    case DataType_Deflate:
        nn::util::InitializeStreamingDecompressDeflateContext(pContext);
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    // ストリーミング伸長処理
    for(;;)
    {
        // バッファのサイズに残りのサイズが満たない時は残りのサイズ分、バッファに入れる。
        size_t inputDataSize = std::min(compressedDataRemainSize, SrcBufSize);
        memcpy(srcBuf, pCompressedBufferPtr, inputDataSize);
        pCompressedBufferPtr     += inputDataSize;
        compressedDataRemainSize -= inputDataSize;

        switch(dataType)
        {
        case DataType_Zlib:
            isSuccess = nn::util::StreamingDecompressZlib(&dstConsumedSize, &srcConsumedSize, pDstCurrent, remainDstSize, srcBuf, SrcBufSize, pContext);
            break;

        case DataType_Gzip:
            isSuccess = nn::util::StreamingDecompressGzip(&dstConsumedSize, &srcConsumedSize, pDstCurrent, remainDstSize, srcBuf, SrcBufSize, pContext);
            break;

        case DataType_Deflate:
            isSuccess = nn::util::StreamingDecompressDeflate(&dstConsumedSize, &srcConsumedSize, pDstCurrent, remainDstSize, srcBuf, SrcBufSize, pContext);
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

        if(isSuccess)
        {
            // 両方とも消費していなければ処理は終了
            if(dstConsumedSize == 0 && srcConsumedSize == 0)
            {
                break;
            }

            // 出力された分、出力バッファのポインタを進める。
            if(remainDstSize >= dstConsumedSize)
            {
                pDstCurrent   += dstConsumedSize;
                remainDstSize -= dstConsumedSize;
            }
            else
            {
                // 元データのサイズが圧縮データを超えている。
                EXPECT_TRUE(remainDstSize >= dstConsumedSize);
                return false;
            }
        }
        else
        {
            // 本来のライブラリとして返り得るエラー
            return false;
        }
    }

    return true;
}

// Src 側も Dst 側もストリーミングするテスト
bool TestSrcAndDstStreaming(DataType dataType, char* pOutputBuf, size_t outputBufSize, const char* pSrcData, size_t srcDataSize, nn::util::StreamingDecompressZlibContext* pContext)
{
    char*          pCompressedBufferPtr     = const_cast<char*>(pSrcData);   // 元の圧縮されたデータを差すポインタ
    size_t         compressedDataRemainSize = srcDataSize;                   // 圧縮データの残りサイズ

    char*          pDstCurrent   = pOutputBuf;               // 最終的に展開されたデータを置くバッファへのポインタ
    size_t         remainDstSize = outputBufSize;            // 展開されたデータを置く場所の残りサイズ

    const size_t   SrcBufSize = 100;                // ストリーミング展開する入力データを一旦置くバッファのサイズ
    char           srcBuf[SrcBufSize];              // ストリーミング展開する入力データを一旦置くバッファ(小さいのでスタックから確保)
    char*          pSrcBufCurrent    = nullptr;     // ストリーミング展開する入力データを一旦置くバッファ上の位置を示すポインタ
    size_t         remainSrcBufSize = 0;            // ストリーミング展開する入力データを一旦置くバッファの残りサイズ

    const size_t   DstBufSize = 100;                // ストリーミング展開する出力データを一旦置くバッファのサイズ
    char           dstBuf[DstBufSize];              // ストリーミング展開する出力データを一旦置くバッファ

    size_t         dstConsumedSize;    // 出力されたデータ量
    size_t         srcConsumedSize;    // 消費されたデータ量

    bool           isSuccess;          // 伸長結果の確認用

    // ストリーミング伸長用コンテキストの初期化
    switch(dataType)
    {
    case DataType_Zlib:
        nn::util::InitializeStreamingDecompressZlibContext(pContext);
        break;

    case DataType_Gzip:
        nn::util::InitializeStreamingDecompressGzipContext(pContext);
        break;

    case DataType_Deflate:
        nn::util::InitializeStreamingDecompressDeflateContext(pContext);
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    // ストリーミング伸長処理
    for(;;)
    {
        // Src 側の temp バッファが空になっていたら補充する
        if(remainSrcBufSize == 0)
        {
            // バッファのサイズに残りのサイズが満たない時は残りのサイズ分、バッファに入れる。
            size_t inputSize = std::min(compressedDataRemainSize, SrcBufSize);
            memcpy(srcBuf, pCompressedBufferPtr, inputSize);
            pCompressedBufferPtr     += inputSize;
            compressedDataRemainSize -= inputSize;
            pSrcBufCurrent            = srcBuf;
            remainSrcBufSize          = inputSize;
        }

        switch(dataType)
        {
        case DataType_Zlib:
            isSuccess = nn::util::StreamingDecompressZlib(&dstConsumedSize, &srcConsumedSize, dstBuf, DstBufSize, pSrcBufCurrent, remainSrcBufSize, pContext);
            break;

        case DataType_Gzip:
            isSuccess = nn::util::StreamingDecompressGzip(&dstConsumedSize, &srcConsumedSize, dstBuf, DstBufSize, pSrcBufCurrent, remainSrcBufSize, pContext);
            break;

        case DataType_Deflate:
            isSuccess = nn::util::StreamingDecompressDeflate(&dstConsumedSize, &srcConsumedSize, dstBuf, DstBufSize, pSrcBufCurrent, remainSrcBufSize, pContext);
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }


        if(isSuccess)
        {
            // 両方とも消費していなければ処理は終了
            if(dstConsumedSize == 0 && srcConsumedSize == 0)
            {
                break;
            }

            if(remainDstSize >= dstConsumedSize)
            {
                // Dst 側は常に吐き出す
                memcpy(pDstCurrent, dstBuf, dstConsumedSize);
                pDstCurrent   += dstConsumedSize;
                remainDstSize -= dstConsumedSize;

                // Src 側は続きからおこなう。
                pSrcBufCurrent   += srcConsumedSize;
                remainSrcBufSize -= srcConsumedSize;
            }
            else
            {
                // 元データのサイズが圧縮データを超えている。
                EXPECT_TRUE(remainDstSize >= dstConsumedSize);
                return false;
            }
        }
        else
        {
            // 本来のライブラリとして返り得るエラー
            return false;
        }
    }

    return true;
}

// Src 側をストリーミング圧縮する関数
bool TestSrcStreamingCompression(DataType dataType, size_t* outConsumedSize, char* pOutputBuf, size_t outputBufSize, const char* pSrcData, size_t srcDataSize, int memLevel, int compressionLevel)
{
    const char*    pOriginalBufferPtr       = pSrcData;                      // オリジナルデータを差すポインタ
    size_t         originalDataRemainSize   = srcDataSize;                   // オリジナルデータの残りサイズ

    char*          pDstCurrent   = pOutputBuf;                               // 最終的に圧縮されたデータを置くバッファへのポインタ
    size_t         remainDstSize = outputBufSize;                            // 圧縮されたデータを置く場所の残りサイズ

    const size_t   SrcBufSize = 100;   // ストリーミング圧縮する入力データを一旦置くバッファのサイズ
    char           srcBuf[SrcBufSize]; // ストリーミング圧縮する入力データを一旦置くバッファ(小さいのでスタックから確保)

    size_t         dstConsumedSize;    // 出力されたデータ量
    size_t         srcConsumedSize;    // 消費されたデータ量

    bool           isSuccess;          // 伸長結果の確認用

    // ストリーミング圧縮用コンテキストの初期化
    nn::util::StreamingCompressZlibContext context;
    void* pWorkBuffer;
    size_t workBufferSize;
    switch(dataType)
    {
    case DataType_Zlib:
        workBufferSize = NN_UTIL_CALCULATE_COMPRESS_ZLIB_WORKBUFFER_SIZE(memLevel);
        pWorkBuffer = std::malloc(workBufferSize);
        nn::util::InitializeStreamingCompressZlibContext(&context, pWorkBuffer, workBufferSize, memLevel, compressionLevel);
        break;

    case DataType_Gzip:
        workBufferSize = NN_UTIL_CALCULATE_COMPRESS_GZIP_WORKBUFFER_SIZE(memLevel);
        pWorkBuffer = std::malloc(workBufferSize);
        nn::util::InitializeStreamingCompressGzipContext(&context, pWorkBuffer, workBufferSize, memLevel, compressionLevel);
        break;

    case DataType_Deflate:
        workBufferSize = NN_UTIL_CALCULATE_COMPRESS_DEFLATE_WORKBUFFER_SIZE(memLevel);
        pWorkBuffer = std::malloc(workBufferSize);
        nn::util::InitializeStreamingCompressDeflateContext(&context, pWorkBuffer, workBufferSize, memLevel, compressionLevel);
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    // ストリーミング圧縮処理
    for(;;)
    {
        // バッファのサイズに残りのサイズが満たない時は残りのサイズ分、バッファに入れる。
        size_t inputDataSize = std::min(originalDataRemainSize, SrcBufSize);
        memcpy(srcBuf, pOriginalBufferPtr, inputDataSize);
        pOriginalBufferPtr     += inputDataSize;
        originalDataRemainSize -= inputDataSize;

        switch(dataType)
        {
        case DataType_Zlib:
            isSuccess = nn::util::StreamingCompressZlib(&dstConsumedSize, &srcConsumedSize, pDstCurrent, remainDstSize, srcBuf, inputDataSize, &context);
            break;

        case DataType_Gzip:
            isSuccess = nn::util::StreamingCompressGzip(&dstConsumedSize, &srcConsumedSize, pDstCurrent, remainDstSize, srcBuf, inputDataSize, &context);
            break;

        case DataType_Deflate:
            isSuccess = nn::util::StreamingCompressDeflate(&dstConsumedSize, &srcConsumedSize, pDstCurrent, remainDstSize, srcBuf, inputDataSize, &context);
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

        if(isSuccess)
        {
            // 両方とも消費していなければ処理は終了
            if(dstConsumedSize == 0 && srcConsumedSize == 0)
            {
                break;
            }

            if(srcConsumedSize != inputDataSize)
            {
                // 出力バッファは十分なサイズ与えているので
                // 入力は全て処理されるはず
                EXPECT_TRUE(srcConsumedSize == inputDataSize);
                return false;
            }

            // 出力された分、出力バッファのポインタを進める。
            if(remainDstSize >= dstConsumedSize)
            {
                pDstCurrent   += dstConsumedSize;
                remainDstSize -= dstConsumedSize;
                *outConsumedSize += dstConsumedSize;
            }
            else
            {
                // 元データのサイズが圧縮データを超えている。
                EXPECT_TRUE(remainDstSize >= dstConsumedSize);
                return false;
            }
        }
        else
        {
            // 本来のライブラリとして返り得るエラー
            return false;
        }
    }

    // ワークバッファを解放
    std::free(pWorkBuffer);

    return true;
}

// Src 側も Dst 側もストリーミング圧縮する関数
bool TestSrcAndDstStreamingCompression(DataType dataType, size_t* outConsumedSize, char* pOutputBuf, size_t outputBufSize, const char* pSrcData, size_t srcDataSize, int memLevel, int compressionLevel)
{
    const char*    pOriginalBufferPtr       = pSrcData;     // オリジナルデータを差すポインタ
    size_t         originalDataRemainSize   = srcDataSize;  // オリジナルデータの残りサイズ

    char*          pDstCurrent   = pOutputBuf;              // 最終的に圧縮されたデータを置くバッファへのポインタ
    size_t         remainDstSize = outputBufSize;           // 圧縮されたデータを置く場所の残りサイズ

    const size_t   SrcBufSize = 100;                        // ストリーミング圧縮する入力データを一旦置くバッファのサイズ
    char           srcBuf[SrcBufSize];                      // ストリーミング圧縮する入力データを一旦置くバッファ(小さいのでスタックから確保)
    char*          pSrcBufCurrent   = nullptr;              // ストリーミング展開する入力データを一旦置くバッファ上の位置を示すポインタ
    size_t         remainSrcBufSize = 0;                    // ストリーミング展開する入力データを一旦置くバッファの残りサイズ

    const size_t   DstBufSize = 100;                        // ストリーミング展開する出力データを一旦置くバッファのサイズ
    char           dstBuf[DstBufSize];                      // ストリーミング展開する出力データを一旦置くバッファ

    size_t         dstConsumedSize;                         // 出力されたデータ量
    size_t         srcConsumedSize;                         // 消費されたデータ量

    bool           isSuccess;                               // 伸長結果の確認用

    // ストリーミング圧縮用コンテキストの初期化
    nn::util::StreamingCompressZlibContext context;
    void* pWorkBuffer;
    size_t workBufferSize;
    switch(dataType)
    {
    case DataType_Zlib:
        workBufferSize = NN_UTIL_CALCULATE_COMPRESS_ZLIB_WORKBUFFER_SIZE(memLevel);
        pWorkBuffer = std::malloc(workBufferSize);
        nn::util::InitializeStreamingCompressZlibContext(&context, pWorkBuffer, workBufferSize, memLevel, compressionLevel);
        break;

    case DataType_Gzip:
        workBufferSize = NN_UTIL_CALCULATE_COMPRESS_GZIP_WORKBUFFER_SIZE(memLevel);
        pWorkBuffer = std::malloc(workBufferSize);
        nn::util::InitializeStreamingCompressGzipContext(&context, pWorkBuffer, workBufferSize, memLevel, compressionLevel);
        break;

    case DataType_Deflate:
        workBufferSize = NN_UTIL_CALCULATE_COMPRESS_DEFLATE_WORKBUFFER_SIZE(memLevel);
        pWorkBuffer = std::malloc(workBufferSize);
        nn::util::InitializeStreamingCompressDeflateContext(&context, pWorkBuffer, workBufferSize, memLevel, compressionLevel);
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    // ストリーミング圧縮処理
    for(;;)
    {
        // Src 側の temp バッファが空になっていたら補充する
        if(remainSrcBufSize == 0)
        {
            // バッファのサイズに残りのサイズが満たない時は残りのサイズ分、バッファに入れる。
            size_t inputDataSize = std::min(originalDataRemainSize, SrcBufSize);
            memcpy(srcBuf, pOriginalBufferPtr, inputDataSize);
            pOriginalBufferPtr     += inputDataSize;
            originalDataRemainSize -= inputDataSize;
            pSrcBufCurrent          = srcBuf;
            remainSrcBufSize        = inputDataSize;
        }

        switch(dataType)
        {
        case DataType_Zlib:
            isSuccess = nn::util::StreamingCompressZlib(&dstConsumedSize, &srcConsumedSize, dstBuf, DstBufSize, pSrcBufCurrent, remainSrcBufSize, &context);
            break;

        case DataType_Gzip:
            isSuccess = nn::util::StreamingCompressGzip(&dstConsumedSize, &srcConsumedSize, dstBuf, DstBufSize, pSrcBufCurrent, remainSrcBufSize, &context);
            break;

        case DataType_Deflate:
            isSuccess = nn::util::StreamingCompressDeflate(&dstConsumedSize, &srcConsumedSize, dstBuf, DstBufSize, pSrcBufCurrent, remainSrcBufSize, &context);
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

        if(isSuccess)
        {
            // 両方とも消費していなければ処理は終了
            if(dstConsumedSize == 0 && srcConsumedSize == 0)
            {
                break;
            }

            // 出力された分、出力バッファのポインタを進める。
            if(remainDstSize >= dstConsumedSize)
            {
                // Dst 側は常に吐き出す
                memcpy(pDstCurrent, dstBuf, dstConsumedSize);
                pDstCurrent      += dstConsumedSize;
                remainDstSize    -= dstConsumedSize;
                *outConsumedSize += dstConsumedSize;

                // Src 側は続きからおこなう。
                pSrcBufCurrent   += srcConsumedSize;
                remainSrcBufSize -= srcConsumedSize;
            }
            else
            {
                // 元データのサイズが圧縮データを超えている。
                EXPECT_TRUE(remainDstSize >= dstConsumedSize);
                return false;
            }
        }
        else
        {
            // 本来のライブラリとして返り得るエラー
            return false;
        }
    }

    // ワークバッファを解放
    std::free(pWorkBuffer);

    return true;
}

// 圧縮データを展開して元データと一致するか確認する関数
bool VerifyCompressedData(DataType dataType, void* pDecompressedBuffer, const void* pCompressedBuffer, size_t compressedBufferSize, const void* originalData, size_t originalDataSize)
{
    // バッファを確保
    void* pDecompressWorkBuffer = std::malloc(nn::util::DecompressZlibWorkBufferSize);

    switch(dataType)
    {
    case DataType_Zlib:
        EXPECT_TRUE(
            nn::util::DecompressZlib(pDecompressedBuffer,       DecompressedBufferSize,
                                       pCompressedBuffer,       compressedBufferSize,
                                       pDecompressWorkBuffer,   nn::util::DecompressZlibWorkBufferSize)
        );
        break;

    case DataType_Gzip:
        EXPECT_TRUE(
            nn::util::DecompressGzip(pDecompressedBuffer,       DecompressedBufferSize,
                                       pCompressedBuffer,       compressedBufferSize,
                                       pDecompressWorkBuffer,   nn::util::DecompressZlibWorkBufferSize)
        );
        break;

    case DataType_Deflate:
        EXPECT_TRUE(
            nn::util::DecompressDeflate(pDecompressedBuffer,        DecompressedBufferSize,
                                           pCompressedBuffer,       compressedBufferSize,
                                           pDecompressWorkBuffer,   nn::util::DecompressZlibWorkBufferSize)
        );
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    // バッファを解放
    std::free(pDecompressWorkBuffer);

    if(originalDataSize != DecompressedBufferSize)
    {
        EXPECT_TRUE(originalDataSize == DecompressedBufferSize);
        return false;
    }

    // データを比較
    return std::memcmp(originalData, pDecompressedBuffer, originalDataSize) == 0;
}


}

// zlib 形式の基本的な圧縮と伸長のテスト
TEST_F(BasicCompressTest, CompressZlibAndDecompressZlib)
{
    // 処理後のサイズ取得用変数
    size_t      outSize;

    // アロケートしたいサイズの合計がヒープサイズを超えていないかチェック
    const size_t AllocateSizeSum = CompressedBufferSize + DecompressedBufferSize + nn::util::CompressZlibWorkBufferSizeDefault + nn::util::DecompressZlibWorkBufferSize;
    ASSERT_LE(AllocateSizeSum, HeapSize);

    // ワークバッファをヒープから取得
    void* pCompressWorkBuffer = std::malloc(nn::util::CompressZlibWorkBufferSizeDefault);
    ASSERT_TRUE(pCompressWorkBuffer != nullptr);

    void* pDecompressWorkBuffer = std::malloc(nn::util::DecompressZlibWorkBufferSize);
    ASSERT_TRUE(pDecompressWorkBuffer != nullptr);

    // ----------------------------------- 圧縮(zlib 形式) --------------------------------------
    bool compressResult = nn::util::CompressZlib(&outSize,
                                  m_pCompressedBuffer,    CompressedBufferSize,
                                  TestData,             sizeof(TestData) / sizeof(TestData[0]),
                                  pCompressWorkBuffer,  nn::util::CompressZlibWorkBufferSizeDefault);

    ASSERT_TRUE(compressResult);

    // zlib の圧縮データをストリーミング展開でも使うため、ファイルグローバルなバッファに保存
    g_ZlibCompressedDataSize = outSize;
    memcpy(g_ZlibCompressedBuffer, m_pCompressedBuffer, outSize);


    // ----------------------------------- 伸長(zlib 形式) -----------------------------------
    bool decompressResult = nn::util::DecompressZlib(m_pDecompressedBuffer,      DecompressedBufferSize,
                                                m_pCompressedBuffer,             outSize,
                                                pDecompressWorkBuffer,         nn::util::DecompressZlibWorkBufferSize);

    ASSERT_TRUE(decompressResult);
    EXPECT_EQ(0, std::memcmp(TestData, m_pDecompressedBuffer, sizeof(TestData)));

    // ワークバッファのメモリブロックの解放
    std::free(pCompressWorkBuffer);
    std::free(pDecompressWorkBuffer);
}


// gzip 形式の基本的な圧縮と伸長のテスト
TEST_F(BasicCompressTest, CompressGzipAndDecompressGzip)
{
    // 処理後のサイズ取得用
    size_t      outSize;

    // アロケートしたいサイズの合計がヒープサイズを超えていないかチェック
    const size_t AllocateSizeSum = CompressedBufferSize + DecompressedBufferSize + nn::util::CompressGzipWorkBufferSizeDefault + nn::util::DecompressGzipWorkBufferSize;
    ASSERT_LE(AllocateSizeSum, HeapSize);

    // ワークバッファをヒープから取得
    void* pCompressWorkBuffer = std::malloc(nn::util::CompressGzipWorkBufferSizeDefault);
    ASSERT_TRUE(pCompressWorkBuffer != nullptr);

    void* pDecompressWorkBuffer = std::malloc(nn::util::DecompressGzipWorkBufferSize);
    ASSERT_TRUE(pDecompressWorkBuffer != nullptr);


    // ----------------------------------- 圧縮(gzip 形式) -------------------------------------
    bool compressResult = nn::util::CompressGzip(&outSize,
                                  m_pCompressedBuffer,      CompressedBufferSize,
                                  TestData,               sizeof(TestData) / sizeof(TestData[0]),
                                  pCompressWorkBuffer,    nn::util::CompressGzipWorkBufferSizeDefault);

    ASSERT_TRUE(compressResult);

    // gzip の圧縮データをストリーミング展開でも使うため、ファイルグローバルなバッファに保存
    g_GzipCompressedDataSize = outSize;
    memcpy(g_GzipCompressedBuffer, m_pCompressedBuffer, outSize);


    // ----------------------------------- 伸長(gzip 形式) ----------------------------------
    // gzip 形式のデータは圧縮前のサイズが取れるため、サイズ取得関数のテストをここで行う
    EXPECT_EQ( sizeof(TestData) / sizeof(TestData[0]) , nn::util::GetGzipDecompressedSize(m_pCompressedBuffer, outSize));


    bool decompressResult = nn::util::DecompressGzip(m_pDecompressedBuffer,    DecompressedBufferSize,
                                                     m_pCompressedBuffer,      outSize,
                                                     pDecompressWorkBuffer,  nn::util::DecompressGzipWorkBufferSize);

    ASSERT_TRUE(decompressResult);
    EXPECT_EQ(0, std::memcmp(TestData, m_pDecompressedBuffer, sizeof(TestData)));

    // ワークバッファのメモリブロックの解放
    std::free(pCompressWorkBuffer);
    std::free(pDecompressWorkBuffer);

}

// deflate 形式の基本的な圧縮と伸長のテスト
TEST_F(BasicCompressTest, CompressDeflateAndDecompressDeflate)
{
    // 処理後のサイズ取得用
    size_t      outSize;

    // アロケートしたいサイズの合計がヒープサイズを超えていないかチェック
    const size_t AllocateSizeSum = CompressedBufferSize + DecompressedBufferSize + nn::util::CompressDeflateWorkBufferSizeDefault + nn::util::DecompressDeflateWorkBufferSize;
    ASSERT_LE(AllocateSizeSum, HeapSize);

    // ワークバッファをヒープから取得
    void* pCompressWorkBuffer = std::malloc(nn::util::CompressGzipWorkBufferSizeDefault);
    ASSERT_TRUE(pCompressWorkBuffer != nullptr);

    void* pDecompressWorkBuffer = std::malloc(nn::util::DecompressGzipWorkBufferSize);
    ASSERT_TRUE(pDecompressWorkBuffer != nullptr);


    // ----------------------------------- 圧縮(deflate 形式) ------------------------------------
    bool compressResult = nn::util::CompressDeflate(&outSize,
                                  m_pCompressedBuffer,       CompressedBufferSize,
                                  TestData,                sizeof(TestData) / sizeof(TestData[0]),
                                  pCompressWorkBuffer,     nn::util::CompressDeflateWorkBufferSizeDefault);

    ASSERT_TRUE(compressResult);

    // Deflate の圧縮データをストリーミング展開でも使うため、ファイルグローバルなバッファに保存
    g_DeflateCompressedDataSize = outSize;
    memcpy(g_DeflateCompressedBuffer, m_pCompressedBuffer, outSize);


    // ----------------------------------- 伸長(deflate 形式) ---------------------------------
    bool decompressResult = nn::util::DecompressDeflate(m_pDecompressedBuffer,  DecompressedBufferSize,
                                                   m_pCompressedBuffer,         outSize,
                                                   pDecompressWorkBuffer,     nn::util::DecompressDeflateWorkBufferSize);

    ASSERT_TRUE(decompressResult);
    EXPECT_EQ(0, std::memcmp(TestData, m_pDecompressedBuffer, sizeof(TestData)));

    // ワークバッファのメモリブロックの解放
    std::free(pCompressWorkBuffer);
    std::free(pDecompressWorkBuffer);
}

// zlib 形式のオプション指定圧縮と伸長のテスト
TEST_F(BasicCompressTest, OptionalCompressZlibAndDecompressZlib)
{
    // 圧縮時に指定する圧縮レベルとメモリ使用量
    const int   CompressionLevel = 3;
    const int   MemLevel         = 4;
    // 処理後のサイズ取得用変数
    size_t      outSize;

    // アロケートしたいサイズの合計がヒープサイズを超えていないかチェック
    const size_t AllocateSizeSum = CompressedBufferSize + DecompressedBufferSize + NN_UTIL_CALCULATE_COMPRESS_ZLIB_WORKBUFFER_SIZE(MemLevel) + nn::util::DecompressZlibWorkBufferSize;
    ASSERT_LE(AllocateSizeSum, HeapSize);

    // ワークバッファをヒープから取得
    void* pCompressWorkBuffer = std::malloc(NN_UTIL_CALCULATE_COMPRESS_ZLIB_WORKBUFFER_SIZE(MemLevel));
    ASSERT_TRUE(pCompressWorkBuffer != nullptr);

    void* pDecompressWorkBuffer = std::malloc(nn::util::DecompressZlibWorkBufferSize);
    ASSERT_TRUE(pDecompressWorkBuffer != nullptr);

    // ----------------------------------- 無圧縮(zlib 形式) --------------------------------------
    bool compressResult = nn::util::CompressZlib(&outSize,
                                  m_pCompressedBuffer, CompressedBufferSize,
                                  TestData,            sizeof(TestData) / sizeof(TestData[0]),
                                  pCompressWorkBuffer, NN_UTIL_CALCULATE_COMPRESS_ZLIB_WORKBUFFER_SIZE(MemLevel),
                                  MemLevel,            0);

    ASSERT_TRUE(compressResult);

    // ヘッダおよびトレイラを除いた無圧縮のデータが圧縮前データと一致するか確認
    char* pCompressedDataPayload = reinterpret_cast<char*>(m_pCompressedBuffer) + ZlibHeaderSize + OptionalHeaderSize;
    EXPECT_EQ(0, std::memcmp(TestData, pCompressedDataPayload, sizeof(TestData)));

    // ----------------------------------- 伸長(zlib 形式) -----------------------------------
    bool decompressResult = nn::util::DecompressZlib(m_pDecompressedBuffer,    DecompressedBufferSize,
                                                m_pCompressedBuffer,           outSize,
                                                pDecompressWorkBuffer,         nn::util::DecompressZlibWorkBufferSize);

    ASSERT_TRUE(decompressResult);
    EXPECT_EQ(0, std::memcmp(TestData, m_pDecompressedBuffer, sizeof(TestData)));

    // ----------------------------------- オプション指定圧縮(zlib 形式) --------------------------------------
    compressResult = nn::util::CompressZlib(&outSize,
                                  m_pCompressedBuffer, CompressedBufferSize,
                                  TestData,            sizeof(TestData) / sizeof(TestData[0]),
                                  pCompressWorkBuffer, NN_UTIL_CALCULATE_COMPRESS_ZLIB_WORKBUFFER_SIZE(MemLevel),
                                  MemLevel,            CompressionLevel);

    ASSERT_TRUE(compressResult);

    // ----------------------------------- 伸長(zlib 形式) -----------------------------------
    decompressResult = nn::util::DecompressZlib(m_pDecompressedBuffer,    DecompressedBufferSize,
                                           m_pCompressedBuffer,           outSize,
                                           pDecompressWorkBuffer,         nn::util::DecompressZlibWorkBufferSize);

    ASSERT_TRUE(decompressResult);
    EXPECT_EQ(0, std::memcmp(TestData, m_pDecompressedBuffer, sizeof(TestData)));

    // ワークバッファのメモリブロックの解放
    std::free(pCompressWorkBuffer);
    std::free(pDecompressWorkBuffer);
}

// gzip 形式のオプション指定圧縮と伸長のテスト
TEST_F(BasicCompressTest, OptionalCompressGzipAndDecompressGzip)
{
    // 圧縮時に指定する圧縮レベルとメモリ使用量
    const int   CompressionLevel = 3;
    const int   MemLevel         = 4;
    // 処理後のサイズ取得用変数
    size_t      outSize;

    // アロケートしたいサイズの合計がヒープサイズを超えていないかチェック
    const size_t AllocateSizeSum = CompressedBufferSize + DecompressedBufferSize + NN_UTIL_CALCULATE_COMPRESS_GZIP_WORKBUFFER_SIZE(MemLevel) + nn::util::DecompressGzipWorkBufferSize;
    ASSERT_LE(AllocateSizeSum, HeapSize);

    // ワークバッファをヒープから取得
    void* pCompressWorkBuffer = std::malloc(NN_UTIL_CALCULATE_COMPRESS_GZIP_WORKBUFFER_SIZE(MemLevel));
    ASSERT_TRUE(pCompressWorkBuffer != nullptr);

    void* pDecompressWorkBuffer = std::malloc(nn::util::DecompressGzipWorkBufferSize);
    ASSERT_TRUE(pDecompressWorkBuffer != nullptr);

    // ----------------------------------- 無圧縮(gzip 形式) --------------------------------------
    bool compressResult = nn::util::CompressGzip(&outSize,
                                  m_pCompressedBuffer, CompressedBufferSize,
                                  TestData,            sizeof(TestData) / sizeof(TestData[0]),
                                  pCompressWorkBuffer, NN_UTIL_CALCULATE_COMPRESS_GZIP_WORKBUFFER_SIZE(MemLevel),
                                  MemLevel,            0);

    ASSERT_TRUE(compressResult);

    // ヘッダおよびトレイラを除いた無圧縮のデータが圧縮前データと一致するか確認
    char* pCompressedDataPayload = reinterpret_cast<char*>(m_pCompressedBuffer) + GzipHeaderSize + OptionalHeaderSize;
    EXPECT_EQ(0, std::memcmp(TestData, pCompressedDataPayload, sizeof(TestData)));

    // ----------------------------------- 伸長(gzip 形式) -----------------------------------
    bool decompressResult = nn::util::DecompressGzip(m_pDecompressedBuffer, DecompressedBufferSize,
                                                m_pCompressedBuffer,        outSize,
                                                pDecompressWorkBuffer,      nn::util::DecompressGzipWorkBufferSize);

    ASSERT_TRUE(decompressResult);
    EXPECT_EQ(0, std::memcmp(TestData, m_pDecompressedBuffer, sizeof(TestData)));

    // ----------------------------------- オプション指定圧縮(gzip 形式) --------------------------------------
    compressResult = nn::util::CompressGzip(&outSize,
                                  m_pCompressedBuffer, CompressedBufferSize,
                                  TestData,            sizeof(TestData) / sizeof(TestData[0]),
                                  pCompressWorkBuffer, NN_UTIL_CALCULATE_COMPRESS_GZIP_WORKBUFFER_SIZE(MemLevel),
                                  MemLevel,            CompressionLevel);

    ASSERT_TRUE(compressResult);

    // ----------------------------------- 伸長(gzip 形式) -----------------------------------
    decompressResult = nn::util::DecompressGzip(m_pDecompressedBuffer, DecompressedBufferSize,
                                           m_pCompressedBuffer,        outSize,
                                           pDecompressWorkBuffer,      nn::util::DecompressGzipWorkBufferSize);

    ASSERT_TRUE(decompressResult);
    EXPECT_EQ(0, std::memcmp(TestData, m_pDecompressedBuffer, sizeof(TestData)));

    // ワークバッファのメモリブロックの解放
    std::free(pCompressWorkBuffer);
    std::free(pDecompressWorkBuffer);
}

// deflate 形式のオプション指定圧縮と伸長のテスト
TEST_F(BasicCompressTest, OptionalCompressDeflateAndDecompressDeflate)
{
    // 圧縮時に指定する圧縮レベルとメモリ使用量
    const int   CompressionLevel = 3;
    const int   MemLevel         = 4;
    // 処理後のサイズ取得用変数
    size_t      outSize;

    // アロケートしたいサイズの合計がヒープサイズを超えていないかチェック
    const size_t AllocateSizeSum = CompressedBufferSize + DecompressedBufferSize + NN_UTIL_CALCULATE_COMPRESS_DEFLATE_WORKBUFFER_SIZE(MemLevel) + nn::util::DecompressDeflateWorkBufferSize;
    ASSERT_LE(AllocateSizeSum, HeapSize);

    // ワークバッファをヒープから取得
    void* pCompressWorkBuffer = std::malloc(NN_UTIL_CALCULATE_COMPRESS_DEFLATE_WORKBUFFER_SIZE(MemLevel));
    ASSERT_TRUE(pCompressWorkBuffer != nullptr);

    void* pDecompressWorkBuffer = std::malloc(nn::util::DecompressDeflateWorkBufferSize);
    ASSERT_TRUE(pDecompressWorkBuffer != nullptr);

    // ----------------------------------- 無圧縮(deflate 形式) --------------------------------------
    bool compressResult = nn::util::CompressDeflate(&outSize,
                                  m_pCompressedBuffer, CompressedBufferSize,
                                  TestData,            sizeof(TestData) / sizeof(TestData[0]),
                                  pCompressWorkBuffer, NN_UTIL_CALCULATE_COMPRESS_DEFLATE_WORKBUFFER_SIZE(MemLevel),
                                  MemLevel,            0);

    ASSERT_TRUE(compressResult);

    // ヘッダを除いた無圧縮のデータが圧縮前データと一致するか確認
    char* pCompressedDataPayload = reinterpret_cast<char*>(m_pCompressedBuffer) + OptionalHeaderSize;
    EXPECT_EQ(0, std::memcmp(TestData, pCompressedDataPayload, sizeof(TestData)));

    // ----------------------------------- 伸長(deflate 形式) -----------------------------------
    bool decompressResult = nn::util::DecompressDeflate(m_pDecompressedBuffer, DecompressedBufferSize,
                                                        m_pCompressedBuffer,   outSize,
                                                        pDecompressWorkBuffer, nn::util::DecompressDeflateWorkBufferSize);

    ASSERT_TRUE(decompressResult);
    EXPECT_EQ(0, std::memcmp(TestData, m_pDecompressedBuffer, sizeof(TestData)));

    // ----------------------------------- オプション指定圧縮(deflate 形式) --------------------------------------
    compressResult = nn::util::CompressDeflate(&outSize,
                                  m_pCompressedBuffer, CompressedBufferSize,
                                  TestData,            sizeof(TestData) / sizeof(TestData[0]),
                                  pCompressWorkBuffer, NN_UTIL_CALCULATE_COMPRESS_DEFLATE_WORKBUFFER_SIZE(MemLevel),
                                  MemLevel,            CompressionLevel);

    ASSERT_TRUE(compressResult);

    // ----------------------------------- 伸長(deflate 形式) -----------------------------------
    decompressResult = nn::util::DecompressDeflate(m_pDecompressedBuffer, DecompressedBufferSize,
                                                   m_pCompressedBuffer,   outSize,
                                                   pDecompressWorkBuffer, nn::util::DecompressDeflateWorkBufferSize);

    ASSERT_TRUE(decompressResult);
    EXPECT_EQ(0, std::memcmp(TestData, m_pDecompressedBuffer, sizeof(TestData)));

    // ワークバッファのメモリブロックの解放
    std::free(pCompressWorkBuffer);
    std::free(pDecompressWorkBuffer);
}

// zlib 形式の ストリーミング伸長のテスト
TEST_F(BasicCompressTest, StreamingZlibTest)
{
    // コンテキストをヒープから領域確保する
    nn::util::StreamingDecompressZlibContext* pContext = static_cast<nn::util::StreamingDecompressZlibContext*>
                                                                    (std::malloc(sizeof(nn::util::StreamingDecompressZlibContext)));
    ASSERT_TRUE(pContext != nullptr);

    // Src 側だけストリーミングして伸長するテスト
    EXPECT_TRUE(TestSrcStreaming(DataType_Zlib, static_cast<char*>(m_pDecompressedBuffer), DecompressedBufferSize, g_ZlibCompressedBuffer, g_ZlibCompressedDataSize, pContext));
    EXPECT_EQ(std::memcmp(TestData, m_pDecompressedBuffer, sizeof(TestData)), 0);

    // Src 側と Dst側の両方をストリーミングして伸長するテスト
    EXPECT_TRUE(TestSrcAndDstStreaming(DataType_Zlib, static_cast<char*>(m_pDecompressedBuffer), DecompressedBufferSize, g_ZlibCompressedBuffer, g_ZlibCompressedDataSize, pContext));
    EXPECT_EQ(std::memcmp(TestData, m_pDecompressedBuffer, sizeof(TestData)), 0);

    // Zlib 形式のストリーミング伸長にGzip のデータを渡して false となることを確認するテスト
    NNT_UTIL_COMPRESSION_ERROR_PATTERN_LOG("< Error Pattern Test > : ");
    EXPECT_FALSE(TestSrcAndDstStreaming(DataType_Zlib, static_cast<char*>(m_pDecompressedBuffer), DecompressedBufferSize, g_GzipCompressedBuffer, g_GzipCompressedDataSize, pContext));

    // 未圧縮のデータを渡して、false となることを確認するテスト
    const char DummyData[] = "abcdefghijklmnopqrstuvwxyz";
    NNT_UTIL_COMPRESSION_ERROR_PATTERN_LOG("< Error Pattern Test > : ");
    EXPECT_FALSE(TestSrcAndDstStreaming(DataType_Zlib, static_cast<char*>(m_pDecompressedBuffer), DecompressedBufferSize, DummyData, sizeof(TestData), pContext));

    // サイズに0を指定して、false となることを確認するテスト
    size_t dstConsumedSize = 0;
    size_t srcConsumedSize = 0;

    nn::util::InitializeStreamingDecompressZlibContext(pContext);
    NNT_UTIL_COMPRESSION_ERROR_PATTERN_LOG("< Error Pattern Test > : ");
    EXPECT_FALSE(nn::util::StreamingDecompressZlib(&dstConsumedSize, &srcConsumedSize, m_pDecompressedBuffer, DecompressedBufferSize, g_ZlibCompressedBuffer, 0, pContext));

    NNT_UTIL_COMPRESSION_ERROR_PATTERN_LOG("< Error Pattern Test > : ");
    EXPECT_FALSE(TestSrcStreaming(DataType_Zlib, static_cast<char*>(m_pDecompressedBuffer), 0, g_ZlibCompressedBuffer, g_ZlibCompressedDataSize, pContext));

    // コンテキスト領域のメモリの解放
    std::free(pContext);
}

// gzip 形式の ストリーミング伸長のテスト
TEST_F(BasicCompressTest, StreamingGzipTest)
{
    // コンテキストをヒープから領域確保する
    nn::util::StreamingDecompressZlibContext* pContext = static_cast<nn::util::StreamingDecompressZlibContext*>
                                                                    (std::malloc(sizeof(nn::util::StreamingDecompressZlibContext)));
    ASSERT_TRUE(pContext != nullptr);

    // Src 側だけストリーミングして伸長するテスト
    EXPECT_TRUE(TestSrcStreaming(DataType_Gzip, static_cast<char*>(m_pDecompressedBuffer), DecompressedBufferSize, g_GzipCompressedBuffer, g_GzipCompressedDataSize, pContext));
    EXPECT_EQ(std::memcmp(TestData, m_pDecompressedBuffer, sizeof(TestData)), 0);

    // Src 側と Dst側の両方をストリーミングして伸長するテスト
    EXPECT_TRUE(TestSrcAndDstStreaming(DataType_Gzip, static_cast<char*>(m_pDecompressedBuffer), DecompressedBufferSize, g_GzipCompressedBuffer, g_GzipCompressedDataSize, pContext));
    EXPECT_EQ(std::memcmp(TestData, m_pDecompressedBuffer, sizeof(TestData)), 0);

    // Gzip 形式のストリーミング伸長に Raw Deflate のデータを渡して false となることを確認するテスト
    NNT_UTIL_COMPRESSION_ERROR_PATTERN_LOG("< Error Pattern Test > : ");
    EXPECT_FALSE(TestSrcAndDstStreaming(DataType_Gzip, static_cast<char*>(m_pDecompressedBuffer), DecompressedBufferSize, g_DeflateCompressedBuffer, g_DeflateCompressedDataSize, pContext));

    // 未圧縮のデータを渡して、false となることを確認するテスト
    const char DummyData[] = "abcdefghijklmnopqrstuvwxyz";
    NNT_UTIL_COMPRESSION_ERROR_PATTERN_LOG("< Error Pattern Test > : ");
    EXPECT_FALSE(TestSrcAndDstStreaming(DataType_Gzip, static_cast<char*>(m_pDecompressedBuffer), DecompressedBufferSize, DummyData, sizeof(TestData), pContext));

    // サイズに0を指定して、false となることを確認するテスト
    size_t dstConsumedSize = 0;
    size_t srcConsumedSize = 0;

    nn::util::InitializeStreamingDecompressGzipContext(pContext);
    NNT_UTIL_COMPRESSION_ERROR_PATTERN_LOG("< Error Pattern Test > : ");
    EXPECT_FALSE(nn::util::StreamingDecompressGzip(&dstConsumedSize, &srcConsumedSize, m_pDecompressedBuffer, DecompressedBufferSize, g_GzipCompressedBuffer, 0, pContext));

    NNT_UTIL_COMPRESSION_ERROR_PATTERN_LOG("< Error Pattern Test > : ");
    EXPECT_FALSE(TestSrcStreaming(DataType_Gzip, static_cast<char*>(m_pDecompressedBuffer), 0, g_GzipCompressedBuffer, g_GzipCompressedDataSize, pContext));

    // コンテキスト領域のメモリの解放
    std::free(pContext);
}

// deflate 形式の ストリーミング伸長のテスト
TEST_F(BasicCompressTest, StreamingDeflateTest)
{
    // コンテキストをヒープから領域確保する
    nn::util::StreamingDecompressZlibContext* pContext = static_cast<nn::util::StreamingDecompressZlibContext*>
                                                                    (std::malloc(sizeof(nn::util::StreamingDecompressZlibContext)));
    ASSERT_TRUE(pContext != nullptr);

    // Src 側だけストリーミングして伸長するテスト
    EXPECT_TRUE(TestSrcStreaming(DataType_Deflate, static_cast<char*>(m_pDecompressedBuffer), DecompressedBufferSize, g_DeflateCompressedBuffer, g_DeflateCompressedDataSize, pContext));
    EXPECT_EQ(std::memcmp(TestData, m_pDecompressedBuffer, sizeof(TestData)), 0);

    // Src 側と Dst側の両方をストリーミングして伸長するテスト
    EXPECT_TRUE(TestSrcAndDstStreaming(DataType_Deflate, static_cast<char*>(m_pDecompressedBuffer), DecompressedBufferSize, g_DeflateCompressedBuffer, g_DeflateCompressedDataSize, pContext));
    EXPECT_EQ(std::memcmp(TestData, m_pDecompressedBuffer, sizeof(TestData)), 0);

    // Deflate 形式のストリーミング伸長に Zlib のデータを渡して false となることを確認するテスト
    NNT_UTIL_COMPRESSION_ERROR_PATTERN_LOG("< Error Pattern Test > : ");
    EXPECT_FALSE(TestSrcAndDstStreaming(DataType_Deflate, static_cast<char*>(m_pDecompressedBuffer), DecompressedBufferSize, g_ZlibCompressedBuffer, g_ZlibCompressedDataSize, pContext));

    // 未圧縮のデータを渡して、false となることを確認するテスト
    const char DummyData[] = "abcdefghijklmnopqrstuvwxyz";
    NNT_UTIL_COMPRESSION_ERROR_PATTERN_LOG("< Error Pattern Test > : ");
    EXPECT_FALSE(TestSrcAndDstStreaming(DataType_Deflate, static_cast<char*>(m_pDecompressedBuffer), DecompressedBufferSize, DummyData, sizeof(TestData), pContext));

    // サイズに0を指定して、false となることを確認するテスト
    size_t dstConsumedSize = 0;
    size_t srcConsumedSize = 0;

    nn::util::InitializeStreamingDecompressDeflateContext(pContext);
    NNT_UTIL_COMPRESSION_ERROR_PATTERN_LOG("< Error Pattern Test > : ");
    EXPECT_FALSE(nn::util::StreamingDecompressDeflate(&dstConsumedSize, &srcConsumedSize, m_pDecompressedBuffer, DecompressedBufferSize, g_DeflateCompressedBuffer, 0, pContext));

    NNT_UTIL_COMPRESSION_ERROR_PATTERN_LOG("< Error Pattern Test > : ");
    EXPECT_FALSE(TestSrcStreaming(DataType_Deflate, static_cast<char*>(m_pDecompressedBuffer), 0, g_DeflateCompressedBuffer, g_DeflateCompressedDataSize, pContext));

    // コンテキスト領域のメモリの解放
    std::free(pContext);
}

// zlib 形式の ストリーミング圧縮のテスト
TEST_F(BasicCompressTest, StreamingCompressionZlibTest)
{
    const size_t testDataSize = sizeof(TestData) / sizeof(TestData[0]);
    size_t compressedDataSize = 0;

    const int   CompressionLevel = 3;
    const int   MemLevel         = 4;

    // Src 側だけストリーミングして圧縮するテスト
    EXPECT_TRUE(TestSrcStreamingCompression(DataType_Zlib, &compressedDataSize, static_cast<char*>(m_pCompressedBuffer), CompressedBufferSize, reinterpret_cast<const char*>(TestData), testDataSize, MemLevel, CompressionLevel));
    EXPECT_TRUE(VerifyCompressedData(DataType_Zlib, m_pDecompressedBuffer, m_pCompressedBuffer, compressedDataSize, TestData, testDataSize));

    // Src 側と Dst側の両方をストリーミングして圧縮するテスト
    EXPECT_TRUE(TestSrcAndDstStreamingCompression(DataType_Zlib, &compressedDataSize, static_cast<char*>(m_pCompressedBuffer), CompressedBufferSize, reinterpret_cast<const char*>(TestData), testDataSize, MemLevel, CompressionLevel));
    EXPECT_TRUE(VerifyCompressedData(DataType_Zlib, m_pDecompressedBuffer, m_pCompressedBuffer, compressedDataSize, TestData, testDataSize));

    nn::util::StreamingCompressZlibContext context;
    std::memset(&context, 0, sizeof(context));
    size_t dstConsumedSize;
    size_t srcConsumedSize;

    // 未初期化のコンテキストを渡すテスト
    // EXPECT_FALSE(nn::util::StreamingCompressZlib(&dstConsumedSize, &srcConsumedSize, m_pCompressedBuffer, CompressedBufferSize, TestData, testDataSize, &context));
    // Assert

    // ワークバッファを確保
    size_t workBufferSize = NN_UTIL_CALCULATE_COMPRESS_ZLIB_WORKBUFFER_SIZE(10);
    void* pWorkBuffer = std::malloc(workBufferSize);

    // サイズの小さなワークバッファを渡すテスト
    // nn::util::InitializeStreamingCompressZlibContext(&context, pWorkBuffer, workBufferSize - 1);
    // Assert

    // コンテキストを初期化
    // オプションなし
    nn::util::InitializeStreamingCompressZlibContext(&context, pWorkBuffer, workBufferSize);
    // オプションあり
    nn::util::InitializeStreamingCompressZlibContext(&context, pWorkBuffer, workBufferSize,  1,  0);
    nn::util::InitializeStreamingCompressZlibContext(&context, pWorkBuffer, workBufferSize,  9,  9);
    // Assert
    // nn::util::InitializeStreamingCompressZlibContext(&context, pWorkBuffer, workBufferSize, 10,  0);
    // nn::util::InitializeStreamingCompressZlibContext(&context, pWorkBuffer, workBufferSize,  1, 10);
    // nn::util::InitializeStreamingCompressZlibContext(&context, pWorkBuffer, workBufferSize, 10, 10);
    // nn::util::InitializeStreamingCompressZlibContext(&context, pWorkBuffer, workBufferSize, -1, -1);

    // 出力バッファのサイズに0を指定して、false となることを確認するテスト
    NNT_UTIL_COMPRESSION_ERROR_PATTERN_LOG("< Error Pattern Test > : ");
    EXPECT_FALSE(nn::util::StreamingCompressZlib(&dstConsumedSize, &srcConsumedSize, m_pCompressedBuffer, 0, TestData, testDataSize, &context));

    // 入力バッファのサイズに0を指定して、true となることを確認するテスト
    EXPECT_TRUE(nn::util::StreamingCompressZlib(&dstConsumedSize, &srcConsumedSize, m_pCompressedBuffer, CompressedBufferSize, TestData, 0, &context));

    // ワークバッファ解放
    std::free(pWorkBuffer);
}

// gzip 形式の ストリーミング圧縮のテスト
TEST_F(BasicCompressTest, StreamingCompressionGzipTest)
{
    const size_t testDataSize = sizeof(TestData) / sizeof(TestData[0]);
    size_t compressedDataSize = 0;

    const int   CompressionLevel = 3;
    const int   MemLevel         = 4;

    // Src 側だけストリーミングして圧縮するテスト
    EXPECT_TRUE(TestSrcStreamingCompression(DataType_Gzip, &compressedDataSize, static_cast<char*>(m_pCompressedBuffer), CompressedBufferSize, reinterpret_cast<const char*>(TestData), testDataSize, MemLevel, CompressionLevel));
    EXPECT_TRUE(VerifyCompressedData(DataType_Gzip, m_pDecompressedBuffer, m_pCompressedBuffer, compressedDataSize, TestData, testDataSize));

    // Src 側と Dst側の両方をストリーミングして圧縮するテスト
    EXPECT_TRUE(TestSrcAndDstStreamingCompression(DataType_Gzip, &compressedDataSize, static_cast<char*>(m_pCompressedBuffer), CompressedBufferSize, reinterpret_cast<const char*>(TestData), testDataSize, MemLevel, CompressionLevel));
    EXPECT_TRUE(VerifyCompressedData(DataType_Gzip, m_pDecompressedBuffer, m_pCompressedBuffer, compressedDataSize, TestData, testDataSize));

    nn::util::StreamingCompressGzipContext context;
    std::memset(&context, 0, sizeof(context));
    size_t dstConsumedSize;
    size_t srcConsumedSize;

    // 未初期化のコンテキストを渡すテスト
    // EXPECT_FALSE(nn::util::StreamingCompressGzip(&dstConsumedSize, &srcConsumedSize, m_pCompressedBuffer, CompressedBufferSize, TestData, testDataSize, &context));
    // Assert

    // ワークバッファを確保
    size_t workBufferSize = NN_UTIL_CALCULATE_COMPRESS_GZIP_WORKBUFFER_SIZE(10);
    void* pWorkBuffer = std::malloc(workBufferSize);

    // サイズの小さなワークバッファを渡すテスト
    // nn::util::InitializeStreamingCompressGzipContext(&context, pWorkBuffer, workBufferSize - 1);
    // Assert

    // コンテキストを初期化
    // オプションなし
    nn::util::InitializeStreamingCompressGzipContext(&context, pWorkBuffer, workBufferSize);
    // オプションあり
    nn::util::InitializeStreamingCompressGzipContext(&context, pWorkBuffer, workBufferSize,  1,  0);
    nn::util::InitializeStreamingCompressGzipContext(&context, pWorkBuffer, workBufferSize,  9,  9);
    // Assert
    // nn::util::InitializeStreamingCompressGzipContext(&context, pWorkBuffer, workBufferSize, 10,  0);
    // nn::util::InitializeStreamingCompressGzipContext(&context, pWorkBuffer, workBufferSize,  1, 10);
    // nn::util::InitializeStreamingCompressGzipContext(&context, pWorkBuffer, workBufferSize, 10, 10);
    // nn::util::InitializeStreamingCompressGzipContext(&context, pWorkBuffer, workBufferSize, -1, -1);

    // 出力バッファのサイズに0を指定して、false となることを確認するテスト
    NNT_UTIL_COMPRESSION_ERROR_PATTERN_LOG("< Error Pattern Test > : ");
    EXPECT_FALSE(nn::util::StreamingCompressGzip(&dstConsumedSize, &srcConsumedSize, m_pCompressedBuffer, 0, TestData, testDataSize, &context));

    // 入力バッファのサイズに0を指定して、true となることを確認するテスト
    EXPECT_TRUE(nn::util::StreamingCompressGzip(&dstConsumedSize, &srcConsumedSize, m_pCompressedBuffer, CompressedBufferSize, TestData, 0, &context));

    // ワークバッファ解放
    std::free(pWorkBuffer);
}

// deflate 形式の ストリーミング圧縮のテスト
TEST_F(BasicCompressTest, StreamingCompressionDeflateTest)
{
    const size_t testDataSize = sizeof(TestData) / sizeof(TestData[0]);
    size_t compressedDataSize = 0;

    const int   CompressionLevel = 3;
    const int   MemLevel         = 4;

    // Src 側だけストリーミングして圧縮するテスト
    EXPECT_TRUE(TestSrcStreamingCompression(DataType_Deflate, &compressedDataSize, static_cast<char*>(m_pCompressedBuffer), CompressedBufferSize, reinterpret_cast<const char*>(TestData), testDataSize, MemLevel, CompressionLevel));
    EXPECT_TRUE(VerifyCompressedData(DataType_Deflate, m_pDecompressedBuffer, m_pCompressedBuffer, compressedDataSize, TestData, testDataSize));

    // Src 側と Dst側の両方をストリーミングして圧縮するテスト
    EXPECT_TRUE(TestSrcAndDstStreamingCompression(DataType_Deflate, &compressedDataSize, static_cast<char*>(m_pCompressedBuffer), CompressedBufferSize, reinterpret_cast<const char*>(TestData), testDataSize, MemLevel, CompressionLevel));
    EXPECT_TRUE(VerifyCompressedData(DataType_Deflate, m_pDecompressedBuffer, m_pCompressedBuffer, compressedDataSize, TestData, testDataSize));

    nn::util::StreamingCompressDeflateContext context;
    std::memset(&context, 0, sizeof(context));
    size_t dstConsumedSize;
    size_t srcConsumedSize;

    // 未初期化のコンテキストを渡すテスト
    // EXPECT_FALSE(nn::util::StreamingCompressDeflate(&dstConsumedSize, &srcConsumedSize, m_pCompressedBuffer, CompressedBufferSize, TestData, testDataSize, &context));
    // Assert

    // ワークバッファを確保
    size_t workBufferSize = NN_UTIL_CALCULATE_COMPRESS_DEFLATE_WORKBUFFER_SIZE(10);
    void* pWorkBuffer = std::malloc(workBufferSize);

    // サイズの小さなワークバッファを渡すテスト
    // nn::util::InitializeStreamingCompressDeflateContext(&context, pWorkBuffer, workBufferSize - 1);
    // Assert

    // コンテキストを初期化
    // オプションなし
    nn::util::InitializeStreamingCompressDeflateContext(&context, pWorkBuffer, workBufferSize);
    // オプションあり
    nn::util::InitializeStreamingCompressDeflateContext(&context, pWorkBuffer, workBufferSize,  1,  0);
    nn::util::InitializeStreamingCompressDeflateContext(&context, pWorkBuffer, workBufferSize,  9,  9);
    // Assert
    // nn::util::InitializeStreamingCompressDeflateContext(&context, pWorkBuffer, workBufferSize, 10,  0);
    // nn::util::InitializeStreamingCompressDeflateContext(&context, pWorkBuffer, workBufferSize,  1, 10);
    // nn::util::InitializeStreamingCompressDeflateContext(&context, pWorkBuffer, workBufferSize, 10, 10);
    // nn::util::InitializeStreamingCompressDeflateContext(&context, pWorkBuffer, workBufferSize, -1, -1);

    // 出力バッファのサイズに0を指定して、false となることを確認するテスト
    NNT_UTIL_COMPRESSION_ERROR_PATTERN_LOG("< Error Pattern Test > : ");
    EXPECT_FALSE(nn::util::StreamingCompressDeflate(&dstConsumedSize, &srcConsumedSize, m_pCompressedBuffer, 0, TestData, testDataSize, &context));

    // 入力バッファのサイズに0を指定して、true となることを確認するテスト
    EXPECT_TRUE(nn::util::StreamingCompressDeflate(&dstConsumedSize, &srcConsumedSize, m_pCompressedBuffer, CompressedBufferSize, TestData, 0, &context));

    // ワークバッファ解放
    std::free(pWorkBuffer);
}

