﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/util/util_Base64.h>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <cstring>
#include <random>

#ifdef NN_BUILD_CONFIG_OS_WIN32
#include <ctime> // std::srand( time() )
#endif


namespace UtilBase64TestSub
{
    void LoopBack(const void* pSrc, size_t sizeSrc, nn::util::Base64::Mode mode);
    void StringLoopBack(const char* src, nn::util::Base64::Mode mode);
    bool IsClean(const uint8_t* pBuf, size_t sizeBuf);

    void PrintBinary(const char* title, const char* buf, size_t size);
    void PrintBinary(const char* title, const uint8_t* buf, size_t size);
};


//------------------------------------------------------------------
// Test Util
//------------------------------------------------------------------
void UtilBase64TestSub::LoopBack(const void* pSrc, size_t sizeSrc, nn::util::Base64::Mode mode)
{
    size_t num;
    char dst[4096];
    unsigned char loopBackBuf[4096];
    nn::util::Base64::Status status;

    status = nn::util::Base64::ToBase64String(dst, sizeof(dst), pSrc, sizeSrc, mode);
    EXPECT_EQ(nn::util::Base64::Status_Success, status);

    {
        // dst size 指定しない版
        status = nn::util::Base64::FromBase64String(&num, loopBackBuf, sizeof(loopBackBuf), dst, mode);
        EXPECT_EQ(nn::util::Base64::Status_Success, status);

        EXPECT_EQ( num , sizeSrc );
        EXPECT_EQ( 0, std::memcmp(pSrc, loopBackBuf, num) );
        if (num != sizeSrc || std::memcmp(pSrc, loopBackBuf, num) != 0)
        {
            NN_LOG("Loopback failed (1)\n");
            size_t i;
            for (i = 0; i < num; ++i)
            {
                NN_LOG("%x, ", loopBackBuf[i]);
            }
        }
    }
    {
        // dst size 指定版
        status = nn::util::Base64::FromBase64String(&num, loopBackBuf, sizeof(loopBackBuf), dst, std::strlen(dst), mode);
        EXPECT_EQ(nn::util::Base64::Status_Success, status);

        EXPECT_EQ( num , sizeSrc );
        EXPECT_EQ( 0, std::memcmp(pSrc, loopBackBuf, num) );
        if (num != sizeSrc || std::memcmp(pSrc, loopBackBuf, num) != 0)
        {
            NN_LOG("Loopback failed (2)\n");
            size_t i;
            for (i = 0; i < num; ++i)
            {
                NN_LOG("%x, ", loopBackBuf[i]);
            }
        }
    }
}

void UtilBase64TestSub::StringLoopBack(const char* src, nn::util::Base64::Mode mode)
{
    LoopBack(src, std::strlen(src), mode);
}

bool UtilBase64TestSub::IsClean(const uint8_t* pBuf, size_t sizeBuf)
{
    bool isClean = true;
    for ( size_t i = 0 ; i < sizeBuf ; ++i )
    {
        if ( pBuf[i] != 0 )
        {
            NN_LOG("%03d : %02X\n", i, pBuf[i]);
            isClean = false;
        }
    }
    return isClean;
}
void UtilBase64TestSub::PrintBinary(const char* title, const char* buf, size_t size)
{
    NN_LOG("%s(size:%zu) -", title, size);
    for( size_t i = 0 ; i < size ; ++i )
    {
        if(i % 16 == 0)
        {
            NN_LOG("\n    ");
        }
        NN_LOG("0x%02x, ", buf[i]);
    }
    NN_LOG("\n");
}
void UtilBase64TestSub::PrintBinary(const char* title, const uint8_t* buf, size_t size)
{
    NN_LOG("%s(size:%zu) -", title, size);
    for( size_t i = 0 ; i < size ; ++i )
    {
        if(i % 16 == 0)
        {
            NN_LOG("\n    ");
        }
        NN_LOG("0x%02x, ", buf[i]);
    }
    NN_LOG("\n");
}

namespace nn { namespace util {

//------------------------------------------------------------------
// Test Functions
//------------------------------------------------------------------

//------------------------------------------------------------------
// Encode <-> Decode LoopBack
//------------------------------------------------------------------
TEST(UtilBase64Test , LoopBackTest)
{
    UtilBase64TestSub::StringLoopBack("ABCDEFG", nn::util::Base64::Mode_Normal);
    UtilBase64TestSub::StringLoopBack("AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA", nn::util::Base64::Mode_Normal);
    UtilBase64TestSub::StringLoopBack("", nn::util::Base64::Mode_Normal);
    UtilBase64TestSub::StringLoopBack("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/", nn::util::Base64::Mode_Normal);

#ifdef NN_BUILD_CONFIG_OS_WIN32
    std::srand(static_cast<unsigned char>(time(NULL)));
#else
    //std::srand(static_cast<s32>(nn::os::Tick::GetSystemCurrent() & 0xFFFFFFFFULL));
    std::srand(1234); // いったん固定値
#endif

    unsigned char bin[1024];
    size_t i, j;
    for (i = 0; i < 1024; ++i)
    {
        size_t len = std::rand() % 1024;
        for (j = 0; j < len; ++j)
        {
            bin[j] = static_cast<unsigned char>(std::rand() & 0xFF);
        }

        UtilBase64TestSub::LoopBack(bin, len, nn::util::Base64::Mode_Normal);
    }
}

TEST(UtilBase64Test , LoopBackTestUrlSafe)
{
    UtilBase64TestSub::StringLoopBack("ABCDEFG", nn::util::Base64::Mode_UrlSafe);
    UtilBase64TestSub::StringLoopBack("AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA", nn::util::Base64::Mode_UrlSafe);
    UtilBase64TestSub::StringLoopBack("", nn::util::Base64::Mode_UrlSafe);
    UtilBase64TestSub::StringLoopBack("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/", nn::util::Base64::Mode_UrlSafe);

#ifdef NN_BUILD_CONFIG_OS_WIN32
    std::srand(static_cast<unsigned char>(time(NULL)));
#else
    //std::srand(static_cast<s32>(nn::os::Tick::GetSystemCurrent() & 0xFFFFFFFFULL));
    std::srand(1234); // いったん固定値
#endif

    unsigned char bin[1024];
    size_t i, j;
    for (i = 0; i < 1024; ++i)
    {
        size_t len = std::rand() % 1024;
        for (j = 0; j < len; ++j)
        {
            bin[j] = static_cast<unsigned char>(std::rand() & 0xFF);
        }

        UtilBase64TestSub::LoopBack(bin, len, nn::util::Base64::Mode_UrlSafe);
    }
}

//------------------------------------------------------------------
// Sample data
//------------------------------------------------------------------
// 別ツールで作成した base64 文字列

static char SampleBase64Txt[] =
    "KQNIIRRLJyBOJixRJThUJERXI1BaIlxdIWhgIHRjH4B2HoxpHZhsHKRvG7ByGrx1Kch4GNR7F+B+\r\n"
    "FuyDFfeAFASHtBCLEqyOESiSEDSUD0CXDkyaDVidDGSgC3WjCnymGYipCJSsB6CvBqyyBbi1BKS4\r\n"
    "A9C7Aty+AejBAPTE7//X/gvL/RfO/CPR+y/U+jvX+Ufa+JPd91/g9mvj9Xfm9IPp8//s8pvu8afy\r\n"
    "8LP177/4/jvr7df+7OMB6+8E6vsG6Qcb6BMO5h8R5isU5TcX4UMa408d4lsg4Wcj4HMm338p3oss\r\n"
    "3Zcv2qMy26812rs42cQ72CM+199B1utE1fZE1A==";

static char SampleBase64TxtNoLinefeed[] =
    "KQNIIRRLJyBOJixRJThUJERXI1BaIlxdIWhgIHRjH4B2HoxpHZhsHKRvG7ByGrx1Kch4GNR7F+B+"
    "FuyDFfeAFASHtBCLEqyOESiSEDSUD0CXDkyaDVidDGSgC3WjCnymGYipCJSsB6CvBqyyBbi1BKS4"
    "A9C7Aty+AejBAPTE7//X/gvL/RfO/CPR+y/U+jvX+Ufa+JPd91/g9mvj9Xfm9IPp8//s8pvu8afy"
    "8LP177/4/jvr7df+7OMB6+8E6vsG6Qcb6BMO5h8R5isU5TcX4UMa408d4lsg4Wcj4HMm338p3oss"
    "3Zcv2qMy26812rs42cQ72CM+199B1utE1fZE1A==";

static char SampleBase64TxtUrlSafe[] =
    "KQNIIRRLJyBOJixRJThUJERXI1BaIlxdIWhgIHRjH4B2HoxpHZhsHKRvG7ByGrx1Kch4GNR7F-B-"
    "FuyDFfeAFASHtBCLEqyOESiSEDSUD0CXDkyaDVidDGSgC3WjCnymGYipCJSsB6CvBqyyBbi1BKS4"
    "A9C7Aty-AejBAPTE7__X_gvL_RfO_CPR-y_U-jvX-Ufa-JPd91_g9mvj9Xfm9IPp8__s8pvu8afy"
    "8LP177_4_jvr7df-7OMB6-8E6vsG6Qcb6BMO5h8R5isU5TcX4UMa408d4lsg4Wcj4HMm338p3oss"
    "3Zcv2qMy26812rs42cQ72CM-199B1utE1fZE1A";

static uint8_t SampleBin[] =
{
    0x29, 0x03, 0x48, 0x21, 0x14, 0x4b, 0x27, 0x20, 0x4e, 0x26, 0x2c, 0x51, 0x25, 0x38, 0x54, 0x24,
    0x44, 0x57, 0x23, 0x50, 0x5a, 0x22, 0x5c, 0x5d, 0x21, 0x68, 0x60, 0x20, 0x74, 0x63, 0x1f, 0x80,
    0x76, 0x1e, 0x8c, 0x69, 0x1d, 0x98, 0x6c, 0x1c, 0xa4, 0x6f, 0x1b, 0xb0, 0x72, 0x1a, 0xbc, 0x75,
    0x29, 0xc8, 0x78, 0x18, 0xd4, 0x7b, 0x17, 0xe0, 0x7e, 0x16, 0xec, 0x83, 0x15, 0xf7, 0x80, 0x14,
    0x04, 0x87, 0xb4, 0x10, 0x8b, 0x12, 0xac, 0x8e, 0x11, 0x28, 0x92, 0x10, 0x34, 0x94, 0x0f, 0x40,
    0x97, 0x0e, 0x4c, 0x9a, 0x0d, 0x58, 0x9d, 0x0c, 0x64, 0xa0, 0x0b, 0x75, 0xa3, 0x0a, 0x7c, 0xa6,
    0x19, 0x88, 0xa9, 0x08, 0x94, 0xac, 0x07, 0xa0, 0xaf, 0x06, 0xac, 0xb2, 0x05, 0xb8, 0xb5, 0x04,
    0xa4, 0xb8, 0x03, 0xd0, 0xbb, 0x02, 0xdc, 0xbe, 0x01, 0xe8, 0xc1, 0x00, 0xf4, 0xc4, 0xef, 0xff,
    0xd7, 0xfe, 0x0b, 0xcb, 0xfd, 0x17, 0xce, 0xfc, 0x23, 0xd1, 0xfb, 0x2f, 0xd4, 0xfa, 0x3b, 0xd7,
    0xf9, 0x47, 0xda, 0xf8, 0x93, 0xdd, 0xf7, 0x5f, 0xe0, 0xf6, 0x6b, 0xe3, 0xf5, 0x77, 0xe6, 0xf4,
    0x83, 0xe9, 0xf3, 0xff, 0xec, 0xf2, 0x9b, 0xee, 0xf1, 0xa7, 0xf2, 0xf0, 0xb3, 0xf5, 0xef, 0xbf,
    0xf8, 0xfe, 0x3b, 0xeb, 0xed, 0xd7, 0xfe, 0xec, 0xe3, 0x01, 0xeb, 0xef, 0x04, 0xea, 0xfb, 0x06,
    0xe9, 0x07, 0x1b, 0xe8, 0x13, 0x0e, 0xe6, 0x1f, 0x11, 0xe6, 0x2b, 0x14, 0xe5, 0x37, 0x17, 0xe1,
    0x43, 0x1a, 0xe3, 0x4f, 0x1d, 0xe2, 0x5b, 0x20, 0xe1, 0x67, 0x23, 0xe0, 0x73, 0x26, 0xdf, 0x7f,
    0x29, 0xde, 0x8b, 0x2c, 0xdd, 0x97, 0x2f, 0xda, 0xa3, 0x32, 0xdb, 0xaf, 0x35, 0xda, 0xbb, 0x38,
    0xd9, 0xc4, 0x3b, 0xd8, 0x23, 0x3e, 0xd7, 0xdf, 0x41, 0xd6, 0xeb, 0x44, 0xd5, 0xf6, 0x44, 0xd4
};

TEST(UtilBase64Test, SampleDataTest)
{
    // Decode
    {
        uint8_t buf[256];
        size_t  num;

        std::memset(buf, 0, sizeof(buf));
        EXPECT_EQ(nn::util::Base64::Status_Success,
            Base64::FromBase64String(&num, buf, sizeof(buf), SampleBase64Txt, Base64::Mode_Normal));
        EXPECT_EQ(num, sizeof(SampleBin));
        EXPECT_EQ(std::memcmp(SampleBin, buf, num), 0);
    }

    // Encode
    {
        char    sbuf[384];

        std::memset(sbuf, 0, sizeof(sbuf));
        EXPECT_EQ(nn::util::Base64::Status_Success,
            Base64::ToBase64String(sbuf, sizeof(sbuf), SampleBin, sizeof(SampleBin), Base64::Mode_Normal));
        EXPECT_EQ(std::strcmp(SampleBase64Txt, sbuf), 0);
    }
}
TEST(UtilBase64Test, SampleDataTestNoLinefeed)
{
    // Decode
    {
        uint8_t buf[256];
        size_t  num;

        std::memset(buf, 0, sizeof(buf));
        EXPECT_EQ(nn::util::Base64::Status_Success,
            Base64::FromBase64String(&num, buf, sizeof(buf), SampleBase64TxtNoLinefeed, Base64::Mode_NormalNoLinefeed));
        EXPECT_EQ(num, sizeof(SampleBin));
        EXPECT_EQ(std::memcmp(SampleBin, buf, num), 0);
    }

    // Encode
    {
        char    sbuf[384];

        std::memset(sbuf, 0, sizeof(sbuf));
        EXPECT_EQ(nn::util::Base64::Status_Success,
            Base64::ToBase64String(sbuf, sizeof(sbuf), SampleBin, sizeof(SampleBin), Base64::Mode_NormalNoLinefeed));
        EXPECT_EQ(std::strcmp(SampleBase64TxtNoLinefeed, sbuf), 0);
    }
}
TEST(UtilBase64Test, SampleDataTestUrlSafe)
{
    // Decode
    {
        uint8_t buf[256];
        size_t  num;

        std::memset(buf, 0, sizeof(buf));
        EXPECT_EQ(nn::util::Base64::Status_Success,
            Base64::FromBase64String(&num, buf, sizeof(buf), SampleBase64TxtUrlSafe, Base64::Mode_UrlSafe));
        EXPECT_EQ(num, sizeof(SampleBin));
        EXPECT_EQ(std::memcmp(SampleBin, buf, num), 0);
    }

    // Encode
    {
        char    sbuf[384];

        std::memset(sbuf, 0, sizeof(sbuf));
        EXPECT_EQ(nn::util::Base64::Status_Success,
            Base64::ToBase64String(sbuf, sizeof(sbuf), SampleBin, sizeof(SampleBin), Base64::Mode_UrlSafe));
        EXPECT_EQ(std::strcmp(SampleBase64TxtUrlSafe, sbuf), 0);
    }
}


//------------------------------------------------------------------
// Skip char test
//------------------------------------------------------------------
static uint8_t BaseData[32] = { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
                             0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF,
                             0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
                             0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF };

// 0x09, 0x0A 0x0D, 0x20 混じり
static char EncodedStr0[]        = "ABEi\nM0RVZne\rImaq7zN3u/wARIj\tNEVWZ3i Jmqu8zd7v8=";
static char EncodedStr0NoLinefeed[] = "ABEi\nM0RVZne\rImaq7zN3u/wARIj\tNEVWZ3i Jmqu8zd7v8=";
static char EncodedStr0UrlSafe[] = "ABEi\nM0RVZne\rImaq7zN3u_wARIj\tNEVWZ3i Jmqu8zd7v8=";
// 連続した 0x20
static char EncodedStr1[]        = "A B E i  M0RVZne   Imaq7zN 3 u / w ARIj      NEVWZ3iJmqu8zd7v8  =";
static char EncodedStr1NoLinefeed[] = "A B E i  M0RVZne   Imaq7zN 3 u / w ARIj      NEVWZ3iJmqu8zd7v8  =";
static char EncodedStr1UrlSafe[] = "A B E i  M0RVZne   Imaq7zN 3 u _ w ARIj      NEVWZ3iJmqu8zd7v8  =";
// = の後に文字列
static char EncodedStr2[]        = "ABEiM0RVZneImaq7zN3u/wARIjNEVWZ3iJmqu8zd7v8=CD01!&#";
static char EncodedStr2NoLinefeed[] = "ABEiM0RVZneImaq7zN3u/wARIjNEVWZ3iJmqu8zd7v8=CD01!&#";
static char EncodedStr2UrlSafe[] = "ABEiM0RVZneImaq7zN3u_wARIjNEVWZ3iJmqu8zd7v8=CD01!&#";

TEST(UtilBase64Test, SkipCharTest)
{
    uint8_t     buf[64];
    size_t      num;

    std::memset(buf, 0, sizeof(buf));
    EXPECT_EQ(Base64::Status_Success,
        Base64::FromBase64String(&num, buf, sizeof(buf), EncodedStr0, Base64::Mode_Normal));
    EXPECT_EQ(num, sizeof(BaseData));
    EXPECT_EQ(std::memcmp(BaseData, buf, num), 0);

    std::memset(buf, 0, sizeof(buf));
    EXPECT_EQ(Base64::Status_Success,
        Base64::FromBase64String(&num, buf, sizeof(buf), EncodedStr1, Base64::Mode_Normal));
    EXPECT_EQ(num, sizeof(BaseData));
    EXPECT_EQ(std::memcmp(BaseData, buf, num), 0);

    std::memset(buf, 0, sizeof(buf));
    EXPECT_EQ(Base64::Status_Success,
        Base64::FromBase64String(&num, buf, sizeof(buf), EncodedStr2, Base64::Mode_Normal));
    EXPECT_EQ(num, sizeof(BaseData));
    EXPECT_EQ(std::memcmp(BaseData, buf, num), 0);
}

TEST(UtilBase64Test, SkipCharTestNoLinefeed)
{
    uint8_t     buf[64];
    size_t      num;

    std::memset(buf, 0, sizeof(buf));
    EXPECT_EQ(Base64::Status_Success,
        Base64::FromBase64String(&num, buf, sizeof(buf), EncodedStr0NoLinefeed, Base64::Mode_NormalNoLinefeed));
    EXPECT_EQ(num, sizeof(BaseData));
    EXPECT_EQ(std::memcmp(BaseData, buf, num), 0);

    std::memset(buf, 0, sizeof(buf));
    EXPECT_EQ(Base64::Status_Success,
        Base64::FromBase64String(&num, buf, sizeof(buf), EncodedStr1NoLinefeed, Base64::Mode_NormalNoLinefeed));
    EXPECT_EQ(num, sizeof(BaseData));
    EXPECT_EQ(std::memcmp(BaseData, buf, num), 0);

    std::memset(buf, 0, sizeof(buf));
    EXPECT_EQ(Base64::Status_Success,
        Base64::FromBase64String(&num, buf, sizeof(buf), EncodedStr2NoLinefeed, Base64::Mode_NormalNoLinefeed));
    EXPECT_EQ(num, sizeof(BaseData));
    EXPECT_EQ(std::memcmp(BaseData, buf, num), 0);
}

TEST(UtilBase64Test, SkipCharTestUrlSafe)
{
    uint8_t     buf[64];
    size_t      num;

    std::memset(buf, 0, sizeof(buf));
    EXPECT_EQ(Base64::Status_Success,
        Base64::FromBase64String(&num, buf, sizeof(buf), EncodedStr0UrlSafe, Base64::Mode_UrlSafe));
    EXPECT_EQ(num, sizeof(BaseData));
    EXPECT_EQ(std::memcmp(BaseData, buf, num), 0);

    std::memset(buf, 0, sizeof(buf));
    EXPECT_EQ(Base64::Status_Success,
        Base64::FromBase64String(&num, buf, sizeof(buf), EncodedStr1UrlSafe, Base64::Mode_UrlSafe));
    EXPECT_EQ(num, sizeof(BaseData));
    EXPECT_EQ(std::memcmp(BaseData, buf, num), 0);

    std::memset(buf, 0, sizeof(buf));
    EXPECT_EQ(Base64::Status_Success,
        Base64::FromBase64String(&num, buf, sizeof(buf), EncodedStr2UrlSafe, Base64::Mode_UrlSafe));
    EXPECT_EQ(num, sizeof(BaseData));
    EXPECT_EQ(std::memcmp(BaseData, buf, num), 0);
}

//------------------------------------------------------------------
// Bad char test
//------------------------------------------------------------------
// BaseData を単純にエンコードしたもの
static char EncodedBase[]        = "ABEiM0RVZneImaq7zN3u/wARIjNEVWZ3iJmqu8zd7v8=";
static char EncodedBaseNoLinefeed[] = "ABEiM0RVZneImaq7zN3u/wARIjNEVWZ3iJmqu8zd7v8=";
static char EncodedBaseUrlSafe[] = "ABEiM0RVZneImaq7zN3u_wARIjNEVWZ3iJmqu8zd7v8";
static char BadChars[]           = "!#\\$%&'()-~|[]`*;:{}<>,.?_^@\\\x01\x02\x03\x04\x05\x06\x07\x08\x0B\x0C\x10\x22\x80\xc0\xff";
static char BadCharsNoLinefeed[] = "!#\\$%&'()-~|[]`*;:{}<>,.?_^@\\\x01\x02\x03\x04\x05\x06\x07\x08\x0B\x0C\x10\x22\x80\xc0\xff";
static char BadCharsUrlSafe[]    = "!#\\$%&'()+~|[]`*;:{}<>,.?/^@\\\x01\x02\x03\x04\x05\x06\x07\x08\x0B\x0C\x10\x22\x80\xc0\xff";

// INFO: \0 は終端のため挟む場所によっては成功してしまう

TEST(UtilBase64Test, BadCharTest)
{
    char        inBuf[64];
    uint8_t     outBuf[64];
    size_t      num;

    const size_t baseLen = std::strlen(EncodedBase);

    for ( size_t i = 0 ; i < baseLen ; ++i )
    {
        std::strcpy(inBuf, EncodedBase);
        std::memset(outBuf, 0, sizeof(outBuf));

        // i 文字目を BadChar で破壊
        inBuf[i] = BadChars[i];

        EXPECT_EQ(Base64::Status_BadData,
            Base64::FromBase64String(&num, outBuf, sizeof(outBuf), inBuf, Base64::Mode_Normal)) << "error i:" << i;
        EXPECT_EQ(num, (i / 4) * 3) << "error i:" << i << " num:" << num;
        EXPECT_EQ(std::memcmp(BaseData, outBuf, num), 0);

        // Debug Print
        if( num != ((i / 4) * 3) ||
            std::memcmp(BaseData, outBuf, num) != 0 )
        {
            NN_LOG("---------------------- Test Failed. i:%zu\n", i);
            NN_LOG("BadChars[%d]:0x%02x, decodeNum:%zu\n", i, BadChars[i], num);

            UtilBase64TestSub::PrintBinary("inBuf", inBuf, baseLen);
            UtilBase64TestSub::PrintBinary("outBuf", outBuf, num);
        }

        EXPECT_TRUE(UtilBase64TestSub::IsClean(&outBuf[num], sizeof(BaseData) - num));
    }
}
TEST(UtilBase64Test, BadCharTestNoLinefeed)
{
    char        inBuf[64];
    uint8_t     outBuf[64];
    size_t      num;

    const size_t baseLen = std::strlen(EncodedBase);

    for ( size_t i = 0 ; i < baseLen ; ++i )
    {
        std::strcpy(inBuf, EncodedBaseNoLinefeed);
        std::memset(outBuf, 0, sizeof(outBuf));

        // i 文字目を BadChar で破壊
        inBuf[i] = BadChars[i];

        EXPECT_EQ(Base64::Status_BadData,
            Base64::FromBase64String(&num, outBuf, sizeof(outBuf), inBuf, Base64::Mode_NormalNoLinefeed)) << "error i:" << i;
        EXPECT_EQ(num, (i / 4) * 3) << "error i:" << i << " num:" << num;
        EXPECT_EQ(std::memcmp(BaseData, outBuf, num), 0);

        // Debug Print
        if( num != ((i / 4) * 3) ||
            std::memcmp(BaseData, outBuf, num) != 0 )
        {
            NN_LOG("---------------------- Test Failed. i:%zu\n", i);
            NN_LOG("BadChars[%d]:0x%02x, decodeNum:%zu\n", i, BadChars[i], num);

            UtilBase64TestSub::PrintBinary("inBuf", inBuf, baseLen);
            UtilBase64TestSub::PrintBinary("outBuf", outBuf, num);
        }

        EXPECT_TRUE(UtilBase64TestSub::IsClean(&outBuf[num], sizeof(BaseData) - num));
    }
}
TEST(UtilBase64Test, BadCharTestUrlSafe)
{
    char        inBuf[64];
    uint8_t     outBuf[64];
    size_t      num;

    const size_t baseLen = std::strlen(EncodedBaseUrlSafe);

    for ( size_t i = 0 ; i < baseLen ; ++i )
    {
        std::strcpy(inBuf, EncodedBaseUrlSafe);
        std::memset(outBuf, 0, sizeof(outBuf));

        // i 文字目を BadCharsUrlSafe で破壊
        inBuf[i] = BadCharsUrlSafe[i];

        EXPECT_EQ(Base64::Status_BadData,
            Base64::FromBase64String(&num, outBuf, sizeof(outBuf),inBuf, Base64::Mode_UrlSafe)) << "error i:" << i;
        EXPECT_EQ(num, (i / 4) * 3) << "error i:" << i << " num:" << num;
        EXPECT_EQ(std::memcmp(BaseData, outBuf, num), 0);

        // Debug Print
        if( num != ((i / 4) * 3) ||
            std::memcmp(BaseData, outBuf, num) != 0 )
        {
            NN_LOG("---------------------- Test Failed. i:%zu\n", i);
            NN_LOG("BadCharsUrlSafe[%d]:0x%02x, decodeNum:%zu\n", i, BadCharsUrlSafe[i], num);

            UtilBase64TestSub::PrintBinary("inBuf", inBuf, baseLen);
            UtilBase64TestSub::PrintBinary("outBuf", outBuf, num);
        }


        EXPECT_TRUE(UtilBase64TestSub::IsClean(&outBuf[num], sizeof(BaseData) - num));
    }
}

//------------------------------------------------------------------
// Buffer full test
//------------------------------------------------------------------
static char EncodedFF[]          = "///////////////////////////////////////////=";
static char EncodedFFNoLinefeed[] = "///////////////////////////////////////////=";
static char EncodedFFUrlSafe[]   = "___________________________________________";

TEST(UtilBase64Test, BufferFullTest)
{
    uint8_t     outBuf[64];
    size_t      num;

    for ( int i = 0 ; i < 32 ; ++i )
    {
        std::memset(outBuf, 0, sizeof(outBuf));

        EXPECT_EQ(Base64::Status_BufferFull,
            Base64::FromBase64String(&num, outBuf, i, EncodedFF, Base64::Mode_Normal));
        EXPECT_EQ(num, (i / 3) * 3);
        if ( num > 0 )
        {
            EXPECT_EQ(0xFF, outBuf[num - 1]);
        }
        EXPECT_TRUE(UtilBase64TestSub::IsClean(&outBuf[num], sizeof(BaseData) - num));
    }
}

TEST(UtilBase64Test, BufferFullTestNoLinefeed)
{
    uint8_t     outBuf[64];
    size_t      num;

    for ( int i = 0 ; i < 32 ; ++i )
    {
        std::memset(outBuf, 0, sizeof(outBuf));

        EXPECT_EQ(Base64::Status_BufferFull,
            Base64::FromBase64String(&num, outBuf, i, EncodedFFNoLinefeed, Base64::Mode_NormalNoLinefeed));
        EXPECT_EQ(num, (i / 3) * 3);
        if ( num > 0 )
        {
            EXPECT_EQ(0xFF, outBuf[num - 1]);
        }
        EXPECT_TRUE(UtilBase64TestSub::IsClean(&outBuf[num], sizeof(BaseData) - num));
    }
}

TEST(UtilBase64Test, BufferFullTestUrlSafe)
{
    uint8_t     outBuf[64];
    size_t      num;

    for ( int i = 0 ; i < 32 ; ++i )
    {
        std::memset(outBuf, 0, sizeof(outBuf));

        EXPECT_EQ(Base64::Status_BufferFull,
            Base64::FromBase64String(&num, outBuf, i, EncodedFFUrlSafe, Base64::Mode_UrlSafe));
        EXPECT_EQ(num, (i / 3) * 3);
        if ( num > 0 )
        {
            EXPECT_EQ(0xFF, outBuf[num - 1]);
        }
        EXPECT_TRUE(UtilBase64TestSub::IsClean(&outBuf[num], sizeof(BaseData) - num));
    }
}

TEST(UtilBase64Test, DecodeOfSpecifiedSrcSize)
{
    const char* encoded = "KQNIIRRL";
    const uint8_t decoded[] = { 0x29, 0x03, 0x48, 0x21, 0x14, 0x4b };
    uint8_t outBuf[64];
    size_t num;

    {
        EXPECT_EQ(Base64::Status_Success,
            Base64::FromBase64String(&num, outBuf, sizeof(outBuf), encoded, 4, Base64::Mode_UrlSafe));
        EXPECT_EQ(3, num);
        EXPECT_EQ(std::memcmp(decoded, outBuf, num), 0);

        EXPECT_EQ(Base64::Status_Success,
            Base64::FromBase64String(&num, outBuf, sizeof(outBuf), encoded, 8, Base64::Mode_UrlSafe));
        EXPECT_EQ(6, num);
        EXPECT_EQ(std::memcmp(decoded, outBuf, num), 0);

        EXPECT_EQ(Base64::Status_Success,
            Base64::FromBase64String(&num, outBuf, sizeof(outBuf), encoded, 12, Base64::Mode_UrlSafe)); // 12 だけどNULL終端で変換終わる
        EXPECT_EQ(6, num);
        EXPECT_EQ(std::memcmp(decoded, outBuf, num), 0);
    }
    {
        EXPECT_EQ(Base64::Status_Success,
            Base64::FromBase64String(&num, outBuf, sizeof(outBuf), encoded, 4, Base64::Mode_Normal));
        EXPECT_EQ(3, num);
        EXPECT_EQ(std::memcmp(decoded, outBuf, num), 0);

        EXPECT_EQ(Base64::Status_Success,
            Base64::FromBase64String(&num, outBuf, sizeof(outBuf), encoded, 8, Base64::Mode_Normal));
        EXPECT_EQ(6, num);
        EXPECT_EQ(std::memcmp(decoded, outBuf, num), 0);

        EXPECT_EQ(Base64::Status_Success,
            Base64::FromBase64String(&num, outBuf, sizeof(outBuf), encoded, 12, Base64::Mode_Normal)); // 12 だけどNULL終端で変換終わる
        EXPECT_EQ(6, num);
        EXPECT_EQ(std::memcmp(decoded, outBuf, num), 0);
    }
    {
        EXPECT_EQ(Base64::Status_Success,
            Base64::FromBase64String(&num, outBuf, sizeof(outBuf), encoded, 4, Base64::Mode_NormalNoLinefeed));
        EXPECT_EQ(3, num);
        EXPECT_EQ(std::memcmp(decoded, outBuf, num), 0);

        EXPECT_EQ(Base64::Status_Success,
            Base64::FromBase64String(&num, outBuf, sizeof(outBuf), encoded, 8, Base64::Mode_NormalNoLinefeed));
        EXPECT_EQ(6, num);
        EXPECT_EQ(std::memcmp(decoded, outBuf, num), 0);

        EXPECT_EQ(Base64::Status_Success,
            Base64::FromBase64String(&num, outBuf, sizeof(outBuf), encoded, 12, Base64::Mode_NormalNoLinefeed)); // 12 だけどNULL終端で変換終わる
        EXPECT_EQ(6, num);
        EXPECT_EQ(std::memcmp(decoded, outBuf, num), 0);
    }
}

//------------------------------------------------------------------
#if 0
{
    char buf[128];
    Base64::ToBase64String(BaseData, sizeof(BaseData), buf, sizeof(buf));

    NN_TLOG_("Encoded:\n%s\n", buf);
}
#endif


}} // namespace nn::util

/*---------------------------------------------------------------------------*
  End of file
 *---------------------------------------------------------------------------*/
