﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nnt/nntest.h>
#include <nn/util/util_MathTypes.h>
#include <nn/util/util_Arithmetic.h>
#include <cmath>

namespace
{
    const float FloatPi = 3.141592653589793f;
    const float Float2Pi = 6.283185307179586f;

    const nn::util::AngleIndex AngleIndexHalfRound = 0x80000000;
    const nn::util::AngleIndex AngleIndex4Ulp = static_cast<nn::util::AngleIndex>(static_cast<int64_t>(4.f * ::std::numeric_limits<float>::epsilon() * (AngleIndexHalfRound / FloatPi)));

    const float ErrorSinCosEst   = 0.00006f;
    const float ErrorTanEst      = 0.00006f;
    const float ErrorAtanEst     = 0.000006f;

    const float ErrorSinCosTable = 0.00006f;
    const float ErrorTanTable    = 0.00006f;
    const float ErrorAtanTable   = 0.000006f;

    ::testing::AssertionResult TestAreEqual(float expected, float actual, float error)
    {
        float diff = ::std::abs(expected - actual);
        float max = ::std::max(::std::abs(expected), ::std::abs(actual));

        error = max <= 1 ? error : error * max;

        if(diff <= error)
        {
            return ::testing::AssertionSuccess();
        }
        else
        {
            return ::testing::AssertionFailure()
                << "\n"
                << " expected: " << expected << "\n"
                << "   actual: " << actual << "\n"
                << "     diff: " << diff << "\n"
                << "    error: " << error << "\n"
                << " exceeded: " << diff - error << "\n";
        }
    }
}

TEST(ArithmeticTest, RadianToAngleIndex)
{
    EXPECT_NEAR(0x00000000, nn::util::RadianToAngleIndex(0.f), AngleIndex4Ulp);
    EXPECT_NEAR(0x80000000, nn::util::RadianToAngleIndex(FloatPi), AngleIndex4Ulp);
}

TEST(ArithmeticTest, DegreeToAngleIndex)
{
    EXPECT_NEAR(0x00000000, nn::util::DegreeToAngleIndex(0.f), AngleIndex4Ulp);
    EXPECT_NEAR(0x80000000, nn::util::DegreeToAngleIndex(180.f), AngleIndex4Ulp);
}

TEST(ArithmeticTest, DegreeToRadian)
{
    EXPECT_FLOAT_EQ(0.f, nn::util::DegreeToRadian(0.f));
    EXPECT_FLOAT_EQ(FloatPi, nn::util::DegreeToRadian(180.f));
}

TEST(ArithmeticTest, RadianToDegree)
{
    EXPECT_FLOAT_EQ(0.f, nn::util::RadianToDegree(0.f));
    EXPECT_FLOAT_EQ(180.f, nn::util::RadianToDegree(FloatPi));
}

TEST(ArithmeticTest, AngleIndexToRadian)
{
    EXPECT_FLOAT_EQ(0.f, nn::util::AngleIndexToRadian(0x00000000));
    EXPECT_FLOAT_EQ(FloatPi, nn::util::AngleIndexToRadian(0x80000000));
}

TEST(ArithmeticTest, AngleIndexToDegree)
{
    EXPECT_FLOAT_EQ(0.f, nn::util::AngleIndexToDegree(0x00000000));
    EXPECT_FLOAT_EQ(180.f, nn::util::AngleIndexToDegree(0x80000000));
}

TEST(ArithmeticTest, SinCosTanEst)
{
    for(uint32_t i = 0; i < 512; i++)
    {
        SCOPED_TRACE(i);

        nn::util::AngleIndex index = i * (AngleIndexHalfRound >> 8);
        float radian = nn::util::AngleIndexToRadian(index);

        EXPECT_NEAR(::std::sin(radian), nn::util::SinEst(radian), ErrorSinCosEst);
        EXPECT_NEAR(::std::cos(radian), nn::util::CosEst(radian), ErrorSinCosEst);

        float sin, cos;
        nn::util::SinCosEst(&sin, &cos, radian);

        EXPECT_FLOAT_EQ(sin, nn::util::SinEst(radian));
        EXPECT_FLOAT_EQ(cos, nn::util::CosEst(radian));

        // 90 度はスキップ
        if(i % 256 != 128)
        {
            EXPECT_TRUE(TestAreEqual(::std::tan(radian), nn::util::TanEst(radian), ErrorTanEst));
        }
    }
}

TEST(ArithmeticTest, AtanEst)
{
    for(uint32_t i = 1; i < 512; i++)
    {
        SCOPED_TRACE(i);

        float tangent = 1.0f / 256 * i;
        EXPECT_NEAR(::std::atan(tangent), nn::util::AtanEst(tangent), ErrorAtanEst);
        EXPECT_NEAR(::std::atan(-tangent), nn::util::AtanEst(-tangent), ErrorAtanEst);
    }
}

TEST(ArithmeticTest, Atan2Est)
{
    EXPECT_NEAR(::std::atan2(1.f, 2.f), nn::util::Atan2Est(1.f, 2.f), ErrorAtanEst);
    EXPECT_NEAR(::std::atan2(2.f, 1.f), nn::util::Atan2Est(2.f, 1.f), ErrorAtanEst);
    EXPECT_NEAR(::std::atan2(2.f, -1.f), nn::util::Atan2Est(2.f, -1.f), ErrorAtanEst);
    EXPECT_NEAR(::std::atan2(1.f, -2.f), nn::util::Atan2Est(1.f, -2.f), ErrorAtanEst);
    EXPECT_NEAR(::std::atan2(-1.f, -2.f), nn::util::Atan2Est(-1.f, -2.f), ErrorAtanEst);
    EXPECT_NEAR(::std::atan2(-2.f, -1.f), nn::util::Atan2Est(-2.f, -1.f), ErrorAtanEst);
    EXPECT_NEAR(::std::atan2(-2.f, 1.f), nn::util::Atan2Est(-2.f, 1.f), ErrorAtanEst);
    EXPECT_NEAR(::std::atan2(-1.f, 2.f), nn::util::Atan2Est(-1.f, 2.f), ErrorAtanEst);
}

TEST(ArithmeticTest, AsinEst)
{
    EXPECT_NEAR(::std::asin(0.f), nn::util::AsinEst(0.f), ErrorAtanEst);
    EXPECT_NEAR(::std::asin(1.f), nn::util::AsinEst(1.f), ErrorAtanEst);
    EXPECT_NEAR(::std::asin(-1.f), nn::util::AsinEst(-1.f), ErrorAtanEst);
    EXPECT_NEAR(::std::asin(0.1f), nn::util::AsinEst(0.1f), ErrorAtanEst);
    EXPECT_NEAR(::std::asin(0.9f), nn::util::AsinEst(0.9f), ErrorAtanEst);
    EXPECT_NEAR(::std::asin(-0.9f), nn::util::AsinEst(-0.9f), ErrorAtanEst);
    EXPECT_NEAR(::std::asin(-0.1f), nn::util::AsinEst(-0.1f), ErrorAtanEst);
}

TEST(ArithmeticTest, AcosEst)
{
    EXPECT_NEAR(::std::acos(0.f), nn::util::AcosEst(0.f), ErrorAtanEst);
    EXPECT_NEAR(::std::acos(1.f), nn::util::AcosEst(1.f), ErrorAtanEst);
    EXPECT_NEAR(::std::acos(-1.f), nn::util::AcosEst(-1.f), ErrorAtanEst);
    EXPECT_NEAR(::std::acos(0.1f), nn::util::AcosEst(0.1f), ErrorAtanEst);
    EXPECT_NEAR(::std::acos(0.9f), nn::util::AcosEst(0.9f), ErrorAtanEst);
    EXPECT_NEAR(::std::acos(-0.9f), nn::util::AcosEst(-0.9f), ErrorAtanEst);
    EXPECT_NEAR(::std::acos(-0.1f), nn::util::AcosEst(-0.1f), ErrorAtanEst);
}


TEST(ArithmeticTest, SinCosTanTable)
{
    for(uint32_t i = 0; i < 512; i++)
    {
        SCOPED_TRACE(i);

        nn::util::AngleIndex index = i * (AngleIndexHalfRound >> 8);
        float radian = nn::util::AngleIndexToRadian(index);

        EXPECT_NEAR(::std::sin(radian), nn::util::SinTable(index), ErrorSinCosTable);
        EXPECT_NEAR(::std::cos(radian), nn::util::CosTable(index), ErrorSinCosTable);

        float sin, cos;
        nn::util::SinCosTable(&sin, &cos, index);

        EXPECT_FLOAT_EQ(sin, nn::util::SinTable(index));
        EXPECT_FLOAT_EQ(cos, nn::util::CosTable(index));

        // 90 度はスキップ
        if(i % 256 != 128)
        {
            EXPECT_TRUE(TestAreEqual(::std::tan(radian), nn::util::TanTable(index), ErrorTanTable));
        }
    }
}

TEST(ArithmeticTest, AtanTable)
{
    for(uint32_t i = 1; i < 512; i++)
    {
        SCOPED_TRACE(i);

        float tangent = 1.0f / 256 * i;
        float radian = ::std::atan(tangent);
        EXPECT_NEAR(radian, nn::util::AngleIndexToRadian(nn::util::AtanTable(tangent)), ErrorAtanTable);
        EXPECT_NEAR(Float2Pi - radian, nn::util::AngleIndexToRadian(nn::util::AtanTable(-tangent)), ErrorAtanTable);
    }
}

TEST(ArithmeticTest, Atan2Table)
{
    EXPECT_NEAR(::std::atan2(1.f, 2.f), nn::util::AngleIndexToRadian(nn::util::Atan2Table(1.f, 2.f)), ErrorAtanTable);
    EXPECT_NEAR(::std::atan2(2.f, 1.f), nn::util::AngleIndexToRadian(nn::util::Atan2Table(2.f, 1.f)), ErrorAtanTable);
    EXPECT_NEAR(::std::atan2(2.f, -1.f), nn::util::AngleIndexToRadian(nn::util::Atan2Table(2.f, -1.f)), ErrorAtanTable);
    EXPECT_NEAR(::std::atan2(1.f, -2.f), nn::util::AngleIndexToRadian(nn::util::Atan2Table(1.f, -2.f)), ErrorAtanTable);
    EXPECT_NEAR(::std::atan2(-1.f, -2.f) + Float2Pi, nn::util::AngleIndexToRadian(nn::util::Atan2Table(-1.f, -2.f)), ErrorAtanTable);
    EXPECT_NEAR(::std::atan2(-2.f, -1.f) + Float2Pi, nn::util::AngleIndexToRadian(nn::util::Atan2Table(-2.f, -1.f)), ErrorAtanTable);
    EXPECT_NEAR(::std::atan2(-2.f, 1.f) + Float2Pi, nn::util::AngleIndexToRadian(nn::util::Atan2Table(-2.f, 1.f)), ErrorAtanTable);
    EXPECT_NEAR(::std::atan2(-1.f, 2.f) + Float2Pi, nn::util::AngleIndexToRadian(nn::util::Atan2Table(-1.f, 2.f)), ErrorAtanTable);
}

TEST(ArithmeticTest, AsinTable)
{
    EXPECT_NEAR(::std::asin(0.f), nn::util::AngleIndexToRadian(nn::util::AsinTable(0.f)), ErrorAtanTable);
    EXPECT_NEAR(::std::asin(1.f), nn::util::AngleIndexToRadian(nn::util::AsinTable(1.f)), ErrorAtanTable);
    EXPECT_NEAR(::std::asin(-1.f) + Float2Pi, nn::util::AngleIndexToRadian(nn::util::AsinTable(-1.f)), ErrorAtanTable);
    EXPECT_NEAR(::std::asin(0.1f), nn::util::AngleIndexToRadian(nn::util::AsinTable(0.1f)), ErrorAtanTable);
    EXPECT_NEAR(::std::asin(0.9f), nn::util::AngleIndexToRadian(nn::util::AsinTable(0.9f)), ErrorAtanTable);
    EXPECT_NEAR(::std::asin(-0.9f) + Float2Pi, nn::util::AngleIndexToRadian(nn::util::AsinTable(-0.9f)), ErrorAtanTable);
    EXPECT_NEAR(::std::asin(-0.1f) + Float2Pi, nn::util::AngleIndexToRadian(nn::util::AsinTable(-0.1f)), ErrorAtanTable);
}

TEST(ArithmeticTest, AcosTable)
{
    EXPECT_NEAR(::std::acos(0.f), nn::util::AngleIndexToRadian(nn::util::AcosTable(0.f)), ErrorAtanTable);
    EXPECT_NEAR(::std::acos(1.f), nn::util::AngleIndexToRadian(nn::util::AcosTable(1.f)), ErrorAtanTable);
    EXPECT_NEAR(::std::acos(-1.f), nn::util::AngleIndexToRadian(nn::util::AcosTable(-1.f)), ErrorAtanTable);
    EXPECT_NEAR(::std::acos(0.1f), nn::util::AngleIndexToRadian(nn::util::AcosTable(0.1f)), ErrorAtanTable);
    EXPECT_NEAR(::std::acos(0.9f), nn::util::AngleIndexToRadian(nn::util::AcosTable(0.9f)), ErrorAtanTable);
    EXPECT_NEAR(::std::acos(-0.9f), nn::util::AngleIndexToRadian(nn::util::AcosTable(-0.9f)), ErrorAtanTable);
    EXPECT_NEAR(::std::acos(-0.1f), nn::util::AngleIndexToRadian(nn::util::AcosTable(-0.1f)), ErrorAtanTable);
}

TEST(ArithmeticTest, Rcp)
{
    EXPECT_FLOAT_EQ(1.f, nn::util::Rcp(1.f));
    EXPECT_FLOAT_EQ(0.5f, nn::util::Rcp(2.f));
    EXPECT_FLOAT_EQ(0.1428571428571429f, nn::util::Rcp(7.f));
}

TEST(ArithmeticTest, RcpLowPrecision)
{
    EXPECT_FLOAT_EQ(1.f, nn::util::RcpLowPrecision<3>(1.f));
    EXPECT_FLOAT_EQ(0.5f, nn::util::RcpLowPrecision<3>(2.f));
    EXPECT_FLOAT_EQ(0.1428571428571429f, nn::util::RcpLowPrecision<3>(7.f));
}

TEST(ArithmeticTest, Rsqrt)
{
    EXPECT_FLOAT_EQ(1.f, nn::util::Rsqrt(1.f));
    EXPECT_FLOAT_EQ(0.5f, nn::util::Rsqrt(4.f));
    EXPECT_FLOAT_EQ(0.70710678118f, nn::util::Rsqrt(2.f));
}

TEST(ArithmeticTest, RsqrtLowPrecision)
{
    EXPECT_FLOAT_EQ(1.f, nn::util::RsqrtLowPrecision<3>(1.f));
    EXPECT_FLOAT_EQ(0.5f, nn::util::RsqrtLowPrecision<3>(4.f));
    EXPECT_FLOAT_EQ(0.70710678118f, nn::util::RsqrtLowPrecision<3>(2.f));
}

TEST(ArithmeticTest, AreEqualAbs)
{
    EXPECT_FALSE(nn::util::AreEqualAbs(1.f, 2.f, 0.f));
    EXPECT_TRUE(nn::util::AreEqualAbs(1.f, 2.f, 1.f));
    EXPECT_FALSE(nn::util::AreEqualAbs(1.f, 2.f, 0.9f));
    EXPECT_FALSE(nn::util::AreEqualAbs(-1.f, -2.f, 0.f));
    EXPECT_TRUE(nn::util::AreEqualAbs(-1.f, -2.f, 1.f));
    EXPECT_FALSE(nn::util::AreEqualAbs(-1.f, -2.f, 0.9f));
}

TEST(ArithmeticTest, AreEqualRelative)
{
    EXPECT_FALSE(nn::util::AreEqualRelative(1.f, 2.f, 0.f));
    EXPECT_TRUE(nn::util::AreEqualRelative(1.f, 2.f, 0.5f));
    EXPECT_FALSE(nn::util::AreEqualRelative(1.f, 2.f, 0.49f));
    EXPECT_FALSE(nn::util::AreEqualRelative(-1.f, -2.f, 0.f));
    EXPECT_TRUE(nn::util::AreEqualRelative(-1.f, -2.f, 0.5f));
    EXPECT_FALSE(nn::util::AreEqualRelative(-1.f, -2.f, 0.49f));
}
