﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>

#include "UsbDiag_Theme.h"
#include "UsbDiag_DeviceView.h"
#include "UsbDiag_IfView.h"
#include "UsbDiag_EpView.h"
#include "UsbDiag_LabelView.h"

using namespace nn::usb;

namespace nnt {
namespace usb {

DeviceView::DeviceView(const Device& device) NN_NOEXCEPT
    : m_Device(device)
    , m_Detail("x-,<-,<<")
{
    const UsbDeviceDescriptor& deviceDesc = m_Device.deviceDesc;
    const UsbConfigDescriptor& configDesc = m_Device.configDesc;

    CreateDetailView();

    disable(glv::Property::FocusToTop);
    enable(glv::Property::DrawBorder);

    nn::util::SNPrintf(
        m_String, sizeof(m_String),
        "Device %04x:%04x\n"
        "Topology: %s\n"
        "Configure: %d (%d in total)",
        deviceDesc.idVendor,
        deviceDesc.idProduct,
        m_Device.name,
        configDesc.bConfigurationValue,
        deviceDesc.bNumConfigurations
    );

    LabelView *deviceLabel = new LabelView(m_String, 10, 10, m_Detail);
    AdaptiveListView *ifList = new AdaptiveListView();

    for (auto& interface : m_Device.interfaces)
    {
        *ifList << new IfView(interface);
    }

    ifList->Adapt(true);
    deviceLabel->width(DeviceWidth - ifList->width());

    *this << deviceLabel << ifList;

    Adapt(false);
}

DeviceView::~DeviceView() NN_NOEXCEPT
{
    // nothing special
}

void DeviceView::CreateDetailView() NN_NOEXCEPT
{
    const UsbDeviceDescriptor& deviceDesc = m_Device.deviceDesc;
    const UsbConfigDescriptor& configDesc = m_Device.configDesc;

    const char *title;
    const char *name;

    title = "Device Descriptor";
    name  =
        "bLength\n"
        "bDescriptorType\n"
        "bcdUSB\n"
        "bDeviceClass\n"
        "bDeviceSubClass\n"
        "bDeviceProtocol\n"
        "bMaxPacketSize0\n"
        "idVendor\n"
        "idProduct\n"
        "bcdDevice\n"
        "iManufacturer\n"
        "iProduct\n"
        "iSerialNumber\n"
        "bNumConfigurations:";

    nn::util::SNPrintf(
        m_String, sizeof(m_String),
        ": %02x\n"
        ": %02x\n"
        ": %04x\n"
        ": %02x\n"
        ": %02x\n"
        ": %02x\n"
        ": %02x\n"
        ": %04x\n"
        ": %04x\n"
        ": %04x\n"
        ": %02x\n"
        ": %02x\n"
        ": %02x\n"
        ": %02x",
        deviceDesc.bLength,
        deviceDesc.bDescriptorType,
        deviceDesc.bcdUSB,
        deviceDesc.bDeviceClass,
        deviceDesc.bDeviceSubClass,
        deviceDesc.bDeviceProtocol,
        deviceDesc.bMaxPacketSize0,
        deviceDesc.idVendor,
        deviceDesc.idProduct,
        deviceDesc.bcdDevice,
        deviceDesc.iManufacturer,
        deviceDesc.iProduct,
        deviceDesc.iSerialNumber,
        deviceDesc.bNumConfigurations
    );

    m_Detail << new glv::Label(title) << new glv::Divider()
             << new glv::Label(name) << new glv::Label(m_String);

    title = "Configuration Descriptor";
    name  =
        "bLength\n"
        "bDescriptorType\n"
        "wTotalLength\n"
        "bNumInterfaces\n"
        "bConfigurationValue\n"
        "iConfiguration\n"
        "bmAttributes\n"
        "bMaxPower";

    nn::util::SNPrintf(
        m_String, sizeof(m_String),
        ": %02x\n"
        ": %02x\n"
        ": %04x\n"
        ": %02x\n"
        ": %02x\n"
        ": %02x\n"
        ": %02x\n"
        ": %02x",
        configDesc.bLength,
        configDesc.bDescriptorType,
        configDesc.wTotalLength,
        configDesc.bNumInterfaces,
        configDesc.bConfigurationValue,
        configDesc.iConfiguration,
        configDesc.bmAttributes,
        configDesc.bMaxPower
    );

    m_Detail << new glv::Label(title) << new glv::Divider()
             << new glv::Label(name) << new glv::Label(m_String);

    m_Detail.arrange();

    m_Detail.disable(glv::Property::HitTest);
}

} // ~usb
} // ~nnt
