﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/omm/omm_Api.h>
#include <nn/os.h>
#include <nn/psm/psm.h>
#include <nn/psm/psm_System.h>
#include <nn/usb/pd/usb_Pd.h>
#include <nn/usb/pd/usb_PdCradle.h>
#include <nn/usb/usb_Host.h>
#include <nn/vi.private.h>
#include <nn/vi/vi_DisplayEvents.h>

#include <nnt/nntest.h>
#include <nnt/usbPdUtil/testUsbPd_util.h>

/*
    @file
    @brief
        本体、クレードル、AC アダプタなどを特定のトポロジで結線した際に、
        USB などの各 API がそれに合った値を返しているかを確かめるテスト
*/

namespace {
    nn::usb::Host g_UsbHost;

    nn::usb::pd::Session g_PdSession;
    nn::usb::pd::Status g_PdStatus;
    int32_t g_PdActiveNoticeCount;

    nn::omm::OperationMode g_OperationMode;

    nn::psm::ChargerType g_ChargerType;

    nn::vi::Display* g_pDisplay;
    nn::vi::HotplugState g_HotplugState;

    const size_t MaxIfCount = 32;
    nn::usb::InterfaceQueryOutput g_IfBuffer[MaxIfCount];

    int32_t g_IfCount;

    bool TestState(nnt::usb::pd::State state)
    {
        nn::util::BitPack32* s = &g_PdStatus.m_Data;
        NN_LOG("[usb::pd] Status = %08x, %08x, %08x, %08x, %08x\n", s[0], s[1], s[2], s[3], s[4]);
        NN_LOG("[usb::host] g_IfCount = %08x\n", static_cast<uint32_t>(g_IfCount));
        NN_LOG("[omm] g_OperationMode = %08x\n", static_cast<uint32_t>(g_OperationMode));
        NN_LOG("[psm] g_ChargerType = %08x\n", static_cast<uint32_t>(g_ChargerType));
        NN_LOG("[vi] g_HotplugState = %08x\n", static_cast<uint32_t>(g_HotplugState));

        return TestState(state, g_PdStatus, g_PdActiveNoticeCount, g_ChargerType, g_IfCount, g_HotplugState, g_OperationMode);
    }
}

class UsbPdStateTest: public ::testing::Test
{
protected:
    virtual void SetUp()
    {
        nn::omm::Initialize();
        nn::psm::Initialize();
        nn::vi::Initialize();
        nn::usb::pd::Initialize();
        nn::usb::pd::InitializeCradle();

        NN_ABORT_UNLESS_RESULT_SUCCESS(g_UsbHost.Initialize());

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::OpenDisplay(&g_pDisplay, "External"));
        nn::usb::pd::OpenSession( &g_PdSession );

        // usb::pd の Status を取得
        nn::usb::pd::GetStatus(&g_PdStatus, &g_PdSession);

        g_PdActiveNoticeCount = 0;
        if ( g_PdStatus.IsActive() )
        {
            g_PdActiveNoticeCount++;
        }

        // 動作モード取得
        g_OperationMode = nn::omm::GetOperationMode();

        // 充電器種別取得
        g_ChargerType = nn::psm::GetChargerType();

        // Display port の HPD 信号取得
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::GetDisplayHotplugState(&g_HotplugState, g_pDisplay));

        // USB ホストが認識している Interface 数を取得
        auto filter = nn::usb::InvalidDeviceFilter;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_UsbHost.QueryAllInterfaces(&g_IfCount, g_IfBuffer, sizeof(g_IfBuffer), &filter));
    }

    virtual void TearDown()
    {
        nn::usb::pd::CloseSession(&g_PdSession);
        nn::vi::CloseDisplay(g_pDisplay);

        g_UsbHost.Finalize();

        nn::usb::pd::FinalizeCradle();
        nn::usb::pd::Finalize();
        nn::vi::Finalize();
        nn::psm::Finalize();
        nn::omm::Finalize();
    }
};

TEST_F( UsbPdStateTest, NoConnection )
{
    ASSERT_TRUE(TestState(nnt::usb::pd::State_NoConnection));
}

TEST_F( UsbPdStateTest, OfficialAcAdaptor )
{
    ASSERT_TRUE(TestState(nnt::usb::pd::State_OfficialAcAdaptor));
}

TEST_F( UsbPdStateTest, AppleCharger61W )
{
    ASSERT_TRUE(TestState(nnt::usb::pd::State_AppleCharger));
}

TEST_F( UsbPdStateTest, CradleWithOfficialAcAdaptor )
{
    ASSERT_TRUE(TestState(nnt::usb::pd::State_CradleWithOfficialAcAdaptor));
}

TEST_F( UsbPdStateTest, CradleWithHdmiTvOn )
{
    ASSERT_TRUE(TestState(nnt::usb::pd::State_CradleWithHdmiTvOn));
}

TEST_F( UsbPdStateTest, CradleWithPc )
{
    ASSERT_TRUE(TestState(nnt::usb::pd::State_CradleWithPc));
}

TEST_F( UsbPdStateTest, CradleWithAppleCharger29W )
{
    ASSERT_TRUE(TestState(nnt::usb::pd::State_CradleWithAppleCharger));
}

TEST_F( UsbPdStateTest, CradleOverVoltage )
{
    ASSERT_TRUE(TestState(nnt::usb::pd::State_CradleOverVoltage));
}

TEST_F( UsbPdStateTest, CradleOverVoltageAfterContract )
{
    ASSERT_TRUE(TestState(nnt::usb::pd::State_CradleOverVoltageAfterContract));
}

TEST_F( UsbPdStateTest, RelayBoxWithHdmiTvOn )
{
    ASSERT_TRUE(TestState(nnt::usb::pd::State_RelayBoxWithHdmiTvOn));
}

TEST_F( UsbPdStateTest, RelayBoxWithPc )
{
    ASSERT_TRUE(TestState(nnt::usb::pd::State_RelayBoxWithPc));
}

TEST_F( UsbPdStateTest, RelayBoxWithAppleCharger29W )
{
    ASSERT_TRUE(TestState(nnt::usb::pd::State_RelayBoxWithAppleCharger));
}

TEST_F( UsbPdStateTest, Pc )
{
    ASSERT_TRUE(TestState(nnt::usb::pd::State_Pc));
}

TEST_F( UsbPdStateTest, UsbEther )
{
    ASSERT_TRUE(TestState(nnt::usb::pd::State_UsbEther));
}

TEST_F( UsbPdStateTest, OverVoltage )
{
    ASSERT_TRUE(TestState(nnt::usb::pd::State_OverVoltage));
}

TEST_F( UsbPdStateTest, OverVoltageAfterContract )
{
    ASSERT_TRUE(TestState(nnt::usb::pd::State_OverVoltageAfterContract));
}
