﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/os/os_UserExceptionHandler.h>
#include <nnt/nntest.h>

#include <nn/usb/pd/usb_PdCradle.h>
#include <nnt/usbPdUtil/testUsbPd_util.h>

//#define ENABLE_WAIT_EVENT

namespace {
    const nn::TimeSpan NoticeWaitTimeoutSpan = nn::TimeSpan::FromMilliSeconds( 1000 );
}

void CheckCradleVdoResult( nn::Result result, nn::usb::pd::Session* pSession )
{
    if ( result <= nn::usb::pd::ResultInactive() )
    {
        NN_SDK_LOG("ResultInactive\n");
    }
    else if ( result <= nn::usb::pd::ResultInvalidDevice() )
    {
        NN_SDK_LOG("ResultInvalidDevice\n");
    }
    else if ( result <= nn::usb::pd::ResultTimeout() )
    {
        NN_SDK_LOG("ResultTimeout\n");
        nn::usb::pd::Status status;
        nn::usb::pd::Notice notice;
        // クレードル挿抜未発生の確認のため Status -> Notice の順に読む必要あり
        nn::usb::pd::GetStatus( &status, pSession );
        nn::usb::pd::GetNotice( &notice, pSession );
        bool hasCradleBeenActive = nnt::usb::pd::IsCradleFamily( status.GetDeviceType() ) &&
                                   status.IsActive() &&
                                   !notice.IsActiveNotice();
        // クレードル接続継続中はタイムアウトしないはず
        if ( hasCradleBeenActive )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }
    else if ( result.IsFailure() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
}

TEST( UsbPdCradleVdm, CradleVdm )
{
    NN_SDK_LOG("[testUsbPd_CradleVdm] Main\n");

    nn::Result result;

    nn::usb::pd::Initialize();
    nn::usb::pd::Initialize();
    nn::usb::pd::InitializeCradle();
    nn::usb::pd::InitializeCradle();
    nn::usb::pd::Session session;
    nn::usb::pd::Session session2;
    nn::usb::pd::OpenSession( &session );
    nn::usb::pd::OpenSession( &session2 );
    nn::usb::pd::CradleSession cradleSession;
    nn::usb::pd::CradleSession cradleSession2;
    nn::usb::pd::OpenCradleSession( &cradleSession );
    nn::usb::pd::OpenCradleSession( &cradleSession2 );
    nn::os::SystemEvent noticeEvent(nn::os::EventClearMode_AutoClear, true);
    nn::os::SystemEvent noticeEvent2(nn::os::EventClearMode_AutoClear, true);
    nn::usb::pd::BindNoticeEvent( noticeEvent.GetBase(), &session );
    nn::usb::pd::BindNoticeEvent( noticeEvent2.GetBase(), &session2 );
    nn::usb::pd::Status status;
    nn::usb::pd::Status status2;
    nn::usb::pd::Notice notice;
    nn::usb::pd::Notice notice2;
    // 通知フラグ空読みによるクリア
    nn::usb::pd::GetNotice( &notice, &session );
    NN_SDK_LOG("[usb::pd|1] Notice = %04x\n", notice.storage);
    nn::usb::pd::Vdo preVdo;
    nn::usb::pd::Vdo vdo;
    nn::usb::pd::Vdo vdo2;
    nn::usb::pd::GetCradleVdo( &preVdo, &cradleSession, nn::usb::pd::CradleVdmCommand_Led );
    NN_SDK_LOG("[usb::pd:c|1] PresetLedVDO = %08x\n", preVdo);
    vdo.Set<nn::usb::pd::VdmLed::Brightness>( 0xdd );
    vdo.Set<nn::usb::pd::VdmLed::OnTime>( 0xaa );
    vdo.Set<nn::usb::pd::VdmLed::OffTime>( 0x55 );
    vdo.Set<nn::usb::pd::VdmLed::DelayTime>( 0x1 );
    nn::usb::pd::SetCradleVdo( &cradleSession, &vdo, nn::usb::pd::CradleVdmCommand_Led );
    result = nn::usb::pd::GetCradleVdo( &vdo2, &cradleSession, nn::usb::pd::CradleVdmCommand_Led );
    NN_SDK_LOG("[usb::pd:c|1] ChangedLedVDO = %08x\n", vdo2);
    // テストで書き換えた LED 設定を元に戻す
    nn::usb::pd::SetCradleVdo( &cradleSession, &preVdo, nn::usb::pd::CradleVdmCommand_Led );
    nn::usb::pd::GetStatus( &status, &session );
    if ( nnt::usb::pd::IsCradleLedControlable( status.GetDeviceType() ) &&
         !(result <= nn::usb::pd::ResultInactive()) &&
         !(result <= nn::usb::pd::ResultInvalidDevice()) &&
         !(result <= nn::usb::pd::ResultTimeout())
       )
    {
        NN_ABORT_UNLESS(vdo.storage == vdo2.storage, "Cradle LED verify is failed!");
    }
    for ( int i = 0; i <= nn::usb::pd::CradleVdmCommand_McuFwVersion; i++ )
    {
        nn::util::BitPack32* s = &status.m_Data;
        nn::util::BitPack32* s2 = &status2.m_Data;
        NN_UNUSED(s);
        NN_UNUSED(s2);
        nn::usb::pd::GetStatus( &status, &session );
        NN_SDK_LOG("[usb::pd|1] Status = %08x, %08x, %08x, %08x, %08x\n", s[0], s[1], s[2], s[3], s[4]);
        nn::usb::pd::GetStatus( &status2, &session2 );
        NN_SDK_LOG("[usb::pd|2] Status = %08x, %08x, %08x, %08x, %08x\n", s2[0], s2[1], s2[2], s2[3], s2[4]);
        result = nn::usb::pd::GetCradleVdo( &vdo, &cradleSession, nn::usb::pd::CradleVdmCommand(i) );
        NN_SDK_LOG("[usb::pd:c|1] VDO[%d] = %08x\n", i, vdo);
        if ( result.IsSuccess() )
        {
            result = nn::usb::pd::GetCradleVdo( &vdo2, &cradleSession2, nn::usb::pd::CradleVdmCommand(i) );
            NN_SDK_LOG("[usb::pd:c|2] VDO[%d] = %08x\n", i, vdo2);
        }
        CheckCradleVdoResult( result, &session );
#ifdef ENABLE_WAIT_EVENT
        noticeEvent.Wait();
#else
        noticeEvent.TimedWait( NoticeWaitTimeoutSpan );
#endif
        noticeEvent.Clear();
        nn::usb::pd::GetNotice( &notice, &session );
        NN_SDK_LOG("[usb::pd|1] Notice = %04x\n", notice.storage);
        nn::usb::pd::GetNotice( &notice, &session );
        NN_SDK_LOG("[usb::pd|1] Notice = %04x\n", notice.storage);
#ifdef ENABLE_WAIT_EVENT
        noticeEvent2.Wait();
#else
        noticeEvent2.TimedWait( NoticeWaitTimeoutSpan );
#endif
        noticeEvent2.Clear();
        nn::usb::pd::GetNotice( &notice2, &session2 );
        NN_SDK_LOG("[usb::pd|2] Notice = %04x\n", notice2.storage);
        nn::usb::pd::GetNotice( &notice2, &session2 );
        NN_SDK_LOG("[usb::pd|2] Notice = %04x\n", notice2.storage);
    }
    nn::usb::pd::UnbindNoticeEvent( &session2 );
    nn::usb::pd::UnbindNoticeEvent( &session );
    nn::usb::pd::CloseCradleSession( &cradleSession2 );
    nn::usb::pd::CloseCradleSession( &cradleSession );
    nn::usb::pd::CloseSession( &session2 );
    nn::usb::pd::CloseSession( &session );
    nn::usb::pd::FinalizeCradle();
    nn::usb::pd::FinalizeCradle();
    nn::usb::pd::Finalize();
    nn::usb::pd::Finalize();
}

