﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_Vibration.h>

#include "VibrationValueDrawer.h"
#include "SpeedChangeableVibrationPlayer.h"

namespace nns {
namespace iodevices {

//!< 振動子の状態を表す構造体です。
struct VibrationState
{
    nn::hid::VibrationDeviceHandle deviceHandle;    //!< 振動子のハンドル
    nn::hid::VibrationDeviceInfo deviceInfo;        //!< 振動子のデバイス情報
    int vibrationPatternId;                         //!< 振動パターン番号
    nn::hid::VibrationValue currentVibrationValue;  //!< 振動子に設定している振動値
    nn::hid::VibrationValue actualVibrationValue;   //!< 振動子で実際に発生している振動値
};

//!< 操作形態に共通の処理を記述した抽象クラスです。
class NpadControllerBase
{
    NN_DISALLOW_COPY(NpadControllerBase);
    NN_DISALLOW_MOVE(NpadControllerBase);

public:
    NpadControllerBase(nn::hid::NpadIdType npadId, const char* name) NN_NOEXCEPT
        : m_NpadId(npadId)
        , m_Name(name)
        , m_IsConnected(false)
        , m_VibrationDeviceCount(0)
        , m_VibrationPlayerCount(0)
        , m_LoopCount(0)
    {
        m_Buttons.Reset();
        //warning C4351対策のためmemsetで配列の初期化を行う
        memset(m_VibrationStateArray,  0x00, VibrationDeviceCountMax * sizeof(VibrationState));
        memset(m_VibrationValueBuffer, 0x00, VibrationDeviceCountMax * sizeof(VibrationValueBuffer));
    }

    virtual ~NpadControllerBase() NN_NOEXCEPT {}

    //!< メインループの前に呼び出す初期化処理です。
    virtual void Initialize() NN_NOEXCEPT;

    //!< 毎フレーム一度だけ呼び出す更新処理です。
    virtual void Update() NN_NOEXCEPT;

    //!< メインループを抜ける必要がある場合に true を返します。
    virtual bool IsQuitRequired() const NN_NOEXCEPT;

    //!< コントローラの接続スタイルを返します。
    virtual nn::hid::NpadStyleSet GetStyleSet() const NN_NOEXCEPT = 0;

    //!< 表示に用いる名称を取得します。
    virtual const char* GetName() const NN_NOEXCEPT
    {
        return m_Name;
    }

    //!< 搭載されている振動子の個数を取得します。
    virtual int GetVibrationDeviceCount() const NN_NOEXCEPT
    {
        return m_VibrationDeviceCount;
    }

    //!< 振動子の状態を取得します。
    virtual const VibrationState& GetVibrationState(int idx) const NN_NOEXCEPT
    {
        return m_VibrationStateArray[idx];
    }

    //!< 時系列の振動値の情報を取得します。
    virtual const VibrationValueBuffer& GetVibrationValueBuffer(int idx) const NN_NOEXCEPT
    {
        return m_VibrationValueBuffer[idx];
    }

    //!< コントローラが接続されている場合に true を返します。
    virtual bool IsConnected() const NN_NOEXCEPT
    {
        return m_IsConnected;
    }

    virtual void SetVibrationPlayers(nns::iodevices::SpeedChangeableVibrationPlayer** playerArray, int playerCount) NN_NOEXCEPT;
    virtual void InitializeVibrationNode() NN_NOEXCEPT;

protected:
    virtual void UpdateButtons() NN_NOEXCEPT = 0;
    virtual void UpdateVibrationPlayer() NN_NOEXCEPT;
    virtual void StopVibration() NN_NOEXCEPT;
    virtual void UpdateVibrationNode() NN_NOEXCEPT;

protected:
    static const int VibrationDeviceCountMax = 2;
    static const int VibrationPlayerCountMax = 2;

    const nn::hid::NpadIdType m_NpadId;
    const char* const m_Name;
    nn::hid::NpadButtonSet m_Buttons;
    bool m_IsConnected;
    int m_VibrationDeviceCount;
    int m_VibrationPlayerCount;
    uint32_t m_LoopCount;
    VibrationState m_VibrationStateArray[VibrationDeviceCountMax];
    VibrationValueBuffer m_VibrationValueBuffer[VibrationDeviceCountMax];

    nns::iodevices::SpeedChangeableVibrationPlayer* m_Player[VibrationPlayerCountMax];
    nn::hid::VibrationWriter m_Writer;
    nn::hid::VibrationMixer  m_Mixer;
    nn::hid::VibrationTarget m_Target[VibrationDeviceCountMax];
    nn::hid::VibrationNodeConnection m_ConnectionFromPlayerToMixer[VibrationPlayerCountMax];
    nn::hid::VibrationNodeConnection m_ConnectionFromWriterToMixer;
    nn::hid::VibrationNodeConnection m_ConnectionFromMixerToTarget[VibrationDeviceCountMax];
};

//!< FullKeyStyleに特化した処理を記述した NpadControllerBase の派生クラスです。
class NpadFullKeyController : public NpadControllerBase
{
    NN_DISALLOW_COPY(NpadFullKeyController);
    NN_DISALLOW_MOVE(NpadFullKeyController);

public:
    NpadFullKeyController(nn::hid::NpadIdType npadId, const char* name) NN_NOEXCEPT
        : NpadControllerBase(npadId, name)
    {}

    virtual ~NpadFullKeyController() NN_NOEXCEPT NN_OVERRIDE {}

    virtual nn::hid::NpadStyleSet GetStyleSet() const NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::hid::NpadStyleFullKey::Mask;
    }

protected:
    virtual void UpdateButtons() NN_NOEXCEPT NN_OVERRIDE;
};

//!< HandHeldStyleに特化した処理を記述した NpadControllerBase の派生クラスです。
class NpadHandheldController : public NpadControllerBase
{
    NN_DISALLOW_COPY(NpadHandheldController);
    NN_DISALLOW_MOVE(NpadHandheldController);

public:
    NpadHandheldController(nn::hid::NpadIdType npadId, const char* name) NN_NOEXCEPT
        : NpadControllerBase(npadId, name)
    {}

    virtual ~NpadHandheldController() NN_NOEXCEPT NN_OVERRIDE {}

    virtual nn::hid::NpadStyleSet GetStyleSet() const NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::hid::NpadStyleHandheld::Mask;
    }

protected:
    virtual void UpdateButtons() NN_NOEXCEPT NN_OVERRIDE;
};

} // namespace iodevices
} // namespace nns
