﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>

#include "NpadController.h"


namespace nns {
namespace iodevices {

void NpadControllerBase::Initialize() NN_NOEXCEPT
{
    // 振動子のハンドルを取得します
    nn::hid::VibrationDeviceHandle handles[VibrationDeviceCountMax];

    m_VibrationDeviceCount = nn::hid::GetVibrationDeviceHandles(
        handles, VibrationDeviceCountMax, m_NpadId, GetStyleSet());

    for (int i = 0; i < m_VibrationDeviceCount; i++)
    {
        VibrationState& v = m_VibrationStateArray[i];
        v.deviceHandle = handles[i];
        v.vibrationPatternId = 0;
        v.currentVibrationValue = nn::hid::VibrationValue::Make();

        // 振動子を初期化します
        nn::hid::InitializeVibrationDevice(v.deviceHandle);

        // 振動子の情報を取得します
        nn::hid::GetVibrationDeviceInfo(&v.deviceInfo, v.deviceHandle);

        // 振動ハンドルを VibrationTarget に設定する
        if (v.deviceInfo.position == nn::hid::VibrationDevicePosition_Left)
        {
            m_Target[0].SetVibrationDeviceHandle(handles[i]);
        }
        else if (v.deviceInfo.position == nn::hid::VibrationDevicePosition_Right)
        {
            m_Target[1].SetVibrationDeviceHandle(handles[i]);
        }
    }
}

void NpadControllerBase::Update() NN_NOEXCEPT
{
    UpdateButtons();
    UpdateVibrationPlayer();
    UpdateVibrationNode();
    m_LoopCount++;
}

bool NpadControllerBase::IsQuitRequired() const NN_NOEXCEPT
{
    // プラスボタンとマイナスボタンが同時に押されていたら終了判定をします
    const nn::hid::NpadButtonSet QuitButtonMask =
        nn::hid::NpadButton::Plus::Mask | nn::hid::NpadButton::Minus::Mask;

    return ((m_Buttons & QuitButtonMask) == QuitButtonMask);
}

void NpadControllerBase::StopVibration() NN_NOEXCEPT
{
    for (int i = 0; i < m_VibrationDeviceCount; i++)
    {
        nn::hid::VibrationModulation modulation = nn::hid::VibrationModulation::Make(0.0f, 1.0f, 0.0f, 1.0f);
        m_ConnectionFromMixerToTarget[i].SetModulation(modulation);
    }
}


bool rWasPressed = false;
bool lWasPressed = false;
void NpadControllerBase::UpdateVibrationPlayer() NN_NOEXCEPT
{
    if(m_Buttons.Test(nn::hid::NpadButton::L::Index))
    {
        if (!lWasPressed)
        {
            if (m_Player[0]->IsPlaying())
            {
                m_Player[0]->Stop();
            }
            else
            {
                m_Player[0]->Play();
            }
        }
        lWasPressed = true;
    }
    else
    {
        lWasPressed = false;
    }

    if (m_VibrationPlayerCount > 1)
    {
        if (m_Buttons.Test(nn::hid::NpadButton::R::Index))
        {
            if (!rWasPressed)
            {
                if (m_Player[1]->IsPlaying())
                {
                    m_Player[1]->Stop();
                }
                else
                {
                    m_Player[1]->Play();
                }
            }
            rWasPressed = true;
        }
        else
        {
            rWasPressed = false;
        }
    }
}

void NpadControllerBase::UpdateVibrationNode() NN_NOEXCEPT
{
    for (int i = 0; i < m_VibrationPlayerCount; i++)
    {
        //再生速度変更
        float speed = 1.0f;
        if (m_Buttons.Test(nn::hid::NpadButton::Up::Index))
        {
            speed = 1.5f;
        }
        else if (m_Buttons.Test(nn::hid::NpadButton::Down::Index))
        {
            speed = 0.5f;
        }
        m_Player[i]->SetPlaySpeed(speed);
    }

    for (int i = 0; i < m_VibrationDeviceCount; i++)
    {
        nn::hid::VibrationModulation modulation = nn::hid::VibrationModulation::Make();
        modulation.pitchHigh = 1.3f;
        modulation.pitchLow = 1.3f;

        //ピッチとゲインの変更
        if (m_Buttons.Test(nn::hid::NpadButton::A::Index))
        {
            modulation.pitchHigh = 1.5f;
            modulation.pitchLow = 1.5f;
        }
        else if (m_Buttons.Test(nn::hid::NpadButton::Y::Index))
        {
            modulation.pitchHigh = 0.8f;
            modulation.pitchLow = 0.8f;
        }

        if (m_Buttons.Test(nn::hid::NpadButton::X::Index))
        {
            modulation.gainHigh = 2.0f;
            modulation.gainLow = 2.0f;
        }
        else if (m_Buttons.Test(nn::hid::NpadButton::B::Index))
        {
            modulation.gainHigh = 0.5f;
            modulation.gainLow = 0.5f;
        }

        //パンニングの変更
        {
            float panning = 0.5f;
            if (m_Buttons.Test(nn::hid::NpadButton::Right::Index))
            {
                panning = 0.1f;
            }
            else if (m_Buttons.Test(nn::hid::NpadButton::Left::Index))
            {
                panning = 0.9f;
            }

            if (i == 0)
            {
                modulation.gainHigh *= panning;
                modulation.gainLow *= panning;
            }
            else
            {
                modulation.gainHigh *= (1.0f - panning);
                modulation.gainLow *= (1.0f - panning);
            }
        }

        m_ConnectionFromMixerToTarget[i].SetModulation(modulation);

        VibrationState& v = m_VibrationStateArray[i];
        m_Target[i].GetActualVibration(&(v.actualVibrationValue));
        v.currentVibrationValue = m_Target[i].GetCurrentVibration();

        m_VibrationValueBuffer[i].AddVibrationValues(
            v.currentVibrationValue, v.actualVibrationValue
        );
    }

    //VibrationWriteの更新
    {
        nn::hid::VibrationValue value = nn::hid::VibrationValue::Make();
        m_Writer.Write(value);
    }
}

void NpadControllerBase::InitializeVibrationNode() NN_NOEXCEPT
{
    for (int i = 0; i < m_VibrationPlayerCount; i++)
    {
        m_ConnectionFromPlayerToMixer[i].Connect(m_Player[i], &m_Mixer);
        m_Player[i]->Play();
    }

    for (int i = 0; i < m_VibrationDeviceCount; i++)
    {
        m_ConnectionFromMixerToTarget[i].Connect(&m_Mixer, &m_Target[i]);
    }

    m_ConnectionFromWriterToMixer.Connect(&m_Writer, &m_Mixer);
}

void NpadControllerBase::SetVibrationPlayers(nns::iodevices::SpeedChangeableVibrationPlayer** playerArray, int playerCount) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(playerArray);
    NN_ASSERT_GREATER(playerCount, 0);

    m_VibrationPlayerCount = std::min<int>(playerCount, NN_ARRAY_SIZE(m_Player));

    for (int i = 0; i < m_VibrationPlayerCount; i++)
    {
        m_Player[i] = playerArray[i];
    }
}

void NpadFullKeyController::UpdateButtons() NN_NOEXCEPT
{
    nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(m_NpadId);

    if (style.Test<nn::hid::NpadStyleFullKey>())
    {
        m_IsConnected = true;
        nn::hid::NpadFullKeyState state;
        nn::hid::GetNpadState(&state, m_NpadId);
        m_Buttons = state.buttons;
    }
    else
    {
        if (m_IsConnected)
        {
            StopVibration();
            m_Buttons.Reset();
            m_IsConnected = false;
        }
    }
}

void NpadHandheldController::UpdateButtons() NN_NOEXCEPT
{
    nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(m_NpadId);

    if (style.Test<nn::hid::NpadStyleHandheld>())
    {
        m_IsConnected = true;
        nn::hid::NpadHandheldState state;
        nn::hid::GetNpadState(&state, m_NpadId);
        m_Buttons = state.buttons;
    }
    else
    {
        if (m_IsConnected)
        {
            StopVibration();
            m_Buttons.Reset();
            m_IsConnected = false;
        }
    }
}

} // namspace iodevices
} // namespace nns

