﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Fx3.h"

namespace nnt {
namespace usb {
namespace hs {

///////////////////////////////////////////////////////////////////////////////
//  static
///////////////////////////////////////////////////////////////////////////////
static nn::usb::Host                    g_HsClient;
static nn::usb::HostInterface           g_IfSessions[FX3_INTERFACE_MAX];
static nn::usb::HostEndpoint            g_EpSession[FX3_ENDPOINT_MAX];
static nn::os::SystemEventType*         g_StateChangeEvent[FX3_INTERFACE_MAX];
static bool                             g_Attached[FX3_INTERFACE_MAX];
static nn::usb::UsbEndpointDescriptor   g_EndpointDescriptor[FX3_ENDPOINT_MAX];
static NN_ALIGNAS(4096) uint8_t         g_CtrlBuffer[FX3_CTRL_SIZE];
static uint8_t                          g_ConnectSpeed;
static uint8_t                          g_MaxPacketSize;
static uint8_t                          g_InterfaceBitmap;
static uint32_t                         g_MaxTransferSize[FX3_ENDPOINT_MAX];
static bool                             g_IsFx3Initialized = false;

///////////////////////////////////////////////////////////////////////////////
static uint32_t Fx3MakeGaloisPattern(uint8_t* pData, uint32_t size, uint32_t initialCondition)
{
    uint32_t processed;
    uint32_t lfsr = initialCondition;

    for (processed = 0; processed < size; processed += 4)
    {
        uint32_t remaining = size - processed;
        /* taps: 32 31 29 1; characteristic polynomial: x^32 + x^31 + x^29 + x + 1 */
        lfsr = (lfsr >> 1) ^ (uint32_t)((0 - (lfsr & 1u)) & 0xd0000001u);
        /* we cannot always have 32-bit alignment in buffer, so it's best
           to memcpy */
        memcpy(pData + processed, &lfsr, (remaining >= 4) ? 4 : remaining);
    }

    return lfsr;
}


///////////////////////////////////////////////////////////////////////////////
static uint32_t Fx3CheckGaloisPattern(uint8_t* pData, uint32_t size, uint32_t initialCondition)
{
    uint32_t processed;
    uint32_t lfsr = initialCondition;

    for (processed = 0; processed < size; processed += 4)
    {
        uint32_t remaining = size - processed;
        /* taps: 32 31 29 1; characteristic polynomial: x^32 + x^31 + x^29 + x + 1 */
        lfsr = (lfsr >> 1) ^ (uint32_t)((0 - (lfsr & 1u)) & 0xd0000001u);
        /* we cannot always have 32-bit alignment in buffer, so it's best
           to memcmp */
        if (memcmp(pData + processed, &lfsr, (remaining >= 4) ? 4 : remaining) != 0)
        {
            FX3_LOG("FAIL: Data varification error!\n");
            ADD_FAILURE();;
            return 0;
        }
    }

    return lfsr;
}


///////////////////////////////////////////////////////////////////////////////
static int Fx3GetEndpointIndex(uint8_t endpointAddress)
{
    int index = 0;

    switch (endpointAddress)
    {
    case FX3_ENDPOINT_BULK_1_IN :    index = 0;     break;
    case FX3_ENDPOINT_BULK_1_OUT:    index = 1;     break;
    case FX3_ENDPOINT_BULK_2_IN :    index = 2;     break;
    case FX3_ENDPOINT_BULK_2_OUT:    index = 3;     break;
    case FX3_ENDPOINT_INTR_3_IN :    index = 4;     break;
    case FX3_ENDPOINT_INTR_3_OUT:    index = 5;     break;
    case FX3_ENDPOINT_INTR_4_IN :    index = 6;     break;
    case FX3_ENDPOINT_INTR_4_OUT:    index = 7;     break;
    case FX3_ENDPOINT_ISOC_5_IN :    index = 8;     break;
    case FX3_ENDPOINT_ISOC_5_OUT:    index = 9;     break;
    case FX3_ENDPOINT_ISOC_6_IN :    index = 10;    break;
    case FX3_ENDPOINT_ISOC_6_OUT:    index = 11;    break;
    default:                                        break;
    }

    return index;
}


///////////////////////////////////////////////////////////////////////////////
static int pow(int base, int exponent)
{
    int solution = base;

    while (exponent-- > 1)
    {
        solution *= base;
    }

    return solution;
}


///////////////////////////////////////////////////////////////////////////////
static uint64_t Fx3ComputeTimeout(int index, uint32_t bytes)
{
    nn::usb::UsbEndpointDescriptor *pUsbEndpointDescriptor = &g_EndpointDescriptor[index];
    uint32_t packets;
    uint64_t totalIntervals;
    uint64_t timeoutUs;

    packets = bytes / pUsbEndpointDescriptor->wMaxPacketSize;

    if (bytes % pUsbEndpointDescriptor->wMaxPacketSize)
    {
        packets++;
    }

    // HS, SS interval in increments of 1 << (bInterval - 1) * 125us
    if ((g_ConnectSpeed & FX3_DEVICE_HIGH_SPEED) || (g_ConnectSpeed & FX3_DEVICE_SUPER_SPEED))
    {
        if (pUsbEndpointDescriptor->bInterval)
        {
            totalIntervals = packets * (1 << (pUsbEndpointDescriptor->bInterval - 1));
        }
        else
        {
            totalIntervals = packets;
        }

        timeoutUs = totalIntervals * 125;
    }
    else
    {
        if (pUsbEndpointDescriptor->bInterval)
        {
            totalIntervals = packets * pUsbEndpointDescriptor->bInterval;
        }
        else
        {
            totalIntervals = packets;
        }

        timeoutUs = totalIntervals * 1000;
    }

    // Lets add 2000us for API call->HS->(transfer time)->HS->return
    timeoutUs += 2000;

    return timeoutUs;
}


///////////////////////////////////////////////////////////////////////////////
static void Fx3InitializeEndpoint(nn::usb::UsbEndpointDescriptor *pUsbEndpointDescriptor, nn::usb::HostInterface *pIfsession)
{
    int index = Fx3GetEndpointIndex(pUsbEndpointDescriptor->bEndpointAddress);
    uint32_t maxTransferSize = g_MaxTransferSize[index];

    memcpy(&g_EndpointDescriptor[index], pUsbEndpointDescriptor, sizeof(nn::usb::UsbEndpointDescriptor));

    FX3_LOG(
        "Initialize endpoint session for endpoint address %02x packet size %d interval %d max transfer %d\n",
        pUsbEndpointDescriptor->bEndpointAddress,
        pUsbEndpointDescriptor->wMaxPacketSize,
        pUsbEndpointDescriptor->bInterval,
        maxTransferSize
        );

    NNT_EXPECT_RESULT_SUCCESS(g_EpSession[index].Initialize(pIfsession, pUsbEndpointDescriptor, 1, maxTransferSize));
}


///////////////////////////////////////////////////////////////////////////////
static void Fx3FinalizeEndpoint(nn::usb::UsbEndpointDescriptor *pUsbEndpointDescriptor)
{
    int index = Fx3GetEndpointIndex(pUsbEndpointDescriptor->bEndpointAddress);

    FX3_LOG("Finalize endpoint session for %02x\n", pUsbEndpointDescriptor->bEndpointAddress);

    NNT_EXPECT_RESULT_SUCCESS(g_EpSession[index].Finalize());

    memset(&g_EndpointDescriptor[index], 0, sizeof(nn::usb::UsbEndpointDescriptor));
}


///////////////////////////////////////////////////////////////////////////////
static void Fx3InitializeEndpoints(uint8_t interfaceNumber)
{
    nn::usb::InterfaceProfile interfaceProfile;

    NNT_EXPECT_RESULT_SUCCESS(g_IfSessions[interfaceNumber].GetInterface(&interfaceProfile));

    // initialize any endpoints
    for (int i = 0; i < nn::usb::UsbLimitMaxEndpointPairCount; i++)
    {
        nn::usb::UsbEndpointDescriptor *pTemp;

        pTemp = interfaceProfile.epInDesc + i;

        if (pTemp->bEndpointAddress)
        {
            Fx3InitializeEndpoint(pTemp, &g_IfSessions[interfaceNumber]);
        }

        pTemp = interfaceProfile.epOutDesc + i;

        if (pTemp->bEndpointAddress)
        {
            Fx3InitializeEndpoint(pTemp, &g_IfSessions[interfaceNumber]);
        }
    }
}


///////////////////////////////////////////////////////////////////////////////
static void Fx3FinalizeEndpoints(uint8_t interfaceNumber)
{
    nn::usb::InterfaceProfile interfaceProfile;

    NNT_EXPECT_RESULT_SUCCESS(g_IfSessions[interfaceNumber].GetInterface(&interfaceProfile));

    // Finalize any endpoints
    for (int i = 0; i < nn::usb::UsbLimitMaxEndpointPairCount; i++)
    {
        nn::usb::UsbEndpointDescriptor *pTemp;

        pTemp = interfaceProfile.epInDesc + i;

        if (pTemp->bEndpointAddress)
        {
            Fx3FinalizeEndpoint(pTemp);
        }

        pTemp = interfaceProfile.epOutDesc + i;

        if (pTemp->bEndpointAddress)
        {
            Fx3FinalizeEndpoint(pTemp);
        }
    }
}


///////////////////////////////////////////////////////////////////////////////
static void Fx3WaitAllInterfacesDisconnect()
{
    int32_t ifCount = 0;
    nn::usb::DeviceFilter filter;
    nn::usb::InterfaceQueryOutput ifList[nn::usb::HsLimitMaxInterfacesCount];

    memset(ifList, 0, sizeof(ifList));
    memset(&filter, 0, sizeof(filter));

    filter.matchFlags   = nn::usb::DeviceFilterMatchFlags_Vendor | nn::usb::DeviceFilterMatchFlags_Product;
    filter.idVendor     = FX3_VID;
    filter.idProduct    = FX3_PID;

    for (;;)
    {
        NN_LOG("Waiting for all Fx3 interfaces to disconnect..\n");

        NNT_ASSERT_RESULT_SUCCESS(
            g_HsClient.QueryAllInterfaces(&ifCount, ifList, sizeof(ifList), &filter)
            );

        NN_LOG("Found %d Fx3 interfaces\n", ifCount);
        if (ifCount == 0)
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
}


///////////////////////////////////////////////////////////////////////////////
static bool Fx3GetAttach()
{
    int32_t ifCount = 0;
    nn::usb::DeviceFilter filter;
    nn::usb::InterfaceQueryOutput ifList[nn::usb::HsLimitMaxInterfacesCount];
    memset(ifList, 0, sizeof(ifList));

    for (int i = 0; i < FX3_INTERFACE_MAX; i++)
    {
        g_Attached[i] = false;
    }

    memset(&filter, 0, sizeof(filter));

    filter.matchFlags   = nn::usb::DeviceFilterMatchFlags_Vendor | nn::usb::DeviceFilterMatchFlags_Product;
    filter.idVendor     = FX3_VID;
    filter.idProduct    = FX3_PID;

    if(g_HsClient.QueryAvailableInterfaces(&ifCount, ifList, sizeof(ifList), &filter).IsFailure())
    {
        ifCount = 0;
    }

    if (ifCount)
    {
        for (int32_t j = 0; j < ifCount; j++)
        {
            nn::Result result;
            nn::usb::InterfaceQueryOutput *pInterfaceQueryOutput = &ifList[j];
            uint8_t interfaceNumber = pInterfaceQueryOutput->ifProfile.ifDesc.bInterfaceNumber;
            uint32_t handle = pInterfaceQueryOutput->ifProfile.handle;



            NNT_EXPECT_RESULT_SUCCESS(g_IfSessions[interfaceNumber].Initialize(&g_HsClient, handle));
            //FX3_LOG("Initialize interface session for %02x\n", interfaceNumber);
            g_StateChangeEvent[interfaceNumber] = g_IfSessions[interfaceNumber].GetStateChangeEvent();
            g_Attached[interfaceNumber] = true;
        }

        return true;
    }

    return false;
}


///////////////////////////////////////////////////////////////////////////////
static void Fx3HandleDetach()
{
    for (int i = 0; i < FX3_ENDPOINT_MAX; i++)
    {
        if (g_EndpointDescriptor[i].bEndpointAddress)
        {
            Fx3FinalizeEndpoint(&g_EndpointDescriptor[i]);
        }
    }

    for (int i = 0; i < FX3_INTERFACE_MAX; i++)
    {
        if (g_Attached[i])
        {
            g_Attached[i] = false;
            nn::os::WaitSystemEvent(g_StateChangeEvent[i]);
            nn::os::ClearSystemEvent(g_StateChangeEvent[i]);

            //FX3_LOG("Finalize interface session for %02x\n", i);
            NNT_EXPECT_RESULT_SUCCESS(g_IfSessions[i].Finalize());

            //FX3_LOG("Finalized interface %d\n", i);
        }
    }
}

static void Fx3InstallFirmware()
{
    nnt::usb::TestFx3Utility fx3Utility; // Install firmware

    FX3_LOG("Installing Fx3 Firmware...\n");
    Fx3FwErrorCode errorCode = Fx3FwErrorCode_Success;
    NNT_EXPECT_RESULT_SUCCESS(fx3Utility.Initialize(WAIT_SECONDS_FOR_ATTACH));
    errorCode = fx3Utility.UpdateFirmware(UsbTestSuiteFx3Firmware, sizeof(UsbTestSuiteFx3Firmware));
    fx3Utility.Finalize();

    if (errorCode != Fx3FwErrorCode_Success)
    {
        FX3_LOG("Warning: failed to UpdateFirmware to Fx3. Error code: %d\n", errorCode);
    }

}

///////////////////////////////////////////////////////////////////////////////
static nn::Result Fx3Ctrl(size_t* pBytesTransferred, uint8_t* pBuffer, uint8_t bmRequestType, uint8_t bRequest, uint16_t wValue, uint16_t wIndex, uint16_t wLength)
{
    nn::Result result;

    if (bmRequestType & 0x80)   // device to host
    {
        result = g_IfSessions[FX3_INTERFACE_CTRL].ControlRequest(
                                                                 pBytesTransferred,
                                                                 g_CtrlBuffer,
                                                                 bmRequestType,
                                                                 bRequest,
                                                                 wValue,
                                                                 wIndex,
                                                                 wLength
                                                                 );

        if (result.IsSuccess() && (pBuffer != g_CtrlBuffer))
        {
            memcpy(pBuffer, g_CtrlBuffer, *pBytesTransferred);
        }
    }
    else
    {
        if (pBuffer != g_CtrlBuffer)
        {
            memcpy(g_CtrlBuffer, pBuffer, wLength);
        }

        result = g_IfSessions[FX3_INTERFACE_CTRL].ControlRequest(
                                                                 pBytesTransferred,
                                                                 g_CtrlBuffer,
                                                                 bmRequestType,
                                                                 bRequest,
                                                                 wValue,
                                                                 wIndex,
                                                                 wLength
                                                                 );
    }

    return result;
}


///////////////////////////////////////////////////////////////////////////////
//  exports
///////////////////////////////////////////////////////////////////////////////
void Fx3Initialize(int seconds)
{
    size_t bytesTransferred;
    int maxRetries = 60;

    FX3_LOG("%s seconds %d\n", __FUNCTION__, seconds);

    g_IsFx3Initialized = false;

    NNT_EXPECT_RESULT_SUCCESS(g_HsClient.Initialize());

    // - set max transfer size to FX3_MAX_TRANSFER_SIZE default
    // - app can override at runtime to perform TRB ring test
    Fx3SetMaxTransfer(FX3_ENDPOINT_BULK_1_IN,   FX3_MAX_TRANSFER_SIZE);
    Fx3SetMaxTransfer(FX3_ENDPOINT_BULK_1_OUT,  FX3_MAX_TRANSFER_SIZE);
    Fx3SetMaxTransfer(FX3_ENDPOINT_BULK_2_IN,   FX3_MAX_TRANSFER_SIZE);
    Fx3SetMaxTransfer(FX3_ENDPOINT_BULK_2_OUT,  FX3_MAX_TRANSFER_SIZE);
    Fx3SetMaxTransfer(FX3_ENDPOINT_INTR_3_IN,   FX3_MAX_TRANSFER_SIZE);
    Fx3SetMaxTransfer(FX3_ENDPOINT_INTR_3_OUT,  FX3_MAX_TRANSFER_SIZE);
    Fx3SetMaxTransfer(FX3_ENDPOINT_INTR_4_IN,   FX3_MAX_TRANSFER_SIZE);
    Fx3SetMaxTransfer(FX3_ENDPOINT_INTR_4_OUT,  FX3_MAX_TRANSFER_SIZE);
    Fx3SetMaxTransfer(FX3_ENDPOINT_ISOC_5_IN,   FX3_MAX_TRANSFER_SIZE);
    Fx3SetMaxTransfer(FX3_ENDPOINT_ISOC_5_OUT,  FX3_MAX_TRANSFER_SIZE);
    Fx3SetMaxTransfer(FX3_ENDPOINT_ISOC_6_IN,   FX3_MAX_TRANSFER_SIZE);
    Fx3SetMaxTransfer(FX3_ENDPOINT_ISOC_6_OUT,  FX3_MAX_TRANSFER_SIZE);

    FX3_LOG("Looking for Fx3 device..");
    for (int i = 0; i < seconds; i++)
    {
        if (Fx3GetAttach())
        {
            g_InterfaceBitmap = 0;

            // Issue reset
            FX3_LOG("Found Fx3. Requesting Hard Reset Fx3 to Bootloader...\n");
            NNT_EXPECT_RESULT_SUCCESS(Fx3Ctrl(
                                        &bytesTransferred,          // bytes transferred
                                        NULL,                       // buffer
                                        0x40,                       // bmRequestType: host to device | vendor | device
                                        FX3_REQUEST_RESET,          // bRequest
                                        0,                          // wValue
                                        0,                          // wIndex
                                        0                           // wLength
                                        ));

            Fx3WaitAllInterfacesDisconnect();
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    FX3_LOG("Fx3 device not found, Installing firmware..\n");

    for (int retry = 0; retry < maxRetries; retry++)
    {
        Fx3InstallFirmware();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

        for (int i = 0; i < seconds; i++)
        {
            if (Fx3GetAttach())
            {
                g_InterfaceBitmap = 0;
                g_IsFx3Initialized = true;
                return;
            }

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        FX3_LOG("FX3 device not attached after %d seconds. Retrying[%d] installation of firmware..\n",
                seconds, retry);
    }

    NN_ABORT("After max retries (%d) Fx3 firmware failed to start.\n", maxRetries);

}

bool IsFx3Initialized()
{
    return g_IsFx3Initialized;
}

///////////////////////////////////////////////////////////////////////////////
void Fx3Finalize()
{
    size_t bytesTransferred;

    FX3_LOG("%s\n", __FUNCTION__);

    NNT_EXPECT_RESULT_SUCCESS(Fx3Ctrl(
                                        &bytesTransferred,          // bytes transferred
                                        NULL,                       // buffer
                                        0x40,                       // bmRequestType: host to device | vendor | device
                                        FX3_REQUEST_RESET,          // bRequest
                                        0,                          // wValue
                                        0,                          // wIndex
                                        0                           // wLength
                                        ));

    Fx3WaitAllInterfacesDisconnect();

    // Make sure all endpoints finalized
    for(int i =0; i < (sizeof(g_EpSession) / sizeof(g_EpSession[0])); i++)
    {
        if(g_EpSession[i].IsInitialized())
        {
            NNT_EXPECT_RESULT_SUCCESS(g_EpSession[i].Finalize());
        }
    }

    // Make sure all interfaces finalized
    for(int i =0; i < (sizeof(g_IfSessions) / sizeof(g_IfSessions[0])); i++)
    {
        if(g_IfSessions[i].IsInitialized())
        {
            NNT_EXPECT_RESULT_SUCCESS(g_IfSessions[i].Finalize());
        }
    }

    NNT_EXPECT_RESULT_SUCCESS(g_HsClient.Finalize());
    g_IsFx3Initialized = false;
}


///////////////////////////////////////////////////////////////////////////////
void Fx3Reset(int seconds)
{
    size_t bytesTransferred;

    FX3_LOG("%s seconds %d\n", __FUNCTION__, seconds);

    NNT_EXPECT_RESULT_SUCCESS(Fx3Ctrl(
                                        &bytesTransferred,          // bytes transferred
                                        NULL,                       // buffer
                                        0x40,                       // bmRequestType: host to device | vendor | device
                                        FX3_REQUEST_RESET,          // bRequest
                                        0,                          // wValue
                                        0,                          // wIndex
                                        0                           // wLength
                                        ));

    Fx3WaitAllInterfacesDisconnect();
    Fx3HandleDetach();

    Fx3InstallFirmware();
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    for (int i = 0; i < seconds; i++)
    {
        if (Fx3GetAttach())
        {
            return;
        }

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    FX3_LOG("FX3 device not attached after %d seconds\n", seconds);
    FAIL();
}


///////////////////////////////////////////////////////////////////////////////
void Fx3SetDeviceMode(Fx3DeviceMode *pFx3DeviceMode, int seconds)
{
    size_t bytesTransferred;
    bool isCorrectSpeed = false;

    FX3_LOG("%s mode %08x\n", __FUNCTION__, pFx3DeviceMode);

    // print intended device speed and interfaces
    switch (pFx3DeviceMode->connectSpeed)
    {
    case FX3_DEVICE_FULL_SPEED:

        FX3_LOG("full speed %d mps ", pFx3DeviceMode->maxPacketSize);

        break;

    case FX3_DEVICE_HIGH_SPEED:

        FX3_LOG("high speed %d mps ", pFx3DeviceMode->maxPacketSize);

        break;

    case FX3_DEVICE_SUPER_SPEED:

        FX3_LOG("super speed %d mps ", pow(2, pFx3DeviceMode->maxPacketSize));

        break;

    default:

        FX3_LOG("unknown speed ");

        break;
    }

    // ctrl interface always present
    FX3_LOG("interface 0 ");

    if (pFx3DeviceMode->interfaceBitmap & FX3_INTERFACE_BULK_1)
    {
        FX3_LOG("1 ");
    }

    if (pFx3DeviceMode->interfaceBitmap & FX3_INTERFACE_BULK_2)
    {
        FX3_LOG("2 ");
    }

    if (pFx3DeviceMode->interfaceBitmap & FX3_INTERFACE_INTR_3)
    {
        FX3_LOG("3 ");
    }

    if (pFx3DeviceMode->interfaceBitmap & FX3_INTERFACE_INTR_4)
    {
        FX3_LOG("4 ");
    }

    FX3_LOG("\n");

    while (!isCorrectSpeed)
    {
        isCorrectSpeed = true;
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        NNT_EXPECT_RESULT_SUCCESS(Fx3Ctrl(
                                &bytesTransferred,          // bytes transferred
                                (uint8_t*)pFx3DeviceMode,   // buffer
                                0x40,                       // host to device | vendor | device
                                FX3_REQUEST_DEVICE_MODE,    // bRequest
                                0,                          // wValue
                                0,                          // wIndex
                                sizeof(Fx3DeviceMode)       // wLength
                                ));
        Fx3HandleDetach();

        for (int i = 0; i < seconds; i++)
        {
            if (Fx3GetAttach())
            {
                uint8_t speed;

                // confirm actual connection speed with FX3
                NNT_EXPECT_RESULT_SUCCESS(Fx3Ctrl(
                    &bytesTransferred,              // bytes transferred
                    &speed,                         // buffer
                    0x80 | 0x40,                    // device to host | vendor | device
                    FX3_REQUEST_DEVICE_SPEED,       // bRequest
                    0,                              // wValue
                    0,                              // wIndex
                    sizeof(uint8_t)                 // wLength
                    ));

                switch (pFx3DeviceMode->connectSpeed)
                {
                case FX3_DEVICE_FULL_SPEED:

                    if (speed != 0)
                    {
                        FX3_LOG("Device is not full speed!\n");
                        isCorrectSpeed = false;
                    }

                    break;

                case FX3_DEVICE_HIGH_SPEED:

                    if (speed != 1)
                    {
                        FX3_LOG("Device is not high speed!\n");
                        isCorrectSpeed = false;
                    }

                    break;

                case FX3_DEVICE_SUPER_SPEED:

                    if (speed != 2)
                    {
                        FX3_LOG("Device is not super speed!\n");
                        isCorrectSpeed = false;
                    }

                    break;

                default:

                    FX3_LOG("Undefined speed!\n");
                    isCorrectSpeed = false;

                    break;
                }

                FX3_LOG("============= %d =? %d\n", pFx3DeviceMode->connectSpeed, speed);

                if (!isCorrectSpeed)
                {
                    NN_LOG("Enumerated as wrong speed! Trying again..\n");
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                    break;
                }

                // confirm expected interfaces are attached
                for (int interfaceNumber = 0; interfaceNumber < FX3_INTERFACE_MAX; interfaceNumber++)
                {
                    if (interfaceNumber && (pFx3DeviceMode->interfaceBitmap & (1 << (interfaceNumber - 1))))
                    {
                        if (g_Attached[interfaceNumber] == false)
                        {
                            FX3_LOG("Interface number %d not attached\n", interfaceNumber);
                            ADD_FAILURE();
                        }
                    }
                }

                g_ConnectSpeed      = pFx3DeviceMode->connectSpeed;
                g_MaxPacketSize     = pFx3DeviceMode->maxPacketSize;
                g_InterfaceBitmap   = pFx3DeviceMode->interfaceBitmap;

                return;
            }

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
    }

    ADD_FAILURE();
    FX3_LOG("FX3 device not attached after %d seconds\n", seconds);
} // NOLINT(impl/function_size)


///////////////////////////////////////////////////////////////////////////////
void Fx3SetMaxTransfer(uint8_t endpointAddress, uint32_t maxTransferSize)
{
    FX3_LOG("%s %02x %d\n", __FUNCTION__, endpointAddress, maxTransferSize);
    g_MaxTransferSize[Fx3GetEndpointIndex(endpointAddress)] = maxTransferSize;
}


///////////////////////////////////////////////////////////////////////////////
void Fx3CheckFirmwareVersion()
{
    size_t bytesTransferred;
    uint32_t version = 0;

    FX3_LOG("%s\n", __FUNCTION__);

    NNT_EXPECT_RESULT_SUCCESS(Fx3Ctrl(
                                        &bytesTransferred,           // bytes transferred
                                        (uint8_t*)&version,          // buffer
                                        0x80 | 0x40,                 // device to host | vendor | device
                                        FX3_REQUEST_FIRMWARE_VERSION,// bRequest
                                        0,                           // wValue
                                        0,                           // wIndex
                                        4                            // wLength
                                        ));

    if (version != FX3_EXPECT_FIRMWARE_VERSION)
    {
        FX3_LOG("FX3 firmware version is %d,  expect %d\n", version, FX3_EXPECT_FIRMWARE_VERSION);
        g_IsFx3Initialized = false;
        FAIL();
    }
}


///////////////////////////////////////////////////////////////////////////////
void Fx3PrintString(uint8_t *pStringData, uint32_t size)
{
    size_t bytesTransferred;

    if (size <= FX3_CTRL_SIZE)
    {
        NNT_EXPECT_RESULT_SUCCESS(Fx3Ctrl(
                                            &bytesTransferred,              // bytes transferred
                                            pStringData,                    // buffer
                                            0x40,                           // bmRequestType: device to host | vendor | device
                                            FX3_REQUEST_PRINT_STRING,       // bRequest
                                            0,                              // wValue
                                            0,                              // wIndex
                                            size                            // wLength
                                            ));
    }
    else
    {
        FX3_LOG("ERROR: String is greater than FX3_CTRL_SIZE buffer (%d).\n", FX3_CTRL_SIZE);
        FAIL();
    }

}


///////////////////////////////////////////////////////////////////////////////
void Fx3SetAltSetting(uint8_t interfaceNumber, uint8_t altSetting)
{
    FX3_LOG("%s interfaceNumber %d altSetting %d\n", __FUNCTION__, interfaceNumber, altSetting);

    Fx3FinalizeEndpoints(interfaceNumber);
    NNT_EXPECT_RESULT_SUCCESS(g_IfSessions[interfaceNumber].SetInterface(altSetting));
    Fx3InitializeEndpoints(interfaceNumber);
}


///////////////////////////////////////////////////////////////////////////////
void Fx3TransferData(Fx3TestDataRarameters *pFx3TestDataParams, uint8_t *pData)
{
    nn::Result result;
    uint8_t     endpointAddress = pFx3TestDataParams->endpointAddress;
    uint8_t     entries         = pFx3TestDataParams->entries;
    uint32_t    seed            = pFx3TestDataParams->dataSeed;

    int index = Fx3GetEndpointIndex(endpointAddress);
    size_t bytesTransferred;

    FX3_LOG(
            "%s 0x%02x 0x%02x 0x%08x 0x%08x\n",
            __FUNCTION__,
            pFx3TestDataParams->endpointAddress,
            pFx3TestDataParams->entries,
            pFx3TestDataParams->dataSeed,
            pData
            );

    // terminate entries where request is greater than max transfer size
    for (int i = 0; i < entries; i++)
    {
        uint32_t requestBytes   = pFx3TestDataParams->params[i].requestBytes;

        if (requestBytes > g_MaxTransferSize[index])
        {
            pFx3TestDataParams->entries = i;

            if (i <= 0)
            {
                FX3_LOG("No valid etries to tranfer, returning to caller\n");
                return;
            }

            break;
        }
    }

    // send params to FX3
    NNT_EXPECT_RESULT_SUCCESS(Fx3Ctrl(
                                        &bytesTransferred,              // bytes transferred
                                        (uint8_t*)pFx3TestDataParams,   // buffer
                                        0x40,                           // host to device | vendor | device
                                        FX3_REQUEST_TRANSFER_DATA,      // bRequest
                                        0,                              // wValue
                                        0,                              // wIndex
                                        sizeof(Fx3TestDataRarameters)   // wLength
                                        ));

    // execute transfers
    for (int i = 0; i < entries; i++)
    {
        uint32_t requestBytes   = pFx3TestDataParams->params[i].requestBytes;
        uint32_t transferBytes  = pFx3TestDataParams->params[i].transferBytes;
        nn::os::Tick tick0;
        nn::os::Tick tick1;

        FX3_LOG("    %02x requestBytes %d transferBytes %d\n", endpointAddress, requestBytes, transferBytes);

        if (((endpointAddress & 0xf) == 0) && (requestBytes > 0xffff))
        {
            FX3_LOG("    Cannot make ctrl request for more than 0xffff!\n");
            return;
        }
        else
        {
            if (requestBytes > g_MaxTransferSize[index])
            {
                FX3_LOG("    requestBytes %d > max transfer size %d skipping transactions\n", requestBytes, g_MaxTransferSize[index]);
                return;
            }
        }

        if (endpointAddress & 0x80)
        {
            bytesTransferred = 0;

            if ((endpointAddress & 0xf) == 0)   // ctrl
            {
                result = Fx3Ctrl(
                                                    &bytesTransferred,              // bytes transferred
                                                    pData,                          // buffer
                                                    0xd0,                           // device to host | vendor | device
                                                    FX3_REQUEST_DATA,               // bRequest
                                                    0,                              // wValue
                                                    0,                              // wIndex
                                                    requestBytes                    // wLength
                                                    );
            }
            else
            {
                tick0 = nn::os::GetSystemTick();
                result = g_EpSession[index].PostBuffer(&bytesTransferred, pData, requestBytes);
                tick1 = nn::os::GetSystemTick();
            }

            if (bytesTransferred == transferBytes)
            {
                seed = Fx3CheckGaloisPattern(pData, transferBytes, seed);
            }
        }
        else
        {
            bytesTransferred = 0;
            seed = Fx3MakeGaloisPattern(pData, transferBytes, seed);

            if ((endpointAddress & 0xf) == 0)   // ctrl
            {
                result = Fx3Ctrl(
                                                    &bytesTransferred,              // bytes transferred
                                                    pData,                          // buffer
                                                    0x40,                           // host to device | vendor | device
                                                    FX3_REQUEST_DATA,               // bRequest
                                                    0,                              // wValue
                                                    0,                              // wIndex
                                                    requestBytes                    // wLength
                                                    );
            }
            else
            {
                tick0 = nn::os::GetSystemTick();
                result = g_EpSession[index].PostBuffer(&bytesTransferred, pData, requestBytes);
                tick1 = nn::os::GetSystemTick();
            }
        }

        // check periodic timing
        if (
            (endpointAddress == FX3_ENDPOINT_INTR_3_IN) || (endpointAddress == FX3_ENDPOINT_INTR_3_OUT) ||
            (endpointAddress == FX3_ENDPOINT_INTR_4_IN) || (endpointAddress == FX3_ENDPOINT_INTR_4_OUT)
            )
        {
            if (bytesTransferred == transferBytes)
            {
                uint64_t elapsedTimeUs      = (tick1 - tick0).ToTimeSpan().GetMicroSeconds();
                uint64_t referenceTimeUs    = Fx3ComputeTimeout(index, bytesTransferred);

                if (elapsedTimeUs > referenceTimeUs)
                {
                    FX3_LOG("WARNING: Transaction took %lld us, should be no more than %lldus\n", elapsedTimeUs, referenceTimeUs);
                    //ADD_FAILURE();
                }
            }
        }

        // Print unsuccessful result here, using NNT_EXPECT_RESULT_SUCCESS will cause stall tests
        // to be failed :0
        if (!result.IsSuccess())
        {
            FX3_LOG("result module %d description %d\n", result.GetModule(), result.GetDescription());
        }

        if (bytesTransferred != transferBytes)
        {
            FX3_LOG("bytesTransferred is %d, should be %d\n", bytesTransferred, transferBytes);
            ADD_FAILURE();
        }
    }

    // For host -> device transfers, check errors on device
    if ((endpointAddress & 0x80) == 0)   // host -> device
    {
        uint32_t errors;

        result = Fx3Ctrl(
                            &bytesTransferred,              // bytes transferred
                            (uint8_t*)&errors,              // buffer
                            0xd0,                           // device to host | vendor | device
                            FX3_REQUEST_ERROR,              // bRequest
                            0,                              // wValue
                            endpointAddress,                // wIndex
                            sizeof(errors)                  // wLength
                            );

        if (result.IsSuccess() && (bytesTransferred == sizeof(errors)))
        {
            if (errors > 0)
            {
                FX3_LOG("Device reports %d errors\n", errors);
                ADD_FAILURE();
            }
        }
        else
        {
            FX3_LOG("Failed to get error status from device\n");
            ADD_FAILURE();
        }
    }

} // NOLINT(impl/function_size)


///////////////////////////////////////////////////////////////////////////////
void Fx3LoopBackTx(void *pContext)
{
    Fx3LoopbackParameters *p = reinterpret_cast<Fx3LoopbackParameters*>(pContext);

    uint8_t     endpointAddress = p->endpointAddressTx;
    uint8_t    *pData           = p->pTxData;
    uint8_t     entries         = p->fx3TestDataParams.entries;

    int index = Fx3GetEndpointIndex(endpointAddress);

    p->txLargestBuffer  = 0;

    for (int i = 0; i < entries; i++)
    {
        size_t bytesTransferred = 0;

        // for loopback requestsBytes == ransferBytes
        uint32_t transferBytes = p->fx3TestDataParams.params[i].transferBytes;

        // store largest buffer for data compare
        if (transferBytes > p->txLargestBuffer)
        {
            p->txLargestBuffer = transferBytes;
        }

        FX3_LOG("    %02x transferBytes %d\n", endpointAddress, transferBytes);

        NNT_EXPECT_RESULT_SUCCESS(g_EpSession[index].PostBuffer(&bytesTransferred, pData, transferBytes));
        p->txBytesTotal += bytesTransferred;
    }
}


///////////////////////////////////////////////////////////////////////////////
void Fx3LoopBackRx(void *pContext)
{
    Fx3LoopbackParameters *p = reinterpret_cast<Fx3LoopbackParameters*>(pContext);

    uint8_t     endpointAddress = p->endpointAddressRx;
    uint8_t    *pData           = p->pRxData;
    uint8_t     entries         = p->fx3TestDataParams.entries;

    int index = Fx3GetEndpointIndex(endpointAddress);

    for (int i = 0; i < entries; i++)
    {
        size_t bytesTransferred = 0;

        // for loopback requestsBytes == ransferBytes
        uint32_t transferBytes = p->fx3TestDataParams.params[i].transferBytes;
        FX3_LOG("    %02x transferBytes %d\n", endpointAddress, transferBytes);
        NNT_EXPECT_RESULT_SUCCESS(g_EpSession[index].PostBuffer(&bytesTransferred, pData, transferBytes));
        p->rxBytesTotal += bytesTransferred;
    }
}


///////////////////////////////////////////////////////////////////////////////
void Fx3LoopbackData(Fx3LoopbackParameters *p)
{
    size_t bytesTransferred;
    nn::os::Tick tick0;
    nn::os::Tick tick1;

    Fx3MakeGaloisPattern(p->pTxData, FX3_MAX_TRANSFER_SIZE, p->fx3TestDataParams.dataSeed);

    NNT_EXPECT_RESULT_SUCCESS(Fx3Ctrl(
                                        &bytesTransferred,              // bytes transferred
                                        (uint8_t*)&p->fx3TestDataParams,// buffer
                                        0x40,                           // device to host | vendor | device
                                        FX3_REQUEST_TRANSFER_DATA,      // bRequest
                                        0,                              // wValue
                                        0,                              // wIndex
                                        sizeof(Fx3TestDataRarameters)   // wLength
                                        ));

    nn::os::ThreadType txThread;
    nn::os::ThreadType rxThread;

    nn::os::CreateThread(
                            &txThread,
                            Fx3LoopBackTx,
                            p,
                            p->pTxThreadStack,
                            p->txThreadStackSize,
                            24
                            );

    nn::os::CreateThread(
                            &rxThread,
                            Fx3LoopBackRx,
                            p,
                            p->pRxThreadStack,
                            p->rxThreadStackSize,
                            24
                            );

    tick0 = nn::os::GetSystemTick();

    nn::os::StartThread(&txThread);
    nn::os::StartThread(&rxThread);

    nn::os::WaitThread(&txThread);
    nn::os::WaitThread(&rxThread);

    tick1 = nn::os::GetSystemTick();

    nn::os::DestroyThread(&txThread);
    nn::os::DestroyThread(&rxThread);

    nn::os::Tick ticks = tick1 - tick0;
    int64_t microseconds = ticks.ToTimeSpan().GetMicroSeconds();
    uint32_t totalBytes = p->txBytesTotal + p->rxBytesTotal;

    FX3_LOG(
            "Elapsed time: %dus %f MB/s for %d bytes loop back.\n",
            microseconds,
            (float)(totalBytes) / ((float)microseconds / (float)1000000) / (1024 * 1024),
            totalBytes
            );

    // Graph data for TeamCity
    ::std::string speedString = "";
    switch (p->speed)
    {
        case FX3_DEVICE_FULL_SPEED:
            speedString = "FS";
            break;
        case FX3_DEVICE_HIGH_SPEED:
            speedString = "HS";
            break;
        case FX3_DEVICE_SUPER_SPEED:
            speedString = "SS";
            break;
        default:
            break;
    }

    const char *cSpeedString = speedString.c_str();

    NN_LOG(
            "##teamcity[buildStatisticValue key='%s_Bandwidth(MB/s)_Loopback_AltSet%d_%dB' value='%f']\n",
            cSpeedString,
            p->altSetting,
            p->txBytesTotal,
            (float)totalBytes / ((float)microseconds / (float)1000000) / (1024 * 1024)
            );

    // compare tx and rx data
    if (memcmp(p->pTxData, p->pRxData, p->txLargestBuffer))
    {
        FX3_LOG("TX and TX data not the same!\n");
        ADD_FAILURE();
    }
}


///////////////////////////////////////////////////////////////////////////////
// Clr stall
void Fx3ClrStall(uint8_t endpointAddress)
{
    FX3_LOG(
            "%s 0x%02x\n",
            __FUNCTION__,
            endpointAddress
            );

    NNT_EXPECT_RESULT_SUCCESS(g_EpSession[Fx3GetEndpointIndex(endpointAddress)].ClearEndpointHalt());
}


} // hs
} // usb
} // nnt

