﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_UserExceptionHandler.h>

#include <nnt.h>

#include <nn/usb/usb_Result.h>
#include <nn/usb/usb_Device.h>

using namespace nn::usb;

namespace nnt {
namespace usb {
namespace ds {

//--------------------------------------------------------------------------

DsClient                  client;
DsInterface               interface;
DsEndpoint                endpoint[2];
UsbState                  state;

UsbStringDescriptor languageStringDescriptor            = {4,   UsbDescriptorType_String, {0x0409}};
UsbStringDescriptor manufacturerStringDescriptor        = {18,  UsbDescriptorType_String, {'N', 'i', 'n', 't', 'e', 'n', 'd', 'o'}};
UsbStringDescriptor productStringDescriptor             = {32,  UsbDescriptorType_String, {'N', 'i', 'n', 't', 'e', 'n', 'd', 'o', 'S', 'd', 'k', 'T', 'e', 's', 't'}};
UsbStringDescriptor serialNumberStringDescriptor        = {26,  UsbDescriptorType_String, {'S', 'e', 'r', 'i', 'a', 'l', 'N', 'u', 'm', 'b', 'e', 'r'}};
UsbStringDescriptor interfaceStringDescriptor           = {16,  UsbDescriptorType_String, {'U', 's', 'b', 'D', 'i', 'a', 'g'}};

UsbDeviceDescriptor fullSpeedDeviceDescriptor =
{
    ::nn::usb::UsbDescriptorSize_Device,                     // bLength
    ::nn::usb::UsbDescriptorType_Device,                     // bDescriptorType
    0x0110,                                                  // bcdUSB 1.1
    0x00,                                                    // bDeviceClass
    0x00,                                                    // bDeviceSubClass
    0x00,                                                    // bDeviceProtocol
    0x40,                                                    // bMaxPacketSize0
    0x057e,                                                  // idVendor
    0x3001,                                                  // idProduct
    0x0100,                                                  // bcdDevice
    0x01,                                                    // iManufacturer
    0x02,                                                    // iProduct
    0x03,                                                    // iSerialNumber
    0x01                                                     // bNumConfigurations
};


UsbDeviceDescriptor highSpeedDeviceDescriptor =
{
    ::nn::usb::UsbDescriptorSize_Device,                     // bLength
    ::nn::usb::UsbDescriptorType_Device,                     // bDescriptorType
    0x0200,                                                  // bcdUSB 2.0
    0x00,                                                    // bDeviceClass
    0x00,                                                    // bDeviceSubClass
    0x00,                                                    // bDeviceProtocol
    0x40,                                                    // bMaxPacketSize0
    0x057e,                                                  // idVendor
    0x3001,                                                  // idProduct
    0x0100,                                                  // bcdDevice
    0x01,                                                    // iManufacturer
    0x02,                                                    // iProduct
    0x03,                                                    // iSerialNumber
    0x01                                                     // bNumConfigurations
};


UsbDeviceDescriptor superSpeedDeviceDescriptor =
{
    ::nn::usb::UsbDescriptorSize_Device,                     // bLength
    ::nn::usb::UsbDescriptorType_Device,                     // bDescriptorType
    0x0300,                                                  // bcdUSB 3.0
    0x00,                                                    // bDeviceClass
    0x00,                                                    // bDeviceSubClass
    0x00,                                                    // bDeviceProtocol
    0x09,                                                    // bMaxPacketSize0, SS 512 or 2^9
    0x057e,                                                  // idVendor
    0x3001,                                                  // idProduct
    0x0100,                                                  // bcdDevice
    0x01,                                                    // iManufacturer
    0x02,                                                    // iProduct
    0x03,                                                    // iSerialNumber
    0x01                                                     // bNumConfigurations
};


uint8_t binaryObjectStore[] =
{
    0x05,                                       // bLength
    UsbDescriptorType_Bos,                      // bDescriptorType
    0x16,0x00,                                  // Length of this descriptor and all sub descriptors
    0x02,                                       // Number of device capability descriptors

    // USB 2.0 extension
    0x07,                                       // bLength
    UsbDescriptorType_DeviceCapability,         // bDescriptorType
    0x02,                                       // USB 2.0 extension capability type
    0x02,0x00,0x00,0x00,                        // Supported device level features: LPM support

    // SuperSpeed device capability
    0x0A,                                       // bLength
    UsbDescriptorType_DeviceCapability,         // bDescriptorType
    0x03,                                       // SuperSpeed device capability type
    0x00,                                       // Supported device level features
    0x0e,0x00,                                  // Speeds supported by the device : SS, HS
    0x03,                                       // Functionality support
    0x00,                                       // U1 Device Exit latency
    0x00,0x00                                   // U2 Device Exit latency
};


UsbInterfaceDescriptor    usbInterfaceDescriptor = {
    UsbDescriptorSize_Interface,                // bLength
    UsbDescriptorType_Interface,                // bDescriptorType
    0,                                          // bInterfaceNumber
    0,                                          // bAlternateSetting
    2,                                          // bNumEndpoints
    0xff,                                       // bInterfaceClass
    0xff,                                       // bInterfaceSubClass
    0xff,                                       // bInterfaceProtocol
    4,                                          // iInterface
};

UsbEndpointDescriptor     fullSpeedUsbEndpointDescriptor[2] = {
    {
        UsbDescriptorSize_Endpoint,             // bLength
        UsbDescriptorType_Endpoint,             // bDescriptorType
        0x81,                                   // bEndpointAddress
        UsbEndpointAttributeMask_XferTypeBulk,  // bmAttributes
        1024,                                   // wMaxPacketSize
        0                                       // bInterval
    },
    {
        UsbDescriptorSize_Endpoint,             // bLength
        UsbDescriptorType_Endpoint,             // bDescriptorType
        0x01,                                   // bEndpointAddress
        UsbEndpointAttributeMask_XferTypeBulk,  // bmAttributes
        1024,                                   // wMaxPacketSize
        0                                       // bInterval
    },
};


UsbEndpointDescriptor     highSpeedUsbEndpointDescriptor[2] = {
    {
        UsbDescriptorSize_Endpoint,             // bLength
        UsbDescriptorType_Endpoint,             // bDescriptorType
        0x81,                                   // bEndpointAddress
        UsbEndpointAttributeMask_XferTypeBulk,  // bmAttributes
        512,                                    // wMaxPacketSize
        0                                       // bInterval
    },
    {
        UsbDescriptorSize_Endpoint,             // bLength
        UsbDescriptorType_Endpoint,             // bDescriptorType
        0x01,                                   // bEndpointAddress
        UsbEndpointAttributeMask_XferTypeBulk,  // bmAttributes
        512,                                    // wMaxPacketSize
        0                                       // bInterval
    },
};


UsbEndpointDescriptor     superSpeedUsbEndpointDescriptor[2] = {
    {
        UsbDescriptorSize_Endpoint,             // bLength
        UsbDescriptorType_Endpoint,             // bDescriptorType
        0x81,                                   // bEndpointAddress
        UsbEndpointAttributeMask_XferTypeBulk,  // bmAttributes
        1024,                                   // wMaxPacketSize
        0                                       // bInterval
    },
    {
        UsbDescriptorSize_Endpoint,             // bLength
        UsbDescriptorType_Endpoint,             // bDescriptorType
        0x01,                                   // bEndpointAddress
        UsbEndpointAttributeMask_XferTypeBulk,  // bmAttributes
        1024,                                   // wMaxPacketSize
        0                                       // bInterval
    },
};


// Use this endpoint companion descriptor for both endpoints :)
UsbEndpointCompanionDescriptor superSpeedUsbEndpointCompanionDescriptor =
{
    0x06,                                       // bLength
    UsbDescriptorType_EndpointCompanion,        // bDescriptorType
    15,                                         // bMaxBurst
    0,                                          // bmAttributes
    0                                           // wBytesPerInterval
};

//--------------------------------------------------------------------------

nn::Result SetStringDescriptors(DsClient *pDsClient)
{
    uint8_t stringIndex;
    NNT_EXPECT_RESULT_SUCCESS(pDsClient->AddUsbStringDescriptor(&stringIndex, &languageStringDescriptor));
    NNT_EXPECT_RESULT_SUCCESS(pDsClient->AddUsbStringDescriptor(&stringIndex, &manufacturerStringDescriptor));
    NNT_EXPECT_RESULT_SUCCESS(pDsClient->AddUsbStringDescriptor(&stringIndex, &productStringDescriptor));
    NNT_EXPECT_RESULT_SUCCESS(pDsClient->AddUsbStringDescriptor(&stringIndex, &serialNumberStringDescriptor));
    NNT_EXPECT_RESULT_SUCCESS(pDsClient->AddUsbStringDescriptor(&stringIndex, &interfaceStringDescriptor));

    return nn::ResultSuccess();
}


nn::Result SetDeviceDescriptors(DsClient *pDsClient)
{
    NNT_EXPECT_RESULT_SUCCESS(pDsClient->SetUsbDeviceDescriptor(&fullSpeedDeviceDescriptor,   UsbDeviceSpeed_Full));
    NNT_EXPECT_RESULT_SUCCESS(pDsClient->SetUsbDeviceDescriptor(&highSpeedDeviceDescriptor,   UsbDeviceSpeed_High));
    NNT_EXPECT_RESULT_SUCCESS(pDsClient->SetUsbDeviceDescriptor(&superSpeedDeviceDescriptor,  UsbDeviceSpeed_Super));
    NNT_EXPECT_RESULT_SUCCESS(pDsClient->SetBinaryObjectStore(binaryObjectStore, sizeof(binaryObjectStore)));

    return nn::ResultSuccess();
}


nn::Result SetConfigurationDescriptors(DsInterface *pDsInterface)
{
    // For each speed, append descriptor in the order to appear in the
    // configuration descriptor stream, this allows any descriptors to appear
    // in the stream as needed :)
    NNT_EXPECT_RESULT_SUCCESS(pDsInterface->AppendConfigurationData(UsbDeviceSpeed_Full, &usbInterfaceDescriptor, sizeof(UsbInterfaceDescriptor)));
    NNT_EXPECT_RESULT_SUCCESS(pDsInterface->AppendConfigurationData(UsbDeviceSpeed_Full, &fullSpeedUsbEndpointDescriptor[0], sizeof(UsbEndpointDescriptor)));
    NNT_EXPECT_RESULT_SUCCESS(pDsInterface->AppendConfigurationData(UsbDeviceSpeed_Full, &fullSpeedUsbEndpointDescriptor[1], sizeof(UsbEndpointDescriptor)));

    NNT_EXPECT_RESULT_SUCCESS(pDsInterface->AppendConfigurationData(UsbDeviceSpeed_High, &usbInterfaceDescriptor, sizeof(UsbInterfaceDescriptor)));
    NNT_EXPECT_RESULT_SUCCESS(pDsInterface->AppendConfigurationData(UsbDeviceSpeed_High, &highSpeedUsbEndpointDescriptor[0], sizeof(UsbEndpointDescriptor)));
    NNT_EXPECT_RESULT_SUCCESS(pDsInterface->AppendConfigurationData(UsbDeviceSpeed_High, &highSpeedUsbEndpointDescriptor[1], sizeof(UsbEndpointDescriptor)));

    NNT_EXPECT_RESULT_SUCCESS(pDsInterface->AppendConfigurationData(UsbDeviceSpeed_Super, &usbInterfaceDescriptor, sizeof(UsbInterfaceDescriptor)));
    NNT_EXPECT_RESULT_SUCCESS(pDsInterface->AppendConfigurationData(UsbDeviceSpeed_Super, &superSpeedUsbEndpointDescriptor[0], sizeof(UsbEndpointDescriptor)));
    NNT_EXPECT_RESULT_SUCCESS(pDsInterface->AppendConfigurationData(UsbDeviceSpeed_Super, &superSpeedUsbEndpointCompanionDescriptor, sizeof(UsbEndpointCompanionDescriptor)));
    NNT_EXPECT_RESULT_SUCCESS(pDsInterface->AppendConfigurationData(UsbDeviceSpeed_Super, &superSpeedUsbEndpointDescriptor[1], sizeof(UsbEndpointDescriptor)));
    NNT_EXPECT_RESULT_SUCCESS(pDsInterface->AppendConfigurationData(UsbDeviceSpeed_Super, &superSpeedUsbEndpointCompanionDescriptor, sizeof(UsbEndpointCompanionDescriptor)));

    return nn::ResultSuccess();
}

enum
{
    ITERATIONS_COUNT    = 100,
    BUFFERS_COUNT       = 5,
    BUFFER_SIZE         = 4096,
};

TEST(DsBasic, CancelTransaction)
{
    NNT_EXPECT_RESULT_SUCCESS(client.Initialize(ComplexId_Tegra21x));
    NNT_EXPECT_RESULT_SUCCESS(client.ClearDeviceData());
    NNT_EXPECT_RESULT_SUCCESS(SetStringDescriptors(&client));
    NNT_EXPECT_RESULT_SUCCESS(SetDeviceDescriptors(&client));
    NNT_EXPECT_RESULT_SUCCESS(interface.Initialize(&client, usbInterfaceDescriptor.bInterfaceNumber));
    NNT_EXPECT_RESULT_SUCCESS(SetConfigurationDescriptors(&interface));
    NNT_EXPECT_RESULT_SUCCESS(endpoint[0].Initialize(&interface, 0x81));
    NNT_EXPECT_RESULT_SUCCESS(endpoint[1].Initialize(&interface, 0x01));

    nn::os::SystemEventType *pStateChangeEvent = client.GetStateChangeEvent();

    NNT_EXPECT_RESULT_SUCCESS(interface.Enable());
    NNT_EXPECT_RESULT_SUCCESS(client.EnableDevice());

    while (true)
    {
        nn::os::WaitSystemEvent(pStateChangeEvent);
        nn::os::ClearSystemEvent(pStateChangeEvent);
        NNT_EXPECT_RESULT_SUCCESS(client.GetState(&state));

        if (state == UsbState_Configured)
        {
            break;
        }
    }

    for (int iteration = 0; iteration < ITERATIONS_COUNT; iteration++)
    {
        NN_LOG("iteration %d / %d\n", iteration + 1, ITERATIONS_COUNT);
        nn::os::SystemEventType* pBufferCompleteEvent = endpoint[0].GetCompletionEvent();

        NN_USB_DMA_ALIGN char buffer[BUFFERS_COUNT][BUFFER_SIZE];
        uint32_t urbId[BUFFERS_COUNT];

        for (int i = 0; i < BUFFERS_COUNT; i++)
        {
            NNT_EXPECT_RESULT_SUCCESS(endpoint[0].PostBufferAsync(&urbId[i], &buffer[i], BUFFER_SIZE));
        }

        if (iteration & 1)
        {
            NNT_EXPECT_RESULT_SUCCESS(endpoint[0].Cancel());
            NN_LOG("Transactions cancelled\n");
        }

        int buffersComplete = 0;

        while (buffersComplete < BUFFERS_COUNT)
        {
            nn::os::WaitSystemEvent(pBufferCompleteEvent);
            nn::os::ClearSystemEvent(pBufferCompleteEvent);
            NN_LOG("complete event\n");
            UrbReport urbReport;
            NNT_EXPECT_RESULT_SUCCESS(endpoint[0].GetUrbReport(&urbReport));

            for (int i = 0; i < urbReport.count; i++)
            {
                int status = urbReport.report[i].status;

                if ((status == UrbStatus::UrbStatus_Finished) || (status == UrbStatus::UrbStatus_Cancelled) || (status == UrbStatus::UrbStatus_Failed))
                {
                    NN_LOG("requestedSize %d transferredSize %d status %08x\n", urbReport.report[i].requestedSize, urbReport.report[i].transferredSize, urbReport.report[i].status);
                    buffersComplete++;
                }
            }
        }
    }

    NNT_EXPECT_RESULT_SUCCESS(client.DisableDevice());
    NNT_EXPECT_RESULT_SUCCESS(interface.Disable());
    NNT_EXPECT_RESULT_SUCCESS(endpoint[1].Finalize());
    NNT_EXPECT_RESULT_SUCCESS(endpoint[0].Finalize());
    NNT_EXPECT_RESULT_SUCCESS(interface.Finalize());
    NNT_EXPECT_RESULT_SUCCESS(client.Finalize());
}

} // namespace ds
} // namespace usb
} // namespace nnt

