﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nnt.h>
#include <nn/nn_Log.h>
#include <nn/nn_Windows.h>
#include <winusb.h>
#include <memory>
#pragma warning(disable: 4668)
#include <setupapi.h>

namespace nnt {
namespace usb {

class StopWatch
{
public:
    void Start()
    {
        QueryPerformanceCounter(&m_Start);
    }

    void Stop()
    {
        QueryPerformanceCounter(&m_End);
    }

    int GetElapsedMicroseconds() const
    {
        LARGE_INTEGER freq;

        QueryPerformanceFrequency(&freq);

        return static_cast<int>(
            (m_End.QuadPart - m_Start.QuadPart) * 1000 * 1000 / freq.QuadPart
        );
    }

private:
    LARGE_INTEGER m_Start;
    LARGE_INTEGER m_End;
};

template <uint32_t N>
class Ring {
public:
    Ring() : m_Head(0), m_Tail(0)
    {
        for (int i = 0; i < RingSize; i++)
        {
            m_Overlap[i].hEvent = CreateEvent(
                NULL,    // default security attribute
                TRUE,    // manual-reset event
                TRUE,    // initial state = signaled
                NULL     // unnamed event object
            );
        }

        for (int i = 0; i < RingSize; i++)
        {
            m_Buffer[i] = new uint8_t[MaxBytesPerTransaction];
        }
    }

    ~Ring()
    {
        for (int i = 0; i < RingSize; i++)
        {
            CloseHandle(m_Overlap[i].hEvent);
        }

        for (int i = 0; i < RingSize; i++)
        {
            delete[] m_Buffer[i];
        }

    }

    uint32_t Next(uint32_t index)
    {
        return (index + 1) % RingSize;
    }

    bool IsEmpty()
    {
        return m_Head == m_Tail;
    }

    bool IsFull()
    {
        return Next(m_Head) == m_Tail;
    }

    bool Enqueue(uint32_t size)
    {
        if (IsFull())
        {
            return false;
        }

        m_Bytes[m_Head] = size;

        m_Head = Next(m_Head);

        return true;
    }

    bool Dequeue()
    {
        if (IsEmpty())
        {
            return false;
        }

        m_Tail = Next(m_Tail);

        return true;
    }

    OVERLAPPED* GetHeadOverlap()
    {
        return &m_Overlap[m_Head];
    }

    OVERLAPPED* GetTailOverlap()
    {
        return &m_Overlap[m_Tail];
    }

    uint8_t* GetHeadBuffer()
    {
        return m_Buffer[m_Head];
    }

    uint8_t* GetTailBuffer()
    {
        return m_Buffer[m_Tail];
    }

    uint32_t GetHeadBytes()
    {
        return m_Bytes[m_Head];
    }

    uint32_t GetTailBytes()
    {
        return m_Bytes[m_Tail];
    }

public:
    static const uint32_t MaxBytesPerTransaction = 1024 * 1024 * 16;

private:
    static const uint32_t RingSize = N;

    OVERLAPPED            m_Overlap[RingSize];
    uint32_t              m_Bytes[RingSize];
    uint8_t               *m_Buffer[RingSize];
    uint32_t              m_Head;
    uint32_t              m_Tail;
};

class UsbDevice
{
public:
    bool Initialize(const GUID *pGuid);
    void Finalize();

    void ReceiveDataSync(uint8_t *buffer, uint32_t size, uint32_t blockSize);
    void SendDataSync(uint8_t *buffer, uint32_t size, uint32_t blockSize);

    void ReceiveDataAsync(uint8_t *buffer, uint32_t size, uint32_t blockSize);
    void SendDataAsync(uint8_t *buffer, uint32_t size, uint32_t blockSize);

private:
    HANDLE                  deviceHandle;
    WINUSB_INTERFACE_HANDLE winUsbHandle;
    USHORT                  interfaceIndex;
    unsigned char           pipeBulkIn;
    unsigned char           pipeBulkOut;

    Ring<16>                m_Ring;
};

class UsbDsTestEnvironment : public ::testing::Environment
{
public:
    virtual void SetUp() override
    {
        NN_LOG("Searching for device.");
        while (!m_Device.Initialize(&TARGET_GUID))
        {
            Sleep(500);
            NN_LOG(".");
        }
        NN_LOG("Done\n");
    }

    virtual void TearDown() override
    {
        m_Device.Finalize();
    }

    UsbDevice* GetDevice()
    {
        return &m_Device;
    }

private:
    // ドライバの GUID
    static const GUID TARGET_GUID;
    UsbDevice m_Device;
};

} // end of namespace usb
} // end of namespace nnt
