﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/init/init_Malloc.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/nn_Log.h>
#include <nnt/gtest/gtest.h>
#include <cstdlib>
#include <nn/nn_Assert.h>

#include <nn/usb/usb_Device.h>
#include "UsbDsInterface.h"

using namespace nn;
using namespace nn::usb;
using namespace nn::usb::test;

const size_t g_HeapSize = 16 * 1024 * 1024;

// UsbDsInterface initialization of static members
DsClient UsbDsInterface::m_DsClient;
DsEndpoint UsbDsInterface::m_EndPoints[DsLimitMaxInterfacesPerConfigurationCount][UsbLimitMaxEndpointsCount];
DsInterface UsbDsInterface::m_DsInterfaces[DsLimitMaxInterfacesPerConfigurationCount];
int UsbDsInterface::m_numInterfaces = 0;

UsbInterfaceDescriptor m_UsbInterfaceDescriptor =
{
    UsbDescriptorSize_Interface,                // bLength
    UsbDescriptorType_Interface,                // bDescriptorType
    0,                                          // bInterfaceNumber
    0,                                          // bAlternateSetting
    0,                                          // bNumEndpoints
    0xff,                                       // bInterfaceClass
    0,                                          // bInterfaceSubClass
    0,                                          // bInterfaceProtocol
    4                                           // iInterface
};


UsbEndpointDescriptor m_UsbEndpointDescriptorsFs[2] =
{
    {
        UsbDescriptorSize_Endpoint,             // bLength;
        UsbDescriptorType_Endpoint,             // bDescriptorType;
        0,                                      // bEndpointAddress;
        UsbEndpointAttributeMask_XferTypeBulk,  // bmAttributes;
        64,                                     // wMaxPacketSize;
        0                                       // bInterval;
    },
    {
        UsbDescriptorSize_Endpoint,             // bLength;
        UsbDescriptorType_Endpoint,             // bDescriptorType;
        0,                                      // bEndpointAddress;
        UsbEndpointAttributeMask_XferTypeBulk,  // bmAttributes;
        64,                                     // wMaxPacketSize;
        0                                       // bInterval;
    }
};


UsbEndpointDescriptor m_UsbEndpointDescriptorsHs[2] =
{
    {
        UsbDescriptorSize_Endpoint,             // bLength;
        UsbDescriptorType_Endpoint,             // bDescriptorType;
        0,                                      // bEndpointAddress;
        UsbEndpointAttributeMask_XferTypeBulk,  // bmAttributes;
        512,                                    // wMaxPacketSize;
        0                                       // bInterval;
    },
    {
        UsbDescriptorSize_Endpoint,             // bLength;
        UsbDescriptorType_Endpoint,             // bDescriptorType;
        0,                                      // bEndpointAddress;
        UsbEndpointAttributeMask_XferTypeBulk,  // bmAttributes;
        512,                                    // wMaxPacketSize;
        0                                       // bInterval;
    }
};


UsbEndpointDescriptor m_UsbEndpointDescriptorsSs[2] =
{
    {
        UsbDescriptorSize_Endpoint,             // bLength;
        UsbDescriptorType_Endpoint,             // bDescriptorType;
        0,                                      // bEndpointAddress;
        UsbEndpointAttributeMask_XferTypeBulk,  // bmAttributes;
        1024,                                   // wMaxPacketSize;
        0                                       // bInterval;
    },
    {
        UsbDescriptorSize_Endpoint,             // bLength;
        UsbDescriptorType_Endpoint,             // bDescriptorType;
        0,                                      // bEndpointAddress;
        UsbEndpointAttributeMask_XferTypeBulk,  // bmAttributes;
        1024,                                   // wMaxPacketSize;
        0                                       // bInterval;
    }
};


UsbEndpointCompanionDescriptor m_UsbEndpointCompanionDescriptors[2] =
{
    {
        0x06,                                   // bLength
        UsbDescriptorType_EndpointCompanion,    // bDescriptorType
        15,                                     // bMaxBurst
        0,                                      // bmAttributes
        0                                       // wBytesPerInterval
    },
    {
        0x06,                                   // bLength
        UsbDescriptorType_EndpointCompanion,    // bDescriptorType
        15,                                     // bMaxBurst
        0,                                      // bmAttributes
        0                                       // wBytesPerInterval
    }
};


//////////////////////////////////////////////////////////////////////////////
class UsbDeviceTestEnvironment : public ::testing::Environment
{
private:
    UsbDsInterface usbTestInterface;
public:
    virtual void SetUp() override
    {
        nn::Result result;

        usbTestInterface.InitializeClient(ComplexId_Tegra21x);

        m_UsbInterfaceDescriptor.bInterfaceNumber       = 0;
        m_UsbInterfaceDescriptor.bNumEndpoints          = 2;
        m_UsbEndpointDescriptorsFs[0].bEndpointAddress  = 0x81;
        m_UsbEndpointDescriptorsFs[1].bEndpointAddress  = 0x01;
        m_UsbEndpointDescriptorsHs[0].bEndpointAddress  = 0x81;
        m_UsbEndpointDescriptorsHs[1].bEndpointAddress  = 0x01;
        m_UsbEndpointDescriptorsSs[0].bEndpointAddress  = 0x81;
        m_UsbEndpointDescriptorsSs[1].bEndpointAddress  = 0x01;
        usbTestInterface.InitializeDescriptors(&m_UsbInterfaceDescriptor, m_UsbEndpointDescriptorsFs, m_UsbEndpointDescriptorsHs, m_UsbEndpointDescriptorsSs, m_UsbEndpointCompanionDescriptors);

        m_UsbInterfaceDescriptor.bInterfaceNumber       = 1;
        m_UsbInterfaceDescriptor.bNumEndpoints          = 2;
        m_UsbEndpointDescriptorsFs[0].bEndpointAddress  = 0x82;
        m_UsbEndpointDescriptorsFs[1].bEndpointAddress  = 0x02;
        m_UsbEndpointDescriptorsHs[0].bEndpointAddress  = 0x82;
        m_UsbEndpointDescriptorsHs[1].bEndpointAddress  = 0x02;
        m_UsbEndpointDescriptorsSs[0].bEndpointAddress  = 0x82;
        m_UsbEndpointDescriptorsSs[1].bEndpointAddress  = 0x02;
        usbTestInterface.InitializeDescriptors(&m_UsbInterfaceDescriptor, m_UsbEndpointDescriptorsFs, m_UsbEndpointDescriptorsHs, m_UsbEndpointDescriptorsSs, m_UsbEndpointCompanionDescriptors);

        m_UsbInterfaceDescriptor.bInterfaceNumber       = 2;
        m_UsbInterfaceDescriptor.bNumEndpoints          = 2;
        m_UsbEndpointDescriptorsFs[0].bEndpointAddress  = 0x83;
        m_UsbEndpointDescriptorsFs[1].bEndpointAddress  = 0x03;
        m_UsbEndpointDescriptorsHs[0].bEndpointAddress  = 0x83;
        m_UsbEndpointDescriptorsHs[1].bEndpointAddress  = 0x03;
        m_UsbEndpointDescriptorsSs[0].bEndpointAddress  = 0x83;
        m_UsbEndpointDescriptorsSs[1].bEndpointAddress  = 0x03;
        usbTestInterface.InitializeDescriptors(&m_UsbInterfaceDescriptor, m_UsbEndpointDescriptorsFs, m_UsbEndpointDescriptorsHs, m_UsbEndpointDescriptorsSs, m_UsbEndpointCompanionDescriptors);

        m_UsbInterfaceDescriptor.bInterfaceNumber       = 3;
        m_UsbInterfaceDescriptor.bNumEndpoints          = 2;
        m_UsbEndpointDescriptorsFs[0].bEndpointAddress  = 0x84;
        m_UsbEndpointDescriptorsFs[1].bEndpointAddress  = 0x04;
        m_UsbEndpointDescriptorsHs[0].bEndpointAddress  = 0x84;
        m_UsbEndpointDescriptorsHs[1].bEndpointAddress  = 0x04;
        m_UsbEndpointDescriptorsSs[0].bEndpointAddress  = 0x84;
        m_UsbEndpointDescriptorsSs[1].bEndpointAddress  = 0x04;
        usbTestInterface.InitializeDescriptors(&m_UsbInterfaceDescriptor, m_UsbEndpointDescriptorsFs, m_UsbEndpointDescriptorsHs, m_UsbEndpointDescriptorsSs, m_UsbEndpointCompanionDescriptors);

        usbTestInterface.WaitForUsbConfigured();
        NN_LOG("Initialized UsbDeviceTestEnvironment\n");
    }
    virtual void TearDown() override
    {
        usbTestInterface.Finalize();
        NN_LOG("Finalizing UsbDeviceTestEnvironment...\n");
    }
};

extern "C" void nninitStartup()
{
    nn::Result result;

    /* set heap size */
    result = nn::os::SetMemoryHeapSize(g_HeapSize);
    if (!result.IsSuccess()) {
        NN_LOG("Failed SetMemoryHeapSize\n");
        return;
    }
    NN_LOG("SetMemoryHeapSize 0x%x OK\n", g_HeapSize);

    uintptr_t address;
    result = nn::os::AllocateMemoryBlock(&address, g_HeapSize / 2);
    NN_ASSERT(result.IsSuccess());

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), g_HeapSize / 2);

    ::testing::AddGlobalTestEnvironment(new UsbDeviceTestEnvironment);
}
