﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <sstream>
#include <iomanip>

#include <nn/nn_Assert.h>
#include <nn/nn_Result.h>

#include <nn/init.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/usb/pd/usb_Pd.h>

#include "Draw.h"
#include "Input.h"
#include <nnt/usbPdUtil/testUsbPd_util.h>
#include "SceneGetStatus.h"
#include "ToString.h"

namespace nnt { namespace usb { namespace pd {

    namespace {
        const State StateList[] = {
            State_NoConnection,
            State_OfficialAcAdaptor,
            State_NoConnection,
            State_AppleCharger,
            State_NoConnection,
            State_CradleWithOfficialAcAdaptor,
            State_NoConnection,
            State_Pc,
            State_NoConnection,
        };
        const int StateListSize = sizeof(StateList) / sizeof(StateList[0]);

        // 1フレームの時間 (msec)
        const int64_t FrameLength = 50;

        nn::TimeSpan g_ApiExecutionTime;
        nn::os::Tick g_SceneStart;

        nn::usb::pd::Session g_Session;
        nn::usb::pd::Notice g_Notice;

        bool g_IsTestFinished;

        void Print(State prev, State current, State next) NN_NOEXCEPT
        {
            nnt::usb::pd::draw::Clear();
            std::stringstream ss;

            if (prev != State_Invalid && current != State_Invalid)
            {
                ss << "State: " << ToString(prev) << " -> " << ToString(current) << std::endl;
                ss << "API Execution Time: " << g_ApiExecutionTime.GetMicroSeconds() << " usec" << std::endl;
                ss << std::endl;

                ss << "IsActiveNotice: " << ToString(g_Notice.IsActiveNotice()) << std::endl;
                ss << "IsErrorNotice: " << ToString(g_Notice.IsErrorNotice()) << std::endl;
                ss << "IsDataRoleNotice: " << ToString(g_Notice.IsDataRoleNotice()) << std::endl;
                ss << "IsPowerRoleNotice: " << ToString(g_Notice.IsPowerRoleNotice()) << std::endl;
                ss << "IsDeviceNotice: " << ToString(g_Notice.IsDeviceNotice()) << std::endl;
                ss << "IsConsumerContractNotice: " << ToString(g_Notice.IsConsumerContractNotice()) << std::endl;
                ss << "IsProviderContractNotice: " << ToString(g_Notice.IsProviderContractNotice()) << std::endl;
                ss << "IsRequestNotice: " << ToString(g_Notice.IsRequestNotice()) << std::endl;
                ss << std::endl;
            }

            ss << "Next: " << ToString(next) << std::endl;
            ss << std::endl;

            float now = (nn::os::GetSystemTick() - g_SceneStart).ToTimeSpan().GetMilliSeconds() / 1000.0f;
            ss << "Time: " << std::fixed << std::setprecision(1) << now << std::endl;
            ss << std::endl;

            ss << "Press B key to go back." << std::endl;
            ss << std::endl;

            nnt::usb::pd::draw::Print(ss.str().c_str());

            if(g_IsTestFinished)
            {
                nnt::usb::pd::draw::Print("Test complete!", nnt::usb::pd::draw::Color_Green);
            }

            nnt::usb::pd::draw::Draw();
        }

        void Update() NN_NOEXCEPT
        {
            auto start = nn::os::GetSystemTick();

            // usb::pd の Status を取得
            nn::usb::pd::GetNotice(&g_Notice, &g_Session);

            auto end = nn::os::GetSystemTick();
            g_ApiExecutionTime = (end - start).ToTimeSpan();
        }

    } // namespace

    void SceneStateTransition(SceneResult *pOutResult, nn::os::SystemEventType *pNoticeEvent)
    {
        g_SceneStart = nn::os::GetSystemTick();
        g_IsTestFinished = false;

        nn::usb::pd::OpenSession(&g_Session);

        nn::usb::pd::BindNoticeEvent(pNoticeEvent, &g_Session);

        nn::os::ClearSystemEvent(pNoticeEvent);

        int index = 0;
        for (;;)
        {
            input::Update();

            State prev = (index - 1 >= 0) ? StateList[index - 1] : State_Invalid;
            State current = StateList[index];
            State next = (index < StateListSize) ? StateList[index + 1] : State_Invalid;

            Print(prev, current, next);

            // Notice イベントが発生したら、テストを進める
            if (nn::os::TimedWaitSystemEvent(pNoticeEvent, nn::TimeSpan::FromMilliSeconds(FrameLength)))
            {
                nn::os::ClearSystemEvent(pNoticeEvent);
                Update();

                if (index < StateListSize - 2)
                {
                    index++;
                }
                else
                {
                    g_IsTestFinished = true;
                }
            }

            // キャンセルボタンが押されたらループを抜ける
            if (input::IsCancelPushed())
            {
                break;
            }
        }

        nn::usb::pd::UnbindNoticeEvent(&g_Session);

        nn::usb::pd::CloseSession(&g_Session);

        pOutResult->nextSceneType = SceneType_Menu;
        pOutResult->nextSceneArg.menu.type = SceneMenuType_Top;
    }
}}}
