﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nnt/gtest/gtest.h>

#include <nn/uart/uart.h>
#include <nn/uart/uart_PortApiDev.h>

#include "../Common/testUart_Common.h"
#include "../Common/testUart_Helper.h"

namespace {

const size_t UartBufferSize = 1 * nn::os::MemoryPageSize;

// 内部送信バッファ
NN_ALIGNAS(nn::os::MemoryPageSize) char s_SendBuffer[UartBufferSize];
// 内部受信バッファ
NN_ALIGNAS(nn::os::MemoryPageSize) char s_ReceiveBuffer[UartBufferSize];

const nn::uart::BaudRate TestedBaudRate = nn::uart::BaudRate_115200;

#if 0 // いつか使うかもしれないのでコメントアウトだけ
// 非同期操作を伴うテスト用のスレッドリソース
const size_t SubThreadStackSize = 1 * 1024 * 1024;
NN_ALIGNAS(4096) char s_SubThreadStack[ SubThreadStackSize ];
nn::os::ThreadType s_SubThread;

void StartSubThread(nn::os::ThreadFunction function, void *arg, int priority)
{
    nn::Result result = nn::os::CreateThread( &s_SubThread, function, arg, s_SubThreadStack, SubThreadStackSize, priority );
    ASSERT_TRUE(result.IsSuccess());
    nn::os::StartThread( &s_SubThread );
}

void WaitAndFinishSubThread()
{
    nn::os::WaitThread( &s_SubThread );
    nn::os::DestroyThread( &s_SubThread );
}
#endif

}

TEST(LibInitTest, Basic)
{
    // 基本
    ASSERT_FALSE(nn::uart::IsInitialized());
    nn::uart::Initialize();
    ASSERT_TRUE(nn::uart::IsInitialized());
    nn::uart::Finalize();
    ASSERT_FALSE(nn::uart::IsInitialized());

    const int LoopCount = 5;
    // 複数回 init が可能
    ASSERT_FALSE(nn::uart::IsInitialized());
    for (int i = 0; i < LoopCount; ++i)
    {
        nn::uart::Initialize();
        ASSERT_TRUE(nn::uart::IsInitialized());
    }
    for (int i = 0; i < LoopCount - 1; ++i)
    {
        nn::uart::Finalize();
        ASSERT_TRUE(nn::uart::IsInitialized());
    }
    nn::uart::Finalize();
    ASSERT_FALSE(nn::uart::IsInitialized());
}

// ポートオープン状態のファイナライズでポートが強制的に閉じられることのテスト
TEST(LibInitTest, OpenShutdown)
{
    // 前準備
    nn::uart::PortSession   portSession;
    nn::uart::PortConfig    portConfig(
        TestedBaudRate,
        s_SendBuffer,
        sizeof(s_SendBuffer),
        s_ReceiveBuffer,
        sizeof(s_ReceiveBuffer)
        );
    nn::uart::Initialize();

    // 最初に見つかった有効なポートをテストに使用
    int devPortIndex = nnt::uart::GetFirstAvailablePortIndex();
    ASSERT_GE(devPortIndex, 0);
    nn::uart::OpenPortForDev(&portSession, devPortIndex, portConfig);

    // Send を開始
    const size_t TestStringSize = UartBufferSize;
    static const char TestString[TestStringSize] = {'A'};
    size_t doneBytes;
    nn::uart::Send(&doneBytes, &portSession, TestString, TestStringSize);
    ASSERT_EQ(TestStringSize, doneBytes);

    // Send は完了していないと思われるが即クローズ・ファイナライズ
    nn::uart::ClosePort(&portSession);
    nn::uart::Finalize();

    // 状態確認
    ASSERT_FALSE(nn::uart::IsInitialized());
}

TEST(PortOpenTest, Basic)
{
    nn::uart::Initialize();

    ASSERT_TRUE(nn::uart::IsInitialized());

    nn::uart::PortConfig portConfig(
        TestedBaudRate,
        s_SendBuffer,
        sizeof(s_SendBuffer),
        nullptr,
        0
        );

    // 通常の open
    for (const auto PortName : TestedPortList)
    {
        if (nn::uart::HasPort(PortName))
        {
            // ポートが存在する場合はオープン確認
            nn::uart::PortSession portSession;

            ASSERT_TRUE(nn::uart::OpenPort(&portSession, PortName, portConfig));
            nn::uart::ClosePort(&portSession);
        }
    }

    // 開発用 API
    for (int portIdx = 0; portIdx < DevPortCountMax + 1; ++portIdx)
    {
        if (portIdx >= DevPortCountMax)
        {
            ASSERT_FALSE(nn::uart::HasPortForDev(portIdx));
        }
        else if (nn::uart::HasPortForDev(portIdx))
        {
            NN_LOG("Testing port %d\n", portIdx);
            // ASSERT_FALSE(nn::uart::IsPortOpen(portName));

            nn::uart::PortSession portSession;

            ASSERT_TRUE(nn::uart::OpenPortForDev(&portSession, portIdx, portConfig));
            // ASSERT_TRUE(nn::uart::IsPortOpen(portName));
            // ASSERT_NE(portSession._handle, nn::uart::InvalidPortHandle._handle);
            nn::uart::ClosePort(&portSession);
            // ASSERT_FALSE(nn::uart::IsPortOpen(portName));

        }
        else
        {
            NN_LOG("Port %d is not supported\n", portIdx);
        }
    }

    nn::uart::Finalize();
}

TEST(ParameterTest, SendReceive)
{
    nn::uart::Initialize();

    nn::uart::PortSession   portSession;
    nn::uart::PortConfig    portConfig(
        TestedBaudRate,
        s_SendBuffer,
        sizeof(s_SendBuffer),
        s_ReceiveBuffer,
        sizeof(s_ReceiveBuffer)
        );

    // 最初に見つかった有効なポートをテストに使用
    int devPortIndex = nnt::uart::GetFirstAvailablePortIndex();
    ASSERT_GE(devPortIndex, 0);
    ASSERT_TRUE(nn::uart::OpenPortForDev(&portSession, devPortIndex, portConfig));

    // Send/Receive の pOutDoneBytes に nullptr を指定しても問題ないことを確認
    const char TestData[] = {'A'};
    nn::uart::Send(nullptr, &portSession, TestData, sizeof(TestData));

    char receiveBuffer[1];
    nn::uart::Receive(nullptr, receiveBuffer, sizeof(receiveBuffer), &portSession);

    nn::uart::ClosePort(&portSession);
    nn::uart::Finalize();
}
