﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_BitTypes.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/os/os_MemoryHeap.h>
#include "eval_ScreenController.h"

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#endif

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/nn_Windows.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/settings/settings_DebugPad.h>
#endif

namespace
{
    const int RenderWidth = 1280;
    const int RenderHeight = 720;
}

namespace nnt { namespace eval { namespace detail {

class GraphicsSystem
{
    NN_DISALLOW_COPY(GraphicsSystem);
    NN_DISALLOW_MOVE(GraphicsSystem);

private:
    //------------------------------------------------------------------------------
    //  レイヤを初期化
    //------------------------------------------------------------------------------
    nn::vi::Display* g_pDisplay;
    nn::vi::Layer* g_pLayer;
    void InitializeLayer() NN_NOEXCEPT
    {
        nn::Result result = nn::vi::OpenDefaultDisplay(&g_pDisplay);
        NN_ASSERT(result.IsSuccess());
        NN_UNUSED(result);

        result = nn::vi::CreateLayer(&g_pLayer, g_pDisplay);
        NN_ASSERT(result.IsSuccess());

        result = nn::vi::SetLayerScalingMode(g_pLayer, nn::vi::ScalingMode_FitToLayer);
        NN_ASSERT(result.IsSuccess());
    }

    //------------------------------------------------------------------------------
    //  デバイスを初期化
    //------------------------------------------------------------------------------
    nn::gfx::Device g_Device;
    void InitializeDevice() NN_NOEXCEPT
    {
        nn::gfx::Device::InfoType info;
        info.SetDefault();
        info.SetApiVersion(nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion);
        g_Device.Initialize(info);
    }

    //------------------------------------------------------------------------------
    //  メモリプールを初期化
    //------------------------------------------------------------------------------
    static const size_t g_VisiblePoolMemorySize = 16 * 1024 * 1024;
    void* g_pVisiblePoolMemory = NULL;
    void* g_pMemoryPoolStart = NULL;
    nn::gfx::MemoryPool g_MemoryPool;
    ptrdiff_t g_MemoryPoolOffset = 0;
    void InitializeMemoryPool()
    {
        nn::gfx::MemoryPool::InfoType info;
        info.SetDefault();
        info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached
            | nn::gfx::MemoryPoolProperty_GpuCached);

        // Determine alignment to allocate space on an alignment boundary
        size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(&g_Device, info);
        g_pVisiblePoolMemory = malloc(g_VisiblePoolMemorySize + alignment);

        g_pMemoryPoolStart = nn::util::BytePtr(g_pVisiblePoolMemory).AlignUp(alignment).Get();
        info.SetPoolMemory(g_pMemoryPoolStart, nn::util::align_down(g_VisiblePoolMemorySize,
            nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&g_Device, info)));
        g_MemoryPool.Initialize(&g_Device, info);

        g_MemoryPoolOffset = 0;
    }

    static const size_t g_InvisiblePoolMemorySize = 20 * 1024 * 1024;
    void* g_pInvisiblePoolMemory = NULL;
    void* g_pInvisibleMemoryPoolStart = NULL;
    ptrdiff_t g_InvisibleMemoryPoolOffset = 0;
    nn::gfx::MemoryPool g_InvisibleMemoryPool;
    void InitializeInvisibleMemoryPool() NN_NOEXCEPT
    {
        nn::gfx::MemoryPool::InfoType info;
        info.SetDefault();
        info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuInvisible
            | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible);

        // Determine alignment to allocate space on an alignment boundary
        size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(&g_Device, info);
        g_pInvisiblePoolMemory = malloc(g_InvisiblePoolMemorySize + alignment);

        g_pInvisibleMemoryPoolStart = nn::util::BytePtr(g_pInvisiblePoolMemory).AlignUp(alignment).Get();
        info.SetPoolMemory(g_pInvisibleMemoryPoolStart, nn::util::align_down(g_InvisiblePoolMemorySize,
            nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&g_Device, info)));
        g_InvisibleMemoryPool.Initialize(&g_Device, info);

        g_InvisibleMemoryPoolOffset = 0;
    }

    //------------------------------------------------------------------------------
    //  スワップチェーンを初期化
    //------------------------------------------------------------------------------
    nn::gfx::SwapChain g_SwapChain;
    void InitializeSwapChain() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(g_pLayer);
        NN_ASSERT_NOT_NULL(g_pDisplay);

        nn::gfx::SwapChain::InfoType info;

        info.SetDefault();
        info.SetLayer(g_pLayer);
        info.SetWidth(RenderWidth);
        info.SetHeight(RenderHeight);
        info.SetFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb);
        info.SetBufferCount(2);
        if (NN_STATIC_CONDITION(nn::gfx::SwapChain::IsMemoryPoolRequired))
        {
            size_t size = g_SwapChain.CalculateScanBufferSize(&g_Device, info);
            g_InvisibleMemoryPoolOffset = nn::util::align_up(g_InvisibleMemoryPoolOffset,
                nn::gfx::SwapChain::GetScanBufferAlignment(&g_Device, info));
            g_SwapChain.Initialize(&g_Device, info, &g_InvisibleMemoryPool, g_InvisibleMemoryPoolOffset, size);
            g_InvisibleMemoryPoolOffset += size;
        }
        else
        {
            g_SwapChain.Initialize(&g_Device, info, NULL, 0, 0);
        }
    }

    //------------------------------------------------------------------------------
    //  キューを初期化
    //------------------------------------------------------------------------------
    nn::gfx::Queue g_Queue;
    void InitializeQueue() NN_NOEXCEPT
    {
        nn::gfx::Queue::InfoType info;
        info.SetDefault();
        info.SetCapability(nn::gfx::QueueCapability_Graphics);
        g_Queue.Initialize(&g_Device, info);
    }

    //------------------------------------------------------------------------------
    //  コマンドバッファを初期化
    //------------------------------------------------------------------------------
    nn::gfx::CommandBuffer g_CommandBuffer;
    void InitializeCommandBuffer() NN_NOEXCEPT
    {
        nn::gfx::CommandBuffer::InfoType info;
        info.SetDefault();
        info.SetQueueCapability(nn::gfx::QueueCapability_Graphics);
        info.SetCommandBufferType(nn::gfx::CommandBufferType_Direct);
        g_CommandBuffer.Initialize(&g_Device, info);
    }

    //
    //------------------------------------------------------------------------------
    //  ビューポートシザーを初期化
    //------------------------------------------------------------------------------
    nn::gfx::ViewportScissorState g_ViewportScissor;
    void InitializeViewport() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(g_pDisplay);

        nn::gfx::ViewportScissorState::InfoType info;
        info.SetDefault();
        info.SetScissorEnabled(true);
        nn::gfx::ViewportStateInfo viewportInfo;
    {
            viewportInfo.SetDefault();
            viewportInfo.SetWidth(static_cast< float >(RenderWidth));
            viewportInfo.SetHeight(static_cast< float >(RenderHeight));
    }
        nn::gfx::ScissorStateInfo scissorInfo;
    {
            scissorInfo.SetDefault();
            scissorInfo.SetWidth(RenderWidth);
            scissorInfo.SetHeight(RenderHeight);
        }
        info.SetViewportStateInfoArray(&viewportInfo, 1);
        info.SetScissorStateInfoArray(&scissorInfo, 1);
        g_ViewportScissor.Initialize(&g_Device, info);
    }

    //------------------------------------------------------------------------------
    //  サンプラディスクリプタプールの初期化
    //------------------------------------------------------------------------------
    nn::gfx::DescriptorPool g_SamplerDescriptorPool;
    int g_SamplerDescriptorBaseIndex = 0;
    void InitializeSamplerDescriptorPool() NN_NOEXCEPT
    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_Sampler);
        info.SetSlotCount(g_SamplerDescriptorBaseIndex + 1);
        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&g_Device, info);
        g_MemoryPoolOffset = nn::util::align_up(g_MemoryPoolOffset,
            nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&g_Device, info));
        g_SamplerDescriptorPool.Initialize(&g_Device, info,
            &g_MemoryPool, g_MemoryPoolOffset, size);
        g_MemoryPoolOffset += size;
    }

    //------------------------------------------------------------------------------
    //  テクスチャディスクリプタプールの初期化
    //------------------------------------------------------------------------------
    nn::gfx::DescriptorPool g_TextureDescriptorPool;
    int g_TextureDescriptorBaseIndex = 0;

    void InitializeTextureDescriptorPool() NN_NOEXCEPT
    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_TextureView);
        info.SetSlotCount(g_TextureDescriptorBaseIndex + 1);
        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&g_Device, info);
        g_MemoryPoolOffset = nn::util::align_up(g_MemoryPoolOffset,
            nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&g_Device, info));
        g_TextureDescriptorPool.Initialize(&g_Device, info, &g_MemoryPool, g_MemoryPoolOffset, size);
        g_MemoryPoolOffset += size;
    }

    //------------------------------------------------------------------------------
    //  gfx オブジェクトの初期化
    //------------------------------------------------------------------------------
    nn::util::BytePtr g_pMemoryHeap;
    nn::util::BytePtr g_pMemory;
    void InitializeGfxObjects() NN_NOEXCEPT
    {
        g_pMemoryHeap.Reset(malloc(1024 * 1024 * 32));
        g_pMemory = g_pMemoryHeap;

        InitializeDevice();

#if NN_GFX_IS_TARGET_NVN
        nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData(g_Device);
        nvnDeviceGetInteger(deviceData.pNvnDevice,
            NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &g_TextureDescriptorBaseIndex);
        nvnDeviceGetInteger(deviceData.pNvnDevice,
            NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &g_SamplerDescriptorBaseIndex);
#endif
        InitializeMemoryPool();
        InitializeInvisibleMemoryPool();

        InitializeSwapChain();
        InitializeQueue();

        InitializeCommandBuffer();
        InitializeViewport();

        InitializeSamplerDescriptorPool();
        InitializeTextureDescriptorPool();

        NN_ASSERT(g_pMemoryHeap.Distance(g_pMemory.Get()) < 1024 * 1024 * 32);
        NN_ASSERT(g_InvisibleMemoryPoolOffset < g_InvisiblePoolMemorySize);
    }

    //------------------------------------------------------------------------------
    //  gfx オブジェクトの終了処理
    //------------------------------------------------------------------------------
    void FinalizeGfxObjects() NN_NOEXCEPT
    {
        // 各オブジェクトを破棄
        g_TextureDescriptorPool.Finalize(&g_Device);
        g_SamplerDescriptorPool.Finalize(&g_Device);

        g_ViewportScissor.Finalize(&g_Device);
        g_CommandBuffer.Finalize(&g_Device);
        g_SwapChain.Finalize(&g_Device);
        g_Queue.Finalize(&g_Device);
        g_InvisibleMemoryPool.Finalize(&g_Device);
        g_MemoryPool.Finalize(&g_Device);
        g_Device.Finalize();

        free(g_pMemoryHeap.Get());
        free(g_pVisiblePoolMemory);
        free(g_pInvisiblePoolMemory);
    }


public:
    GraphicsSystem() NN_NOEXCEPT
        : g_pDisplay(NULL)
        , g_pLayer(NULL)
        , g_pVisiblePoolMemory(NULL)
        , g_pMemoryPoolStart(NULL)
        , g_pInvisiblePoolMemory(NULL)
        , g_pInvisibleMemoryPoolStart(NULL)
        , g_pMemoryHeap(NULL)
        , g_pMemory(NULL)
    {
    }

    ~GraphicsSystem() NN_NOEXCEPT
    {
    }

    void Initialize() NN_NOEXCEPT
    {
        nn::vi::Initialize();
        InitializeLayer();

        nn::gfx::Initialize();
        InitializeGfxObjects();
    }

    void Finalize() NN_NOEXCEPT
    {
        FinalizeGfxObjects();
        nn::gfx::Finalize();

        nn::vi::DestroyLayer(g_pLayer);
        nn::vi::CloseDisplay(g_pDisplay);
        nn::vi::Finalize();
    }

    void BeginDraw() NN_NOEXCEPT
    {
        // コマンド生成
        g_CommandBuffer.Reset();
        g_MemoryPoolOffset = nn::util::align_up(g_MemoryPoolOffset,
            nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&g_Device));
        g_CommandBuffer.AddCommandMemory(&g_MemoryPool, g_MemoryPoolOffset, 1024 * 1024);
        g_pMemory.AlignUp(256);
        g_CommandBuffer.AddControlMemory(g_pMemory.Get(), 256);
        g_CommandBuffer.Begin();
        {
            nn::gfx::ColorTargetView* pTarget = g_SwapChain.AcquireNextScanBufferView();
            g_CommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Descriptor | nn::gfx::GpuAccess_ShaderCode);
            g_CommandBuffer.SetDescriptorPool(&g_TextureDescriptorPool);
            g_CommandBuffer.SetDescriptorPool(&g_SamplerDescriptorPool);
            g_CommandBuffer.ClearColor(pTarget, 0.1f, 0.1f, 0.1f, 1.0f, NULL);
            g_CommandBuffer.SetRenderTargets(1, &pTarget, NULL);
            g_CommandBuffer.SetViewportScissorState(&g_ViewportScissor);
    }
    }

    void EndDraw() NN_NOEXCEPT
    {
        g_CommandBuffer.End();

        // コマンドの実行
        g_Queue.ExecuteCommand(&g_CommandBuffer, NULL);

        // 結果の表示
        g_Queue.Present(&g_SwapChain, 1);
        g_Queue.Sync();
    }

    //下記メンバは受け側のアクセス範囲が広すぎてconst付けられなかったため直接返す
    nn::gfx::Device* GetInitializedDevice()
    {
        return &g_Device;
    }
    nn::vi::Display* GetInitializedDisplay()
    {
        return g_pDisplay;
    }
    nn::gfx::DescriptorPool* GetTextureDescriptorPool()
    {
        return &g_TextureDescriptorPool;
    }
    int GetTextureDescriptorBaseIndex()
        {
        return g_TextureDescriptorBaseIndex;
        }
    nn::gfx::DescriptorPool* GetSamplerDescriptorPool()
    {
        return &g_SamplerDescriptorPool;
    }
    int GetSamplerDescriptorBaseIndex()
    {
        return g_SamplerDescriptorBaseIndex;
    }
    nn::gfx::CommandBuffer* GetCommandBuffer()
    {
        return &g_CommandBuffer;
    }
};

class FontSystem
{
    NN_DISALLOW_COPY(FontSystem);
    NN_DISALLOW_MOVE(FontSystem);

private:
    nn::util::BytePtr* m_pDebugFontHeap;
    nn::gfx::util::DebugFontTextWriter m_TextWriter;
public:
    FontSystem() NN_NOEXCEPT
        : m_pDebugFontHeap(NULL)
    {
    }

    ~FontSystem() NN_NOEXCEPT
    {

    }

    void Initialize(GraphicsSystem* pGraphicsSystem) NN_NOEXCEPT
    {
        nn::gfx::Device* pInitializedDevice = pGraphicsSystem->GetInitializedDevice();
        nn::gfx::DescriptorPool* pTextureDescriptorPool = pGraphicsSystem->GetTextureDescriptorPool();
        int textureDescriptorPoolBaseIndex = pGraphicsSystem->GetTextureDescriptorBaseIndex();
        nn::gfx::DescriptorPool* pSamplerDescriptorPool = pGraphicsSystem->GetSamplerDescriptorPool();
        int samplerDescriptorPoolBaseIndex = pGraphicsSystem->GetSamplerDescriptorBaseIndex();


        // デバッグフォント初期化
        const int charCountMax = 1024;
        nn::gfx::util::DebugFontTextWriterInfo info;
        info.SetDefault();
        info.SetCharCountMax(charCountMax);
        info.SetUserMemoryPoolEnabled(false);

        size_t debugFontHeapSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(pInitializedDevice, info);
        m_pDebugFontHeap = new nn::util::BytePtr(new uint8_t[debugFontHeapSize]);

        m_TextWriter.Initialize(
            pInitializedDevice,
            info,
            m_pDebugFontHeap->Get(),
            debugFontHeapSize,
            nullptr,
            0,
            0
            );

        m_TextWriter.SetDisplayWidth(RenderWidth);
        m_TextWriter.SetDisplayHeight(RenderHeight);
        m_TextWriter.SetTextureDescriptor(pTextureDescriptorPool, textureDescriptorPoolBaseIndex);
        m_TextWriter.SetSamplerDescriptor(pSamplerDescriptorPool, samplerDescriptorPoolBaseIndex);
    }

    void Finalize() NN_NOEXCEPT
    {
        m_TextWriter.Finalize();
        delete[] reinterpret_cast<uint8_t*>(m_pDebugFontHeap->Get());
        m_pDebugFontHeap->Reset(nullptr);
        delete m_pDebugFontHeap;
    }


    nn::gfx::util::DebugFontTextWriter* GetDebugFontTextWriter()
    {
        return &m_TextWriter;
    }

    void Draw(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
    {
        m_TextWriter.Draw(pCommandBuffer);
    }
};

FontSystem*     g_pFontSystem;
GraphicsSystem* g_pGraphicsSystem;

void Initialize() NN_NOEXCEPT
{
    g_pGraphicsSystem = new GraphicsSystem();
    g_pGraphicsSystem->Initialize();

    g_pFontSystem = new FontSystem();
    g_pFontSystem->Initialize(g_pGraphicsSystem);
}

void Finalize() NN_NOEXCEPT
{
    g_pFontSystem->Finalize();
    delete g_pFontSystem;

    g_pGraphicsSystem->Finalize();
    delete g_pGraphicsSystem;
}

void Draw() NN_NOEXCEPT
{
    g_pGraphicsSystem->BeginDraw();
    g_pFontSystem->Draw(g_pGraphicsSystem->GetCommandBuffer());
    g_pGraphicsSystem->EndDraw();

#if defined(NN_BUILD_CONFIG_OS_WIN)
    MSG msg;
    while (::PeekMessageA(&msg, NULL, 0, 0, PM_REMOVE))
    {
        ::TranslateMessage(&msg);
        ::DispatchMessageA(&msg);
    }
#endif

}

nn::gfx::util::DebugFontTextWriter* GetDebugFontTextWriter() NN_NOEXCEPT
{
    return g_pFontSystem->GetDebugFontTextWriter();
}

}}} // namespace nnt::eval::detail

