﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using ZarfCreator.VersionData;

namespace ZarfCreatorTest.VersionTest
{
    /// <summary>
    /// VersionTest の概要の説明
    /// </summary>
    [TestClass]
    public class VersionTest
    {
        private string ResoucesPath = "../../../../../Tools/Resources/PackageData";

        [TestMethod]
        public void TestMethod1()
        {
            var reader = VersionDataFactory.Create(".date", "");
            var result = reader.Read();

            Assert.AreEqual(result, DateTime.Now.ToString("yyyy.M.d"));

            reader = VersionDataFactory.Create("1.0.0", "");
            result = reader.Read();

            Assert.AreEqual(result, "1.0.0");

            reader = VersionDataFactory.Create("1", "");
            result = reader.Read();

            Assert.AreEqual(result, "1");

            reader = VersionDataFactory.Create("*", "");
            result = reader.Read();

            Assert.AreEqual(result, "*");

            reader = VersionDataFactory.Create(">=4.4.0", "");
            result = reader.Read();

            Assert.AreEqual(result, ">=4.4.0");
        }

        [TestMethod]
        public void ReadFromHeaderTest()
        {
            var reader = VersionDataFactory.Create("DummyVersion.h", Path.Combine(ResoucesPath, "VersionHeader"));
            var result = reader.Read();
            Assert.AreEqual(result, "0.9.0");


            reader = VersionDataFactory.Create("VersionHeader/DummyVersion.h", ResoucesPath);
            result = reader.Read();
            Assert.AreEqual(result, "0.9.0");


            reader = VersionDataFactory.Create("VersionHeader/InvalidVersion.h", ResoucesPath);
            TestUtil.ExpectException<ArgumentException>(() => reader.Read(), "Invalid file has specified.");


            reader = VersionDataFactory.Create(@"VersionHeader\InvalidVersion2.h", ResoucesPath);
            result = reader.Read();
            Assert.AreEqual(result, "0.5.7");


            reader = VersionDataFactory.Create("version.h", "");
            TestUtil.ExpectException<IOException>(() => reader.Read(), "見つかりませんでした。");


            reader = VersionDataFactory.Create("hoge.h", @"D:\home");
            TestUtil.ExpectException<IOException>(() => reader.Read(), "見つかりませんでした。");
        }

        [TestMethod]
        public void ReadFromTextTest()
        {
            var reader = VersionDataFactory.Create("VersionText/DummyVersion.txt", ResoucesPath);
            var result = reader.Read();
            Assert.AreEqual("0.9.0", result);

            reader = VersionDataFactory.Create("VersionText/DummyVersion2.txt", ResoucesPath);
            result = reader.Read();
            Assert.AreEqual("373.53.0", result);

            reader = VersionDataFactory.Create("VersionText/DummyVersion3.txt", ResoucesPath);
            result = reader.Read();
            Assert.AreEqual("2.0.0+Build6", result);

            reader = VersionDataFactory.Create("VersionText/DummyVersion4.txt", ResoucesPath);
            result = reader.Read();
            Assert.AreEqual("2.0.0-beta.1", result);

            reader = VersionDataFactory.Create("VersionText/InvalidVersion.txt", ResoucesPath);
            TestUtil.ExpectException<ArgumentException>(() => reader.Read(), "Invalid format has specified.");

            reader = VersionDataFactory.Create("VersionText/InvalidVersion2.txt", ResoucesPath);
            TestUtil.ExpectException<ArgumentException>(() => reader.Read(), "Invalid format has specified.");
        }

        [TestMethod]
        public void ReadFromXmlTest()
        {
            var reader = VersionDataFactory.Create("VersionXml/DummyVersion.xml", ResoucesPath, false);
            var result = reader.Read();
            Assert.AreEqual("1.2.3", result);

            reader = VersionDataFactory.Create("VersionXml/DummyVersion2.xml", ResoucesPath, true);
            result = reader.Read();
            Assert.AreEqual("1.2.3+Build4", result);

            reader = VersionDataFactory.Create("VersionXml/DummyVersion3.xml", ResoucesPath, true);
            result = reader.Read();
            Assert.AreEqual("1.2.3+Build4.5", result);

            reader = VersionDataFactory.Create("VersionXml/DummyVersion4.xml", ResoucesPath, true);
            result = reader.Read();
            Assert.AreEqual("1.2.3+testVer", result);

            reader = VersionDataFactory.Create("VersionXml/DummyVersion5.xml", ResoucesPath, true);
            result = reader.Read();
            Assert.AreEqual("1.2.3+Build4.5.testVer", result);

            reader = VersionDataFactory.Create("VersionXml/InvalidVersion.xml", ResoucesPath);
            TestUtil.ExpectException<ArgumentException>(() => reader.Read(), "Invalid file has specified.");
        }

        [TestMethod]
        public void ReplaceTest()
        {
            // 'R'を使ってない場合は、参照の値を取り出すデリゲートを呼び出さないことを確認する
            var actual =
                VersionReplace.Replace(
                    new[] { "4.3.2" },
                    () => throw new InvalidOperationException())
                    .ToArray();
            CollectionAssert.AreEqual(new[] { "4.3.2" }, actual);

            // 一通り 'R' を使ったパターン
            actual =
                VersionReplace.Replace(
                    new[] {
                        "R.R.0",
                        "R.R",
                        "R.R.*",
                        ">=R.R.R",
                        "R.R.R - 4.99.99",
                        "4.1.0-beta - R.R.R"
                    },
                    () => "4.3.2")
                    .ToArray();
            CollectionAssert.AreEqual(
                new[] {
                    "4.3.0",
                    "4.3",
                    "4.3.*",
                    ">=4.3.2",
                    "4.3.2 - 4.99.99",
                    "4.1.0-beta - 4.3.2"
                },
                actual);

            // 'R' の方が参照するバージョン値より多い場合
            actual =
                VersionReplace.Replace(
                    new[] { "R.R.R" },
                    () => "4.3")
                    .ToArray();
            CollectionAssert.AreEqual(new[] { "4.3" }, actual);

            actual =
                VersionReplace.Replace(
                    new[] { "R.R" },
                    () => "4")
                    .ToArray();
            CollectionAssert.AreEqual(new[] { "4" }, actual);

            // バージョンを ' ', '||' で複数記述した場合
            actual =
                VersionReplace.Replace(
                    new[] { ">=R.R.R R.R.R - 4.99.99 || R.R" },
                    () => "4.3.2")
                    .ToArray();
            CollectionAssert.AreEqual(new[] { ">=4.3.2 4.3.2 - 4.99.99 || 4.3" }, actual);

            // 末尾が'||'で終わっていたらエラー
            TestUtil.ExpectException<FormatException>(
                () => VersionReplace.Replace(new[] { "R.R.* ||" }, () => "4.3.2")
                    .ToArray(),
                "Invalid version format.");
        }

        [TestMethod]
        public void FailTest()
        {
            TestUtil.ExpectException<ArgumentException>(() => VersionDataFactory.Create("", ""));
        }
    }
}
