﻿using System;
using System.Text;
using System.Collections.Generic;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using ZarfCreator.Parser;
using ZarfCreator.ZarfDefinitionData;
using System.Collections;
using System.Linq;
using Nintendo.Zarf.v1;

namespace ZarfCreatorTest.ParserTest
{
    /// <summary>
    /// InstructionParseTest の概要の説明
    /// </summary>
    [TestClass]
    public class InstructionParseTest
    {
        private List<object> commandList;
        private List<InstructionCommandInfo> parsedCommandList;

        public InstructionParseTest()
        {
            var command1 = new Dictionary<string, object>()
            {
                { "Handler", "Internal-Parser" },
                { "CmdSpecifier", "unzip" },
                { "CmdArgs", new List<object>()
                    {
                        @"{fulldir-deploy-bin}\\${File1}"
                    }
                }
            };

            var command2 = new Dictionary<string, object>()
            {
                { "Handler", "Internal-Parser" },
                { "CmdSpecifier", "replacementCopy" },
                { "CmdArgs", new List<object>()
                    {
                        @"{fulldir-temp}\\NintendoSDK",
                        "{release-installdir}"
                    }
                }
            };

            var command3 = new Dictionary<string, object>()
            {
                { "Handler", "External-Exe" },
                { "CmdSpecifier", "msiexec" },
                { "CmdArgs", new List<object>()
                    {
                        "/i",
                        @"{fulldir-deploy-bin}\\Setup.msi",
                        "/qn",
                        "/promptrestart"
                    }
                },
                { "SuccessReturnCodes", new List<object>() { "0" }}
            };

            this.commandList = new List<object>() { command1, command2, command3 };
            this.parsedCommandList = new InstructionCommandParser().Parse(commandList).ToList();
        }

        [TestMethod]
        public void ParseSuccessTest1()
        {
            var parser = new InstructionParser();

            var data = new Dictionary<string, object>()
            {
                { "Install", commandList },
                { "Uninstall", commandList },
                { "InstallDirectory", "NintendoSDK" }
            };


            var result = parser.Parse(data);

            CollectionAssert.AreEquivalent(result.Install.ToList(), this.parsedCommandList);
            CollectionAssert.AreEquivalent(result.Uninstall.ToList(), this.parsedCommandList);
            CollectionAssert.AreEquivalent(result.Prerequisites.ToList(), new List<InstructionCommandInfo>());
            CollectionAssert.AreEquivalent(result.PostInstall.ToList(), new List<InstructionCommandInfo>());
            Assert.AreEqual(result.InstallDirectory, "NintendoSDK");

            var data2 = new Dictionary<string, object>()
            {
                { "Install", commandList },
                { "Uninstall", commandList },
                { "Prerequisites", commandList },
                { "InstallDirectory", @".\" },
                { "RollBack", "tryDelete" },
                { "LogicRequiresContentsFromBinFolder", "true" }
            };

            result = parser.Parse(data2);

            CollectionAssert.AreEquivalent(result.Install.ToList(), this.parsedCommandList);
            CollectionAssert.AreEquivalent(result.Uninstall.ToList(), this.parsedCommandList);
            CollectionAssert.AreEquivalent(result.Prerequisites.ToList(), this.parsedCommandList);
            Assert.AreEqual(result.InstallDirectory, @".\");
            Assert.AreEqual(result.RollBackType, Install.RollbackAction.TryDelete);
            Assert.AreEqual(result.RequireBinDir, true);

            var data3 = new Dictionary<string, object>()
            {
                { "Install", commandList },
                { "Uninstall", commandList },
                { "InstallDirectory", @".\" },
                { "RollBack", "tryUninstall" },
                { "PostInstall", commandList },
            };

            result = parser.Parse(data3);

            CollectionAssert.AreEquivalent(result.Install.ToList(), this.parsedCommandList);
            CollectionAssert.AreEquivalent(result.Uninstall.ToList(), this.parsedCommandList);
            CollectionAssert.AreEquivalent(result.PostInstall.ToList(), this.parsedCommandList);
            Assert.AreEqual(result.InstallDirectory, @".\");
            Assert.AreEqual(result.RollBackType, Install.RollbackAction.TryUninstall);
        }

        /// <summary>
        /// 必須要素が足りない場合
        /// </summary>
        [TestMethod]
        public void ParseFailByNotMeetRequirementsTest()
        {
            var parser = new InstructionParser();

            Console.Error.WriteLine("---");
            var data1 = new Dictionary<string, object>() { };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data1), "Required items have not been completed.");

            Console.Error.WriteLine("---");
            var data2 = new Dictionary<string, object>()
            {
                { "Uninstall", commandList },
                { "RollBack", "tryDelete" },
                { "InstallDirectory", "NintendoSDK" }
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data2), "Required items have not been completed.");

            Console.Error.WriteLine("---");
            var data3 = new Dictionary<string, object>()
            {
                { "Install", commandList },
                { "LogicRequiresContentsFromBinFolder", "true" },
                { "InstallDirectory", "NintendoSDK" }
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data3), "Required items have not been completed.");

            Console.Error.WriteLine("---");
            var data4 = new Dictionary<string, object>()
            {
                { "Install", commandList },
                { "Uninstall", commandList },
                { "Prerequisites", commandList },
                { "PostInstall", commandList },
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data4), "Required items have not been completed.");
        }

        [TestMethod]
        public void ParseFailAtRollBackTypeTest()
        {
            var parser = new InstructionParser();

            Console.Error.WriteLine("---");
            var data1 = new Dictionary<string, object>()
            {
                { "Install", commandList },
                { "Uninstall", commandList },
                { "InstallDirectory", @".\" },
                { "RollBack", "" },
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data1), "Invalid parameter in the 'Rollback' field.");

            Console.Error.WriteLine("---");
            var data2 = new Dictionary<string, object>()
            {
                { "PostInstall", commandList },
                { "Install", commandList },
                { "Uninstall", commandList },
                { "InstallDirectory", @".\" },
                { "RollBack", "TryDelete" },
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data2), "Invalid parameter in the 'Rollback' field.");
        }

        [TestMethod]
        public void ParseFailAtRequireBinDirTest()
        {
            var parser = new InstructionParser();

            Console.Error.WriteLine("---");
            var data1 = new Dictionary<string, object>()
            {
                { "Install", commandList },
                { "Uninstall", commandList },
                { "InstallDirectory", @".\" },
                { "LogicRequiresContentsFromBinFolder", "" },
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data1), "文字列が有効な Boolean 型として認識されませんでした。");

            Console.Error.WriteLine("---");
            var data2 = new Dictionary<string, object>()
            {
                { "Install", commandList },
                { "Uninstall", commandList },
                { "InstallDirectory", @".\" },
                { "LogicRequiresContentsFromBinFolder", "ture" },
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data2), "文字列が有効な Boolean 型として認識されませんでした。");
        }

        /// <summary>
        /// typo している場合
        /// </summary>
        [TestMethod]
        public void ParseFailByTypographicalErrorTest()
        {
            var parser = new InstructionParser();

            Console.Error.WriteLine("---");
            var data1 = new Dictionary<string, object>()
            {
                { "install", commandList },
                { "Uninstall", commandList },
                { "LogicRequiresContentsFromBinFolder", "" },
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data1), "Unknown key was specified.");

            Console.Error.WriteLine("---");
            var data2 = new Dictionary<string, object>()
            {
                { "Install", commandList },
                { "Uninstall", commandList },
                { "InstallDirectory", @".\" },
                { "Rollback", "tryDelete" },
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data2), "Unknown key was specified.");
        }
    }
}
