﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using VsSolutionLibrary;
using System.Collections.Generic;

namespace VsSolutionLibraryTest
{
    /// <summary>
    /// 空の VsSolution オブジェクトに操作を加えていき、
    /// 各段階でのオブジェクトの状態が正しいか検査するテストです。
    /// </summary>
    [TestClass]
    public class TypicalConstructionTest
    {
        private const string TextDefault =
@"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Global
EndGlobal
";
        private const string TextBeforeAddSolutionProperty = TextDefault;
        private const string TextAfterAddSolutionProperty = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Global
    GlobalSection(SolutionProperties) = preSolution
        HideSolutionNode = FALSE
    EndGlobalSection
EndGlobal
";
        private const string TextBeforeChangeVersion = TextAfterAddSolutionProperty;
        private const string TextAfterChangeVersion2013 = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2013
VisualStudioVersion = 12.0.40629.0
MinimumVisualStudioVersion = 10.0.40219.1
Global
    GlobalSection(SolutionProperties) = preSolution
        HideSolutionNode = FALSE
    EndGlobalSection
EndGlobal
";
        private const string TextAfterChangeVersion2015 = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 14
VisualStudioVersion = 14.0.25123.0
MinimumVisualStudioVersion = 10.0.40219.1
Global
    GlobalSection(SolutionProperties) = preSolution
        HideSolutionNode = FALSE
    EndGlobalSection
EndGlobal
";
        private const string TextAfterChangeVersion2017 = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 15
VisualStudioVersion = 15.0.26228.10
MinimumVisualStudioVersion = 10.0.40219.1
Global
    GlobalSection(SolutionProperties) = preSolution
        HideSolutionNode = FALSE
    EndGlobalSection
EndGlobal
";
        private const string TextBeforeAddSolutionConfiguration = TextAfterAddSolutionProperty;
        private const string TextAfterAddSolutionConfiguration = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Global
    GlobalSection(SolutionConfigurationPlatforms) = preSolution
        Debug|Siglo 0 = Debug|Siglo 0
        Debug|Siglo 1 = Debug|Siglo 1
        Develop|Siglo 0 = Develop|Siglo 0
        Develop|Siglo 1 = Develop|Siglo 1
        Release|Siglo 0 = Release|Siglo 0
        Release|Siglo 1 = Release|Siglo 1
    EndGlobalSection
    GlobalSection(SolutionProperties) = preSolution
        HideSolutionNode = FALSE
    EndGlobalSection
EndGlobal
";
        private const string TextBeforeRemoveSolutionConfiguration = TextAfterAddSolutionConfiguration;
        private const string TextAfterRemoveSolutionConfiguration = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Global
    GlobalSection(SolutionConfigurationPlatforms) = preSolution
        Debug|Siglo 0 = Debug|Siglo 0
        Debug|Siglo 1 = Debug|Siglo 1
        Release|Siglo 0 = Release|Siglo 0
        Release|Siglo 1 = Release|Siglo 1
    EndGlobalSection
    GlobalSection(SolutionProperties) = preSolution
        HideSolutionNode = FALSE
    EndGlobalSection
EndGlobal
";
        private const string TextBeforeAddProject = TextAfterRemoveSolutionConfiguration;
        private const string TextAfterAddProject = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Project(""{8BC9CEB8-8B4A-11D0-8D11-00A0C91BC942}"") = ""Project0"", ""Project0\Project0.vcxproj"", ""{11111111-1111-1111-1111-111111111111}""
EndProject
Project(""{FAE04EC0-301F-11D3-BF4B-00C04F79EFBC}"") = ""Project 1"", ""全角\Project 1.csproj"", ""{22222222-2222-2222-2222-222222222222}""
    ProjectSection(ProjectDependencies) = postProject
        {11111111-1111-1111-1111-111111111111} = {11111111-1111-1111-1111-111111111111}
    EndProjectSection
EndProject
Global
    GlobalSection(SolutionConfigurationPlatforms) = preSolution
        Debug|Siglo 0 = Debug|Siglo 0
        Debug|Siglo 1 = Debug|Siglo 1
        Release|Siglo 0 = Release|Siglo 0
        Release|Siglo 1 = Release|Siglo 1
    EndGlobalSection
    GlobalSection(ProjectConfigurationPlatforms) = postSolution
        {11111111-1111-1111-1111-111111111111}.Debug|Siglo 0.ActiveCfg = Debug|Siglo 0
        {11111111-1111-1111-1111-111111111111}.Debug|Siglo 1.ActiveCfg = Debug|Siglo 1
        {11111111-1111-1111-1111-111111111111}.Release|Siglo 0.ActiveCfg = Release|Siglo 0
        {11111111-1111-1111-1111-111111111111}.Release|Siglo 1.ActiveCfg = Release|Siglo 1
        {22222222-2222-2222-2222-222222222222}.Debug|Siglo 0.ActiveCfg = Debug|Siglo 0
        {22222222-2222-2222-2222-222222222222}.Debug|Siglo 1.ActiveCfg = Debug|Siglo 1
        {22222222-2222-2222-2222-222222222222}.Release|Siglo 0.ActiveCfg = Release|Siglo 0
        {22222222-2222-2222-2222-222222222222}.Release|Siglo 1.ActiveCfg = Release|Siglo 1
    EndGlobalSection
    GlobalSection(SolutionProperties) = preSolution
        HideSolutionNode = FALSE
    EndGlobalSection
EndGlobal
";
        private const string TextBeforeRemoveProject = TextAfterAddProject;
        private const string TextAfterRemoveProject = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Project(""{8BC9CEB8-8B4A-11D0-8D11-00A0C91BC942}"") = ""Project0"", ""Project0\Project0.vcxproj"", ""{11111111-1111-1111-1111-111111111111}""
EndProject
Global
    GlobalSection(SolutionConfigurationPlatforms) = preSolution
        Debug|Siglo 0 = Debug|Siglo 0
        Debug|Siglo 1 = Debug|Siglo 1
        Release|Siglo 0 = Release|Siglo 0
        Release|Siglo 1 = Release|Siglo 1
    EndGlobalSection
    GlobalSection(ProjectConfigurationPlatforms) = postSolution
        {11111111-1111-1111-1111-111111111111}.Debug|Siglo 0.ActiveCfg = Debug|Siglo 0
        {11111111-1111-1111-1111-111111111111}.Debug|Siglo 1.ActiveCfg = Debug|Siglo 1
        {11111111-1111-1111-1111-111111111111}.Release|Siglo 0.ActiveCfg = Release|Siglo 0
        {11111111-1111-1111-1111-111111111111}.Release|Siglo 1.ActiveCfg = Release|Siglo 1
    EndGlobalSection
    GlobalSection(SolutionProperties) = preSolution
        HideSolutionNode = FALSE
    EndGlobalSection
EndGlobal
";
        private const string TextBeforeChangeProjectProperty = TextAfterRemoveProject;
        private const string TextAfterChangeProjectProperty = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Project(""{8BC9CEB8-8B4A-11D0-8D11-00A0C91BC942}"") = ""NewProject"", ""NewProject\NewProject.vcxproj"", ""{11111111-1111-1111-1111-111111111111}""
EndProject
Global
    GlobalSection(SolutionConfigurationPlatforms) = preSolution
        Debug|Siglo 0 = Debug|Siglo 0
        Debug|Siglo 1 = Debug|Siglo 1
        Release|Siglo 0 = Release|Siglo 0
        Release|Siglo 1 = Release|Siglo 1
    EndGlobalSection
    GlobalSection(ProjectConfigurationPlatforms) = postSolution
        {11111111-1111-1111-1111-111111111111}.Debug|Siglo 0.ActiveCfg = Debug|Siglo 0
        {11111111-1111-1111-1111-111111111111}.Debug|Siglo 1.ActiveCfg = Debug|Siglo 1
        {11111111-1111-1111-1111-111111111111}.Release|Siglo 0.ActiveCfg = Release|Siglo 0
        {11111111-1111-1111-1111-111111111111}.Release|Siglo 1.ActiveCfg = Release|Siglo 1
    EndGlobalSection
    GlobalSection(SolutionProperties) = preSolution
        HideSolutionNode = FALSE
    EndGlobalSection
EndGlobal
";
        private const string TextBeforeChangeProjectConfiguration = TextAfterChangeProjectProperty;
        private const string TextAfterChangeProjectConfiguration = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Project(""{8BC9CEB8-8B4A-11D0-8D11-00A0C91BC942}"") = ""NewProject"", ""NewProject\NewProject.vcxproj"", ""{11111111-1111-1111-1111-111111111111}""
EndProject
Global
    GlobalSection(SolutionConfigurationPlatforms) = preSolution
        Debug|Siglo 0 = Debug|Siglo 0
        Debug|Siglo 1 = Debug|Siglo 1
        Release|Siglo 0 = Release|Siglo 0
        Release|Siglo 1 = Release|Siglo 1
    EndGlobalSection
    GlobalSection(ProjectConfigurationPlatforms) = postSolution
        {11111111-1111-1111-1111-111111111111}.Debug|Siglo 0.ActiveCfg = Debug|Siglo 0
        {11111111-1111-1111-1111-111111111111}.Debug|Siglo 1.ActiveCfg = Debug|Siglo 1
        {11111111-1111-1111-1111-111111111111}.Release|Siglo 0.ActiveCfg = Debug|Siglo 0
        {11111111-1111-1111-1111-111111111111}.Release|Siglo 0.Build.0 = Debug|Siglo 0
        {11111111-1111-1111-1111-111111111111}.Release|Siglo 1.ActiveCfg = Release|Siglo 1
    EndGlobalSection
    GlobalSection(SolutionProperties) = preSolution
        HideSolutionNode = FALSE
    EndGlobalSection
EndGlobal
";

        private VsSolutionLibrary.VsSolution m_Solution;

        /// <summary>
        /// ソリューション・プロパティの取得と変更
        /// </summary>
        private void ChangeSolutionProperty()
        {
            Assert.IsTrue(TestUtility.CompareSolutionString(TextBeforeAddSolutionProperty, m_Solution.ToString()));

            IDictionary<string, string> solutionProperties = m_Solution.SolutionProperties;
            solutionProperties.Add("HideSolutionNode", "FALSE");
            m_Solution.SolutionProperties = solutionProperties;
            Assert.IsTrue(TestUtility.CompareAsSet(solutionProperties, m_Solution.SolutionProperties));

            Assert.IsTrue(TestUtility.CompareSolutionString(TextAfterAddSolutionProperty, m_Solution.ToString()));
        }

        /// <summary>
        /// バージョンの変更
        /// </summary>
        private void ChangeVersion()
        {
            Assert.IsTrue(TestUtility.CompareSolutionString(TextBeforeChangeVersion, m_Solution.ToString()));
            Assert.AreEqual(VsSolutionVersion.VisualStudio2012, m_Solution.Version);

            m_Solution.Version = VsSolutionVersion.VisualStudio2013;

            Assert.IsTrue(TestUtility.CompareSolutionString(TextAfterChangeVersion2013, m_Solution.ToString()));
            Assert.AreEqual(VsSolutionVersion.VisualStudio2013, m_Solution.Version);

            m_Solution.Version = VsSolutionVersion.VisualStudio2015;

            Assert.IsTrue(TestUtility.CompareSolutionString(TextAfterChangeVersion2015, m_Solution.ToString()));
            Assert.AreEqual(VsSolutionVersion.VisualStudio2015, m_Solution.Version);

            m_Solution.Version = VsSolutionVersion.VisualStudio2017;

            Assert.IsTrue(TestUtility.CompareSolutionString(TextAfterChangeVersion2017, m_Solution.ToString()));
            Assert.AreEqual(VsSolutionVersion.VisualStudio2017, m_Solution.Version);

            m_Solution.Version = VsSolutionVersion.VisualStudio2012;

            Assert.IsTrue(TestUtility.CompareSolutionString(TextBeforeChangeVersion, m_Solution.ToString()));
            Assert.AreEqual(VsSolutionVersion.VisualStudio2012, m_Solution.Version);
        }

        /// <summary>
        /// ソリューション構成の追加と削除
        /// </summary>
        private void ChangeSolutionConfiguration()
        {
            List<VsConfigurationPair> expected = new List<VsConfigurationPair>();

            // ソリューション構成の追加
            Assert.IsTrue(TestUtility.CompareSolutionString(TextBeforeAddSolutionConfiguration, m_Solution.ToString()));
            foreach (string platform in new string[] { "Siglo 0", "Siglo 1" })
            {
                foreach (string configuration in new string[] { "Debug", "Develop", "Release" })
                {
                    VsConfigurationPair pair = new VsConfigurationPair(configuration, platform);

                    m_Solution.AddSolutionConfiguration(pair);
                    expected.Add(pair);
                }
            }
            Assert.IsTrue(TestUtility.CompareSolutionString(TextAfterAddSolutionConfiguration, m_Solution.ToString()));

            Assert.IsTrue(TestUtility.CompareAsSet(expected, m_Solution.Configurations));

            // ソリューション構成の削除
            Assert.IsTrue(TestUtility.CompareSolutionString(TextBeforeRemoveSolutionConfiguration, m_Solution.ToString()));
            foreach (string platform in new string[] { "Siglo 0", "Siglo 1" })
            {
                VsConfigurationPair pair = new VsConfigurationPair("Develop", platform);

                m_Solution.RemoveSolutionConfiguration(pair);
                expected.Remove(pair);
            }
            Assert.IsTrue(TestUtility.CompareSolutionString(TextAfterRemoveSolutionConfiguration, m_Solution.ToString()));

            Assert.IsTrue(TestUtility.CompareAsSet(expected, m_Solution.Configurations));
        }

        private void ChangeProject()
        {
            List<Guid> projects = new List<Guid>();
            projects.Add(new Guid("{11111111-1111-1111-1111-111111111111}"));
            projects.Add(new Guid("{22222222-2222-2222-2222-222222222222}"));

            VsSolutionProjectProperty[] properties = new VsSolutionProjectProperty[] {
                new VsSolutionProjectProperty(VsProjectType.VisualC, "Project0", "Project0\\Project0.vcxproj", null),
                new VsSolutionProjectProperty(VsProjectType.CSharp, "Project 1", "全角\\Project 1.csproj", new Guid[] { projects[0] })
            };

            // プロジェクトの追加
            Assert.IsTrue(TestUtility.CompareSolutionString(TextBeforeAddProject, m_Solution.ToString()));

            m_Solution.AddProject(projects[0], properties[0]);
            m_Solution.AddProject(projects[1], properties[1]);

            Assert.IsTrue(TestUtility.CompareAsSet(projects, m_Solution.Projects));
            Assert.IsTrue(TestUtility.CompareSolutionString(TextAfterAddProject, m_Solution.ToString()));

            // プロジェクトの削除
            Assert.IsTrue(TestUtility.CompareSolutionString(TextBeforeRemoveProject, m_Solution.ToString()));

            m_Solution.RemoveProject(projects[1]);
            projects.RemoveAt(1);

            Assert.IsTrue(TestUtility.CompareAsSet(projects, m_Solution.Projects));
            Assert.IsTrue(TestUtility.CompareSolutionString(TextAfterRemoveProject, m_Solution.ToString()));
        }

        /// <summary>
        /// プロジェクト・プロパティの取得と変更
        /// </summary>
        private void ChangeProjectProperty()
        {
            Assert.IsTrue(TestUtility.CompareSolutionString(TextBeforeChangeProjectProperty, m_Solution.ToString()));

            Guid project = new Guid("{11111111-1111-1111-1111-111111111111}");

            VsSolutionProjectProperty oldProperty = m_Solution.GetProjectProperty(project);
            VsSolutionProjectProperty newProperty = new VsSolutionProjectProperty(oldProperty.ProjectType, "NewProject", "NewProject\\NewProject.vcxproj", oldProperty.Dependencies);
            m_Solution.SetProjectProperty(project, newProperty);

            Assert.AreEqual(newProperty, m_Solution.GetProjectProperty(project));

            Assert.IsTrue(TestUtility.CompareSolutionString(TextAfterChangeProjectProperty, m_Solution.ToString()));
        }

        /// <summary>
        /// プロジェクト構成の取得と変更
        /// </summary>
        private void ChangeProjectConfiguration()
        {
            Assert.IsTrue(TestUtility.CompareSolutionString(TextBeforeChangeProjectConfiguration, m_Solution.ToString()));

            Guid project = new Guid("{11111111-1111-1111-1111-111111111111}");

            VsConfigurationPair solutionConfiguration = new VsConfigurationPair("Release", "Siglo 0");

            VsProjectConfiguration projectConfiguration = new VsProjectConfiguration(new VsConfigurationPair("Debug", "Siglo 0"), true);
            m_Solution.SetProjectConfiguration(solutionConfiguration, project, projectConfiguration);

            Assert.AreEqual(projectConfiguration, m_Solution.GetProjectConfiguration(solutionConfiguration, project));

            Assert.IsTrue(TestUtility.CompareSolutionString(TextAfterChangeProjectConfiguration, m_Solution.ToString()));
        }

        [TestMethod]
        public void TestConstruction()
        {
            m_Solution = new VsSolutionLibrary.VsSolution();
            Assert.IsTrue(TestUtility.CompareSolutionString(m_Solution.ToString(), TextDefault));

            ChangeSolutionProperty();

            ChangeVersion();

            ChangeSolutionConfiguration();

            ChangeProject();

            ChangeProjectProperty();

            ChangeProjectConfiguration();
        }
    }
}
