﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.IO;
using System.Diagnostics;
using System.Net.Sockets;

namespace UpdaterTest
{
    class RunningOptions
    {
        public int SuccessTimeout { get; set; }
        public int FailureTimeout { get; set; }
        public string SuccessPattern { get; set; }
        public string FailurePattern { get; set; }
        public bool IgnoreApplicationTerminate { get; set; }
        public bool WithReset { get; set; }
        public bool NoWait { get; set; }
        public RunningOptions()
        {
            SuccessTimeout = 0;
            FailureTimeout = 100;
            SuccessPattern = null;
            FailurePattern = null;
            IgnoreApplicationTerminate = false;
            WithReset = false;
            NoWait = false;
        }
        public string GetRunOnTargetArgument()
        {
            var list = new List<string>();
            if(WithReset)
            {
                list.Add("--reset");
            }
            if (SuccessTimeout > 0)
            {
                list.Add(string.Format("--success-timeout {0}", SuccessTimeout));
            }
            if (FailureTimeout > 0)
            {
                list.Add(string.Format("--failure-timeout {0}", FailureTimeout));
            }
            if (SuccessPattern != null)
            {
                list.Add(string.Format("--pattern-success-exit {0}", SuccessPattern));
            }
            if (FailurePattern != null)
            {
                list.Add(string.Format("--pattern-failure-exit {0}", FailurePattern));
            }
            if(NoWait == true)
            {
                list.Add("--no-wait");
            }

            return String.Join(" ", list);
        }
        public string GetRunOnTargetPrivateArgument()
        {
            var list = new List<string>();
            list.Add(GetRunOnTargetArgument());

            if(IgnoreApplicationTerminate)
            {
                list.Add(" --suppress-polling-process");
            }
            return String.Join(" ", list);
        }
    }

    // 実機上でのテストを補助するライブラリ
    // とりあえず UpdateTest で抱える
    class TestOnTargetLibrary
    {
        private TestUtility.TestPath m_TestPath = null;
        private string m_RunOnTarget = null;
        private string m_RunOnTargetPrivate = null;
        private string m_ControlTarget = null;
        private string m_ControlTargetPrivate = null;

        private string m_DevMenuCommand = null;
        private string m_DevMenuCommandSystem = null;

        private TcpClient m_Client = null;
        private Task m_UartTask = null;
        private StringBuilder m_UartStringBuilder = null;

        // 別の種類を使いたい場合、ここを修正する
        private string m_KeyType = "K5";
        private string m_SignedType = "Unsigned";
        private string m_BuildType = "Develop";

        // 別の場所でビルド済みの BootImagePackage を使いたい場合
        private string m_CiBuiltRootPath = null;

        public TestOnTargetLibrary(TestContext testContext)
        {
            m_TestPath = new TestUtility.TestPath(testContext);
            m_RunOnTarget = Path.Combine(m_TestPath.GetSigloRoot(), "Tools/CommandLineTools/RunOnTarget.exe");
            m_RunOnTargetPrivate = Path.Combine(m_TestPath.GetSigloRoot(), "Tools/CommandLineTools/RunOnTargetPrivate.exe");
            m_ControlTarget = Path.Combine(m_TestPath.GetSigloRoot(), "Tools/CommandLineTools/ControlTarget.exe");
            m_ControlTargetPrivate = Path.Combine(m_TestPath.GetSigloRoot(), "Tools/CommandLineTools/ControlTargetPrivate.exe");

            m_DevMenuCommand = Path.Combine(m_TestPath.GetSigloRoot(), "Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommand/Develop/DevMenuCommand.nca");
            m_DevMenuCommandSystem = Path.Combine(m_TestPath.GetSigloRoot(), "Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nca");

            // CI から取得した NintendoSDK_SetupIntegratedDebugKit-ForNX の展開パスを、必要なら設定する
            // m_CiBuiltRootPath = Path.Combine(m_TestPath.GetSigloRoot(), "prebuilt/NintendoSDK-516");
        }
        private string ConvertParameters(string plain)
        {
            // KeyType, SignedType, BuildType を置換する
            return plain.Replace("$KEY", m_KeyType)
                        .Replace("$SIGNED", m_SignedType)
                        .Replace("$BUILD", m_BuildType);
        }
        public string GetPath(string relativePath)
        {
            return ConvertParameters(Path.Combine(m_TestPath.GetSigloRoot(), relativePath));
        }
        public string GetBipPath(string bipType)
        {
            var relativePath = string.Format("Programs/Eris/Outputs/NX-NXFP2-a64/SystemData/{0}-$KEY-$SIGNED/$BUILD/{0}-$KEY-$SIGNED.nsp", bipType);
            if(m_CiBuiltRootPath != null)
            {
                return ConvertParameters(Path.Combine(m_CiBuiltRootPath, relativePath));
            }
            else
            {
                return ConvertParameters(Path.Combine(m_TestPath.GetSigloRoot(), relativePath));
            }
        }

        public Tuple<String, String> ExecuteCommand(string filename, string arguments)
        {
            using (var process = new Process())
            {
                var standardOutput = new StringBuilder();
                var standardError = new StringBuilder();

                process.StartInfo.FileName = filename;
                process.StartInfo.Arguments = arguments;
                process.StartInfo.CreateNoWindow = true;
                process.StartInfo.UseShellExecute = false;
                process.StartInfo.RedirectStandardOutput = true;
                process.StartInfo.RedirectStandardError = true;
                process.StartInfo.StandardOutputEncoding = System.Text.Encoding.UTF8;
                process.OutputDataReceived += (object sender, DataReceivedEventArgs e) =>
                {
                    standardOutput.AppendLine(e.Data);
                };
                process.ErrorDataReceived += (object sender, DataReceivedEventArgs e) =>
                {
                    standardError.AppendLine(e.Data);
                };
                process.Start();
                process.BeginOutputReadLine();
                process.BeginErrorReadLine();
                process.WaitForExit();

                if(process.ExitCode != 0)
                {
                    throw new Exception(string.Format("{0} failed: {1}", filename, process.ExitCode));
                }

                return new Tuple<String, String>(standardOutput.ToString(), standardError.ToString());
            }
        }

        // 実機の IP を取得する
        private string GetIpAddressOfTarget()
        {
            var log = ExecuteCommand(m_ControlTarget, "get-default");
            return log.Item1.Trim().Split('\t').Last();
        }
        private TcpClient OpenUartLog()
        {
            var ip = GetIpAddressOfTarget();
            return  new TcpClient(ip, 10023);
        }

        private void StartReadUart()
        {
            m_Client = OpenUartLog();
            m_UartStringBuilder = new StringBuilder();
            m_UartTask = System.Threading.Tasks.Task.Run(() =>
            {
                using (var stream = m_Client.GetStream())
                using (var reader = new StreamReader(stream))
                {
                    try
                    {
                        while (m_Client.Connected)
                        {
                            var l = reader.ReadLine();
                            m_UartStringBuilder.AppendLine(l);
                        }
                    }
                    catch (IOException)
                    {
                        // nothing to do
                    }
                }
            });
        }
        private void StopReadUart()
        {
            System.Threading.Thread.Sleep(100);
            m_Client.Close();
            System.Threading.Thread.Sleep(100);
        }

        public TargetLogTester Run(string nca, string argument, RunningOptions options = null)
        {
            if (options == null)
            {
                options = new RunningOptions();
            }
            StartReadUart();
            var outputs = ExecuteCommand(m_RunOnTargetPrivate, string.Format("run {0} {1} -- {2}", nca, options.GetRunOnTargetPrivateArgument(), argument));
            StopReadUart();

            return new TargetLogTester(new Tuple<string, string>(outputs.Item1, m_UartStringBuilder.ToString()), m_BuildType);
        }

        // 出力と、UART ログとを返す
        public TargetLogTester RunDevMenuCommand(string argument, RunningOptions options = null)
        {
            return Run(m_DevMenuCommand, argument, options);
        }
        public TargetLogTester RunDevMenuCommandSystem(string argument, RunningOptions options = null)
        {
            return Run(m_DevMenuCommandSystem, argument, options);
        }
        public TargetLogTester WriteToCard(string app, string upp)
        {
            // 安定しないので、3sec ほど待つ
            System.Threading.Thread.Sleep(3000);
            return Run(m_DevMenuCommand, string.Format("gamecard write {0} --update-partition {1}", app, upp));
        }
    }
}
