﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.Serialization.Formatters.Binary;
using System.Runtime.InteropServices;
using System.Diagnostics;

namespace UartTestHost
{
    /// <summary>
    /// ターゲットとのハンドシェイク機能の実装クラス
    /// </summary>
    public class TestHandshake
    {
        /// <summary>
        /// ACK に使用するデータのサイズ [bytes]
        /// </summary>
        private const int AckSize = 4;

        /// <summary>
        /// ACK を送信します。
        /// </summary>
        /// <param name="stream">オープン済のシリアルストリーム</param>
        /// <returns>処理結果</returns>
        private static TestResult SendAck(SerialStream stream)
        {
            byte[] signiture = new byte[AckSize];
            signiture[0] = Convert.ToByte('A');
            signiture[1] = Convert.ToByte('B');
            signiture[2] = Convert.ToByte('C');
            signiture[3] = Convert.ToByte('D');
            Console.WriteLine("Sending Ack");
            return stream.Send(signiture, AckSize);
        }

        /// <summary>
        /// ACK をタイムアウトつきで受信します。
        /// </summary>
        /// <param name="stream">オープン済のシリアルストリーム</param>
        /// <param name="timeoutMsec">タイムアウト時間</param>
        /// <returns>処理結果</returns>
        private static TestResult WaitAck(SerialStream stream, int timeoutMsec)
        {
            Console.WriteLine("Waiting Ack");

            byte[] receivedAck;
            TestResult result = stream.ReceiveSync(out receivedAck, AckSize, timeoutMsec);
            if (result != TestResult.Success)
            {
                return result;
            }
            else if (receivedAck[0] != Convert.ToByte('A') ||
                        receivedAck[1] != Convert.ToByte('B') ||
                        receivedAck[2] != Convert.ToByte('C') ||
                        receivedAck[3] != Convert.ToByte('D'))
            {
                Console.WriteLine("Handshake failed. receivedAck=" + receivedAck[0] + " " + receivedAck[1] +
                                    " " + receivedAck[2] + " " + receivedAck[3]);
                return TestResult.DataCorrupted;
            }
            Console.WriteLine("ACK Received. receivedAck=" + receivedAck[0] + " " + receivedAck[1] +
                                " " + receivedAck[2] + " " + receivedAck[3]);
            return TestResult.Success;
        }

        /// <summary>
        /// ACK を交換し、PC とターゲット間で同期を取ります。
        /// 先にターゲットからの ACK を待機し、ACK を確認後応答 ACK を返します。
        /// </summary>
        /// <param name="stream">オープン済のシリアルストリーム</param>
        /// <returns>処理結果</returns>
        private static TestResult SyncSignature(SerialStream stream)
        {
            const int SignitureTimeoutMsec = 600000; // 10 min
            const int SignitureWaitInterval = 2000;
            int signitureWaitedTimeMsec = 0;
            while (signitureWaitedTimeMsec < SignitureTimeoutMsec)
            {
                TestResult result;

                // ターゲット側でのほかのテスト実行でゴミを受信している可能性があるので破棄
                stream.DiscardInBuffer();

                // ACK 受信待ち
                result = WaitAck(stream, SignitureWaitInterval);
                if (result == TestResult.TargetNotResponding)
                {
                    signitureWaitedTimeMsec += SignitureWaitInterval;
                    continue;
                }
                else if (result != TestResult.Success)
                {
                    continue;
                }

                // ACK 送信
                result = SendAck(stream);
                if (result != TestResult.Success)
                {
                    return result;
                }
                return result;
            }

            // TIMEOUT
            return TestResult.TargetNotResponding;
        }

        /// <summary>
        /// ターゲットとのハンドシェイクを行います。
        /// ハンドシェイク成功時にテストパラメータが得られます。
        /// </summary>
        /// <param name="testParam">受信するテストパラメータ</param>
        /// <returns>処理結果</returns>
        public static TestResult HandshakeWithTarget(out TestParam testParam)
        {
            TestResult result;
            testParam = new TestParam(TestScenario.TestTermination, TestBaudRateType.Baud_57600, TestFlowControlMode.FlowControlMode_None);
            SerialStream stream = new SerialStream();
            SerialConfig handshakeConfig = new SerialConfig(testParam);

            result = stream.Open(handshakeConfig);
            if (result != TestResult.Success)
            {
                return result;
            }

            Console.WriteLine("Handshake - Sync");
            result = SyncSignature(stream);
            if (result != TestResult.Success)
            {
                stream.Close();
                return result;
            }

            // シナリオ番号やテストパラメータを含んだパケットを受信
            Console.WriteLine("Handshake - Receiving Test Parameters");
            int size = Marshal.SizeOf(testParam);
            Trace.Assert(size == 4); // 暗黙のパディングが入っていないかチェック

            byte[] receivedData;
            const int TimeoutMsec = 1000;
            result = stream.ReceiveSync(out receivedData, size, TimeoutMsec);
            if (result != TestResult.Success)
            {
                stream.Close();
                return result;
            }

            // パケットを構造体として復元（各フィールドの値域チェックを兼ねる）
            if (!TestParamUtil.ToTestParam(out testParam, receivedData, size))
            {
                stream.Close();
                return TestResult.DataCorrupted;
            }

            // ACK 送信
            result = SendAck(stream);
            if (result != TestResult.Success)
            {
                stream.Close();
                return result;
            }

            Console.WriteLine("Handshake - Done");

            result = stream.Close();
            if (result != TestResult.Success)
            {
                return result;
            }

            return TestResult.Success;
        }
    }
}
