﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TargetShellTest
{
    using System;
    using System.Collections.Generic;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using TargetShell;
    using TargetShell.PluginInterface;

    /// <summary>
    /// SearchLib のテスト
    /// </summary>
    [TestClass]
    public class SearchLibTest
    {
        /// <summary>
        /// 検索フラグ
        /// </summary>
        private readonly bool detectDevice = true;
        /// <summary>
        /// ダミーのデバイスプラグイン情報
        /// </summary>
        private List<DevicePluginInfo> dummyDevicePluginInfo;
        /// <summary>
        /// テストに使用する SearchLib を保持する
        /// </summary>
        private SearchLib searchLib;

        /// <summary>
        /// テストの前準備を行う
        /// </summary>
        [TestInitialize]
        public void TestInitialize()
        {
            SetNintendoSdkRootPath();
            this.searchLib = new SearchLib();
            this.dummyDevicePluginInfo = new List<DevicePluginInfo>();
            this.dummyDevicePluginInfo = GetDummyDevicePluginInfo();
            // 接続されているデバイスをすべて対象とする
            var preSearchCondition = new SearchCondition
            {
                TargetEntry = new SearchTargetEntry(),
                DetectDevice = this.detectDevice,
                DevicePluginInfo = this.dummyDevicePluginInfo,
            };
            this.searchLib.SetSearchCondition(preSearchCondition);
        }

        /// <summary>
        /// テストの終了処理を行う
        /// </summary>
        [TestCleanup]
        public void TestCleanup()
        {
            this.searchLib = null;
        }

        /// <summary>
        /// フィルタリングなしで検索するテスト。
        /// </summary>
        [TestMethod]
        public void Search()
        {
            Assert.AreEqual(this.searchLib.Run(), Result.Success);
        }

        /// <summary>
        /// IPアドレスのフィルタリングありで検索するテスト
        /// </summary>
        [TestMethod]
        public void SearchFilterByIPAddress()
        {
            var searchCondition = PrepareSearchCondition();
            searchCondition.TargetEntry.IpAddress.Add("169.254.9.247");

            this.searchLib.SetSearchCondition(searchCondition);
            Assert.AreEqual(this.searchLib.Run(), Result.Success);
        }

        /// <summary>
        /// デバイス名のフィルタリングありで検索するテスト
        /// </summary>
        [TestMethod]
        public void SearchFilterByDeviceName()
        {
            var searchCondition = PrepareSearchCondition();
            searchCondition.TargetEntry.DeviceName = "XAL02100026706";

            this.searchLib.SetSearchCondition(searchCondition);
            Assert.AreEqual(this.searchLib.Run(), Result.Success);
        }

        /// <summary>
        /// シリアル番号のフィルタリングありで検索するテスト
        /// </summary>
        [TestMethod]
        public void SearchFilterBySerialNumber()
        {
            var searchCondition = PrepareSearchCondition();
            searchCondition.TargetEntry.SerialNumber = "XAL02100026706";

            this.searchLib.SetSearchCondition(searchCondition);
            Assert.AreEqual(this.searchLib.Run(), Result.Success);
        }

        /// <summary>
        /// ハードウェアタイプのフィルタリングなしで検索するテスト
        /// </summary>
        [TestMethod]
        public void SearchFilterByHardwareTypeNotSpecified()
        {
            var searchCondition = PrepareSearchCondition();
            searchCondition.TargetEntry.HardwareType = "NotSpecified";

            this.searchLib.SetSearchCondition(searchCondition);
            Assert.AreEqual(this.searchLib.Run(), Result.Success);
        }

        /// <summary>
        /// 個別にハードウェアタイプを限定してフィルタリングありで検索するテスト
        /// </summary>
        [TestMethod]
        public void SearchFilterByHardwareType()
        {
            var hardwareTypeList = new List<string>
            {
                "Sdev", "Edev"
            };
            foreach (var hardwareType in hardwareTypeList)
            {
                var searchCondition = PrepareSearchCondition();
                searchCondition.TargetEntry.HardwareType = hardwareType;

                this.searchLib.SetSearchCondition(searchCondition);
                Assert.AreEqual(this.searchLib.Run(), Result.Success);
            }
        }

        /// <summary>
        /// ハードウェアコンフィグのフィルタリングなしで検索するテスト
        /// </summary>
        [TestMethod]
        public void SearchFilterByHardwareConfigNONE()
        {
            var searchCondition = PrepareSearchCondition();
            searchCondition.TargetEntry.HardwareConfig = string.Empty;

            this.searchLib.SetSearchCondition(searchCondition);
            Assert.AreEqual(this.searchLib.Run(), Result.Success);
        }

        /// <summary>
        /// ハードウェアコンフィグを限定してフィルタリングありで検索するテスト
        /// </summary>
        [TestMethod]
        public void SearchFilterByHardwareConfig()
        {
            var HardwareConfigList = new List<string>
            {
                "SDEV_00_01_08", "SDEV_00_02_00_00"
            };
            foreach (string hardwareConfig in HardwareConfigList)
            {
                var searchCondition = PrepareSearchCondition();
                searchCondition.TargetEntry.HardwareConfig = hardwareConfig;

                this.searchLib.SetSearchCondition(searchCondition);
                Assert.AreEqual(this.searchLib.Run(), Result.Success);
            }
        }

        /// <summary>
        /// コネクションパスのフィルタリングなしで検索するテスト
        /// </summary>
        [TestMethod]
        public void SearchFilterByConnectionPathNotSpecified()
        {
            var searchCondition = PrepareSearchCondition();
            searchCondition.TargetEntry.ConnectionPath = ConnectionPath.NotSpecified;

            this.searchLib.SetSearchCondition(searchCondition);
            Assert.AreEqual(this.searchLib.Run(), Result.Success);
        }

        /// <summary>
        /// コネクションパスを限定してフィルタリングありで検索するテスト
        /// </summary>
        [TestMethod]
        public void SearchFilterByConnectionPath()
        {
            foreach (ConnectionPath connectionPath in Enum.GetValues(typeof(ConnectionPath)))
            {
                if (connectionPath == ConnectionPath.NotSpecified)
                {
                    continue;
                }
                var searchCondition = PrepareSearchCondition();
                searchCondition.TargetEntry.ConnectionPath = connectionPath;

                this.searchLib.SetSearchCondition(searchCondition);
                Assert.AreEqual(this.searchLib.Run(), Result.Success);
            }
        }

        /// <summary>
        /// ワイルドカードのIPアドレスのフィルタリングありで検索するテスト
        /// </summary>
        [TestMethod]
        public void SearchFilterByIPAddressWithWildCard()
        {
            var searchCondition = PrepareSearchCondition();
            searchCondition.TargetEntry.IpAddress.Add("169.*");

            this.searchLib.SetSearchCondition(searchCondition);
            Assert.AreEqual(this.searchLib.Run(), Result.Success);
        }

        /// <summary>
        /// ワイルドカードのデバイス名のフィルタリングありで検索するテスト
        /// </summary>
        [TestMethod]
        public void SearchFilterByDeviceNameWithWildCard()
        {
            var searchCondition = PrepareSearchCondition();
            searchCondition.TargetEntry.DeviceName = "*";

            this.searchLib.SetSearchCondition(searchCondition);
            Assert.AreEqual(this.searchLib.Run(), Result.Success);
        }

        /// <summary>
        /// ワイルドカードのシリアル番号のフィルタリングありで検索するテスト
        /// </summary>
        [TestMethod]
        public void SearchFilterBySerialNumberWithWildCard()
        {
            var searchCondition = PrepareSearchCondition();
            searchCondition.TargetEntry.SerialNumber = "*";

            this.searchLib.SetSearchCondition(searchCondition);
            Assert.AreEqual(this.searchLib.Run(), Result.Success);
        }

        /// <summary>
        /// デバイスプラグインのダミー情報
        /// </summary>
        private List<DevicePluginInfo> GetDummyDevicePluginInfo()
        {
            var devicePluginInfo = new List<DevicePluginInfo>();
            var tempSdevDevicePluginInfo = new DevicePluginInfo
            {
                SearchDeviceKeyword = "SDEV",
                SupportHardwareType = "Sdev"
            };
            devicePluginInfo.Add(tempSdevDevicePluginInfo);
            var tempEdevDevicePluginInfo = new DevicePluginInfo
            {
                SearchDeviceKeyword = "EDEV",
                SupportHardwareType = "Edev"
            };
            devicePluginInfo.Add(tempEdevDevicePluginInfo);
            return devicePluginInfo;
        }

        /// <summary>
        /// SDKのルートパス設定
        /// </summary>
        private void SetNintendoSdkRootPath()
        {
            // UtilityにSDKルートを設定
            Utility.NintendoSdkRootPath = Utility.GetNintendoSdkRootPath();
        }

        /// <summary>
        /// 検索条件の準備
        /// </summary>
        /// <returns>SearchCondition</returns>
        private SearchCondition PrepareSearchCondition()
        {
            var searchCondition = new SearchCondition
            {
                TargetEntry = new SearchTargetEntry(),
                DetectDevice = this.detectDevice,
                DevicePluginInfo = this.dummyDevicePluginInfo,
            };
            return searchCondition;
        }
    }
}
