﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using Nintendo.ServiceFramework;
using Nintendo.ServiceFramework.Validation;
using Nintendo.ServiceFramework.Hipc;

namespace Nintendo.ServiceFrameworkTest.Validation
{
    [TestClass]
    public class InterfaceValidatorTest
    {
        /// <summary>
        /// AssertInvalidInterface の格納用属性
        /// </summary>
        [AttributeUsage(AttributeTargets.All)]
        private class AssertInvalidInterfaceAttribute : Attribute
        {
            public string ErrorCode { get; private set; }
            public string[] AdditionalStrings { get; private set; }

            public AssertInvalidInterfaceAttribute(string errorCode, params object[] additionalStrings)
            {
                this.ErrorCode = errorCode;
                this.AdditionalStrings = additionalStrings.Select(x => x.ToString()).ToArray();
            }
        }

        /// <summary>
        /// 指定したインターフェイスの validate において、指定した例外が起きて失敗しその例外内容が想定通りであることを assert する
        /// </summary>
        /// <typeparam name="ExpectedException">期待する例外型</typeparam>
        /// <param name="t">validate 対象の型(ExpectedInvalidInterfaceExceptionAttribute による修飾が必要)</param>
        private void AssertInvalidInterface<ExpectedException>(Type t)
            where ExpectedException : UserErrorException
        {
            var a = t.GetCustomAttribute<AssertInvalidInterfaceAttribute>();
            AssertionUtility.AssertException<ExpectedException>(
                () => new InterfaceValidator().Validate(t),
                e => AssertionUtility.AssertUserErrorException(e, a.ErrorCode, a.AdditionalStrings));
        }

        /// <summary>
        /// 指定したインターフェイスの validate に成功することを assert する
        /// </summary>
        /// <param name="t">validate 対象の型</param>
        private void AssertValidInterface(Type t)
        {
            AssertionUtility.AssertNoException(() => new InterfaceValidator().Validate(t));
        }

        #region インターフェイスに関するテスト

        private interface IChild : nn.sf.IServiceObject
        {
        }

        [AssertInvalidInterface("SF-IF-NotSupported:GenericTypeDefinition")]
        private interface IGenericTypeDefinition<T> : nn.sf.IServiceObject
        {
        }

        [AssertInvalidInterface("SF-IF-NotSupported:GenericInterface")]
        private interface IGenericInterface<T> : nn.sf.IServiceObject
        {
        }

        [AssertInvalidInterface("SF-IF-InvalidBaseInterface:NotIServiceObject", typeof(nn.sf.IServiceObject))]
        private interface INotServiceObjectInterface
        {
        }

        [AssertInvalidInterface("SF-IF-MethodNameDuplicated", "DuplicatedMethod")]
        private interface IMethodNameDuplicatedInterface : nn.sf.IServiceObject
        {
            [MethodId(0)]
            int DuplicatedMethod();
            [MethodId(1)]
            int DuplicatedMethod(int a);
        }

        [AssertInvalidInterface("SF-IF-MethodNameDuplicated", "Release")]
        private interface IMethodNameDuplicatedInterface_AddReference : nn.sf.IServiceObject
        {
            [MethodId(0)]
            void Release(int n);
        }

        [AssertInvalidInterface("SF-IF-MethodNameDuplicated", "AddReference")]
        private interface IMethodNameDuplicatedInterface_Release : nn.sf.IServiceObject
        {
            [MethodId(0)]
            void AddReference(int n);
        }

        [AssertInvalidInterface("SF-IF-MethodIdDuplicated", "12345")]
        private interface IMethodIdDuplicatedInterface : nn.sf.IServiceObject
        {
            [MethodId(12345)]
            void A();
            [MethodId(12345)]
            void B();
        }

        private interface IMethodIdDuplicatedInterface1Base : nn.sf.IServiceObject
        {
            [MethodId(6789)]
            void A();
        }

        [AssertInvalidInterface("SF-IF-MethodIdDuplicated", "6789")]
        private interface IMethodIdDuplicatedInterface1 : IMethodIdDuplicatedInterface1Base
        {
            [MethodId(6789)]
            void B();
        }

        private interface IValidInterface : nn.sf.IServiceObject
        {
            [MethodId(0)]
            void A();
            [MethodId(1)]
            void B();
        }

        private interface IValidInterface1 : nn.sf.IServiceObject
        {
            [MethodId(0)]
            void C();
        }

        private interface ISingleInheritedInterface : IValidInterface
        {
            [MethodId(2)]
            void D();
        }

        [AssertInvalidInterface("SF-IF-InvalidBaseInterface:MultipleBase")]
        private interface IMultiInheritedInterface : IValidInterface, IValidInterface1
        {
            [MethodId(10)]
            void E();
        }

        private static Type[] InvalidInterfaceTypes = new[]
        {
            typeof(IGenericTypeDefinition<>),
            typeof(IGenericInterface<int>),
            typeof(INotServiceObjectInterface),
            typeof(IMethodNameDuplicatedInterface),
            typeof(IMethodNameDuplicatedInterface_AddReference),
            typeof(IMethodNameDuplicatedInterface_Release),
            typeof(IMethodIdDuplicatedInterface),
            typeof(IMethodIdDuplicatedInterface1),
            typeof(IMultiInheritedInterface),
        };

        private static Type[] ValidInterfaceTypes = new[]
        {
            typeof(IValidInterface),
            typeof(ISingleInheritedInterface),
        };

        [TestMethod]
        public void TestValidateForInterfaces()
        {
            foreach (var type in InvalidInterfaceTypes)
            {
                AssertInvalidInterface<InvalidInterfaceException>(type);
            }
            foreach (var type in ValidInterfaceTypes)
            {
                AssertValidInterface(type);
            }
        }

        #endregion

        #region メソッドに関するテスト

        [AssertInvalidInterface("SF-IF-NotSupported:ReturnType")]
        private interface IOutReturnParameterMethod : nn.sf.IServiceObject
        {
            nn.sf.Out<int> F();
        }

        [AssertInvalidInterface("SF-IF-NotSupported:ReturnType")]
        private interface INativeHandleReturnParameterMethod : nn.sf.IServiceObject
        {
            nn.sf.NativeHandle F();
        }

        private interface IValidMethods : nn.sf.IServiceObject
        {
            [MethodId(0)]
            void Void();
            [MethodId(1)]
            nn.Result Result();
            [MethodId(2)]
            sbyte Sbyte();
            [MethodId(3)]
            byte Byte();
            [MethodId(4)]
            short Short();
            [MethodId(5)]
            ushort UShotr();
            [MethodId(6)]
            int Int();
            [MethodId(7)]
            uint UInt();
            [MethodId(8)]
            long Long();
            [MethodId(9)]
            ulong ULong();
            [MethodId(10)]
            IChild Child();
        }

        private static Type[] InvalidMethodTypes = new[]
        {
            typeof(IOutReturnParameterMethod),
            typeof(INativeHandleReturnParameterMethod),
        };

        [TestMethod]
        public void TestValidateForMethods()
        {
            foreach (var type in InvalidMethodTypes)
            {
                AssertInvalidInterface<InvalidMethodException>(type);
            }
            AssertValidInterface(typeof(IValidMethods));
        }

        #endregion

        #region パラメータに関するテスト

        [AssertInvalidInterface("SF-IF-NotSupported:NativeOutParameter", @"out ", typeof(nn.sf.Out<>))]
        private interface IIncludingNativeOutParameter : nn.sf.IServiceObject
        {
            [MethodId(0)]
            nn.Result F(out int a);
        }

        [AssertInvalidInterface("SF-IF-NotSupported:ParameterType", typeof(nn.sf.Out<nn.sf.InBuffer>))]
        private interface IInvalidOutParameter_InBuffer : nn.sf.IServiceObject
        {
            [MethodId(0)]
            nn.Result F(nn.sf.Out<nn.sf.InBuffer> a);
        }

        [AssertInvalidInterface("SF-IF-NotSupported:ParameterType", typeof(nn.sf.Out<nn.sf.OutBuffer>))]
        private interface IInvalidOutParameter_OutBuffer : nn.sf.IServiceObject
        {
            [MethodId(0)]
            nn.Result F(nn.sf.Out<nn.sf.OutBuffer> a);
        }

        [AssertInvalidInterface("SF-IF-NotSupported:ParameterType", typeof(string))]
        private interface IInvalidParameterType : nn.sf.IServiceObject
        {
            [MethodId(0)]
            nn.Result F(string s);
        }

        [AssertInvalidInterface("SF-IF-InvalidParameterType:NativeHandle", typeof(string))]
        private interface INoMoveCopyNativeHandle : nn.sf.IServiceObject
        {
            [MethodId(0)]
            nn.Result F(nn.sf.NativeHandle s);
        }

        [AssertInvalidInterface("SF-IF-InvalidParameterType:HipcProcessId")]
        private interface IOutProcessId : nn.sf.IServiceObject
        {
            [MethodId(0)]
            nn.Result F([ProcessId] nn.sf.Out<nn.Bit64> s);
        }

        [AssertInvalidInterface("SF-IF-InvalidParameterAttribute")]
        private interface IInvalidTypeOfInProcessId : nn.sf.IServiceObject
        {
            [MethodId(0)]
            nn.Result F([ProcessId] std.uint64_t s);
        }

        [AssertInvalidInterface("SF-IF-InvalidParameterAttribute", typeof(LargeDataAttribute))]
        private interface IInvalidTypeOfLargeData : nn.sf.IServiceObject
        {
            [MethodId(0)]
            nn.Result F([LargeData] nn.sf.InBuffer b);
        }

        [AssertInvalidInterface("SF-IF-InvalidParameterAttribute:HipcMapTransferAttribute")]
        private interface IInvalidHipcMapTransferSecurity1 : nn.sf.IServiceObject
        {
            [MethodId(0)]
            nn.Result F([MapTransferSecurity(MapTransferSecurity.Secure)] int x);
        }

        [AssertInvalidInterface("SF-IF-InvalidParameterAttribute:HipcMapTransferAttribute")]
        private interface IInvalidHipcMapTransferSecurity2 : nn.sf.IServiceObject
        {
            [MethodId(0)]
            nn.Result F([MapTransferSecurity(MapTransferSecurity.Secure)][NonSecureMapTransfer] nn.sf.InBuffer x);
        }

        public struct S
        {
            public int A;
        }

        [LargeData]
        public struct T
        {
            public int A;
        }

        private interface IValidParameters : nn.sf.IServiceObject
        {
            [MethodId(0)]
            void SByte(sbyte x, nn.sf.Out<sbyte> o);
            [MethodId(1)]
            void Byte(byte x, nn.sf.Out<byte> o);
            [MethodId(2)]
            void Short(short x, nn.sf.Out<short> o);
            [MethodId(3)]
            void UShort(ushort x, nn.sf.Out<ushort> o);
            [MethodId(4)]
            void Int(int x, nn.sf.Out<int> o);
            [MethodId(5)]
            void UInt(uint x, nn.sf.Out<uint> o);
            [MethodId(6)]
            void Long(long x, nn.sf.Out<long> o);
            [MethodId(7)]
            void ULong(ulong x, nn.sf.Out<ulong> o);
            [MethodId(8)]
            void Child(IChild x, nn.sf.Out<IChild> o);
            [MethodId(9)]
            void NativeHandle([MoveHandle] nn.sf.NativeHandle moveIn, nn.sf.NativeHandle copyIn, nn.sf.Out<nn.sf.NativeHandle> o);
            [MethodId(10)]
            void MoveNativeHandle([MoveHandle] nn.sf.Out<nn.sf.NativeHandle> o);
            [MethodId(11)]
            void InProcessId([ProcessId] nn.Bit64 pid);
            [MethodId(12)]
            void ArrayInt(nn.sf.InArray<int> inArray, nn.sf.OutArray<int> outArray);
            [MethodId(13)]
            void ArrayS(nn.sf.InArray<S> inArray, nn.sf.OutArray<S> outArray);
            [MethodId(14)]
            void Large(S s, T t, [LargeData] S ls, [LargeData] T lt);
            [MethodId(15)]
            void LargeOut(nn.sf.Out<S> s, nn.sf.Out<T> t, [LargeData] nn.sf.Out<S> ls, [LargeData] nn.sf.Out<T> lt);
        }

        private static Type[] InvalidMethodParameterTypes = new[]
        {
            typeof(IIncludingNativeOutParameter),
            typeof(IInvalidOutParameter_InBuffer),
            typeof(IInvalidOutParameter_OutBuffer),
            typeof(IInvalidParameterType),
            typeof(IOutProcessId),
            typeof(IInvalidTypeOfInProcessId),
            typeof(IInvalidTypeOfLargeData),
            typeof(IInvalidHipcMapTransferSecurity1),
            typeof(IInvalidHipcMapTransferSecurity2),
        };

        [TestMethod]
        public void TestValidateForParameters()
        {
            foreach (var type in InvalidMethodParameterTypes)
            {
                AssertInvalidInterface<InvalidMethodParameterException>(type);
            }
            AssertValidInterface(typeof(IValidParameters));
        }

        #endregion
    }
}
