﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Runtime.InteropServices;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using Nintendo.ServiceFramework;

namespace Nintendo.ServiceFrameworkTest
{
    [TestClass]
    public class SfStructTest
    {
        #region CanAccept/GetEntity

        private struct StructGetEntity
        {
        }

        private struct StructGetEntity2
        {
        }

        [TestMethod]
        public void TestCanAccept()
        {
            Assert.IsTrue(SfStruct.CanAccept(typeof(StructGetEntity)));
        }

        [TestMethod]
        public void TestGetEntity()
        {
            // 共通テスト
            SfEntityTestUtility.TestGetEntityCommon<StructGetEntity, StructGetEntity2>();
            // エンティティの型が SfStruct であることテスト
            var e = typeof(StructGetEntity).GetSfEntity();
            Assert.AreSame(typeof(SfStruct), e.GetType());
        }

        #endregion

        #region GetFields

        public struct StructGetFields
        {
            public int A;
            public long B;
        }

        [TestMethod]
        public void TestGetFields()
        {
            var t = typeof(StructGetFields);
            var e = new SfStruct(t);
            var fields = e.GetFields().Select(m => m.Name).ToArray();
            AssertionUtility.AssertAreSetEqual(new[] { "A", "B" }, e.GetFields().Select(m => m.Name));
        }

        #endregion

        #region TestTypes

        private class ExpectedReferredTypesAttribute : Attribute
        {
            public Type[] Types { get; private set; }
            public ExpectedReferredTypesAttribute(params Type[] types)
            {
                this.Types = types;
            }
        }

        private class ExpectedSizeAndAlignmentAttribute : Attribute
        {
            public int Size { get; private set; }
            public int Alignment { get; private set; }
            public ExpectedSizeAndAlignmentAttribute(int size, int alignment)
            {
                this.Size = size;
                this.Alignment = alignment;
            }
        }

        [ExpectedReferredTypes()]
        [ExpectedSizeAndAlignment(0, 1)]
        public struct StructGetReferredTypes0
        {
        }

        [ExpectedReferredTypes(typeof(int))]
        [ExpectedSizeAndAlignment(4, 4)]
        public struct StructGetReferredTypes1
        {
            public int A;
        }

        [ExpectedReferredTypes(typeof(int))]
        [ExpectedSizeAndAlignment(8, 4)]
        public struct StructGetReferredTypes2
        {
            public int A;
            public int B;
        }

        [ExpectedReferredTypes(typeof(int), typeof(long))]
        [ExpectedSizeAndAlignment(16, 8)]
        public struct StructGetReferredTypes3
        {
            public int A;
            public long B;
        }

        [ExpectedReferredTypes(typeof(bool))]
        [ExpectedSizeAndAlignment(1, 1)]
        public struct StructGetReferredTypes4
        {
            public bool A;
        }

        [ExpectedReferredTypes(typeof(StructGetReferredTypes1))]
        [ExpectedSizeAndAlignment(4, 4)]
        public struct NestedStruct1
        {
            public StructGetReferredTypes1 Nested;
        }

        [ExpectedReferredTypes(typeof(StructGetReferredTypes1), typeof(StructGetReferredTypes3))]
        [ExpectedSizeAndAlignment(24, 8)]
        public struct NestedStruct2
        {
            public StructGetReferredTypes1 Nested1;
            public StructGetReferredTypes3 Nested3;
        }

        [ExpectedReferredTypes(typeof(int))]
        [ExpectedSizeAndAlignment(40, 4)]
        public struct FixedArrayStruct
        {
            [FixedArray(10)]
            public int[] A;
        }

        [ExpectedReferredTypes(typeof(char_t), typeof(std.int32_t), typeof(std.int64_t), typeof(NestedStruct2))]
        [ExpectedSizeAndAlignment(96, 8)]
        public struct FixedArrayStruct2
        {
            [FixedArray(3)]
            public char_t[] A; // (1 * 3 = 3, 1)

            // padding 5

            [FixedArray(2)]
            public std.int64_t[] B; // (8 * 2 = 16, 8)

            // padding 0

            [FixedArray(5)]
            public std.int32_t[] C; // (4 * 5 = 20, 4)

            // padding 4

            [FixedArray(2)]
            public NestedStruct2[] D; // (24 * 2 = 48, 8)
        }

        [ExpectedReferredTypes()]
        [ExpectedSizeAndAlignment(56, 16)]
        [ExternalStruct(56, 16)]
        public struct ExternalStruct
        {
        }

        private static Type[] TestTypes = new[]
        {
            typeof(StructGetReferredTypes0),
            typeof(StructGetReferredTypes1),
            typeof(StructGetReferredTypes2),
            typeof(StructGetReferredTypes3),
            typeof(StructGetReferredTypes4),
            typeof(NestedStruct1),
            typeof(NestedStruct2),
            typeof(FixedArrayStruct),
            typeof(FixedArrayStruct2),
            typeof(ExternalStruct),
        };

        #endregion

        [TestMethod]
        public void TestGetReferredEntities()
        {
            foreach (var type in TestTypes)
            {
                var expected = type.GetCustomAttribute<ExpectedReferredTypesAttribute>().Types;
                var referred = new SfStruct(type).GetReferredEntities().Select(x => x.InnerType);
                AssertionUtility.AssertAreSetEqual(expected, referred);
                var stronglyReferred = new SfStruct(type).GetStronglyReferredEntities().Select(x => (Type)x);
                AssertionUtility.AssertAreSetEqual(expected, stronglyReferred);
            }
        }

        [TestMethod]
        public void TestSizeAndAlignment()
        {
            foreach (var type in TestTypes)
            {
                var expected = type.GetCustomAttribute<ExpectedSizeAndAlignmentAttribute>();
                SfEntityTestUtility.TestSizeAndAlignment(expected.Size, expected.Alignment, type);
            }
        }

        #region TestUnsafeSizeof

        // unsafe sizeof と SfStruct での解釈が一致しているかどうかのテスト

        private void AssertStructSize<T>(int expectedSize)
        {
            var type = typeof(T);
            Assert.AreEqual(expectedSize, ((SfValueType)(type.GetSfEntity())).Size);
            var expected = type.GetCustomAttribute<ExpectedSizeAndAlignmentAttribute>();
            SfEntityTestUtility.TestSizeAndAlignment(expected.Size, expected.Alignment, type);
        }

        [ExpectedSizeAndAlignment(1, 1)]
        public struct SizeOf1
        {
            public bool A;
        }

        [ExpectedSizeAndAlignment(2, 1)]
        public struct SizeOf2
        {
            public bool A;
            public bool B;
        }

        [ExpectedSizeAndAlignment(12, 4)]
        public struct SizeOf3
        {
            public bool A;
            public int B;
            public bool C;
        }

        [ExpectedSizeAndAlignment(24, 8)]
        public struct SizeOf4
        {
            public bool A;
            public long B;
            public bool C;
        }

        [ExpectedSizeAndAlignment(16, 8)]
        public struct SizeOf5
        {
            public bool A;
            public bool C;
            public long B;
        }

        [ExpectedSizeAndAlignment(56, 8)]
        public struct SizeOf6
        {
            public SizeOf1 A;
            public SizeOf2 B;
            public SizeOf3 C;
            public SizeOf4 D;
            public SizeOf5 E;
        }

        [TestMethod]
        public unsafe void TestUnsafeSizeof()
        {
            AssertStructSize<SizeOf1>(sizeof(SizeOf1));
            AssertStructSize<SizeOf2>(sizeof(SizeOf2));
            AssertStructSize<SizeOf3>(sizeof(SizeOf3));
            AssertStructSize<SizeOf4>(sizeof(SizeOf4));
            AssertStructSize<SizeOf5>(sizeof(SizeOf5));
            AssertStructSize<SizeOf6>(sizeof(SizeOf6));
        }

        #endregion
    }
}
