﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Nintendo.ServiceFramework;
using Nintendo.ServiceFramework.CppCode;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace Nintendo.ServiceFrameworkTest.CppCode
{
    [TestClass]
    public class StructCppCodeEmitterTest
    {
        [CppRefPath(@"<nnt/Struct.h>")]
        [CppFullName(@"nnt::Struct")]
        public struct Struct
        {
            public int A;
        }

        [CppRefPath(@"<nnt/BigStruct.h>")]
        [CppFullName(@"nnt::BigStruct")]
        public struct BigStruct
        {
            public long A;
            public long B;
        }

        [CppRefPath(@"<nnt/NestedStruct.h>")]
        [CppFullName(@"nnt::NestedStruct")]
        public struct NestedStruct
        {
            public Struct Nested;
            public int B;
        }

        [CppRefPath(@"<nnt/BigNestedStruct.h>")]
        [CppFullName(@"nnt::BigNestedStruct")]
        public struct BigNestedStruct
        {
            public BigStruct Nested;
            public int B;
        }

        [CppRefPath(@"<nnt/BuiltInMemberStruct.h>")]
        [CppFullName(@"nnt::BuiltInMemberStruct")]
        public struct BuiltInMemberStruct
        {
            public char_t C;
            public char16_t C16;
            public char32_t C32;
        }

        [CppRefPath(@"<nnt/FixedArrayStruct.h>")]
        [CppFullName(@"nnt::FixedArrayStruct")]
        public struct FixedArrayStruct
        {
            [FixedArray(10)]
            public int[] I10;

            [FixedArray(20)]
            public Struct[] S20;
        }

        [TestMethod]
        public void TestGetCppTypeString()
        {
            Assert.AreEqual(@"::nnt::Struct", typeof(Struct).GetSfEntity().GetCppTypeString(false));
            Assert.AreEqual(@"nnt::Struct", typeof(Struct).GetSfEntity().GetCppTypeString(true));
        }

        [TestMethod]
        public void TestGetParameterString()
        {
            Assert.AreEqual(@"::nnt::Struct foo", typeof(Struct).GetSfEntity().GetParameterString(@"foo", InOutType.In, false));
            Assert.AreEqual(@"nnt::Struct foo", typeof(Struct).GetSfEntity().GetParameterString(@"foo", InOutType.In, true));
            Assert.AreEqual(@"::nn::sf::Out<::nnt::Struct> foo", typeof(Struct).GetSfEntity().GetParameterString(@"foo", InOutType.Out, false));
            Assert.AreEqual(@"nn::sf::Out<nnt::Struct> foo", typeof(Struct).GetSfEntity().GetParameterString(@"foo", InOutType.Out, true));
            Assert.AreEqual(@"const ::nnt::BigStruct& foo", typeof(BigStruct).GetSfEntity().GetParameterString(@"foo", InOutType.In, false));
            Assert.AreEqual(@"const nnt::BigStruct& foo", typeof(BigStruct).GetSfEntity().GetParameterString(@"foo", InOutType.In, true));
            Assert.AreEqual(@"::nn::sf::Out<::nnt::BigStruct> foo", typeof(BigStruct).GetSfEntity().GetParameterString(@"foo", InOutType.Out, false));
            Assert.AreEqual(@"nn::sf::Out<nnt::BigStruct> foo", typeof(BigStruct).GetSfEntity().GetParameterString(@"foo", InOutType.Out, true));
        }

        [TestMethod]
        public void TestGetExternalIncludes()
        {
            AssertionUtility.AssertAreSetEqual(new[] { @"<nnt/Struct.h>" }, typeof(Struct).GetSfEntity().GetExternalIncludes());
        }

        [TestMethod]
        public void TestGetAdditionalIncludes()
        {
            AssertionUtility.AssertAreSetEqual(Enumerable.Empty<string>(), typeof(Struct).GetSfEntity().GetAdditionalIncludes());
        }

        [TestMethod]
        public void TestEmitForwardDeclarationCode()
        {
            GenerateTestUtility.AssertGeneratedString(g =>
            {
                g.Newline();
                typeof(Struct).GetSfEntity().EmitForwardDeclarationCode(g);
                g.AutoNewLine();
                g.CloseCurrentNamespace();
            }, @"
namespace nnt {

struct Struct;

}
");
        }

        [TestMethod]
        public void TestEmitDefinitionCode()
        {
            GenerateTestUtility.AssertGeneratedString(g =>
            {
                g.Newline();
                typeof(Struct).GetSfEntity().EmitDefinitionCode(g);
                g.AutoNewLine();
                g.CloseCurrentNamespace();
            }, @"
namespace nnt {

struct Struct
{
    std::int32_t A;
};

}
");
            GenerateTestUtility.AssertGeneratedString(g =>
            {
                g.Newline();
                typeof(NestedStruct).GetSfEntity().EmitDefinitionCode(g);
                g.AutoNewLine();
                g.CloseCurrentNamespace();
            }, @"
namespace nnt {

struct NestedStruct
{
    nnt::Struct Nested;
    std::int32_t B;
};

}
");
            GenerateTestUtility.AssertGeneratedString(g =>
            {
                g.Newline();
                typeof(BigNestedStruct).GetSfEntity().EmitDefinitionCode(g);
                g.AutoNewLine();
                g.CloseCurrentNamespace();
            }, @"
namespace nnt {

struct BigNestedStruct
{
    nnt::BigStruct Nested;
    std::int32_t B;
};

}
");
            GenerateTestUtility.AssertGeneratedString(g =>
            {
                g.Newline();
                typeof(BuiltInMemberStruct).GetSfEntity().EmitDefinitionCode(g);
                g.AutoNewLine();
                g.CloseCurrentNamespace();
            }, @"
namespace nnt {

struct BuiltInMemberStruct
{
    char C;
    char16_t C16;
    char32_t C32;
};

}
");
            GenerateTestUtility.AssertGeneratedString(g =>
            {
                g.Newline();
                typeof(FixedArrayStruct).GetSfEntity().EmitDefinitionCode(g);
                g.AutoNewLine();
                g.CloseCurrentNamespace();
            }, @"
namespace nnt {

struct FixedArrayStruct
{
    std::int32_t I10[10];
    nnt::Struct S20[20];
};

}
");
        }

        [TestMethod]
        public void TestEmitReferenceCheckCode()
        {
            GenerateTestUtility.AssertGenerateEmpty(typeof(Struct).GetSfEntity().EmitReferenceCheckCode);
        }
    }
}
