﻿namespace SampleLogFileComparatorTest
{
    using System;
    using System.Diagnostics;
    using System.IO;
    using System.Reflection;
    using System.Text;
    using Microsoft.VisualStudio.TestTools.UnitTesting;

    [TestClass]
    public class SampleLogFileComparatorTestUnit
    {
        private const string LogSequenceSequenceMatch = "SequenceMatch";
        private const string LogSequenceSequenceBetweenLinesDontCare = "SequenceBetweenLinesDontCare";
        private const string LogSequenceSequenceRandom = "SequenceRandom";
        private const string MatchPertternPerfectMatch = "PerfectMatch";
        private const string MatchPertternRegexMatch = "RegexMatch";

        [TestMethod]
        // No.1
        public void TestLogCompareSequenceMatchPerfectMatch()
        {
            var patternFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                true, patternFileLines, logFileLines,
                LogSequenceSequenceMatch,
                MatchPertternPerfectMatch);
        }

        [TestMethod]
        // No.2
        public void TestLogCompareSequenceMatchRegexMatch()
        {
            var patternFileLines =
@"BEGIN
[A-Z]emaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                true, patternFileLines, logFileLines,
                LogSequenceSequenceMatch,
                MatchPertternRegexMatch);
        }

        [TestMethod]
        // No.3
        public void TestLogCompareSequenceMatchPerfectUnmatch()
        {
            var patternFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"BEGIN
semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                false, patternFileLines, logFileLines,
                LogSequenceSequenceMatch,
                MatchPertternPerfectMatch);
        }

        [TestMethod]
        // No.4
        public void TestLogCompareSequenceMatchRegextUnmatch()
        {
            var patternFileLines =
@"BEGIN
[A-Z]emaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"BEGIN
semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                false, patternFileLines, logFileLines,
                LogSequenceSequenceMatch,
                MatchPertternRegexMatch);
        }
        [TestMethod]
        // No.5
        public void TestLogCompareSequenceMatchPerfectUnmatchTooLong()
        {
            var patternFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END";
            var logFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
aaa
";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                false, patternFileLines, logFileLines,
                LogSequenceSequenceMatch,
                MatchPertternPerfectMatch);
        }
        [TestMethod]
        // No.6
        public void TestLogCompareSequenceMatchRegexUnmatchTooLong()
        {
            var patternFileLines =
@"BEGIN
[A-Z]emaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END";
            var logFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
aaa
";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                false, patternFileLines, logFileLines,
                LogSequenceSequenceMatch,
                MatchPertternRegexMatch);
        }
        [TestMethod]
        // No.7
        public void TestLogCompareSequenceMatchPerfectUnmatchShort()
        {
            var patternFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                false, patternFileLines, logFileLines,
                LogSequenceSequenceMatch,
                MatchPertternPerfectMatch);
        }
        [TestMethod]
        // No.8
        public void TestLogCompareSequenceMatchRegexUnmatchShort()
        {
            var patternFileLines =
@"BEGIN
[A-Z]emaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                false, patternFileLines, logFileLines,
                LogSequenceSequenceMatch,
                MatchPertternRegexMatch);
        }
        [TestMethod]
        // No.9
        public void TestLogCompareSequenceBetweenLinesDontCarePerfectMatch()
        {
            var patternFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"BEGIN
aaa
Semaphore is releasing.
aaa
Event is signaling.
aaa
Message Queue is not empty.
aaa
SignalThread was finished.
aaa
END
";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                true, patternFileLines, logFileLines,
                LogSequenceSequenceBetweenLinesDontCare,
                MatchPertternPerfectMatch);
        }
        [TestMethod]
        // No.10
        public void TestLogCompareSequenceBetweenLinesDontCareRegexMatch()
        {
            var patternFileLines =
@"BEGIN
[A-Z]emaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"BEGIN
aaa
Semaphore is releasing.
aaa
Event is signaling.
aaa
Message Queue is not empty.
aaa
SignalThread was finished.
aaa
END
";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                true, patternFileLines, logFileLines,
                LogSequenceSequenceBetweenLinesDontCare,
                MatchPertternRegexMatch);
        }
        [TestMethod]
        // No.11
        public void TestLogCompareSequenceBetweenLinesDontCarePerfectUnmatchString()
        {
            var patternFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"BEGIN
aaa
semaphore is releasing.
aaa
Event is signaling.
aaa
Message Queue is not empty.
aaa
SignalThread was finished.
aaa
END
";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                false, patternFileLines, logFileLines,
                LogSequenceSequenceBetweenLinesDontCare,
                MatchPertternPerfectMatch);
        }
        [TestMethod]
        // No.12
        public void TestLogCompareSequenceBetweenLinesDontCareRegexUnmatchString()
        {
            var patternFileLines =
@"BEGIN
[A-Z]emaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"BEGIN
aaa
semaphore is releasing.
aaa
Event is signaling.
aaa
Message Queue is not empty.
aaa
SignalThread was finished.
aaa
END
";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                false, patternFileLines, logFileLines,
                LogSequenceSequenceBetweenLinesDontCare,
                MatchPertternRegexMatch);
        }
        [TestMethod]
        // No.13
        public void TestLogCompareSequenceBetweenLinesDontCarePerfectUnmatch()
        {
            var patternFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"BEGIN
aaa
aaa
Event is signaling.
aaa
Message Queue is not empty.
aaa
SignalThread was finished.
aaa
END
";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                false, patternFileLines, logFileLines,
                LogSequenceSequenceBetweenLinesDontCare,
                MatchPertternPerfectMatch);
        }
        [TestMethod]
        // No.14
        public void TestLogCompareSequenceBetweenLinesDontCareRegexUnmatch()
        {
            var patternFileLines =
@"BEGIN
[A-Z]emaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"BEGIN
aaa
aaa
Event is signaling.
aaa
Message Queue is not empty.
aaa
SignalThread was finished.
aaa
END
";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                false, patternFileLines, logFileLines,
                LogSequenceSequenceBetweenLinesDontCare,
                MatchPertternRegexMatch);
        }
        [TestMethod]
        // No.15
        public void TestLogCompareSequenceBetweenLinesDontCarePerfectUnmatchShort()
        {
            var patternFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                false, patternFileLines, logFileLines,
                LogSequenceSequenceBetweenLinesDontCare,
                MatchPertternPerfectMatch);
        }
        [TestMethod]
        // No.16
        public void TestLogCompareSequenceBetweenLinesDontCareRegexUnmatchShort()
        {
            var patternFileLines =
@"BEGIN
[A-Z]emaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                false, patternFileLines, logFileLines,
                LogSequenceSequenceBetweenLinesDontCare,
                MatchPertternRegexMatch);
        }
        [TestMethod]
        // No.17
        public void TestLogCompareSequenceRandomPerfectMatch()
        {
            var patternFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"END
Semaphore is releasing.
BEGIN
Message Queue is not empty.
Event is signaling.
SignalThread was finished.
";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                true, patternFileLines, logFileLines,
                LogSequenceSequenceRandom,
                MatchPertternPerfectMatch);
        }
        [TestMethod]
        // No.18
        public void TestLogCompareSequenceRandomRegexMatch()
        {
            var patternFileLines =
@"BEGIN
[A-Z]emaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"END
Semaphore is releasing.
BEGIN
Message Queue is not empty.
Event is signaling.
SignalThread was finished.
";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                true, patternFileLines, logFileLines,
                LogSequenceSequenceRandom,
                MatchPertternRegexMatch);
        }
        [TestMethod]
        // No.19
        public void TestLogCompareSequenceRandomPerfectUnmatchString()
        {
            var patternFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"END
semaphore is releasing.
BEGIN
Message Queue is not empty.
Event is signaling.
SignalThread was finished.
";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                false, patternFileLines, logFileLines,
                LogSequenceSequenceRandom,
                MatchPertternPerfectMatch);
        }
        [TestMethod]
        // No.20
        public void TestLogCompareSequenceRandomRegexUnmatchString()
        {
            var patternFileLines =
@"BEGIN
[A-Z]emaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"END
semaphore is releasing.
BEGIN
Message Queue is not empty.
Event is signaling.
SignalThread was finished.
";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                false, patternFileLines, logFileLines,
                LogSequenceSequenceRandom,
                MatchPertternRegexMatch);
        }

        [TestMethod]
        // No.21
        public void TestLogCompareSequenceRandomPerfectUnmatch()
        {
            var patternFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"END
BEGIN
Message Queue is not empty.
Event is signaling.
SignalThread was finished.
";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                false, patternFileLines, logFileLines,
                LogSequenceSequenceRandom,
                MatchPertternPerfectMatch);
        }
        [TestMethod]
        // No.22
        public void TestLogCompareSequenceRandomRegexUnmatch()
        {
            var patternFileLines =
@"BEGIN
[A-Z]emaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
            var logFileLines =
@"END
BEGIN
Message Queue is not empty.
Event is signaling.
SignalThread was finished.
";
            TestRunMethod(MethodBase.GetCurrentMethod().Name,
                false, patternFileLines, logFileLines,
                LogSequenceSequenceRandom,
                MatchPertternRegexMatch);
        }
        private void TestRunMethod(string functionName, bool expentedResult,
            string patternFileLines, string logFileLines,
            string logSequence, string matchPerttern)
        {
            try
            {
                var patternFileName = "pattern.txt";
                var logFileName = "logfile.txt";
                SaveTestLogFile(patternFileName, patternFileLines);
                SaveTestLogFile(logFileName, logFileLines);
                var compareResult = ExecuteSampleLogFileComparator(patternFileName, logFileName, logSequence, matchPerttern);
                File.Delete(patternFileName);
                File.Delete(logFileName);
                Console.WriteLine(functionName);
                if (expentedResult == true)
                {
                    if (compareResult == true)
                    {
                        Console.WriteLine("期待通りログファイルに差異はありませんでした。");
                    }
                    Assert.IsTrue(compareResult);
                }
                else
                {
                    if (compareResult == false)
                    {
                        Console.WriteLine("期待通りログファイルの差異を検出しました。");
                    }
                    Assert.IsFalse(compareResult);
                }
            }
            catch(Exception exception)
            {
                Console.WriteLine($"予期しない例外が発生しました。 {exception.ToString()}");
            }
        }
        /// <summary>
        /// SampleLogFileComparatorTestUnit のカレントフォルダから sdk のルートパスを探し、
        /// SampleLogFileComparator の実行ファイルパスを取得する。
        /// </summary>
        /// <returns>SampleLogFileComparator の実行ファイルパス</returns>
        private static string GetSampleLogFileComparatorPath()
        {
            string sdkRootPath = string.Empty;
            var path = Directory.GetCurrentDirectory();
            var ppath = Path.GetDirectoryName(path);
            while (ppath != null && ppath != string.Empty)
            {
                if (File.Exists(Path.Combine(ppath, "NintendoSdkRootMark")) == true)
                {
                    sdkRootPath = ppath;
                }
                path = ppath;
                ppath = Path.GetDirectoryName(path);
            }
            if (sdkRootPath == string.Empty)
            {
                return string.Empty;
            }
            return (sdkRootPath + "\\Integrate\\Outputs\\AnyCPU\\Tools\\SampleLogFileComparator\\Release\\SampleLogFileComparator.exe");
        }
        /// <summary>
        /// Prosess()を介して、SampleLogFileComparatorを実行する。
        /// </summary>
        /// <param name="solutionFile">ソリューションファイル</param>
        /// <param name="configuration">コンフィグレーション</param>
        /// <param name="platform">プラットフォーム</param>
        private bool ExecuteSampleLogFileComparator(string patternFileName,
            string logFileName, string logSequence, string matchPerttern)
        {
            patternFileName = "--PatternFile=" + patternFileName;
            logFileName = "--LogFile=" + logFileName;
            logSequence = "--LogSequenceType=" + logSequence;
            matchPerttern = "--MatchPerttern=" + matchPerttern;
            string[] args = { patternFileName, logFileName, logSequence, matchPerttern };
            var exitCode = ExecuteProgram(GetSampleLogFileComparatorPath(), string.Join(" ", args));
            if (exitCode == 0)
            {
                return true;
            }
            return false;
        }
        /// <summary>
        /// Prosess()を介して、指定したプログラムを実行する。
        /// </summary>
        /// <param name="filename">実行ファイル</param>
        /// <param name="argument">実行時引数</param>
        private int ExecuteProgram(string filename, string argument)
        {
            using (var proc = new Process())
            {
                proc.StartInfo = new ProcessStartInfo()
                {
                    FileName = filename,
                    Arguments = argument,
                    UseShellExecute = false,
                    CreateNoWindow = true,
                    RedirectStandardOutput = true,
                    RedirectStandardError = true,
                };

                var handler = new DataReceivedEventHandler(
                    (object obj, DataReceivedEventArgs args) =>
                    {
                        if (args.Data == null)
                        {
                            return;
                        }
                        Console.WriteLine(args.Data);
                    });

                proc.OutputDataReceived += handler;
                proc.ErrorDataReceived += handler;
                proc.Start();
                proc.BeginOutputReadLine();
                proc.BeginErrorReadLine();
                proc.WaitForExit();

                return proc.ExitCode;
            }
        }
        private void SaveTestLogFile(string fileName, string logFileLines)
        {
            var streamWriter = new StreamWriter(
                fileName,
                false,
                System.Text.Encoding.UTF8);
            streamWriter.WriteLine(logFileLines);
            streamWriter.Close();
        }

        [TestMethod]
        // No.101
        public void TestLogCompareArgumentPatternFileEmpty()
        {
            try
            {
                var logFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
                var logFileName = "logfile.txt";
                SaveTestLogFile(logFileName, logFileLines);
                var compareResult = ExecuteSampleLogFileComparator(string.Empty, logFileName,
                    LogSequenceSequenceMatch,
                    MatchPertternPerfectMatch);
                File.Delete(logFileName);
                Console.WriteLine(MethodBase.GetCurrentMethod());
                if (compareResult == false)
                {
                    Console.WriteLine("期待通り関数の戻り値はfalseでした。");
                }
                Assert.IsFalse(compareResult);
            }
            catch (Exception exception)
            {
                Console.WriteLine($"予期しない例外が発生しました。 {exception.ToString()}");
            }
        }
        [TestMethod]
        // No.102
        public void TestLogCompareArgumentLogFileEmpty()
        {
            try
            {
                var patternFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
                var patternFileName = "pattern.txt";
                SaveTestLogFile(patternFileName, patternFileLines);
                var compareResult = ExecuteSampleLogFileComparator(patternFileName, string.Empty,
                    LogSequenceSequenceMatch,
                    MatchPertternPerfectMatch);
                File.Delete(patternFileName);
                Console.WriteLine(MethodBase.GetCurrentMethod());
                if (compareResult == false)
                {
                    Console.WriteLine("期待通り関数の戻り値はfalseでした。");
                }
                Assert.IsFalse(compareResult);
            }
            catch (Exception exception)
            {
                Console.WriteLine($"予期しない例外が発生しました。 {exception.ToString()}");
            }
        }
        [TestMethod]
        // No.103
        public void TestLogCompareArgumentPatternFileNotExist()
        {
            try
            {
                var logFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
                var logFileName = "logfile.txt";
                SaveTestLogFile(logFileName, logFileLines);
                var compareResult = ExecuteSampleLogFileComparator("notexist.txt", logFileName,
                    LogSequenceSequenceMatch,
                    MatchPertternPerfectMatch);
                File.Delete(logFileName);
                Console.WriteLine(MethodBase.GetCurrentMethod());
                if (compareResult == false)
                {
                    Console.WriteLine("期待通り関数の戻り値はfalseでした。");
                }
                Assert.IsFalse(compareResult);
            }
            catch (Exception exception)
            {
                Console.WriteLine($"予期しない例外が発生しました。 {exception.ToString()}");
            }
        }
        [TestMethod]
        // No.104
        public void TestLogCompareArgumentLogFileNotExist()
        {
            try
            {
                var patternFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
                var patternFileName = "pattern.txt";
                SaveTestLogFile(patternFileName, patternFileLines);
                var compareResult = ExecuteSampleLogFileComparator(patternFileName, "notexist.txt",
                    LogSequenceSequenceMatch,
                    MatchPertternPerfectMatch);
                File.Delete(patternFileName);
                Console.WriteLine(MethodBase.GetCurrentMethod());
                if (compareResult == false)
                {
                    Console.WriteLine("期待通り関数の戻り値はfalseでした。");
                }
                Assert.IsFalse(compareResult);
            }
            catch (Exception exception)
            {
                Console.WriteLine($"予期しない例外が発生しました。 {exception.ToString()}");
            }
        }
        [TestMethod]
        // No.105
        public void TestLogCompareArgumentPatternFileNameTooLong()
        {
            try
            {
                var logFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
                var patternFileName = string.Empty;
                var stringBuilder = new StringBuilder();
                for (var i = 0; i < 1000; i++)
                {
                    stringBuilder.Append("a");
                }
                patternFileName = (stringBuilder.ToString() + ".txt");
                var logFileName = "logfile.txt";

                SaveTestLogFile(logFileName, logFileLines);
                var compareResult = ExecuteSampleLogFileComparator(patternFileName, logFileName,
                    LogSequenceSequenceMatch,
                    MatchPertternPerfectMatch);
                File.Delete(logFileName);
                Console.WriteLine(MethodBase.GetCurrentMethod());
                if (compareResult == false)
                {
                    Console.WriteLine("期待通り関数の戻り値はfalseでした。");
                }
                Assert.IsFalse(compareResult);
            }
            catch (Exception exception)
            {
                Console.WriteLine($"予期しない例外が発生しました。 {exception.ToString()}");
            }
        }
        [TestMethod]
        // No.106
        public void TestLogCompareArgumentLogFileNameTooLong()
        {
            try
            {
                var patternFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
                var patternFileName = "pattern.txt";
                var logFileName = string.Empty;
                var stringBuilder = new StringBuilder();
                for (var i = 0; i < 1000; i++)
                {
                    stringBuilder.Append("a");
                }
                logFileName = (stringBuilder.ToString() + ".txt");

                SaveTestLogFile(patternFileName, patternFileLines);
                var compareResult = ExecuteSampleLogFileComparator(patternFileName, logFileName,
                    LogSequenceSequenceMatch,
                    MatchPertternPerfectMatch);
                File.Delete(patternFileName);
                Console.WriteLine(MethodBase.GetCurrentMethod());
                if (compareResult == false)
                {
                    Console.WriteLine("期待通り関数の戻り値はfalseでした。");
                }
                Assert.IsFalse(compareResult);
            }
            catch (Exception exception)
            {
                Console.WriteLine($"予期しない例外が発生しました。 {exception.ToString()}");
            }
        }
        [TestMethod]
        // No.107
        public void TestLogCompareArgumentPatternFileisFolder()
        {
            try
            {
                var logFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
                var folderName = "pattern.txt";
                if (Directory.Exists(folderName) == false)
                {
                    Directory.CreateDirectory(folderName);
                }

                var logFileName = "logfile.txt";
                SaveTestLogFile(logFileName, logFileLines);
                var compareResult = ExecuteSampleLogFileComparator(folderName, logFileName,
                    LogSequenceSequenceMatch,
                    MatchPertternPerfectMatch);
                File.Delete(logFileName);
                Directory.Delete(folderName);
                Console.WriteLine(MethodBase.GetCurrentMethod());
                if (compareResult == false)
                {
                    Console.WriteLine("期待通り関数の戻り値はfalseでした。");
                }
                Assert.IsFalse(compareResult);
            }
            catch (Exception exception)
            {
                Console.WriteLine($"予期しない例外が発生しました。 {exception.ToString()}");
            }
        }
        [TestMethod]
        // No.108
        public void TestLogCompareArgumentLogFileisFolder()
        {
            try
            {
                var patternFileLines =
@"BEGIN
Semaphore is releasing.
Event is signaling.
Message Queue is not empty.
SignalThread was finished.
END
";
                var patternFileName = "pattern.txt";
                SaveTestLogFile(patternFileName, patternFileLines);

                var folderName = "logfile.txt";
                if (Directory.Exists(folderName) == false)
                {
                    Directory.CreateDirectory(folderName);
                }
                var compareResult = ExecuteSampleLogFileComparator(patternFileName, folderName,
                    LogSequenceSequenceMatch,
                    MatchPertternPerfectMatch);
                File.Delete(patternFileName);
                Directory.Delete(folderName);
                Console.WriteLine(MethodBase.GetCurrentMethod());
                if (compareResult == false)
                {
                    Console.WriteLine("期待通り関数の戻り値はfalseでした。");
                }
                Assert.IsFalse(compareResult);
            }
            catch (Exception exception)
            {
                Console.WriteLine($"予期しない例外が発生しました。 {exception.ToString()}");
            }
        }
    }
}
