﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.ControlTarget;
using Nintendo.RunOnTarget;
using System.IO;
using System.IO.Ports;
using System.Diagnostics;
using System.Threading;
using System.Net;
using System.Net.Sockets;
using System.Collections.Generic;
using System.Text;
using System.Linq;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.Threading.Tasks.Dataflow;
using TestUtility;

namespace RunnerToolsTest
{
    [TestClass]
    public class RunOnTargetTest
    {
        public TestContext TestContext { get; set; }

        [ClassInitialize]
        public static void ClassInit(TestContext context)
        {
            ConsoleApplicationTraceListener.SetGlobal(true);
        }

        [Ignore]
        [TestMethod]
        public void TestIsRunningUBoot()
        {
            using (var port = new SerialPortWrapper("COM4"))
            {
                var uBoot = new UBoot(port);
                Console.WriteLine("U-Boot is running: {0}", uBoot.IsRunning());
            }
        }

        [Ignore]
        [TestMethod]
        public void TestUBoot()
        {
            using (var port = new SerialPortWrapper("COM4"))
            {
                var uBoot = new UBoot(port);
                uBoot.EnsureBoot();
            }
        }

        [Ignore]
        [TestMethod]
        public void TestHtcDaemonIsRunning()
        {
            Console.WriteLine("Htc Daemon is running: {0}", new HtcDaemonAccessor().IsRunning());
        }

        [Ignore]
        [TestMethod]
        public void TestEnsureStartHtcDaemon()
        {
            new HtcDaemonAccessor().EnsureStart();
        }

        [Ignore]
        [TestMethod]
        public void TestGetHtcDaemonPath()
        {
            Console.WriteLine(PathUtility.GetHtcDaemonPath());
        }

        [Ignore]
        [TestMethod]
        public void TestHtcAccessor()
        {
            var htc = new ShellAccessor(new IPEndPoint(IPAddress.Parse("192.168.0.10"), 23));
            Console.WriteLine(htc.SendCommand(string.Empty));
        }

        [TestMethod]
        public void TestLineSplitBlock()
        {
            var splitter = DataflowUtility.CreateHalfwayLineSplitter();
            splitter.Post("a");
            splitter.Post("bcd\n");
            splitter.Post("efg");
            splitter.Post("h\ni");
            splitter.Post("jkl\nmn");

            Assert.AreEqual("a", splitter.Receive());
            Assert.AreEqual("abcd", splitter.Receive());
            Assert.AreEqual("efg", splitter.Receive());
            Assert.AreEqual("efgh", splitter.Receive());
            Assert.AreEqual("i", splitter.Receive());
            Assert.AreEqual("ijkl", splitter.Receive());
            Assert.AreEqual("mn", splitter.Receive());
        }

        [TestMethod]
        public void TestSerialPortNames()
        {
            Console.Write(string.Join("\n", SerialPortWrapper.GetPortNames()));
        }

        [Ignore]
        [TestMethod]
        public void TestProcessId()
        {
            Console.WriteLine(Trace.Listeners.Count);
            TestPath testPath = new TestPath(TestContext);
            var sampleKip = new FileInfo(Path.Combine(testPath.GetSigloRoot(), @"Programs\Chris\Outputs\jetson-tk1\Processes\DummyProcess\Develop\DummyProcess.kip"));

            var htc = new ShellAccessor(new HtcDaemonAccessor().FindService("Shell0"));
            var result = htc.LoadProgram(sampleKip);

            Console.WriteLine("ProcessId: {0}", result.ResultCode);
        }

        [Ignore]
        [TestMethod]
        public void TestListProcessId()
        {
            var shell = new ShellAccessor(new HtcDaemonAccessor().FindService("Shell0"));

            Console.WriteLine("RawResult: {0}", shell.SendCommand("P").Message);

            foreach (var pid in shell.ListProcess().Item2)
            {
                Console.WriteLine("ProcessId: {0}", pid);
            }
        }

        [Ignore]
        [TestMethod]
        public void TestHostBridgeAccessor()
        {
            using (var uart = new HostBridgeAccessor(NetworkUtility.Parse("169.254.8.99", 10023)))
            {
                for (int i = 0; i < 100; i++)
                {
                    Console.WriteLine("Result: {0}", uart.ReadPort.Receive(TimeSpan.FromSeconds(10)));
                    Thread.Sleep(100);
                }
            }
        }

        public Tuple<string, string> MakeLog(string meta, string log)
        {
            return new Tuple<string, string>(meta, log);
        }

        [TestMethod]
        public void TestAggregateMultiLog()
        {
            var logA = new BufferBlock<Tuple<string, string>>();
            var logB = new BufferBlock<Tuple<string, string>>();
            var logC = new BufferBlock<Tuple<string, string>>();
            var builder = new StringBuilder();

            var stringBuilderBlock = new ActionBlock<string>(s => { builder.Append(s); });
            var arrangementBlock = DataflowUtility.CreateArrangementMultiLogBlock();

            logA.LinkTo(arrangementBlock);
            logB.LinkTo(arrangementBlock);
            logC.LinkTo(arrangementBlock);
            arrangementBlock.LinkTo(stringBuilderBlock);

            logC.Post(MakeLog("logC", "Wait long time..."));
            Task.Delay(1).Wait();
            logA.Post(MakeLog("logA", "He"));
            Task.Delay(1).Wait();
            logA.Post(MakeLog("logA", "llo"));
            Task.Delay(1).Wait();
            logB.Post(MakeLog("logB", "Interrupt\r\n"));
            Task.Delay(1).Wait();
            logA.Post(MakeLog("logA", "World\r\n"));
            Task.Delay(1).Wait();
            logC.Post(MakeLog("logA", "Many\r\nMany\r\nLi"));
            Task.Delay(1).Wait();
            logB.Post(MakeLog("logB", "Interrupt\r\n"));
            Task.Delay(1).Wait();
            logC.Post(MakeLog("logC", "Done\r\n"));
            Task.Delay(1).Wait();
            logC.Post(MakeLog("logA", "nes\r\n"));

            arrangementBlock.Complete();
            Task.Delay(1).Wait();

            Assert.AreEqual(@"[logC] Wait long time...
[logA] Hello
[logB] Interrupt
[logA] HelloWorld
[logA] Many
[logA] Many
[logA] Li
[logB] Interrupt
[logC] Wait long time...Done
[logA] Lines
", builder.ToString());
        }

        [Ignore]
        [TestMethod]
        public void TestRunProgramAsCommand()
        {
            TestPath testPath = new TestPath(TestContext);
            var sampleKip = new FileInfo(Path.Combine(testPath.GetSigloRoot(), @"Programs\Chris\Outputs\jetson-tk1\Processes\DummyProcess\Develop\DummyProcess.kip"));

            Assert.AreEqual(0, CommandInterface.Main(new string[] {
                "run-kip",
                sampleKip.FullName,
                "--pattern-success-exit",
                "KProcess::Finalize",
                "--success-timeout",
                "5",
                "--port",
                "COM4",
                "--varbose"
            }));
        }

        [TestMethod]
        public void TestTargetAccessor()
        {
            using (var targetManager = new TargetManagerAccessor())
            {
                targetManager.EnsureStart();

                var targetId = targetManager.FindTarget(null);

                Console.WriteLine("targetId: {0}", targetId);

                var targetId2 = targetManager.FindTarget("XAWF0111100014");

                Console.WriteLine("targetId2: {0}", targetId2);

                var targetId3 = targetManager.FindTarget("DefaultUSBTarget");

                Console.WriteLine("targetId3: {0}", targetId3);

                targetManager.ConnectTarget("XAWF0111100014");
            }
        }
    }
}
