﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Collections.Generic;
using System.IO;
using System.Security.Cryptography;
using System.Linq;

namespace PatchTest
{
    [TestClass]
    public class PatchTest
    {
        public TestContext TestContext { get; set; }

        public static bool IsGameCardInserted { get; set; } = false;

        [ClassInitialize]
        public static void ClassInit(TestContext context)
        {
            var lib = new TestOnTargetLibrary(context);

            IsGameCardInserted = !lib.GetStatusGameCard().ContainsLog("Not Inserted");
            if (IsGameCardInserted)
            {
                lib.EraseGameCard();
            }
            lib.RunDevMenuCommandSystem("sdcard format;");
            lib.Reboot();

            lib.RunDevMenuCommandSystem("application uninstall 0x0100f6900049a000;"
                                      + "application uninstall 0x010000000000b148;"
                                      + "application uninstall 0x010000000000b147;"
                                      + "application reset-required-version --all;"
                                      + "patch remove-update-relates;"
                                      + "patch set-prefer-delta true;"
                                      + "application clear-task-status-list;"
                                      + "debug empty-nand-free-space; debug empty-sdcard-free-space;"
                                      + "ticket delete-all --ticket-type common;");
            // reset の強制電源断でタスクは cleanup されるので、これは不要
            //            lib.Run(lib.GetPath("./Tests/Outputs/NX-NXFP2-a64/Tests/testPatch_PatchTestRuntimeHelper/Develop/testPatch_PatchTestRuntimeHelper.nsp"), "CleanupTask 0x0100f6900049a000");
        }


        private string GetHelperPath()
        {
            var lib = new TestOnTargetLibrary(this.TestContext);
            return lib.GetPath("./Tests/Outputs/NX-NXFP2-a64/Tests/testPatch_PatchTestRuntimeHelper/Develop/testPatch_PatchTestRuntimeHelper.nsp");
        }
        private string GetOutputApplicationPath(string name)
        {
            var lib = new TestOnTargetLibrary(this.TestContext);
            return lib.GetPath(string.Format("./Tests/Outputs/NX-NXFP2-a64/Patch/BaseApplications/{0}/Develop/{0}.nsp", name));
        }
        private string GetOutputPatchPath(string name)
        {
            var lib = new TestOnTargetLibrary(this.TestContext);
            return lib.GetPath(string.Format("./Tests/Outputs/NX-NXFP2-a64/Patch/{0}/Develop/{0}.nsp", name));
        }
        private string GetOutputNonOptimizedPatchPath(string name)
        {
            var lib = new TestOnTargetLibrary(this.TestContext);
            return lib.GetPath(string.Format("./Tests/Outputs/NX-NXFP2-a64/Patch/{0}/Develop/{0}.intermediate.nsp", name));
        }
        private string GetOutputDeltaPath(string name)
        {
            var lib = new TestOnTargetLibrary(this.TestContext);
            return lib.GetPath(string.Format("./Tests/Outputs/NX-NXFP2-a64/Patch/{0}/Develop/{0}.delta.nsp", name));
        }
        private string GetDtlPath(string name)
        {
            var lib = new TestOnTargetLibrary(this.TestContext);
            return lib.GetPath(string.Format("Tests/Tools/Sources/Tests/PatchTest/{0}.json", name));
        }
        private string GetVersionListPath(string name)
        {
            var lib = new TestOnTargetLibrary(this.TestContext);
            return lib.GetPath(string.Format("Tests/Tools/Sources/Tests/PatchTest/{0}.json", name));
        }

        [TestCleanup()]
        public void Cleanup()
        {
            var lib = new TestOnTargetLibrary(this.TestContext);

            var command = string.Empty;

            if (IsGameCardInserted)
            {
                command += "gamecard erase;";
            }

            command += "application uninstall 0x0100f6900049a000;"
                     + "application uninstall 0x010000000000b148;"
                     + "application uninstall 0x010000000000b147;"
                     + "application reset-required-version --all;"
                     + "patch remove-update-relates;"
                     + "application clear-task-status-list;"
                     + "debug empty-nand-free-space; debug empty-sdcard-free-space;"
                     + "ticket delete-all --ticket-type common;";

            lib.RunDevMenuCommandSystem(command);
            lib.Run(GetHelperPath(), "CleanupTask 0x0100f6900049a000");
            lib.Run(GetHelperPath(), "CleanupTask 0x010000000000b148");
            lib.Run(GetHelperPath(), "CleanupTask 0x010000000000b147");
        }

        private static bool VerifyProgress(string log)
        {
            long current = 0;
            long total = 0;
            foreach (var l in log.Split('\n'))
            {
                if (l.StartsWith("[Progress]"))
                {
                    var split = l.Split(' ');
                    var c = long.Parse(split[1].Trim());
                    var t = long.Parse(split[3].Trim());

                    if (total == 0)
                    {
                        // 最初だけ設定
                        total = t;
                    }
                    if (c < current)
                    {
                        // 巻き戻りが起きたらおかしい
                        return false;
                    }
                    if (t != total)
                    {
                        // total が変わったらおかしい
                        return false;
                    }
                    current = c;
                }
            }
            return true;
        }

        private Func<string, bool> GetOccupiedSizeVerifier(int deltaStep, string patchStorage)
        {
            return (string log) =>
            {
                long current = 0;
                int steps = 0;
                foreach (var l in log.Split('\n'))
                {
                    if (l.StartsWith("[Occupied]"))
                    {
                        var split = l.Split(' ');
                        var storageSize = patchStorage == "SdCard" ? long.Parse(split[4].Trim()) : long.Parse(split[2].Trim().TrimEnd(','));

                        // 占有サイズが上昇する回数をカウントしておく
                        if (storageSize > current)
                        {
                            steps++;
                        }
                        current = storageSize;
                    }
                }
                // 想定回数よりも上昇回数が少なければ OK
                // パッチがサイズ減少する場合は必ずしも増加しないので、一致保証はできない
                return 1 <= steps && steps <= deltaStep;
            };
        }

        private Func<string, bool> GetApplicationVerifier(int version)
        {
            var lib = new TestOnTargetLibrary(this.TestContext);
            var dict = new Dictionary<int, Tuple<string, string>>();

            // パスは PatchSamples/!.nact と同期する必要がある
            dict.Add(0, new Tuple<string, string>("Hello", lib.GetPath("Tests/Tools/Sources/Tests/PatchTest/PatchSamples/data0/")));
            dict.Add(2, new Tuple<string, string>("Goodbye", lib.GetPath("Tests/Tools/Sources/Tests/PatchTest/PatchSamples/data0/")));
            dict.Add(4, new Tuple<string, string>("Goodnight", lib.GetPath("Tests/Tools/Sources/Tests/PatchTest/PatchSamples/data1/")));
            dict.Add(5, new Tuple<string, string>("Goodmorning", lib.GetPath("Tests/Tools/Sources/Tests/PatchTest/PatchSamples/data2/")));
            dict.Add(7, new Tuple<string, string>("Spring", lib.GetPath("Tests/Tools/Sources/Tests/PatchTest/PatchSamples/data3/")));
            dict.Add(8, new Tuple<string, string>("Summer", lib.GetPath("Tests/Tools/Sources/Tests/PatchTest/PatchSamples/data3/")));
            dict.Add(9, new Tuple<string, string>("Autumn", lib.GetPath("Tests/Tools/Sources/Tests/PatchTest/PatchSamples/data2/")));
            dict.Add(10, new Tuple<string, string>("Winter", lib.GetPath("Tests/Tools/Sources/Tests/PatchTest/PatchSamples/data1/")));

            return (string log) => {
                var setting = dict[version];
                if (log.IndexOf(setting.Item1) < 0)
                {
                    return false;
                }
                if (log.IndexOf(string.Format("DisplayVersion: {0}.0", version)) < 0)
                {
                    return false;
                }
                foreach (var path in Directory.EnumerateFiles(setting.Item2, "*", SearchOption.AllDirectories))
                {
                    var relative = path.Replace(setting.Item2, "rom:/");
                    var hash = SHA256CryptoServiceProvider.Create().ComputeHash(File.ReadAllBytes(path));
                    var hashStr = string.Join(string.Empty, hash.Select(b => b.ToString("x2")));
                    if (log.IndexOf(string.Format("{0} {1}", relative, hashStr)) < 0)
                    {
                        return false;
                    }
                }
                return true;
            };
        }
#if false // record に基づいた cleanup が行われるようになった関係で、このテストは泣く泣く封印
        [TestMethod]
        public void DeltaMetaExtendedDataTest()
        {
            var lib = new TestOnTargetLibrary(this.TestContext);

            // 何もインストールしていないときは、何も出ない
            lib.Run(GetHelperPath(), "ReadDeltaMetaExtendedData 0x0100f6900049a000")
                .ExpectInLog("[Failed] id not found");

            // アプリ本編だけインストールしても、何も出ない
            lib.RunDevMenuCommand("application install " + GetOutputApplicationPath("049a-0"));
            lib.Run(GetHelperPath(), "ReadDeltaMetaExtendedData 0x0100f6900049a000")
                .ExpectInLog("[Failed] id not found");

            // デルタをインストールすると、上記ログがでる
            lib.RunDevMenuCommandSystem("patch install-delta " + GetOutputDeltaPath("049a-4"));
            lib.Run(GetHelperPath(), "ReadDeltaMetaExtendedData 0x0100f6900049ac00")
                .ExpectInLog("Source", "ID: 0100f6900049a800", "Version: 131072",
                             "Destination", "ID: 0100f6900049a800", "Version: 262144");
        }
#endif
        [TestMethod]
        public void PatchMetaExtendedDataTest()
        {
            var lib = new TestOnTargetLibrary(this.TestContext);

            // 何もインストールしていないときは、何も出ない
            lib.Run(GetHelperPath(), "ReadPatchMetaExtendedData 0x0100f6900049a800")
                .ExpectInLog("[Failed] id not found");

            // アプリ本編だけインストールしても、何も出ない
            lib.RunDevMenuCommand("application install " + GetOutputApplicationPath("049a-0"));
            lib.Run(GetHelperPath(), "ReadPatchMetaExtendedData 0x0100f6900049a800")
                .ExpectInLog("[Failed] id not found");

            // マージしたパッチの場合、history などがでる
            lib.RunDevMenuCommand("patch install " + GetOutputPatchPath("049a-2"));
            lib.Run(GetHelperPath(), "ReadPatchMetaExtendedData 0x0100f6900049a800")
                .ExpectInLog("Key: 0100f6900049a000, 0, 128")
                .NotExpectInLog("TargetContentType: 0")
                .ExpectInLog("[Success]");

            lib.RunDevMenuCommand("patch install " + GetOutputPatchPath("049a-4"));
            lib.Run(GetHelperPath(), "ReadPatchMetaExtendedData 0x0100f6900049a800")
                .ExpectInLog("Key: 0100f6900049a000, 0, 128",
                             "Key: 0100f6900049a800, 131072, 129")
                .ExpectInLog("Delta History", "Source", "ID: 0100f6900049a800", "Version: 131072", "Destination", "ID: 0100f6900049a800", "Version: 262144",
                             "Delta", "Source", "ID: 0100f6900049a800", "Version: 131072", "Destination", "ID: 0100f6900049a800", "Version: 262144")
                .NotExpectInLog("TargetContentType: 0")
                .ExpectInLog("[Success]");

            lib.RunDevMenuCommand("patch install " + GetOutputPatchPath("049a-5"));
            lib.Run(GetHelperPath(), "ReadPatchMetaExtendedData 0x0100f6900049a800")
                .ExpectInLog("Key: 0100f6900049a000, 0, 128",
                             "Key: 0100f6900049a800, 131072, 129",
                             "Key: 0100f6900049a800, 262144, 129")
                .ExpectInLog("Delta History", "Source", "ID: 0100f6900049a800", "Version: 131072", "Destination", "ID: 0100f6900049a800", "Version: 262144",
                             "Delta History", "Source", "ID: 0100f6900049a800", "Version: 262144", "Destination", "ID: 0100f6900049a800", "Version: 327680",
                             "Delta", "Source", "ID: 0100f6900049a800", "Version: 262144", "Destination", "ID: 0100f6900049a800", "Version: 327680")
                .NotExpectInLog("TargetContentType: 0")
                .ExpectInLog("[Success]");

            lib.RunDevMenuCommand("patch install " + GetOutputPatchPath("049a-10"));
            lib.Run(GetHelperPath(), "ReadPatchMetaExtendedData 0x0100f6900049a800")
                .ExpectInLog("Key: 0100f6900049a000, 0, 128",
                             "Key: 0100f6900049a800, 131072, 129",
                             "Key: 0100f6900049a800, 262144, 129",
                             "Key: 0100f6900049a800, 327680, 129",
                             "Key: 0100f6900049a800, 458752, 129",
                             "Key: 0100f6900049a800, 524288, 129",
                             "Key: 0100f6900049a800, 589824, 129")
                .ExpectInLog("Delta History", "Source", "ID: 0100f6900049a800", "Version: 131072", "Destination", "ID: 0100f6900049a800", "Version: 262144",
                             "Delta History", "Source", "ID: 0100f6900049a800", "Version: 262144", "Destination", "ID: 0100f6900049a800", "Version: 327680",
                             "Delta History", "Source", "ID: 0100f6900049a800", "Version: 327680", "Destination", "ID: 0100f6900049a800", "Version: 458752",
                             "Delta History", "Source", "ID: 0100f6900049a800", "Version: 327680", "Destination", "ID: 0100f6900049a800", "Version: 589824",
                             "Delta History", "Source", "ID: 0100f6900049a800", "Version: 458752", "Destination", "ID: 0100f6900049a800", "Version: 524288",
                             "Delta History", "Source", "ID: 0100f6900049a800", "Version: 524288", "Destination", "ID: 0100f6900049a800", "Version: 589824",
                             "Delta History", "Source", "ID: 0100f6900049a800", "Version: 589824", "Destination", "ID: 0100f6900049a800", "Version: 655360",
                             "Delta", "Source", "ID: 0100f6900049a800", "Version: 589824", "Destination", "ID: 0100f6900049a800", "Version: 655360")
                .NotExpectInLog("TargetContentType: 0")
                .ExpectInLog("[Success]");
        }
        [TestMethod]
        public void RunWithPatchTest()
        {
            // パッチでの更新動作
            var lib = new TestOnTargetLibrary(this.TestContext);

            // アプリ本編、パッチをインストール
            lib.RunDevMenuCommand("application install " + GetOutputApplicationPath("049a-0"));
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(0));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "0.0");

            lib.RunDevMenuCommand("patch install " + GetOutputPatchPath("049a-2"));
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(2));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "2.0");

            lib.RunDevMenuCommand("patch install " + GetOutputNonOptimizedPatchPath("049a-4"));
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(4));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "4.0");

            lib.RunDevMenuCommand("patch install " + GetOutputNonOptimizedPatchPath("049a-5"));
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(5));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "5.0");

            lib.RunDevMenuCommand("patch uninstall 0x0100f6900049a000;"
                                + "application reset-required-version --all;");

            // マージ済みのパッチでも正しいかの確認
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(0));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "0.0");

            lib.RunDevMenuCommand("patch install " + GetOutputPatchPath("049a-2"));
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(2));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "2.0");

            lib.RunDevMenuCommand("patch install " + GetOutputPatchPath("049a-4"));
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(4));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "4.0");

            lib.RunDevMenuCommand("patch install " + GetOutputPatchPath("049a-5"));
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(5));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "5.0");

            lib.RunDevMenuCommand("application reset-required-version --all");

            // パッチの実体が存在しないときでも正しく起動できるかの確認
            lib.RunDevMenuCommandSystem("application delete-entity 0x0100f6900049a000 --patch");
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(0));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "5.0");
        }
        [TestMethod]
        public void DownloadPatchAndRun()
        {
            // 本当は、先に ContentsUploader で最新のものをアップロードするべき
            var lib = new TestOnTargetLibrary(this.TestContext);

            // アプリをローカルインストール
            lib.RunDevMenuCommandSystem("application create-download-task 0x0100f6900049a000;"
                                      + "application wait-download 0x0100f6900049a000");
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(0));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "0.0");

            // パッチをダウンロードして実行
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v2"),
                                        new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("No patches are installed, download directly");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(2));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "2.0");

            // パッチの差分更新
            lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v5"),
                                        new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("register dependency: 0100f6900049a800, 327680")
                .ExpectInLog("register dependency: 0100f6900049a800, 262144");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // 実行できることの確認
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(5));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "5.0");

            // v5 -> v10 の更新 (v5 -> v9 は急行パッチ)
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v10"),
                            new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
               .ExpectInLog("register dependency: 0100f6900049a800, 655360")
               .ExpectInLog("register dependency: 0100f6900049a800, 589824");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // 実行できることの確認
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(10));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "10.0");

            // アンインストール
            lib.RunDevMenuCommandSystem("application uninstall 0100f6900049a000");

            // 再度アプリをローカルインストール
            lib.RunDevMenuCommandSystem("application reset-required-version --all;"
                                      + "application create-download-task 0x0100f6900049a000;"
                                      + "application wait-download 0x0100f6900049a000");
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(0));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "0.0");

            // パッチをダウンロードして実行
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v2"),
                                        new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("No patches are installed, download directly");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(2));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "2.0");

            // パッチの強制直接更新
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v4-force-direct-update"),
                                        new RunningOptions { IgnoreApplicationTerminate = true, FailurePattern = "\"\\[DeltaUpdate\\] Resolved\"", SuccessTimeout = 30 })
               .ExpectInLog("[NetworkInstallTask] Indirect update check is skipped");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(4));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "4.0");

            // チケットを削除したら実行できない = チケットをダウンロードして使っていることの確認
            lib.RunDevMenuCommandSystem("ticket delete-all --ticket-type common;");
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000")
                .ExpectInLog("Unexpected error. result = 0x00057810.");

        }
        [TestMethod]
        public void DownloadPatchNoRoute()
        {
            // 更新経路が見つからない場合は、直接ダウンロードすることの確認
            var lib = new TestOnTargetLibrary(this.TestContext);

            // アプリをローカルインストール
            lib.RunDevMenuCommandSystem("application create-download-task 0x0100f6900049a000;"
                                      + "application wait-download 0x0100f6900049a000");
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(0));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "0.0");


            // 歴史上連続していないパッチをローカルインストール
            lib.RunDevMenuCommand("patch install " + GetOutputPatchPath("049a-2-other") + " --skip-patch-combination-check");
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "2.0");

            // 鍵世代が混ざると動作しないため、パッチの common チケットを削除してしまう
            // このテストは経路を見たいため、common チケットを消しても特に問題は無い
            lib.RunDevMenuCommandSystem("ticket delete 0x0100f6900049a800;");

            // パッチを直接ダウンロードすることの確認
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v5"),
                                        new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("Installed patch is NOT registered in the history, download directly");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // 実行できることの確認
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(5));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "5.0");

            // 途中のパッチがサーバに登録されていない場合も、直接更新に移る
            lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v8"),
                            new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
               .ExpectInLog("dependency 404", "Do rollback");

            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // 実行できることの確認
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(8));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "8.0");
        }
        [TestMethod]
        public void DownloadPatchAndSuspendRepeatly()
        {
            var lib = new TestOnTargetLibrary(this.TestContext);

            // アプリをローカルインストール
            lib.RunDevMenuCommandSystem("application create-download-task 0x0100f6900049a000;"
                                      + "application wait-download 0x0100f6900049a000");
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(0));


            // パッチをダウンロード
            lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v2"),
                                        new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("No patches are installed, download directly");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // パッチの差分更新
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v9"));
            lib.Run(GetHelperPath(), "WaitAndSuspendApplyDelta 0x0100f6900049a000", new RunningOptions { FailureTimeout = 360 })
                .ExpectInLog("[apply] stop request server");

            // 更新完了することの確認
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(9));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "9.0");
        }
        [TestMethod]
        public void DownloadPatchAndPoweroffRepeatly()
        {
            var lib = new TestOnTargetLibrary(this.TestContext);

            lib.RunDevMenuCommandSystem("application create-download-task 0x0100f6900049a000;"
                                      + "application wait-download 0x0100f6900049a000");
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(0));

            // パッチをダウンロード
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v2"),
                                        new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("No patches are installed, download directly");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // パッチの差分更新
            lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v9"));

            // 4 回くらい電源断をして問題ないか見る
            for (int i = 0; i < 4; ++i)
            {
                lib.Run(GetHelperPath(), "WaitAndSuspendApplyDeltaPermanently 0x0100f6900049a000", new RunningOptions { SuccessPattern = "\\[apply\\]", SuccessTimeout = 20 });
                lib.Reboot();
            }
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // 更新完了することの確認
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(9));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "9.0");
        }
        [TestMethod]
        public void DownloadPatchAndSleepRepeatly()
        {
            var lib = new TestOnTargetLibrary(this.TestContext);

            lib.RunDevMenuCommandSystem("application create-download-task 0x0100f6900049a000;"
                                      + "application wait-download 0x0100f6900049a000");
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(0));

            // パッチをダウンロード
            lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v2"),
                                        new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("No patches are installed, download directly");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // パッチの差分更新
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v9"));

            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180, SuccessPattern = "\\[Apply\\]" });
            // 10 回くらいスリープをして問題ないか見る
            for (int i = 0; i < 10; ++i)
            {
                lib.Sleep();
                System.Threading.Thread.Sleep(500);
            }
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // 更新完了することの確認
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(9));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "9.0");
        }

#if false
        [TestMethod]
        public void DownloadPatchAndSleepRepeatlyAging()
        {
            var lib = new TestOnTargetLibrary(this.TestContext);

            for (var j = 0; j < 10000; ++j)
            {
                lib.RunDevMenuCommandSystem("application uninstall 0x0100f6900049a000; application install " + GetOutputApplicationPath("049a-0") + ";");
                lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                   .ExpectInLog(GetApplicationVerifier(0));

                // パッチをダウンロード
                lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v2"),
                                            new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                    .ExpectInLog("No patches are installed, download directly");
                lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

                // パッチの差分更新
                lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v9"));

                lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180, SuccessPattern = "\\[Apply\\]" });
                // 3 回くらいスリープをして問題ないか見る
                for (int i = 0; i < 10; ++i)
                {
                    lib.Sleep();
                    System.Threading.Thread.Sleep(500);
                }
                lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });
            }
        }
#endif


        [TestMethod]
        public void DownloadPatchAndCancel()
        {
            var lib = new TestOnTargetLibrary(this.TestContext);

            // アプリをローカルインストール
            lib.RunDevMenuCommandSystem("application install " + GetOutputApplicationPath("049a-0") + ";");
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(0));


            // パッチをダウンロード
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v2"),
                                        new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("No patches are installed, download directly");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // パッチの差分更新
            lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v9"));
            lib.Run(GetHelperPath(), "WaitAndCancelApplyDelta 0x0100f6900049a000", new RunningOptions { FailureTimeout = 90 })
                .ExpectInLog("[apply] call cancel");

            // meta only のデータが残らないことの確認
            lib.RunDevMenuCommandSystem("patch list-update-relates")
                .NotExpectInLog("FragmentOnly");
        }

        [TestMethod]
        public void PatchToCardTest()
        {
            // パッチでの更新動作
            var lib = new TestOnTargetLibrary(this.TestContext);

            // アプリ本編、パッチ、デルタをインストール
            lib.WriteToCard(GetOutputApplicationPath("049a-0"));

            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(0));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "0.0");

            lib.RunDevMenuCommand("patch install " + GetOutputPatchPath("049a-2"));
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(2));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "2.0");

            lib.RunDevMenuCommand("patch install " + GetOutputPatchPath("049a-4"));
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(4));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "4.0");

            lib.RunDevMenuCommand("patch install " + GetOutputPatchPath("049a-5"));
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(5));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "5.0");
        }
        [TestMethod]
        public void OnCardPatchTest()
        {
            // パッチでの更新動作
            var lib = new TestOnTargetLibrary(this.TestContext);

            lib.WriteToCardWithPatch(GetOutputApplicationPath("049a-0"), GetOutputPatchPath("049a-2"));
            lib.RunDevMenuCommandSystem("application check-launch-rights 0x0100f6900049a000")
                .ExpectInLog("[SUCCESS]");
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog("[ApplicationLaunchManager] Import ticket on card")
                .ExpectInLog(GetApplicationVerifier(2));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "2.0");

            // すでにインポート済みなので、ログはでない
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .NotExpectInLog("[ApplicationLaunchManager] Import ticket on card")
                .ExpectInLog(GetApplicationVerifier(2));

            lib.WriteToCardWithPatch(GetOutputApplicationPath("049a-0"), GetOutputPatchPath("049a-4"));
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(4));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "4.0");

            lib.WriteToCardWithPatch(GetOutputApplicationPath("049a-0"), GetOutputPatchPath("049a-5"));
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(5));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "5.0");
        }
#if false // record に基づいた cleanup が行われるようになった関係で、このテストは泣く泣く封印
        [TestMethod]
        public void ApplyDeltaTest()
        {
            // パッチをパッチ間差分で更新したときの動作
            var lib = new TestOnTargetLibrary(this.TestContext);

            // アプリ本編、パッチ、デルタをインストール
            lib.RunDevMenuCommandSystem("application install " + GetOutputApplicationPath("049a-0") + ";"
                                      + "patch install " + GetOutputPatchPath("049a-2") + ";"
                                      + "patch install-delta " + GetOutputDeltaPath("049a-4"));

            // 適用
            lib.Run(GetHelperPath(), "ApplyDelta 0x0100f6900049a000 0x0100f6900049ac00")
                .NotExpectInLog("id not found")
                .ExpectInLog("[Progress] 2", "[Progress] 3");

            // 適用後は、version があがっている
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(4));
            lib.RunDevMenuCommand("application list")
                .ExpectInLog("0x0100f6900049a000", "4.0");

            // デルタおよびパッチの削除、再インストール
            // デルタの歴史をマージしてしまうと、patch の meta が変わるため、連続適用ができない
            // 次のバージョン (サイズが変わる)
            lib.RunDevMenuCommandSystem("patch uninstall-delta 0x0100f6900049ac00;"
                                      + "patch uninstall 0x0100f6900049a000;"
                                      + "patch install " + GetOutputPatchPath("049a-4") + ";"
                                      + "patch install-delta " + GetOutputDeltaPath("049a-5"));
            lib.Run(GetHelperPath(), "ApplyDelta 0x0100f6900049a000 0x0100f6900049ac00")
                .NotExpectInLog("id not found")
                .ExpectInLog("[ApplyDelta] Resize")
                .ExpectInLog("[Progress] 2", "[Progress] 3");

            // 適用後は、version があがっている
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(5));
            lib.RunDevMenuCommand("application list")
                .ExpectInLog("0x0100f6900049a000", "5.0");


            // デルタおよびパッチの削除、再インストール
            lib.RunDevMenuCommandSystem("patch uninstall 0x0100f6900049a000;"
                                      + "patch install " + GetOutputPatchPath("049a-5") + ";"
                                      + "patch install-delta " + GetOutputDeltaPath("049a-7"));
            lib.Run(GetHelperPath(), "ApplyDeltaWithRepeatedCancel 0x0100f6900049a000 0x0100f6900049ac00")
                .NotExpectInLog("id not found")
                .ExpectInLog("[Progress] 3");

            // 適用後は、version があがっている
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(7));
            lib.RunDevMenuCommand("application list")
                .ExpectInLog("0x0100f6900049a000", "7.0");

            // いったん削除
            // アプリ本編、パッチをインストールし、DevMenuCommand から更新
            lib.RunDevMenuCommandSystem("application uninstall 0x0100f6900049a000;"
                                      + "application reset-required-version --all;"
                                      + "patch uninstall-delta 0x0100f6900049ac00;"
                                      + "application install " + GetOutputApplicationPath("049a-0") + ";"
                                      + "patch install " + GetOutputPatchPath("049a-2") + ";"
                                      + "patch apply-delta " + GetOutputDeltaPath("049a-4"));

            // 適用後は、version があがっている
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(4));
            lib.RunDevMenuCommand("application list")
                .ExpectInLog("0x0100f6900049a000", "4.0");

            // デルタおよびパッチの削除、再インストール
            lib.RunDevMenuCommandSystem("patch uninstall 0x0100f6900049a000;"
                                      + "patch install " + GetOutputPatchPath("049a-4") + ";"
                                      + "patch apply-delta " + GetOutputDeltaPath("049a-5"));

            // 適用後は、version があがっている
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(5));
            lib.RunDevMenuCommand("application list")
                .ExpectInLog("0x0100f6900049a000", "5.0");
        }
#endif

        [TestMethod]
        public void ApplyPatchDeltaTest()
        {
            // パッチをパッチ間差分で更新したときの動作。パッチ間差分の情報はパッチから取得する
            var lib = new TestOnTargetLibrary(this.TestContext);

            // アプリ本編、パッチ、デルタをインストール (デルタは、ダウンロードする代わり)
            lib.RunDevMenuCommand("application install " + GetOutputApplicationPath("049a-0") + ";"
                                + "patch install " + GetOutputPatchPath("049a-2"));

            // 更新後のパッチのメタを、placeholder に入れる
            {
                lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-4"));

                lib.Run(GetHelperPath(), "ApplyPatchDelta 0x0100f6900049a000")
                    .NotExpectInLog("id not found")
                    .ExpectInLog("[Progress] 2", "[Progress] 3");

                // 適用後は、version があがっている
                lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                    .ExpectInLog(GetApplicationVerifier(4));
                lib.RunDevMenuCommand("application list")
                    .ExpectInLog("0x0100f6900049a000", "4.0");
            }
            // 次のバージョン
            {
                lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-5"));

                lib.Run(GetHelperPath(), "ApplyPatchDelta 0x0100f6900049a000")
                    .NotExpectInLog("id not found")
                    .ExpectInLog("[Progress] 2", "[Progress] 3");

                // 適用後は、version があがっている
                lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                    .ExpectInLog(GetApplicationVerifier(5));
                lib.RunDevMenuCommand("application list")
                    .ExpectInLog("0x0100f6900049a000", "5.0");
            }
            // タスクの作り直し付き
            {
                lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-7"));
                lib.Run(GetHelperPath(), "ApplyPatchDeltaWithRepeatedCancel 0x0100f6900049a000")
                    .NotExpectInLog("id not found")
                    .ExpectInLog("[Progress] 3");


                // 適用後は、version があがっている
                lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                    .ExpectInLog(GetApplicationVerifier(7));
                lib.RunDevMenuCommand("application list")
                    .ExpectInLog("0x0100f6900049a000", "7.0");
            }
        }
        [TestMethod]
        public void ApplyPatchDeltaTestInOtherStorage()
        {
            // パッチをパッチ間差分で更新したときの動作。パッチ間差分の情報はパッチから取得する
            var lib = new TestOnTargetLibrary(this.TestContext);

            // アプリ本編、パッチ、デルタをインストール (デルタは、ダウンロードする代わり)
            lib.RunDevMenuCommand("application install " + GetOutputApplicationPath("049a-0") + ";"
                                + "patch install " + GetOutputPatchPath("049a-2"));

            // 更新後のパッチのメタを、placeholder に入れる
            {
                lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-4") + " sdcard");

                lib.Run(GetHelperPath(), "ApplyPatchDelta 0x0100f6900049a000")
                    .NotExpectInLog("id not found")
                    .ExpectInLog("[Progress] 2", "[Progress] 3");

                // 適用後は、version があがっている
                lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                    .ExpectInLog(GetApplicationVerifier(4));
                lib.RunDevMenuCommand("application list")
                    .ExpectInLog("0x0100f6900049a000", "4.0");
            }
            // 次のバージョン
            {
                lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-5") + " sdcard");

                lib.Run(GetHelperPath(), "ApplyPatchDelta 0x0100f6900049a000")
                    .NotExpectInLog("id not found")
                    .ExpectInLog("[Progress] 2", "[Progress] 3");

                // 適用後は、version があがっている
                lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                    .ExpectInLog(GetApplicationVerifier(5));
                lib.RunDevMenuCommand("application list")
                    .ExpectInLog("0x0100f6900049a000", "5.0");
            }
            // タスクの作り直し付き
            {
                lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-7") + " sdcard");
                lib.Run(GetHelperPath(), "ApplyPatchDeltaWithRepeatedCancel 0x0100f6900049a000")
                    .NotExpectInLog("id not found")
                    .ExpectInLog("[Progress] 3");


                // 適用後は、version があがっている
                lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                    .ExpectInLog(GetApplicationVerifier(7));
                lib.RunDevMenuCommand("application list")
                    .ExpectInLog("0x0100f6900049a000", "7.0");
            }
        }
        Func<string, bool> GetRequiredSizeVerifier(long clusterCountMin, long clusterCountMax)
        {
            const long ClusterSizeMax = 256 * 1024; // = nn::ncm::MaxClusterSize
            return (log) =>
                {
                    var match = new System.Text.RegularExpressions.Regex("Required size: (\\d+)").Match(log);
                    if(!match.Success)
                    {
                        return false;
                    }
                    long clusterCount = (long.Parse(match.Groups[1].Value) + (ClusterSizeMax - 1)) / ClusterSizeMax;
                    var result = clusterCountMin <= clusterCount && clusterCount <= clusterCountMax;
                    if (!result)
                    {
                        Utility.WriteLine("required size verify failed: " + clusterCount + " expeted (" + clusterCountMin + ", " + clusterCountMax + ")");
                    }
                    return result;
                };
        }
        [TestMethod]
        public void ApplyPatchCausesNotEnoughSpace()
        {
            var lib = new TestOnTargetLibrary(this.TestContext);

            // アプリ本編、パッチ、デルタをインストール (デルタは、ダウンロードする代わり)
            lib.RunDevMenuCommand("application install " + GetOutputApplicationPath("049a-0") + ";"
                                + "patch install " + GetOutputPatchPath("049a-2"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-4"));
            lib.RunDevMenuCommand("debug fill-nand-free-space");

            lib.Run(GetHelperPath(), "RegisterAndExpectNotEnoughSpace 0x0100f6900049a000")
                .ExpectInLog("Detect not enough space")
                .NotExpectInLog("Storage: 6")
                .ExpectInLog(GetRequiredSizeVerifier(44, 45));
            lib.RunDevMenuCommandSystem("debug empty-nand-free-space;"
                                      + "application resume-download 0x0100f6900049a000");

            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(4));
            lib.RunDevMenuCommand("application list")
                .ExpectInLog("0x0100f6900049a000", "4.0");

        }
        [TestMethod]
        public void DownloadPatchCausesNotEnoughSpaceAndVerifyStorage()
        {
            // Detect 時に Storage: 6 が出力される = StorageId::Any が返ってきている
            var lib = new TestOnTargetLibrary(this.TestContext);

            lib.RunDevMenuCommandSystem("application create-download-task 0x0100f6900049a000;"
                          + "application wait-download 0x0100f6900049a000");
            lib.RunDevMenuCommandSystem("debug fill-nand-free-space; debug fill-sdcard-free-space;"
                                      + "application launch 0x0100f6900049a000");
            lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v2"),
                            new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("No patches are installed, download directly");
            lib.Run(GetHelperPath(), "ExpectNotEnoughSpace 0x0100f6900049a000")
                .ExpectInLog("Detect not enough space")
                .ExpectInLog("Storage: 6");

            // 容量を復活させてダウンロードしきる
            lib.RunDevMenuCommandSystem("debug empty-nand-free-space;"
                                      + "application resume-download 0x0100f6900049a000");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // 更新ダウンロード
            lib.RunDevMenuCommand("debug fill-nand-free-space;");
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v4"),
                                        new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("register dependency: 0100f6900049a800, 262144");

            lib.Run(GetHelperPath(), "ExpectNotEnoughSpace 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 })
                .ExpectInLog("Detect not enough space")
                .ExpectInLog("Storage: 6"); // パッチがすでにインストールされているので、Any が返ってきてはいけない

            // 容量を復活させてダウンロードしきる
            lib.RunDevMenuCommandSystem("debug empty-nand-free-space;"
                                      + "application resume-download 0x0100f6900049a000");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // 更新ダウンロード、ApplyDelta 時の容量不足。v4 -> v5 が必要容量が大きいのでそこで行う。
            // 異なる ID のパッチを FragmentOnly でインストールし、fill して削除する
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("0148-5"));
            lib.RunDevMenuCommandSystem("debug fill-nand-free-space --margin 6144;"
                                      + "patch uninstall 0x010000000000b148");
            lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v5"),
                                        new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("register dependency: 0100f6900049a800, 327680");

            // download は成功するが、Apply で容量不足になる (ExpectNotEnoughSpace で待つと、ダウンロードの commit がされないのでこうする)
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 })
                 .ExpectInLog("Apply suspended, state is NotEnoughSpace");
            lib.Run(GetHelperPath(), "ExpectNotEnoughSpace 0x0100f6900049a000")
                .ExpectInLog("Detect not enough space")
                .NotExpectInLog("Storage: 6") // パッチがすでにインストールされているので、Any が返ってきてはいけない
                .ExpectInLog(GetRequiredSizeVerifier(63, 64));

            // この状態で電断も大丈夫か確認
            lib.Reboot();

            // 容量を復活させて、適用再開するかを確認
            lib.RunDevMenuCommandSystem("debug empty-nand-free-space;"
                                      + "application resume-download 0x0100f6900049a000");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // 実行も確認
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(5));
            lib.RunDevMenuCommand("application list")
                .ExpectInLog("0x0100f6900049a000", "5.0");

            // 直接更新も確認
            lib.RunDevMenuCommand("patch uninstall 0x0100f6900049a000;"
                                + "patch install " + GetOutputPatchPath("049a-2-other") + " --skip-patch-combination-check");
            lib.RunDevMenuCommand("debug fill-nand-free-space;");
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v9"),
                                        new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("Installed patch is NOT registered in the history, download directly");

            lib.Run(GetHelperPath(), "ExpectNotEnoughSpace 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 })
                .ExpectInLog("Detect not enough space")
                .ExpectInLog("Storage: 6"); // 3.0NUP 以降、ダウンロードストレージは常に任意

            // 容量を復活させてダウンロードしきる
            lib.RunDevMenuCommandSystem("debug empty-nand-free-space;"
                                      + "application resume-download 0x0100f6900049a000");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });
        }
        [TestMethod]
        public void DownloadPatchCausesNotEnoughSpaceAndVerifyStorage2()
        {
            // SD がささっている前提で SD と出るか、NAND と出るかをちゃんとみる
            var lib = new TestOnTargetLibrary(this.TestContext);

            lib.RunDevMenuCommandSystem("application create-download-task 0x0100f6900049a000;"
                                      + "application wait-download 0x0100f6900049a000");
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(0));


            // SD にダウンロード
            lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v2"),
                new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("No patches are installed, download directly");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // SD を埋める
            lib.RunDevMenuCommand("debug fill-sdcard-free-space;");

            // NAND にダウンロードされて、適用時にサイズエラーになる
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v4"),
                                        new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("register dependency: 0100f6900049a800, 262144");

            lib.Run(GetHelperPath(), "ExpectNotEnoughSpace 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 })
                .ExpectInLog("Detect not enough space")
                .ExpectInLog("Storage: 5"); // 適用時なので、SD が返ってくる

            // 容量を復活させて適用しきる
            lib.RunDevMenuCommandSystem("debug empty-sdcard-free-space;"
                                      + "application resume-download 0x0100f6900049a000");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // 実行確認
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(4));


            // 逆に SD が空いていて NAND が埋まっている場合
            lib.RunDevMenuCommandSystem("application reset-required-version --all;"
                                      + "patch uninstall 0x0100f6900049a000 -s sdcard;"
                                      + "debug fill-sdcard-free-space;");
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(0));


            // SD が埋まっているので NAND にダウンロードするはず
            lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v5"),
                new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
               .ExpectInLog("No patches are installed, download directly");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // SD を開けて NAND を埋める
            lib.RunDevMenuCommandSystem("debug empty-sdcard-free-space;"
                                      + "debug fill-nand-free-space;");

            // SD にダウンロードされて、適用時にサイズエラーになることを確認
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v9"),
                            new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 });

            lib.Run(GetHelperPath(), "ExpectNotEnoughSpace 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 })
                .ExpectInLog("Detect not enough space")
                .ExpectInLog("Storage: 4"); // パッチがインストールされている NAND の容量不足

            // 容量を復活させてダウンロードしきる
            lib.RunDevMenuCommandSystem("debug empty-nand-free-space;"
                                      + "application resume-download 0x0100f6900049a000");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // 実行確認
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(9));
        }
        [TestMethod]
        public void CannotCreateTaskSituationTest()
        {
            // 何かのダウンロード中/パッチ間差分適用中に、dtl でパッチをさらにダウンロードしようとしても登録されない

            var lib = new TestOnTargetLibrary(this.TestContext);

            lib.RunDevMenuCommandSystem("application create-download-task 0x0100f6900049a000;"
                                      + "application wait-download 0x0100f6900049a000");
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(0));


            // ダウンロード中
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v2"));
            lib.Run(GetHelperPath(), "WaitAndSuspendDownloadPermanently 0x0100f6900049a000", new RunningOptions { SuccessPattern = "\"\\[apply\\] stop request server\"" });
            lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v5"), new RunningOptions { SuppressAutoKill = true })
                .ExpectInLog("[ApplicationVersionManager] Found recommended version 327680", "[CanCreateNetworkInstallTask] Download task exists, cannot create task", "[ApplicationVersionManager] Skip create install task")
                .NotExpectInLog("[ApplicationVersionManager] Created");

            // 電断 -> 再投入をし、ダウンロードしきる (stop request server の解除)
            lib.Reboot();
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // 実行も確認
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(2));
            lib.RunDevMenuCommand("application list")
                .ExpectInLog("0x0100f6900049a000", "2.0");

            // パッチ間差分適用中
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v9"));
            lib.Run(GetHelperPath(), "WaitAndSuspendApplyDeltaPermanently 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180, SuccessPattern = "\"\\[apply\\] stop request server\"" });
            lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v10"), new RunningOptions { SuppressAutoKill = true })
                .ExpectInLog("[ApplicationVersionManager] Found recommended version 655360", "[CanCreateNetworkInstallTask] Apply delta task exists, cannot create task", "[ApplicationVersionManager] Skip create install task")
                .NotExpectInLog("[ApplicationVersionManager] Created");

            // 電断 -> 再投入をし、適用しきる (stop request server の解除)
            lib.Reboot();
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // 実行も確認
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(9));
            lib.RunDevMenuCommand("application list")
                .ExpectInLog("0x0100f6900049a000", "9.0");

            // パッチ以外は登録されることの確認
            lib.RunDevMenuCommandSystem("application uninstall 0x0100f6900049a000;"
                                      + "application reset-required-version --all;"
                                      + "application push-download-task-list " + GetDtlPath("dtl-patch-v2"));
            lib.Run(GetHelperPath(), "WaitAndSuspendDownloadPermanently 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180, SuccessPattern = "\"\\[apply\\] stop request server\"" });

            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-app-patch-v8"), new RunningOptions { SuppressAutoKill = true })
                .ExpectInLog("[DownloadTaskListManager] Patch task exists, skip key 0100f6900049a800, 524288")
                .ExpectInLog("[DownloadTaskListManager] Add to task: appId 0x0100f6900049a000 keyCount 1")
                .NotExpectInLog("[DownloadTaskListManager]     key id 0x0100f6900049a800")
                .ExpectInLog("[DownloadTaskListManager]     key id 0x0100f6900049a000 version 0 type 128");

            // 電断 -> 再投入をし、ダウンロードしきる (stop request server の解除)
            lib.Reboot();
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // 実行も確認
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(2));
            lib.RunDevMenuCommand("application list")
                .ExpectInLog("0x0100f6900049a000", "2.0");
        }
        [TestMethod]
        public void CannotCreateTaskSituationInRequestUpdate()
        {
            // 何かのダウンロード中/パッチ間差分適用中に RequestUpdate しても登録されない

            var lib = new TestOnTargetLibrary(this.TestContext);
            lib.RunDevMenuCommandSystem("application install " + GetOutputApplicationPath("049a-0") + ";");
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(0));


            // ダウンロード中
            lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v2"));
            lib.Run(GetHelperPath(), "SuspendDownloadAndRequestUpdateApplication 0x0100f6900049a000")
                .ExpectInLog("[Failed] 00010410");

            // タスク完了を待つ
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // パッチ間差分適用中
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v9"));
            lib.Run(GetHelperPath(), "SuspendApplyDeltaAndRequestUpdateApplication 0x0100f6900049a000", new RunningOptions { FailureTimeout = 90 })
                .ExpectInLog("[Failed] 00010410");

            // タスク完了を待つ
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });
        }
        [TestMethod]
        public void CannotCreateTaskSituationInAutoUpdate()
        {
            // 何かのダウンロード中/パッチ間差分適用中に version list を更新してもパッチタスクは登録されない

            var lib = new TestOnTargetLibrary(this.TestContext);
            lib.RunDevMenuCommandSystem("application install " + GetOutputApplicationPath("049a-0") + ";");
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(0));


            // ダウンロード中
            lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v2"), new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"" });
            lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v5"), new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"Skip create install task\"" })
                .ExpectInLog("[CanCreateNetworkInstallTask] Download task exists, cannot create tas");

            // タスク完了を待つ
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // パッチ間差分適用中
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v9"), new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"" });
            lib.Run(GetHelperPath(), "WaitAndSuspendApplyDeltaPermanently 0x0100f6900049a000", new RunningOptions { SuccessPattern = "\"\\[apply\\] stop request server\"" });
            lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v10"), new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"Skip create install task\"" })
                .ExpectInLog("[CanCreateNetworkInstallTask] Apply delta task exists, cannot create tas");

            // タスク完了を待つ
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });
        }
        [TestMethod]
        public void FormatSdTest()
        {
            // NAND に存在するパッチ間差分適用タスクは、SD のフォーマットで消える。SD のものは削除される

            // SD をフォーマット
            var lib = new TestOnTargetLibrary(this.TestContext);
            lib.RunDevMenuCommandSystem("sdcard format;");
            lib.Reboot();

            // SD および NAND に異なるパッチをインストール
            lib.RunDevMenuCommandSystem("application install " + GetOutputApplicationPath("049a-0") + " -s sdcard;"
                                      + "application install " + GetOutputApplicationPath("0148-0") + " -s builtin;"
                                      + "patch install " + GetOutputPatchPath("049a-2") + " -s builtin;"
                                      + "patch install " + GetOutputPatchPath("0148-2") + " -s sdcard;");

            // 起動確認 (SD と NAND コンボの意味を含む)
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(2));
            lib.RunDevMenuCommand("application list")
                .ExpectInLog("0x0100f6900049a000", "2.0")
                .ExpectInLog("0x010000000000b148", "2.0");

            // ローカルでパッチ間差分適用タスクを作成、その後 SD をフォーマット
            // 適用後のタスクの状態を確認
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-4"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("0148-4") + " sdcard");
            lib.Run(GetHelperPath(), "FormatSdTest 0x0100f6900049a000 0x010000000000b148")
                .NotExpectInLog("[Failure] NAND task killed unexpectedly")
                .NotExpectInLog("[Failure] SD task exists unexpectedly");

            // 再起動
            lib.Reboot();
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000;"
                                      + "application wait-download 0x010000000000b148;", new RunningOptions { FailureTimeout = 180 });

            // NAND 側は更新されていることを確認
            lib.RunDevMenuCommand("application list")
                .ExpectInLog("0x0100f6900049a000              4.0")
                .NotExpectInLog("0x010000000000b148              4.0"); // NotExpect に連続でヒットかどうかがないのでこうする

            // アプリをカードに焼いて起動を確認 (アプリケーションのレコードがあるので、インストールできない)
            lib.WriteToCard(GetOutputApplicationPath("049a-0"));
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(4));
        }
        [TestMethod]
        public void CleanupIntermediately()
        {
            // nim のタスクを作り、キャンセルして、余計なものが残らないかを見る
            var lib = new TestOnTargetLibrary(this.TestContext);

            // すぐ容量不足になったときは、元のパッチは残る
            lib.RunDevMenuCommandSystem("patch install " + GetOutputPatchPath("049a-4"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-5"));
            lib.RunDevMenuCommandSystem("debug fill-nand-free-space;"
                                      + "patch list-update-relates;")
               .ExpectInLog("0100f6900049a800", "262144", "(Full)")
               .ExpectInLog("0100f6900049a800", "327680", "(FragmentOnly)");

            lib.Run(GetHelperPath(), "RegisterAndExpectNotEnoughSpace 0x0100f6900049a000")
                .ExpectInLog("Detect not enough space")
                .NotExpectInLog("Storage: 6")
                .ExpectInLog(GetRequiredSizeVerifier(75, 76));
            lib.RunDevMenuCommandSystem("application cancel-download 0x0100f6900049a000;"
                                      + "patch list-update-relates")
                .ExpectInLog("0100f6900049a800", "262144", "(Full)")
                .NotExpectInLog("not installed");
            lib.RunDevMenuCommand("debug empty-nand-free-space;");

            // それ以外のタイミングでは何も残らない
            // 適用開始後、容量不足
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-5"));
            lib.RunDevMenuCommandSystem("debug fill-nand-free-space --margin 6144;"
                                      + "patch list-update-relates;")
               .ExpectInLog("0100f6900049a800", "262144", "(Full)")
               .ExpectInLog("0100f6900049a800", "327680", "(FragmentOnly)");

            lib.Run(GetHelperPath(), "RegisterAndExpectNotEnoughSpace 0x0100f6900049a000")
                .ExpectInLog("Detect not enough space")
                .NotExpectInLog("Storage: 6")
                .ExpectInLog(GetRequiredSizeVerifier(63, 64));
            lib.RunDevMenuCommandSystem("application cancel-download 0x0100f6900049a000;"
                                      + "patch list-update-relates")
                .NotExpectInLog("0100f6900049a800");
            lib.RunDevMenuCommand("debug empty-nand-free-space;");

            // タスク適用完了、コミット前
            lib.RunDevMenuCommandSystem("patch install " + GetOutputPatchPath("049a-4") + " --force");
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-5"));
            lib.RunDevMenuCommandSystem("patch list-update-relates;")
               .ExpectInLog("0100f6900049a800", "262144", "(Full)")
               .ExpectInLog("0100f6900049a800", "327680", "(FragmentOnly)");

            lib.Run(GetHelperPath(), "RegisterApplyDeltaTaskAndDestryWithoutCommit 0x0100f6900049a000")
                .ExpectInLog("Destroyed");

            lib.RunDevMenuCommandSystem("patch list-update-relates")
                .NotExpectInLog("0100f6900049a800");

            // タスク適用途中
            lib.RunDevMenuCommandSystem("patch install " + GetOutputPatchPath("049a-4") + " --force");
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-5"));
            lib.RunDevMenuCommandSystem("patch list-update-relates;")
               .ExpectInLog("0100f6900049a800", "262144", "(Full)")
               .ExpectInLog("0100f6900049a800", "327680", "(FragmentOnly)");

            lib.Run(GetHelperPath(), "RegisterApplyDeltaTaskAndDestryIntermediately 0x0100f6900049a000")
                .ExpectInLog("Destroyed");

            lib.RunDevMenuCommandSystem("patch list-update-relates")
                .NotExpectInLog("0100f6900049a800");
        }

#if false // 自動テストではさすがに時間がかかるので実行させたくない
        [TestMethod]
        public void DownloadPatchCausesNotEnoughSpaceAndVerifyStorageRepeat()
        {
            // Detect 時に Storage: 6 が出力される = StorageId::Any が返ってきている
            var lib = new TestOnTargetLibrary(this.TestContext);

            lib.RunDevMenuCommand("application install " + GetOutputApplicationPath("049a-0"));

            for (int i = 0; i < 100; ++i)
            {
                lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v4"),
                            new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 });
                lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000");

                // 更新ダウンロード、ApplyDelta 時の容量不足。v4 -> v5 が必要容量が大きいのでそこで行う。
                // 一度 delta をインストールした状態で fill し、delta を消してダウンロードする
                lib.RunDevMenuCommandSystem("patch install-delta " + GetOutputDeltaPath("049a-5") + ";"
                                      + "debug fill-nand-free-space;"
                                      + "patch uninstall-delta 0x0100f6900049a000");
                lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v5"),
                                            new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                    .ExpectInLog("register dependency: 0100f6900049a800, 327680");

                // download は成功するが、Apply で容量不足になる (ExpectNotEnoughSpace で待つと、ダウンロードの commit がされないのでこうする)
                lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000")
                     .ExpectInLog("Apply suspended, state is NotEnoughSpace");
                lib.Run(GetHelperPath(), "ExpectNotEnoughSpace 0x0100f6900049a000")
                    .ExpectInLog("Detect not enough space")
                    .NotExpectInLog("Storage: 6"); // パッチがすでにインストールされているので、Any が返ってきてはいけない

                // この状態で電断も大丈夫か確認
               lib.Reboot();

                // 容量を復活させて、適用再開するかを確認
                lib.RunDevMenuCommand("debug empty-nand-free-space");
                lib.Run(GetHelperPath(), "resume-download 0x0100f6900049a000");
                lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000");

                // 実行も確認
                lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                    .ExpectInLog(GetApplicationVerifier(5));
                lib.RunDevMenuCommand("application list")
                    .ExpectInLog("0x0100f6900049a000", "5.0");
                lib.RunDevMenuCommand("patch uninstall 0x0100f6900049a000");
            }
        }
#endif
#if false
        // もうダウンロードが終わったらタスクが作られるので要らない
        [TestMethod]
        public void DownloadPatchAndApplyInNim()
        {
            var lib = new TestOnTargetLibrary(this.TestContext);

            // アプリ本編をインストール
            lib.RunDevMenuCommand("application install " + GetOutputApplicationPath("049a-0"));

            // パッチをダウンロード
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v2"),
                            new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("No patches are installed, download directly");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000");

            lib.RunDevMenuCommandSystem("patch set-prefer-delta true");

            // パッチ間差分をダウンロード
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v5"),
                            new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("register dependency: 0100f6900049a800, 327680")
                .ExpectInLog("register dependency: 0100f6900049ac00, 327680")
                .ExpectInLog("register dependency: 0100f6900049a800, 262144")
                .ExpectInLog("register dependency: 0100f6900049ac00, 262144");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000");

            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(5));
            lib.RunDevMenuCommand("application list")
                .ExpectInLog("0x0100f6900049a000", "5.0");
        }
#endif
        [TestMethod]
        public void OrderCheck()
        {
            // パッチをパッチ間差分で更新したときの動作。パッチ間差分の情報はパッチから取得する
            var lib = new TestOnTargetLibrary(this.TestContext);

            // アプリ本編、パッチ、デルタをインストール (デルタは、ダウンロードする代わり)
            lib.RunDevMenuCommand("application install " + GetOutputApplicationPath("049a-0") + ";"
                                + "patch install " + GetOutputPatchPath("049a-2") + ";"
                                + "application install " + GetOutputApplicationPath("0148-0") + ";"
                                + "patch install " + GetOutputPatchPath("0148-2") + ";"
                                + "application install " + GetOutputApplicationPath("0147-0") + ";"
                                + "patch install " + GetOutputPatchPath("0147-2") + ";");

            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-4"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("0148-4"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("0147-4"));

            lib.Run(GetHelperPath(), "OrderChecker 0x0100f6900049a000 0x010000000000b148 0x010000000000b147")
                .ExpectInLog("[ApplyDelta] key: id: 010000000000b948,", "[ApplyDelta] key: id: 010000000000b947,");
        }

        [TestMethod]
        public void CleanupWhenForcePowerOff()
        {
            // 強制電源断で、クリーンナップされるか
            var lib = new TestOnTargetLibrary(this.TestContext);

            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-4"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("0148-4"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("0147-4"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-5"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("0148-5"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("0147-5"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-7"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("0148-7"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("0147-7"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-8"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("0148-8"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("0147-8"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-9"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("0148-9"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("0147-9"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-10"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("0148-10"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("0147-10"));

            lib.RunDevMenuCommandSystem("patch list-update-relates")
                .ExpectInLog("FragmentOnly");

            // タスクを作成 (存在しないバージョンのダウンロードをして、fatal にする。タスクが無いと再起動時に削除されるため)
            lib.RunDevMenuCommandSystem("application create-download-task 0x0100f6900049a000 --type Patch --id 0x0100f6900049a800 --version 1;");

            // 正しい電源断では消えないことを確認
            lib.Reboot();

            lib.RunDevMenuCommandSystem("patch list-update-relates")
                .ExpectInLog("FragmentOnly");

            // runontarget --reset (強制電源断) では消えることを確認
            lib.RunDevMenuCommandSystem("patch list-update-relates", new RunningOptions() { WithReset = true })
                .NotExpectInLog("FragmentOnly");
        }
        [TestMethod]
        public void CleanupSdcardInserted()
        {
            var lib = new TestOnTargetLibrary(this.TestContext);

            // fragment をインストール
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("0148-8"));
            lib.Run(GetHelperPath(), "InstallFragmentsOnly " + GetOutputPatchPath("049a-5") + " sdcard");
            lib.RunDevMenuCommandSystem("patch list-update-relates -s sdcard;")
                .ExpectInLog("0100f6900049a800", "327680", "(FragmentOnly)");
            lib.RunDevMenuCommandSystem("patch list-update-relates;")
                .ExpectInLog("010000000000b948", "524288", "(FragmentOnly)");

            // タスクを作成 (存在しないバージョンのダウンロードをして、fatal にする。タスクが無いと再起動時に削除されるため)
            lib.RunDevMenuCommandSystem("application create-download-task 0x0100f6900049a000 --type Patch --id 0x0100f6900049a800 --version 1;");

            // 通常電源断では消えないことを確認
            lib.Reboot();
            lib.RunDevMenuCommandSystem("patch list-update-relates -s sdcard;")
                .ExpectInLog("0100f6900049a800", "327680", "(FragmentOnly)");
            lib.RunDevMenuCommandSystem("patch list-update-relates;")
                .ExpectInLog("010000000000b948", "524288", "(FragmentOnly)");

            // SD を debug disable して再起動 -> debug enable して再起動
            lib.RunDevMenuCommandSystem("debug disable-mount-sdcard;");
            lib.Reboot();
            lib.RunDevMenuCommandSystem("debug enable-mount-sdcard;");
            lib.Reboot();

            // SD の fragment が消えていることを確認。NAND は消えないことを確認
            lib.RunDevMenuCommandSystem("patch list-update-relates -s sdcard;")
                .NotExpectInLog("0100f6900049a800", "327680", "(FragmentOnly)");
            lib.RunDevMenuCommandSystem("patch list-update-relates;")
                .ExpectInLog("010000000000b948", "524288", "(FragmentOnly)");
        }
        [TestMethod]
        public void PatchCommandTest()
        {
            var lib = new TestOnTargetLibrary(this.TestContext);

            // 元アプリケーションがインストールされていない場合
            lib.RunDevMenuCommand("patch install " + GetOutputPatchPath("049a-2"));
            lib.RunDevMenuCommand("patch list")
                .ExpectInLog("0x0100f6900049a000", "2.0", "2");

            lib.RunDevMenuCommand("patch install " + GetOutputPatchPath("049a-5"));
            lib.RunDevMenuCommand("patch list")
                .ExpectInLog("0x0100f6900049a000", "5.0", "5");

            // 普通では、低いバージョンはインストールできない
            lib.RunDevMenuCommand("patch install " + GetOutputPatchPath("049a-4"))
                .ExpectInLog("Same or higher version nsp is already installed");
            lib.RunDevMenuCommand("patch list")
                .ExpectInLog("0x0100f6900049a000", "5.0", "5");

            // --force を付けるとインストールできる
            lib.RunDevMenuCommand("patch install " + GetOutputPatchPath("049a-4") + " --force");
            lib.RunDevMenuCommand("patch list")
                .ExpectInLog("0x0100f6900049a000", "4.0", "4");

            // パッチの削除
            lib.RunDevMenuCommand("patch uninstall 0x0100f6900049a000");
            lib.RunDevMenuCommand("patch list")
                .NotExpectInLog("0x0100f6900049a000");

            // パッチを先にインストールしてから、アプリのインストール
            lib.RunDevMenuCommand("patch install " + GetOutputPatchPath("049a-2"));
            lib.RunDevMenuCommand("patch list")
                .ExpectInLog("0x0100f6900049a000", "2.0", "2");

            lib.RunDevMenuCommand("application install " + GetOutputApplicationPath("049a-0"));
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
              .ExpectInLog(GetApplicationVerifier(2));
            lib.RunDevMenuCommand("application list")
                .ExpectInLog("0x0100f6900049a000", "2.0");

            // パッチだけ削除しても、アプリは残る
            lib.RunDevMenuCommand("patch uninstall 0x0100f6900049a000;"
                                + "application reset-required-version --all;");

            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(0));
            lib.RunDevMenuCommand("application list")
                .ExpectInLog("0x0100f6900049a000", "0.0");

            // 起動すると必須バージョンが書き換わること、reset-required-version でクリアされることの確認
            lib.RunDevMenuCommand("patch install " + GetOutputPatchPath("049a-2"));
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(2));
            lib.RunDevMenuCommandSystem("application dump-required-version")
                .ExpectInLog("0x0100f6900049a000             131072");
            lib.RunDevMenuCommand("application reset-required-version 0x0100f6900049a000")
                .ExpectInLog("Reset: 0x0100f6900049a000");
            lib.RunDevMenuCommandSystem("application dump-required-version")
                .ExpectInLog("0x0100f6900049a000                  0");

            // --all でも削除できるか
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(2));
            lib.RunDevMenuCommandSystem("application dump-required-version")
                .ExpectInLog("0x0100f6900049a000             131072");
            lib.RunDevMenuCommand("application reset-required-version --all")
                .ExpectInLog("Reset: 0x0100f6900049a000");
            lib.RunDevMenuCommandSystem("application dump-required-version")
                .ExpectInLog("0x0100f6900049a000                  0");
        }
        [TestMethod]
        public void EjectSdCardInNotEnoughSpace()
        {
            var lib = new TestOnTargetLibrary(this.TestContext);

            lib.RunDevMenuCommandSystem("application install " + GetOutputApplicationPath("049a-0") + ";");
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(0));


            // SD にダウンロード
            lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v2"),
                new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("No patches are installed, download directly");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // SD を埋める
            lib.RunDevMenuCommand("debug fill-sdcard-free-space;");

            // 更新をして、適用タスクがサイズエラーになることを確認
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v4"),
                                        new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("register dependency: 0100f6900049a800, 262144");

            lib.Run(GetHelperPath(), "ExpectNotEnoughSpace 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 })
                .ExpectInLog("Detect not enough space")
                .ExpectInLog("Storage: 5");

            // SD を抜く
            try
            {
                lib.RunDevMenuCommandSystem("debug disable-mount-sdcard;");
                lib.Reboot();

                // タスクを待って、適用タスクが fatal することを確認
                // TODO: 本来は、SD カードが抜かれた時点でタスクが fatal になるべき
                lib.RunDevMenuCommandSystem("application resume-download 0x0100f6900049a000;"
                                          + "application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 })
                    .ExpectInLog("Apply suspended, state is Fatal")
                    .ExpectInLog("Fatal: 00025889"); // nim::ResultSdCardNotActive
            }
            finally
            {
                lib.RunDevMenuCommandSystem("debug enable-mount-sdcard;");
                lib.Reboot();
            }
        }
        [TestMethod]
        public void EjectSdCardWhenDownloadingDelta()
        {
            var lib = new TestOnTargetLibrary(this.TestContext);

            lib.RunDevMenuCommandSystem("application install " + GetOutputApplicationPath("049a-0") + ";");
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(0));


            // SD にダウンロード
            lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v2"),
                new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("No patches are installed, download directly");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // NAND/SD を埋める (差分での経路決定は、埋まっていても完了する)
            lib.RunDevMenuCommand("debug fill-nand-free-space;");
            lib.RunDevMenuCommand("debug fill-sdcard-free-space;");

            // 更新をして、ダウンロードタスクがサイズエラーになることを確認
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v5"),
                                        new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("register dependency: 0100f6900049a800, 262144")
                .ExpectInLog("register dependency: 0100f6900049a800, 327680");
            lib.Run(GetHelperPath(), "ExpectNotEnoughSpace 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 })
                .ExpectInLog("Detect not enough space");

            // NAND を開ける
            lib.RunDevMenuCommand("debug empty-nand-free-space;");

            // SD を抜く
            try
            {
                lib.RunDevMenuCommandSystem("debug disable-mount-sdcard;");
                lib.Reboot();

                // タスクを待って、適用タスクがパッチが見つからないエラーになることを確認
                lib.RunDevMenuCommandSystem("application resume-download 0x0100f6900049a000;"
                                          + "application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 })
                    .ExpectInLog("Apply suspended, state is Fatal")
                    .ExpectInLog("Fatal: 00023489"); // nim::ResultSourcePatchNotFound

                // この時点でごみがあることを確認
                lib.RunDevMenuCommandSystem("patch list-update-relates")
                    .ExpectInLog("FragmentOnly");

                // キャンセル時のクリーンナップを起こさないように、適当なタスクを作成 (fatal してよし)
                lib.RunDevMenuCommandSystem("application create-download-task 0x010000000000b148 --type Patch --id 0x010000000000b948 --version 1;");

                // キャンセルして、ごみが残らないことを確認
                lib.RunDevMenuCommandSystem("application cancel-download 0x0100f6900049a000;"
                                          + "patch list-update-relates")
                    .NotExpectInLog("FragmentOnly");

            }
            finally
            {
                lib.RunDevMenuCommandSystem("debug enable-mount-sdcard;");
                lib.Reboot();
            }
        }
        [TestMethod]
        public void PrepareAgainToDeltaDownloadTask()
        {
            var lib = new TestOnTargetLibrary(this.TestContext);
            try
            {
                lib.RunDevMenuCommandSystem("debug set-boolean-fwdbg --name devmenu --key enable_application_update false;"
                                          + "debug set-boolean-fwdbg --name contents_delivery --key disable_auto_commit_forcibly true;");
                lib.Reboot();

                // パッチ間差分適用待ちの状態にして、その状態で Application を追加ダウンロードして prepare agin させる
                lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v2"));
                lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

                lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v5"));
                lib.RunDevMenuCommandSystem("application wait-download-without-commit --timeout 180000 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

                lib.RunDevMenuCommandSystem("application create-download-task 0x0100f6900049a000");
                lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 })
                    .NotExpectInLog("[FAILURE]");
                lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                    .ExpectInLog(GetApplicationVerifier(5));
                lib.RunDevMenuCommand("application list")
                    .ExpectInLog("0x0100f6900049a000", "5.0");
            }
            finally
            {
                lib.RunDevMenuCommandSystem("debug set-boolean-fwdbg --name devmenu --key enable_application_update true;"
                                          + "debug set-boolean-fwdbg --name contents_delivery --key disable_auto_commit_forcibly false;");
                lib.Reboot();
            }
        }
        [TestMethod]
        public void CommitFailed()
        {
            // commit が失敗するようにすると、view からきちんとエラーが観測できるか
            var lib = new TestOnTargetLibrary(this.TestContext);
            try
            {
                lib.RunDevMenuCommandSystem("debug set-boolean-fwdbg --name devmenu --key enable_application_update false;"
                                          + "debug set-boolean-fwdbg --name contents_delivery --key disable_auto_commit_forcibly true;");
                lib.Reboot();

                // ダウンロード中のコミット、placeholder を消してあげる
                lib.RunDevMenuCommandSystem("application create-download-task 0x0100f6900049a000;"
                                          + "application wait-download-without-commit --timeout 180000 0x0100f6900049a000");
                lib.Run(GetHelperPath(), "CleanupPlaceholders");

                lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000")
                    .ExpectInLog("Download suspended as 0x00000605");
                lib.RunDevMenuCommandSystem("application cancel-download 0x0100f6900049a000");

                // 適用中のコミット、SD にダウンロード済みでコミット待ちの状態にし、SD を抜く
                lib.RunDevMenuCommandSystem("application create-download-task 0x0100f6900049a000;"
                                          + "application create-download-task 0x0100f6900049a000 --type Patch --id 0x0100f6900049a800 --version 131072;"
                                          + "application wait-download 0x0100f6900049a000");
                lib.RunDevMenuCommandSystem("application create-download-task 0x0100f6900049a000 --type Patch --id 0x0100f6900049a800 --version 262144;");
                lib.Run(GetHelperPath(), "WaitApplyDeltaWithoutCommit 0x0100f6900049a000")
                    .ExpectInLog("Waiting commit");

                lib.RunDevMenuCommandSystem("debug disable-mount-sdcard;");
                lib.Reboot();

                // この状態で view を取得すると、タスクが fatal になっているはず
                lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000")
                    .ExpectInLog("Apply suspended, state is Fatal", "Fatal: 00025889");
            }
            finally
            {
                lib.RunDevMenuCommandSystem("debug enable-mount-sdcard;");
                lib.RunDevMenuCommandSystem("debug set-boolean-fwdbg --name devmenu --key enable_application_update true;"
                                          + "debug set-boolean-fwdbg --name contents_delivery --key disable_auto_commit_forcibly false;");
                lib.Reboot();
            }
        }

        [TestMethod]
        public void ApplyingOccupiedSizeTest()
        {
            var lib = new TestOnTargetLibrary(this.TestContext);

            // アプリをインストール
            lib.RunDevMenuCommandSystem("application create-download-task 0x0100f6900049a000;"
                                      + "application wait-download 0x0100f6900049a000");
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(0));

            // SD での 1 段階更新
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v2"),
                                        new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("No patches are installed, download directly");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v4"),
                            new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("register dependency: 0100f6900049a800, 262144");

            lib.Run(GetHelperPath(), "WaitApplyDeltaAndReportOccupiedSize 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 })
                .ExpectInLog("Success")
                .ExpectInLog(GetOccupiedSizeVerifier(2, "SdCard"));

            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(4));
            lib.RunDevMenuCommandSystem("patch uninstall 0x0100f6900049a000 -s sdcard");

            // SD での多段階更新
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v2"),
                                        new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("No patches are installed, download directly");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v10"),
                            new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("register dependency: 0100f6900049a800, 262144")
                .ExpectInLog("register dependency: 0100f6900049a800, 327680")
                .ExpectInLog("register dependency: 0100f6900049a800, 589824")
                .ExpectInLog("register dependency: 0100f6900049a800, 655360");

            lib.Run(GetHelperPath(), "WaitApplyDeltaAndReportOccupiedSize 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 })
                .ExpectInLog("Success")
                .ExpectInLog(GetOccupiedSizeVerifier(4, "SdCard"));

            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(10));
            lib.RunDevMenuCommandSystem("patch uninstall 0x0100f6900049a000 -s sdcard");

            // NAND での多段階
            lib.RunDevMenuCommand("debug fill-sdcard-free-space");
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v2"),
                            new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("No patches are installed, download directly");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            lib.RunDevMenuCommand("debug empty-sdcard-free-space");

            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v10"),
                new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("register dependency: 0100f6900049a800, 262144")
                .ExpectInLog("register dependency: 0100f6900049a800, 327680")
                .ExpectInLog("register dependency: 0100f6900049a800, 589824")
                .ExpectInLog("register dependency: 0100f6900049a800, 655360");

            lib.Run(GetHelperPath(), "WaitApplyDeltaAndReportOccupiedSize 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 })
                .ExpectInLog("Success")
                .ExpectInLog(GetOccupiedSizeVerifier(4, "BuiltInUser"))
                .ExpectInLog(GetOccupiedSizeVerifier(1, "SdCard")); // SD CARD は差分がダウンロードされるが、それは単純に減少していくだけとなる

            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(10));
            lib.RunDevMenuCommandSystem("patch uninstall 0x0100f6900049a000");

            // NAND で容量不足
            lib.RunDevMenuCommand("debug fill-sdcard-free-space");
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v2"),
                            new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("No patches are installed, download directly");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            lib.RunDevMenuCommand("debug empty-sdcard-free-space");
            lib.RunDevMenuCommand("debug fill-nand-free-space");

            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v10"),
                new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("register dependency: 0100f6900049a800, 262144")
                .ExpectInLog("register dependency: 0100f6900049a800, 327680")
                .ExpectInLog("register dependency: 0100f6900049a800, 589824")
                .ExpectInLog("register dependency: 0100f6900049a800, 655360");

            lib.Run(GetHelperPath(), "WaitApplyDeltaAndReportOccupiedSize 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 })
                .ExpectInLog("Task state is NotEnoughSpace")
                .ExpectInLog(GetOccupiedSizeVerifier(1, "BuiltInUser")) // どちらも容量が計上されることのチェック
                .ExpectInLog(GetOccupiedSizeVerifier(1, "SdCard"));     // どちらも容量が計上されることのチェック
        }
        [TestMethod]
        public void DuplicatedContentMetaEntryTest()
        {
            var lib = new TestOnTargetLibrary(this.TestContext);

            // アプリをインストール
            lib.RunDevMenuCommandSystem("application create-download-task 0x0100f6900049a000;"
                                      + "application wait-download 0x0100f6900049a000");
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
                .ExpectInLog(GetApplicationVerifier(0));

            // パッチをダウンロードして実行
            lib.RunDevMenuCommandSystem("application push-download-task-list " + GetDtlPath("dtl-patch-v4"),
                                        new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("No patches are installed, download directly");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // 古い ContentMetaDatabase エントリの作成
            lib.Run(GetHelperPath(), "CreateContentMetaDatabasePatchEntry " + GetOutputPatchPath("049a-2") + " sdcard");

            // この状態で実行できることを確認
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(4));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "4.0");

            // パッチの差分更新
            lib.RunDevMenuCommandSystem("application update-version-list " + GetVersionListPath("versionlist-v5"),
                                        new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"\\[DeltaUpdate\\] Resolved\"", FailureTimeout = 30 })
                .ExpectInLog("register dependency: 0100f6900049a800, 327680");
            lib.RunDevMenuCommandSystem("application wait-download 0x0100f6900049a000", new RunningOptions { FailureTimeout = 180 });

            // 実行できることの確認
            lib.RunDevMenuCommand("application launch 0x0100f6900049a000", new RunningOptions { IgnoreApplicationTerminate = true, SuccessPattern = "\"execute finished\"", FailureTimeout = 30 })
               .ExpectInLog(GetApplicationVerifier(5));
            lib.RunDevMenuCommand("application list")
               .ExpectInLog("0x0100f6900049a000", "5.0");
        }

        [TestMethod]
        public void MeasureApplyDeltaTime()
        {
            var lib = new TestOnTargetLibrary(this.TestContext);

            // 前者と後者の時間の差分が、適用の時間と考える
            lib.RunDevMenuCommandSystem("patch install " + GetOutputPatchPath("049a-4") + ";"
                                      + "--time patch apply-delta " + GetOutputDeltaPath("049a-5"));

            lib.RunDevMenuCommandSystem("patch uninstall 0x0100f6900049a000;"
                                      + "patch install " + GetOutputPatchPath("049a-4") + ";"
                                      + "--time patch install-delta " + GetOutputDeltaPath("049a-5"));
        }
    }
}
