﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using NactBuildLogParser;

namespace NactBuildLogParserTest
{
    [TestClass]
    public class UnitTest
    {
        [TestMethod]
        public void TestGetTargetRange()
        {
            var text = @"
[15:19:08]:	 [Step 3/4] ビルド完了
[15:19:08]:	 [Step 3/4]   indent 1
[15:19:08]:	 [Step 3/4]     indent 2
[15:19:09]:	 [Step 3/4] Process exited with code 0";

            var targets = LogParser.GetTargetRange(text);

            CollectionAssert.AreEqual(targets,
                new List<string> {
                    "ビルド完了",
                    "  indent 1",
                    "    indent 2"
                });
        }

        /// <summary>
        /// 等しいことを比較するだけなので、正しい順序関係は返さない
        /// </summary>
        public class LineWithLevelComparer : IComparer
        {
            #region IComparer メンバー

            public int Compare(object x, object y)
            {
                var x2 = (LineWithLevel)x;
                var y2 = (LineWithLevel)y;

                return (x2.Level == y2.Level && x2.Line == y2.Line) ? 0 : 1;
            }

            #endregion
        }

        [TestMethod]
        public void TestConvertToLineWithLevel()
        {
            var data = new List<string> {
                    "indent 0",
                    "  indent 1",
                    "    indent 2",
                    "  indent 1",
                    "indent 0",
                };

            var lineWithLevels = LogParser.ConvertToLineWithLevel(data, 2);

            CollectionAssert.AreEqual(lineWithLevels,
                new List<LineWithLevel>
                {
                    new LineWithLevel{ Level = 0, Line = "indent 0" },
                    new LineWithLevel{ Level = 1, Line = "indent 1" },
                    new LineWithLevel{ Level = 2, Line = "indent 2" },
                    new LineWithLevel{ Level = 1, Line = "indent 1" },
                    new LineWithLevel{ Level = 0, Line = "indent 0" },
                },
                new LineWithLevelComparer());
        }

        [TestMethod]
        public void TestParseNactNumber()
        {
            Assert.AreEqual(0, LogParser.ParseNactNumber("--"));
            Assert.AreEqual(1, LogParser.ParseNactNumber("-1"));
            Assert.AreEqual(12, LogParser.ParseNactNumber("12"));
        }

        [TestMethod]
        public void TestParseNactTime()
        {
            Assert.AreEqual(
                new TimeSpan(0, 0, 12, 3, 45).Ticks,
                LogParser.ParseNactTime("--:12:-3.045").Ticks);
        }

        [TestMethod]
        public void TestConvertToKeyDictionary()
        {
            var data = new List<LineWithLevel>
                {
                    new LineWithLevel{ Level = 0, Line = "ビルド完了" },
                    new LineWithLevel{ Level = 1, Line = "実行対象ルール数 = 401" },
                    new LineWithLevel{ Level = 2, Line = "処理時間:              --:-2:10.050" },
                    new LineWithLevel{ Level = 1, Line = "ExecuteProgram:        --:11:44.153   388 回" },
                    new LineWithLevel{ Level = 0, Line = "cl.exe                   - --:10:29.319   285 回  02.208" },
                };

            {
                var pattern = LogParser.ConvertToKeyDictionary(string.Empty, "ビルド完了");
                Assert.AreEqual(pattern["name"], "ビルド完了");
                Assert.AreEqual(pattern["path"], string.Empty);
            }
            {
                var pattern = LogParser.ConvertToKeyDictionary("path", "実行対象ルール数 = 401");
                Assert.AreEqual(pattern["name"], "実行対象ルール数");
                Assert.AreEqual(pattern["num"], 401);
                Assert.AreEqual(pattern["path"], "path");
            }
            {
                var pattern = LogParser.ConvertToKeyDictionary("path/path", "処理時間:              --:-2:10.050");
                Assert.AreEqual(pattern["name"], "処理時間");
                Assert.AreEqual(pattern["msec"], (double)((2 * 60 + 10) * 1000 + 50));
                Assert.AreEqual(pattern["path"], "path/path");
            }
            {
                var pattern = LogParser.ConvertToKeyDictionary(string.Empty, "ExecuteProgram:        --:11:44.153   388 回");
                Assert.AreEqual(pattern["name"], "ExecuteProgram");
                Assert.AreEqual(pattern["msec"], (double)((11 * 60 + 44) * 1000 + 153));
                Assert.AreEqual(pattern["num"], 388);
            }
            {
                var pattern = LogParser.ConvertToKeyDictionary(string.Empty, "cl.exe                   - --:10:29.319   285 回  02.208");
                Assert.AreEqual(pattern["name"], "cl.exe");
                Assert.AreEqual(pattern["msec"], (double)((10 * 60 + 29) * 1000 + 319));
                Assert.AreEqual(pattern["num"], 285);
                Assert.AreEqual(pattern["average_msec"], (double)(2 * 1000 + 208));
            }
        }

        [TestMethod]
        public void TestConvertToJsonText()
        {
            var data = new List<Dictionary<string, object>>
            {
                new Dictionary<string, object>
                {
                    { "name", "test1" },
                },
                new Dictionary<string, object>
                {
                    { "name", "test2" },
                    { "num", 1 },
                    { "msec", (double)1.1 },
                },
            };

            Assert.AreEqual(
                LogParser.ConvertToJsonText(data),
                "[{\"name\":\"test1\"},{\"name\":\"test2\",\"num\":1,\"msec\":1.1}]");
        }

        [TestMethod]
        public void TestParseToJsonTextFromText()
        {
            var data = @"
[17:19:56]:	 [Step 2/3] ビルド完了
[17:19:56]:	 [Step 2/3]   実行対象ルール数 = 401
[17:19:56]:	 [Step 2/3]     実行ルール数     = 400
[17:19:56]:	 [Step 2/3]     スキップルール数 = 1
[17:19:56]:	 [Step 2/3] 処理時間:              --:-2:10.050
[17:19:56]:	 [Step 2/3]   パラメータの処理:      --:--:-0.565
[17:19:56]:	 [Step 2/3]   ルールの読み込み:      --:--:-0.450
[17:19:56]:	 [Step 2/3]   インスタンスの構築:    --:--:-0.058
[17:19:56]:	 [Step 2/3]   ルールの構築:          --:--:-6.370
[17:19:56]:	 [Step 2/3]   依存関係の解決:        --:--:-0.255
[17:19:56]:	 [Step 2/3]   ルールの実行:          --:-2:-2.351
[17:19:56]:	 [Step 2/3]     全スレッドの延べ時間:  --:12:-9.623
[17:19:56]:	 [Step 2/3]     ExecuteProgram:        --:11:44.153   388 回
[17:19:56]:	 [Step 2/3]       cl.exe                   - --:10:29.319   285 回  02.208
[17:19:56]:	 [Step 2/3]       lib.exe                  - --:--:-2.144    39 回  00.054
[17:19:56]:	 [Step 2/3]       link.exe                 - --:-1:11.321    60 回  01.188
[17:19:56]:	 [Step 2/3]       MakeResultDefinition.exe - --:--:-1.368     4 回  00.342
[17:19:56]:	 [Step 2/3]     GetProgramOutput:      --:--:-0.000     0 回
[17:19:56]:	 [Step 2/3]     Script:                --:--:-0.000     0 回
[17:19:56]:	 [Step 2/3]     MsBuildCs:             --:--:-4.344
[17:19:56]:	 [Step 2/3]     その他:                --:--:21.125
[17:19:56]:	 [Step 2/3] Process exited with code 0";

            var expected = "[{\"name\":\"ビルド完了\",\"path\":\"\"},{\"name\":\"実行対象ルール数\",\"num\":401,\"path\":\"ビルド完了\"},{\"name\":\"実行ルール数\",\"num\":400,\"path\":\"ビルド完了\\/実行対象ルール数\"},{\"name\":\"スキップルール数\",\"num\":1,\"path\":\"ビルド完了\\/実行対象ルール数\"},{\"name\":\"処理時間\",\"msec\":130050,\"path\":\"\"},{\"name\":\"パラメータの処理\",\"msec\":565,\"path\":\"処理時間\"},{\"name\":\"ルールの読み込み\",\"msec\":450,\"path\":\"処理時間\"},{\"name\":\"インスタンスの構築\",\"msec\":58,\"path\":\"処理時間\"},{\"name\":\"ルールの構築\",\"msec\":6370,\"path\":\"処理時間\"},{\"name\":\"依存関係の解決\",\"msec\":255,\"path\":\"処理時間\"},{\"name\":\"ルールの実行\",\"msec\":122351,\"path\":\"処理時間\"},{\"name\":\"全スレッドの延べ時間\",\"msec\":729623,\"path\":\"処理時間\\/ルールの実行\"},{\"name\":\"ExecuteProgram\",\"msec\":704153,\"num\":388,\"path\":\"処理時間\\/ルールの実行\"},{\"name\":\"cl.exe\",\"msec\":629319,\"num\":285,\"average_msec\":2208,\"path\":\"処理時間\\/ルールの実行\\/ExecuteProgram\"},{\"name\":\"lib.exe\",\"msec\":2144,\"num\":39,\"average_msec\":54,\"path\":\"処理時間\\/ルールの実行\\/ExecuteProgram\"},{\"name\":\"link.exe\",\"msec\":71321,\"num\":60,\"average_msec\":1188,\"path\":\"処理時間\\/ルールの実行\\/ExecuteProgram\"},{\"name\":\"MakeResultDefinition.exe\",\"msec\":1368,\"num\":4,\"average_msec\":342,\"path\":\"処理時間\\/ルールの実行\\/ExecuteProgram\"},{\"name\":\"GetProgramOutput\",\"msec\":0,\"num\":0,\"path\":\"処理時間\\/ルールの実行\"},{\"name\":\"Script\",\"msec\":0,\"num\":0,\"path\":\"処理時間\\/ルールの実行\"},{\"name\":\"MsBuildCs\",\"msec\":4344,\"path\":\"処理時間\\/ルールの実行\"},{\"name\":\"その他\",\"msec\":21125,\"path\":\"処理時間\\/ルールの実行\"}]";

            Assert.AreEqual(
                LogParser.ParseToJsonTextFromText(data),
                expected);
        }
    }
}
