﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.MakeVisualStudioProject.Converter;

namespace Nintendo.MakeVisualStudioProjectTest.Converter
{
    [TestClass]
    public class OptionStringConverterTest
    {
        [TestMethod]
        public void TestConvertToOptionValue()
        {
            var converter = new TestOptionStringConverter();
            var testInstances = new Tuple<IEnumerable<string>, IEnumerable<OptionValue>, IEnumerable<string>>[]
            {
                new Tuple<IEnumerable<string>, IEnumerable<OptionValue>, IEnumerable<string>>(
                    new[] { "/a" },
                    new[] { new OptionValue("/a") },
                    new string[0]),
                new Tuple<IEnumerable<string>, IEnumerable<OptionValue>, IEnumerable<string>>(
                    new[] { "/b1", "b1Arg", "/b2", "b2Arg" },
                    new[] { new OptionValue("/b1", "b1Arg"), new OptionValue("/b2", "b2Arg") },
                    new string[0]),
                new Tuple<IEnumerable<string>, IEnumerable<OptionValue>, IEnumerable<string>>(
                    new[] { "/b1", "/b2" },
                    new[] { new OptionValue("/b1") },
                    new[] { "/b2" }),
                new Tuple<IEnumerable<string>, IEnumerable<OptionValue>, IEnumerable<string>>(
                    new[] { "/c1c1Arg", "/c2c2Arg" },
                    new[] { new OptionValue("/c1", "c1Arg"), new OptionValue("/c2", "c2Arg") },
                    new string[0]),
                new Tuple<IEnumerable<string>, IEnumerable<OptionValue>, IEnumerable<string>>(
                    new[] { "/c2", "/c1" },
                    new[] { new OptionValue("/c1") },
                    new[] { "/c2" }),
                new Tuple<IEnumerable<string>, IEnumerable<OptionValue>, IEnumerable<string>>(
                    new[] { "/d1", "arg1;arg2;arg3", "/d2\"a r g\"", "/d1arg4" },
                    new[] { new OptionValue("/d1", "arg1;arg2;arg3"), new OptionValue("/d2", "a r g"), new OptionValue("/d1", "arg4") },
                    new string[0]),
                new Tuple<IEnumerable<string>, IEnumerable<OptionValue>, IEnumerable<string>>(
                    new[] { "/d2", "/d1", "/unknownSwitch", "/d1", "d1arg" },
                    new[] { new OptionValue("/d1"), new OptionValue("/d1", "d1arg") },
                    new[] { "/d2", "/unknownSwitch" }),
                new Tuple<IEnumerable<string>, IEnumerable<OptionValue>, IEnumerable<string>>(
                    new[] { "/e1;;arg1;;arg2;;arg3;;", "/e2", "arg1  arg2" },
                    new[] { new OptionValue("/e1", ";;arg1;;arg2;;arg3;;"), new OptionValue("/e2", "arg1  arg2") },
                    new string[0]),
                new Tuple<IEnumerable<string>, IEnumerable<OptionValue>, IEnumerable<string>>(
                    new[] { "/e1", "/e2", "/e2", "e2Arg", "/e2", "/e3" },
                    new[] { new OptionValue("/e1"), new OptionValue("/e2", "e2Arg") },
                    new[] { "/e2", "/e2", "/e3" }),
                new Tuple<IEnumerable<string>, IEnumerable<OptionValue>, IEnumerable<string>>(
                    new[] { "/f1", "arg", "/f2 \"arg1;arg2\"" },
                    new[] { new OptionValue("/f1", "arg"), new OptionValue("/f2", "arg1;arg2") },
                    new string[0]),
                new Tuple<IEnumerable<string>, IEnumerable<OptionValue>, IEnumerable<string>>(
                    new[] { "/f1arg", "/f2arg1;arg2" },
                    new OptionValue[0],
                    new[] { "/f1arg", "/f2arg1;arg2" }),
                new Tuple<IEnumerable<string>, IEnumerable<OptionValue>, IEnumerable<string>>(
                    new[] { "/g:arg", ":arg" },
                    new[] { new OptionValue("/g", "arg") },
                    new[] { ":arg" }),
            };

            foreach (var testInstance in testInstances)
            {
                var input = testInstance.Item1;
                var expected = testInstance.Item2;
                var expectedUnknowns = testInstance.Item3;

                IEnumerable<string> actualUnknowns;
                var actual = converter.ConvertToOptionValue(input, out actualUnknowns);

                Assert.IsTrue(
                    actual.SequenceEqual(expected),
                    "expected = {0}, actual = {1}",
                    Utility.ToArrayString(expected),
                    Utility.ToArrayString(actual));
                Assert.IsTrue(
                    actualUnknowns.SequenceEqual(expectedUnknowns),
                    "expected = {0}, actual = {1}",
                    Utility.ToArrayString(expectedUnknowns),
                    Utility.ToArrayString(actualUnknowns));
            }
        }

        [TestMethod]
        public void TestConvertToOptionString()
        {
            var converter = new TestOptionStringConverter();
            var testInstances = new Tuple<IEnumerable<OptionValue>, IEnumerable<string>, IEnumerable<OptionValue>>[]
            {
                new Tuple<IEnumerable<OptionValue>, IEnumerable<string>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/a") },
                    new[] { "/a" },
                    new OptionValue[0]),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<string>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/b1", "b1Arg"), new OptionValue("/b2", "b2Arg") },
                    new[] { "/b1\"b1Arg\"", "/b2\"b2Arg\"" },
                    new OptionValue[0]),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<string>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/b1"), new OptionValue("/b2") },
                    new[] { "/b1" },
                    new[] { new OptionValue("/b2") }),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<string>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/c1", "c1Arg"), new OptionValue("/unknownSwitch"), new OptionValue("/c2", "c2Arg") },
                    new[] { "/c1\"c1Arg\"", "/c2\"c2Arg\"" },
                    new[] { new OptionValue("/unknownSwitch") }),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<string>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/c2") },
                    new string[0],
                    new[] { new OptionValue("/c2") }),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<string>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/d1", "arg1;arg2;arg3"), new OptionValue("/d2", "a r g") },
                    new[] { "/d1\"arg1;arg2;arg3\"", "/d2\"a r g\"" },
                    new OptionValue[0]),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<string>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/d2"), new OptionValue("/d1"), new OptionValue("/unknownSwitch"), new OptionValue("/d1"), new OptionValue("/d1", "d1arg") },
                    new[] { "/d1", "/d1", "/d1\"d1arg\"" },
                    new[] { new OptionValue("/d2"), new OptionValue("/unknownSwitch") }),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<string>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/e1", ";;arg1;;arg2;;arg3;;"), new OptionValue("/e2", "arg1  arg2") },
                    new[] { "/e1\";;arg1;;arg2;;arg3;;\"", "/e2\"arg1  arg2\"" },
                    new OptionValue[0]),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<string>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/e1"), new OptionValue("/e2"), new OptionValue("/e2"), new OptionValue("/e2", "e2Arg"), new OptionValue("/e2"), new OptionValue("/e3") },
                    new[] { "/e1", "/e2\"e2Arg\"" },
                    new[] { new OptionValue("/e2"), new OptionValue("/e2"), new OptionValue("/e2"), new OptionValue("/e3") }),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<string>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/f1", "arg"), new OptionValue("/f2", "arg1;arg2") },
                    new[] { "/f1 \"arg\"", "/f2 \"arg1;arg2\"" },
                    new OptionValue[0]),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<string>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/f1"), new OptionValue("/f2") },
                    new string[0],
                    new[] { new OptionValue("/f1"), new OptionValue("/f2") }),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<string>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/g", "arg") },
                    new[] { "/g:\"arg\"" },
                    new OptionValue[0]),
            };

            foreach (var testInstance in testInstances)
            {
                var input = testInstance.Item1;
                var expected = testInstance.Item2;
                var expectedUnknowns = testInstance.Item3;

                IEnumerable<OptionValue> actualUnknowns;
                var actual = converter.ConvertToOptionString(input, out actualUnknowns);

                Assert.IsTrue(
                    actual.SequenceEqual(expected),
                    "expected = {0}, actual = {1}",
                    Utility.ToArrayString(expected),
                    Utility.ToArrayString(actual));
                Assert.IsTrue(
                    actualUnknowns.SequenceEqual(expectedUnknowns),
                    "expected = {0}, actual = {1}",
                    Utility.ToArrayString(expectedUnknowns),
                    Utility.ToArrayString(actualUnknowns));
            }
        }
    }
}
