﻿using System;
using System.IO;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace MakeNroTest
{
    [TestClass]
    public class BinaryTest : TestBase
    {
        private byte[] PublicKey = { 0x01, 0x00, 0x01 };

        byte[] ReadFile(FileManager fd)
        {
            using (FileStream fs = new FileStream(fd.FilePath, FileMode.Open, FileAccess.Read))
            {
                BinaryReader br = new BinaryReader(fs);
                using (MemoryStream ms = new MemoryStream())
                {
                    fs.CopyTo(ms);
                    return ms.ToArray();
                }
            }
        }

        [TestMethod]
        public void CheckBinary()
        {
            string inputFilePath = GetTestDsoFilePath();
            FileManager nro = SuccessExecute(inputFilePath);
            byte[] contents = ReadFile(nro);

            // Rocrt 用のオフセットが入っている
            uint rocrtOffset = BitConverter.ToUInt32(contents, NroBinaryFormat.RocrtOffsetOffset);
            uint rocrtSignature = BitConverter.ToUInt32(contents, (int)rocrtOffset + RocrtModuleHeaderFormat.SignatureOffset);
            Assert.AreEqual(rocrtSignature, RocrtModuleHeaderFormat.Signature);
            int headerOffset = (int)NroBinaryFormat.HeaderOffset;

            // シグネチャが正しい
            uint signature = BitConverter.ToUInt32(contents, NroHeaderFormat.SignatureOffset + headerOffset);
            Assert.AreEqual(signature, NroHeaderFormat.Signature);

            // バージョンが正しい
            uint version = BitConverter.ToUInt32(contents, NroHeaderFormat.VersionOffset + headerOffset);
            Assert.AreEqual(version, 0u);

            // サイズが正しい
            int fileSize = BitConverter.ToInt32(contents, NroHeaderFormat.SizeOffset + headerOffset);
            Assert.AreEqual(fileSize, contents.Length);

            // 雑多なフラグが正しい
            uint miscFlags = BitConverter.ToUInt32(contents, NroHeaderFormat.MiscFlagOffset + headerOffset);
            Assert.AreEqual(miscFlags, 0u);

            // Text 領域のアドレスが正しい
            uint exAddr = BitConverter.ToUInt32(contents, NroHeaderFormat.TextAddressOffset + headerOffset);
            Assert.AreEqual(exAddr, 0u);

            // Text領域のサイズが正しい
            uint exSize = BitConverter.ToUInt32(contents, NroHeaderFormat.TextSizeOffset + headerOffset);
            Assert.AreEqual(exSize, 0x634u);

            // RoData 領域のアドレスが正しい
            uint roAddr = BitConverter.ToUInt32(contents, NroHeaderFormat.RoDataAddressOffset + headerOffset);
            Assert.AreEqual(roAddr, 0x1000u);

            // RoData領域のサイズが正しい
            uint roSize = BitConverter.ToUInt32(contents, NroHeaderFormat.RoDataSizeOffset + headerOffset);
            Assert.AreEqual(roSize, 0x7c4u);

            // Data領域のアドレスが正しい
            uint dataAddr = BitConverter.ToUInt32(contents, NroHeaderFormat.DataAddressOffset + headerOffset);
            Assert.AreEqual(dataAddr, 0x2000u);

            // Data領域のサイズが正しい
            uint dataSize = BitConverter.ToUInt32(contents, NroHeaderFormat.DataSizeOffset + headerOffset);
            Assert.AreEqual(dataSize, 0x1c0u);

            // BSS領域のサイズが正しい
            uint bssSize = BitConverter.ToUInt32(contents, NroHeaderFormat.BssSizeOffset + headerOffset);
            Assert.AreEqual(bssSize, 0xe40u);

            // モジュールID が正しい
            byte[] moduleIdData = { 0x76, 0xe5, 0xa7, 0xcd, 0xdc, 0x2a, 0x6f, 0xd2, 0xce, 0x2f, 0x5e, 0x9a, 0xfc, 0x79, 0x30, 0xbd, 0xd2, 0xc1, 0x3b, 0x2c };
            byte[] moduleId = new byte[0x20];
            Array.Copy(moduleIdData, moduleId, moduleIdData.Length);
            byte[] nroModuleId = new byte[0x20];

            Array.Copy(contents, NroHeaderFormat.ModuleIdOffset + headerOffset, nroModuleId, 0, moduleId.Length);
            Assert.IsTrue(moduleId.SequenceEqual(nroModuleId));
        }

        [TestMethod]
        public void TestInputFile()
        {
            string inputFilePath = GetTestInvalidDsoDirPath() + InvalidTextAddressFile;
            FailExecute(string.Format(MakeNro.Properties.Resources.Message_InvalidTextAddress, 0x200000), inputFilePath);

            inputFilePath = GetTestInvalidDsoDirPath() + InvalidSegmentAlignFile;
            FailExecute(string.Format(MakeNro.Properties.Resources.Message_InvalidSegmentAlign, "RO", 0x700), inputFilePath);

            inputFilePath = GetTestInvalidDsoDirPath() + InvalidBssAddressFile;
            FailExecute(MakeNro.Properties.Resources.Message_InvalidBssSegments, inputFilePath);

            inputFilePath = GetTestInvalidDsoDirPath() + InvalidBssAlignFile;
            FailExecute(string.Format(MakeNro.Properties.Resources.Message_InvalidBssSegmentAlign, 0x21D8), inputFilePath);

            inputFilePath = GetTestInvalidDsoDirPath() + InvalidSectionAlignFile;
            FailExecute(string.Format(MakeNro.Properties.Resources.Message_InvalidSectionAlign, "ZI", 0x10000), inputFilePath);
        }
    }
}
