﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using TestUtility;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace MakeMetaTest
{
    [TestClass]
    public class TestMetaBinary : TestBase
    {
        private const int NameMaxSize = 0xf;
        private const int ProductCodeMaxSize = 0xf;
        private const int AcidContentDefaultSize = 0x2c;
        private const int AciContentDefaultSize = 0x1c;

        private const int BinaryAlign = 0x10;

        public byte[] CreateDefaultMetaFile(XmlElement targetElem = null)
        {
            XmlFile meta = new XmlFile();
            XmlFile descFile = CreateDefaultDescFile();
            XmlElement elem = new XmlElement("Meta");
            XmlElement core = new XmlElement("Core");
            elem.AddChild(core);
            XmlElement defaultElem = CreateDefaultMetaParam();
            core.AddChild(defaultElem);

            if (targetElem != null)
            {
                core.AddChild(targetElem);
            }

            meta.AddLine(elem.GetXml());
            meta.WriteData();
            FileManager outputFile = SuccessExecute(descFile, meta, null, null);

            return outputFile.ReadFile(MetaHeaderSize, MetaHeaderOffset);
        }

       [TestMethod]
        public void SignatureTest()
        {
            byte[] binary = CreateDefaultMetaFile();
            byte[] target = new byte[MetaSignature.Length];
            Array.Copy(binary, MetaSignatureOffset, target, 0, MetaSignature.Length);
            Assert.IsTrue(System.Text.Encoding.ASCII.GetString(target) == MetaSignature);
        }

        // 予約領域は0 埋めされている
        [TestMethod]
        public void ReservedArea()
        {
            byte[] binary = CreateDefaultMetaFile();
            byte[] target = new byte[MetaReserved1Size];
            Array.Copy(binary, MetaReserved1Offset, target, 0, MetaReserved1Size);
            foreach (var c in target)
            {
                Assert.IsTrue(c == 0);
            }

            target = new byte[MetaReserved2Size];
            Array.Copy(binary, MetaReserved2Offset, target, 0, MetaReserved2Size);
            foreach (var c in target)
            {
                Assert.IsTrue(c == 0);
            }

            target = new byte[MetaReserved3Size];
            Array.Copy(binary, MetaReserved3Offset, target, 0, MetaReserved3Size);
            foreach (var c in target)
            {
                Assert.IsTrue(c == 0);
            }
        }

        [TestMethod]
        public void FlagTest()
        {
            // Is64BitInstruction = False
            {
                XmlElement is64BitInstruction = new XmlElement("Is64BitInstruction");
                is64BitInstruction.InnerText = "false";

                byte[] binary = CreateDefaultMetaFile(is64BitInstruction);
                byte[] target = new byte[UInt32Size];
                Array.Copy(binary, MetaFlagsOffset, target, 0, UInt32Size);
                Assert.IsTrue((target[0] & 0x1) == 0);
            }

            // Is64BitInstruction = True
            {
                XmlElement is64BitInstruction = new XmlElement("Is64BitInstruction");
                is64BitInstruction.InnerText = "true";

                byte[] binary = CreateDefaultMetaFile(is64BitInstruction);
                byte[] target = new byte[UInt32Size];
                Array.Copy(binary, MetaFlagsOffset, target, 0, UInt32Size);
                Assert.IsTrue((target[0] & 0x1) == 1);
            }

            // AddressSpace32Bit
            {
                XmlElement addressSpace = new XmlElement("ProcessAddressSpace");
                addressSpace.InnerText = "AddressSpace32Bit";

                byte[] binary = CreateDefaultMetaFile(addressSpace);
                byte[] target = new byte[UInt32Size];
                Array.Copy(binary, MetaFlagsOffset, target, 0, UInt32Size);
                byte space = (byte)((target[0] >> 1) & ((1 << 3) - 1));
                Assert.IsTrue(space == 0);
            }

            // AddressSpace64BitOld
            {
                XmlElement addressSpace = new XmlElement("ProcessAddressSpace");
                addressSpace.InnerText = "AddressSpace64BitOld";

                byte[] binary = CreateDefaultMetaFile(addressSpace);
                byte[] target = new byte[UInt32Size];
                Array.Copy(binary, MetaFlagsOffset, target, 0, UInt32Size);
                byte space = (byte)((target[0] >> 1) & ((1 << 3) - 1));
                Assert.IsTrue(space == 1);
            }

            // AddressSpace32BitNoReserved
            {
                XmlElement addressSpace = new XmlElement("ProcessAddressSpace");
                addressSpace.InnerText = "AddressSpace32BitNoReserved";

                byte[] binary = CreateDefaultMetaFile(addressSpace);
                byte[] target = new byte[UInt32Size];
                Array.Copy(binary, MetaFlagsOffset, target, 0, UInt32Size);
                byte space = (byte)((target[0] >> 1) & ((1 << 3) - 1));
                Assert.IsTrue(space == 2);
            }

            // AddressSpace64Bit
            {
                XmlElement addressSpace = new XmlElement("ProcessAddressSpace");
                addressSpace.InnerText = "AddressSpace64Bit";

                byte[] binary = CreateDefaultMetaFile(addressSpace);
                byte[] target = new byte[UInt32Size];
                Array.Copy(binary, MetaFlagsOffset, target, 0, UInt32Size);
                byte space = (byte)((target[0] >> 1) & ((1 << 3) - 1));
                Assert.IsTrue(space == 3);
            }

            // Flags[0] の bit 3-7 は未使用
            {
                XmlElement addressSpace = new XmlElement("ProcessAddressSpace");
                addressSpace.InnerText = "AddressSpace32BitNoReserved";

                byte[] binary = CreateDefaultMetaFile(addressSpace);
                byte[] target = new byte[UInt32Size];
                Array.Copy(binary, MetaFlagsOffset, target, 0, UInt32Size);
                byte space = (byte)((target[0] >> 3) & ((1 << 5) - 1));
                Assert.IsTrue(space == 0);
            }

            // Flags[1] は未使用
            {
                XmlElement addressSpace = new XmlElement("ProcessAddressSpace");
                addressSpace.InnerText = "AddressSpace32BitNoReserved";

                byte[] binary = CreateDefaultMetaFile(addressSpace);
                byte[] target = new byte[UInt32Size];
                Array.Copy(binary, MetaFlagsOffset, target, 0, UInt32Size);
                Assert.IsTrue(target[1] == 0);
            }

            // Flags[2] はメインスレッドの優先度
            {
                XmlElement mainThreadPriority = new XmlElement("MainThreadPriority");
                mainThreadPriority.InnerText = "24";

                byte[] binary = CreateDefaultMetaFile(mainThreadPriority);
                byte[] target = new byte[UInt32Size];
                Array.Copy(binary, MetaFlagsOffset, target, 0, UInt32Size);
                Assert.IsTrue(target[2] == 24);
            }

            // Flags[3] はメインスレッドの優先コア番号
            {
                XmlElement mainThreadCoreNumber = new XmlElement("MainThreadCoreNumber");
                mainThreadCoreNumber.InnerText = "1";

                byte[] binary = CreateDefaultMetaFile(mainThreadCoreNumber);
                byte[] target = new byte[4];
                Array.Copy(binary, MetaFlagsOffset, target, 0, UInt32Size);
                Assert.IsTrue(target[3] == 1);
            }
        }

        [TestMethod]
        public void TestVersion()
        {
            // Version 指定
            {
                XmlElement version = new XmlElement("Version");
                version.InnerText = "1";

                byte[] binary = CreateDefaultMetaFile(version);
                byte[] target = new byte[UInt32Size];
                Array.Copy(binary, MetaVersionOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == 1);
            }

            // Version 未指定
            {
                byte[] binary = CreateDefaultMetaFile();
                byte[] target = new byte[UInt32Size];
                Array.Copy(binary, MetaVersionOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == 0);
            }
        }

        [TestMethod]
        public void TestMainThreadStackSize()
        {
            {
                XmlElement stackSize = new XmlElement("MainThreadStackSize");
                stackSize.InnerText = "0x1000";

                byte[] binary = CreateDefaultMetaFile(stackSize);
                byte[] target = new byte[UInt32Size];
                Array.Copy(binary, MetaStackSizeOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == 0x1000);
            }
        }

        [TestMethod]
        public void TestName()
        {
            // Name を指定
            {
                XmlElement name = new XmlElement("Name");
                name.InnerText = "テスト";
                int size = System.Text.Encoding.UTF8.GetByteCount(name.InnerText);

                byte[] binary = CreateDefaultMetaFile(name);
                byte[] target = new byte[MetaNameSize];
                Array.Copy(binary, MetaNameOffset, target, 0, MetaNameSize);
                string test = System.Text.Encoding.UTF8.GetString(target, 0, size);
                Assert.IsTrue(test == name.InnerText);

                // 残り部分が 0 埋めされている
                for (int i = size; i < MetaNameSize; i++)
                {
                    Assert.IsTrue(target[i] == 0);
                }
            }

            // Name を指定しないときは .nmeta のファイル名
            {
                XmlFile meta = new XmlFile();
                XmlFile descFile = CreateDefaultDescFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                meta.FileName = "あああああa.nmeta";
                string correctName = "あああああ";
                int size = System.Text.Encoding.UTF8.GetByteCount(correctName);
                Assert.IsTrue(size == NameMaxSize);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FileManager outputFile = SuccessExecute(descFile, meta, null, null);
                byte[] binary = outputFile.ReadFile(MetaHeaderSize, MetaHeaderOffset);
                byte[] target = new byte[MetaNameSize];
                Array.Copy(binary, MetaNameOffset, target, 0, MetaNameSize);
                string test = System.Text.Encoding.UTF8.GetString(target, 0, size);
                Assert.IsTrue(test == correctName);

                // 残り部分が 0 埋めされている
                for (int i = size; i < MetaNameSize; i++)
                {
                    Assert.IsTrue(target[i] == 0);
                }
            }
        }

        [TestMethod]
        public void TestProductCode()
        {
            // ProductCode を指定
            {
                XmlElement productCode = new XmlElement("ProductCode");
                productCode.InnerText = "ProductCode";
                int size = System.Text.Encoding.ASCII.GetByteCount(productCode.InnerText);

                byte[] binary = CreateDefaultMetaFile(productCode);
                byte[] target = new byte[MetaProductCodeSize];
                Array.Copy(binary, MetaProductCodeOffset, target, 0, MetaProductCodeSize);
                string test = System.Text.Encoding.ASCII.GetString(target, 0, size);
                Assert.IsTrue(test == productCode.InnerText);

                // 残り部分が 0 埋めされている
                for (int i = size; i < MetaProductCodeSize; i++)
                {
                    Assert.IsTrue(target[i] == 0);
                }
            }

            // ProductCode を指定しないときは 0埋めされている
            {
                byte[] binary = CreateDefaultMetaFile();
                byte[] target = new byte[MetaProductCodeSize];
                Array.Copy(binary, MetaProductCodeOffset, target, 0, MetaProductCodeSize);

                // 0 埋めされている
                for (int i = 0; i < MetaProductCodeSize; i++)
                {
                    Assert.IsTrue(target[i] == 0);
                }
            }
        }

        [TestMethod]
        public void TestAci()
        {
            const int dummyCapablitySize = 16;

            // ACID のディスクリプタが空
            {
                byte[] binary = CreateDefaultMetaFile();
                byte[] target = new byte[UInt32Size];

                // ACI オフセット
                Array.Copy(binary, MetaAciOffsetOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == MetaHeaderSize + AcidHeaderSize + RoundUp(AcidContentDefaultSize, BinaryAlign));

                // ACI サイズ
                Array.Copy(binary, MetaAciSizeOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == AciHeaderSize + AciContentDefaultSize);
            }

            // ACID のディスクリプタがアライメントに沿っていない
            {
                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                XmlElement handleTableSize = new XmlElement("HandleTableSize");
                handleTableSize.InnerText = "256";
                kcd.AddChild(handleTableSize);

                XmlFile descFile = CreateDefaultDescFileWithAdditionalTags(kcd);
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FileManager outputFile = SuccessExecute(descFile, meta, null, null);

                byte[] binary = outputFile.ReadFile(MetaHeaderSize, MetaHeaderOffset);
                byte[] target = new byte[UInt32Size];

                // ACI オフセット
                Array.Copy(binary, MetaAciOffsetOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == MetaHeaderSize + AcidHeaderSize + RoundUp(AcidContentDefaultSize, BinaryAlign) + RoundUp(KernelCapabilityEntrySize, BinaryAlign));

                // ACI サイズ
                Array.Copy(binary, MetaAciSizeOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == AciHeaderSize + AciContentDefaultSize);
            }

            // ACI のアクセス制御データがアライメントに沿っていない
            // SrvAccessControlData
            {
                XmlElement entry = TestDataForSrv.MakeSrvEntry("Test", "True");
                XmlElement[] srvData = new XmlElement[] { entry };

                FileManager descFile = new FileManager();
                MakeDesc(ref descFile, MakeDescriptorElem(srvData, CapabilityType.Service), null);

                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);
                core.AddChild(MakeDataElem(srvData, CapabilityType.Service));

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FileManager outputFile = SuccessExecute(descFile, meta, null, null);

                byte[] binary = outputFile.ReadFile(MetaHeaderSize, MetaHeaderOffset);
                byte[] target = new byte[UInt32Size];

                // ACI オフセット
                Array.Copy(binary, MetaAciOffsetOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == MetaHeaderSize + AcidHeaderSize + RoundUp(AcidContentDefaultSize, BinaryAlign) + dummyCapablitySize);

                // ACI サイズ
                Array.Copy(binary, MetaAciSizeOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == AciHeaderSize + RoundUp(AciContentDefaultSize, BinaryAlign) + "Test".Length + 1);
            }

            // ACI のアクセス制御データがアライメントに沿っていない
            // KernelCapability
            {
                XmlElement handleTableSize = TestDataForKc.MakeHandleTableSize("256");
                XmlElement[] kcData = new XmlElement[] { handleTableSize };

                FileManager descFile = new FileManager();
                MakeDesc(ref descFile, MakeDescriptorElem(kcData, CapabilityType.Kernel), null);

                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);
                core.AddChild(MakeDataElem(kcData, CapabilityType.Kernel));

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FileManager outputFile = SuccessExecute(descFile, meta, null, null);

                byte[] binary = outputFile.ReadFile(MetaHeaderSize, MetaHeaderOffset);
                byte[] target = new byte[UInt32Size];

                // ACI オフセット
                Array.Copy(binary, MetaAciOffsetOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == MetaHeaderSize + AcidHeaderSize + RoundUp(AcidContentDefaultSize, BinaryAlign) + dummyCapablitySize);

                // ACI サイズ
                Array.Copy(binary, MetaAciSizeOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == AciHeaderSize + RoundUp(AciContentDefaultSize, BinaryAlign) + KernelCapabilityEntrySize);
            }
        }

        [TestMethod]
        public void TestAcid()
        {
            // ACID のディスクリプタが空
            {
                byte[] binary = CreateDefaultMetaFile();
                byte[] target = new byte[UInt32Size];

                // ACID オフセット
                Array.Copy(binary, MetaAcidOffsetOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == MetaHeaderSize);

                // ACID サイズ
                Array.Copy(binary, MetaAcidSizeOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == AcidHeaderSize + AcidContentDefaultSize);
            }

            // ACID のディスクリプタがアライメントに沿っていない
            // SrvAccessControlDescriptor
            {
                XmlElement sad = new XmlElement("SrvAccessControlDescriptor");
                XmlElement entry = new XmlElement("Entry");
                sad.AddChild(entry);
                XmlElement isServer = new XmlElement("IsServer");
                isServer.InnerText = "true";
                entry.AddChild(isServer);
                XmlElement name = new XmlElement("Name");
                name.InnerText = "Test";
                entry.AddChild(name);

                XmlFile descFile = CreateDefaultDescFileWithAdditionalTags(sad);
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FileManager outputFile = SuccessExecute(descFile, meta, null, null);

                byte[] binary = outputFile.ReadFile(MetaHeaderSize, MetaHeaderOffset);
                byte[] target = new byte[UInt32Size];

                // ACID オフセット
                Array.Copy(binary, MetaAcidOffsetOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == MetaHeaderSize);

                // ACID サイズ
                Array.Copy(binary, MetaAcidSizeOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == AcidHeaderSize + RoundUp(AcidContentDefaultSize, BinaryAlign) + name.InnerText.Length + 1);
            }

            // ACID のディスクリプタがアライメントに沿っていない
            // KernelCapabilityDescriptor
            {
                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                XmlElement handleTableSize = new XmlElement("HandleTableSize");
                handleTableSize.InnerText = "256";
                kcd.AddChild(handleTableSize);

                XmlFile descFile = CreateDefaultDescFileWithAdditionalTags(kcd);
                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FileManager outputFile = SuccessExecute(descFile, meta, null, null);

                byte[] binary = outputFile.ReadFile(MetaHeaderSize, MetaHeaderOffset);
                byte[] target = new byte[UInt32Size];

                // ACID オフセット
                Array.Copy(binary, MetaAcidOffsetOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == MetaHeaderSize);

                // ACID サイズ
                Array.Copy(binary, MetaAcidSizeOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == AcidHeaderSize + RoundUp(AcidContentDefaultSize, BinaryAlign) + KernelCapabilityEntrySize);
            }
        }
    }
}
