﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using TestUtility;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace MakeMetaTest
{
    [TestClass]
    public class TestAciBinary : TestBase
    {
        private const int AciFacDefaultSize = 0x1C;

        [TestMethod]
        public void TestSignature()
        {
            byte[] aci = CreateDefaultAciFile(CapabilityType.None);
            Assert.IsTrue(System.Text.Encoding.ASCII.GetString(aci, 0, AciSignature.Length) == AciSignature);
        }

        [TestMethod]
        public void TestReservedArea()
        {
            byte[] aci = CreateDefaultAciFile(CapabilityType.None);

            for (int i = 0; i < AciReserved1Size; i++)
            {
                Assert.IsTrue(aci[AciReserved1Offset + i] == 0);
            }

            for (int i = 0; i < AciReserved2Size; i++)
            {
                Assert.IsTrue(aci[AciReserved2Offset + i] == 0);
            }

            for (int i = 0; i < AciReserved3Size; i++)
            {
                Assert.IsTrue(aci[AciReserved3Offset + i] == 0);
            }
        }

        [TestMethod]
        public void TestProgramId()
        {
            byte[] aci = CreateDefaultAciFile(CapabilityType.None);
            ulong correct = ulong.Parse(DefaultProgramId.Replace("0x", string.Empty), System.Globalization.NumberStyles.HexNumber);
            Assert.IsTrue(BitConverter.ToUInt64(aci, AciProgramIdOffset) ==  correct);
        }

        [TestMethod]
        public void TestFac()
        {
            byte[] aci = CreateDefaultAciFile(CapabilityType.None);
            Assert.IsTrue(BitConverter.ToUInt32(aci, AciFacOffsetOffset) == AciHeaderSize);
            Assert.IsTrue(BitConverter.ToUInt32(aci, AciFacSizeOffset) == AciFacDefaultSize);
        }

        [TestMethod]
        public void TestSac()
        {
            // sac が空
            {
                byte[] aci = CreateDefaultAciFile(CapabilityType.None);
                Assert.IsTrue(BitConverter.ToUInt32(aci, AciSacOffsetOffset) == AciHeaderSize + RoundUp(AciFacDefaultSize, BinaryAlignSize));
                Assert.IsTrue(BitConverter.ToUInt32(aci, AciSacSizeOffset) == 0);
            }

            // 1 エントリ
            {
                XmlElement entry = new XmlElement("Entry");
                XmlElement isServer = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "True";
                XmlElement name = new XmlElement("Name");
                entry.AddChild(name);
                name.InnerText = "Test";

                int size = name.InnerText.Length + 1;
                byte[] aci = CreateDefaultAciFile(entry, CapabilityType.Service);
                Assert.IsTrue(BitConverter.ToUInt32(aci, AciSacOffsetOffset) == AciHeaderSize + RoundUp(AciFacDefaultSize, BinaryAlignSize));
                Assert.IsTrue(BitConverter.ToUInt32(aci, AciSacSizeOffset) == size);

                byte[] entryBinary = new byte[size];
                Array.Copy(aci, AciHeaderSize + RoundUp(AciFacDefaultSize, BinaryAlignSize), entryBinary, 0, size);

                Assert.IsTrue(entryBinary[0] == 0x83);
                Assert.IsTrue(System.Text.Encoding.ASCII.GetString(entryBinary, 1, size - 1) == name.InnerText);
            }

            // 複数エントリ
            {
                XmlElement entry1 = new XmlElement("Entry");
                XmlElement isServer1 = new XmlElement("IsServer");
                entry1.AddChild(isServer1);
                isServer1.InnerText = "True";
                XmlElement name1 = new XmlElement("Name");
                entry1.AddChild(name1);
                name1.InnerText = "Server";

                XmlElement entry2 = new XmlElement("Entry");
                XmlElement isServer2 = new XmlElement("IsServer");
                entry2.AddChild(isServer2);
                isServer2.InnerText = "False";
                XmlElement name2 = new XmlElement("Name");
                entry2.AddChild(name2);
                name2.InnerText = "Client";

                int size = name1.InnerText.Length + 1 + name2.InnerText.Length + 1;
                int entry1Offset = 0;
                int entry2Offset = name1.InnerText.Length + 1;
                byte[] aci = CreateDefaultAciFile(new XmlElement[] { entry1, entry2 }, CapabilityType.Service);
                Assert.IsTrue(BitConverter.ToUInt32(aci, AciSacOffsetOffset) == AciHeaderSize + RoundUp(AciFacDefaultSize, BinaryAlignSize));
                Assert.IsTrue(BitConverter.ToUInt32(aci, AciSacSizeOffset) == size);

                byte[] entryBinary = new byte[size];
                Array.Copy(aci, AciHeaderSize + RoundUp(AciFacDefaultSize, BinaryAlignSize), entryBinary, 0, size);

                Assert.IsTrue(entryBinary[entry1Offset] == 0x85);
                Assert.IsTrue(System.Text.Encoding.ASCII.GetString(entryBinary, entry1Offset + 1, name1.InnerText.Length) == name1.InnerText);

                Assert.IsTrue(entryBinary[entry2Offset] == 0x05);
                Assert.IsTrue(System.Text.Encoding.ASCII.GetString(entryBinary, entry2Offset + 1, name2.InnerText.Length) == name2.InnerText);
            }
        }

        [TestMethod, TestCategory("MiddleTest")]
        public void TestKc()
        {
            // 空の場合
            {
                byte[] aci = CreateDefaultAciFile(CapabilityType.None);
                Assert.IsTrue(BitConverter.ToUInt32(aci, AciKcOffsetOffset) == AciHeaderSize + RoundUp(AciFacDefaultSize, BinaryAlignSize));
                Assert.IsTrue(BitConverter.ToUInt32(aci, AciKcSizeOffset) == 0);
            }

            // ThreadInfo のデータが正しく書き込まれている
            {
                byte[] aci = MakeKcEntryAci(TestDataForKc.MakeThreadInfoElement("63", "24", "0", "1"));

                // ケイパビリティが正しく書き込まれている
                int offset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset) == 0x10063F7);
            }

            // MapMemory のデータが正しく書き込まれている
            // Permission: Ro, Type: Static
            {
                byte[] aci = MakeKcEntryAci(TestDataForKc.MakeMemoryMap("0x100000", "0x2000", "Ro", "Static"));

                // ケイパビリティが正しく書き込まれている
                int offset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset) == 0x8000803F);
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset + KernelCapabilityEntrySize) == 0x8000013F);
            }

            // MapMemory のデータが正しく書き込まれている
            // Permission: Rw, Type: Static
            {
                byte[] aci = MakeKcEntryAci(TestDataForKc.MakeMemoryMap("0x100000", "0x2000", "Rw", "Static"));

                // ケイパビリティが正しく書き込まれている
                int offset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset) == 0x0000803F);
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset + KernelCapabilityEntrySize) == 0x8000013F);
            }

            // MapMemory のデータが正しく書き込まれている
            // Permission: Rw, Type: Io
            {
                byte[] aci = MakeKcEntryAci(TestDataForKc.MakeMemoryMap("0x100000", "0x2000", "Rw", "Io"));

                // ケイパビリティが正しく書き込まれている
                int offset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset) == 0x0000803F);
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset + KernelCapabilityEntrySize) == 0x13F);
            }

            // MapMemory のデータが正しく書き込まれている
            // Permission: Ro, Type: Io
            {
                byte[] aci = MakeKcEntryAci(TestDataForKc.MakeMemoryMap("0x100000", "0x2000", "Ro", "Io"));

                // ケイパビリティが正しく書き込まれている
                int offset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset) == 0x8000803F);
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset + KernelCapabilityEntrySize) == 0x13F);
            }


            // Io の MapMemory のデータが正しく書き込まれている
            {
                byte[] aci = MakeKcEntryAci(TestDataForKc.MakeMemoryMap("0x100000", "0x1000", "Rw", "Io"));

                // ケイパビリティが正しく書き込まれている
                int offset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset) == 0x1007F);
            }

            // EnableInterrupts のデータが正しく書き込める
            {
                byte[] aci = MakeKcEntryAci(TestDataForKc.MakeEnableInterrupts( new string[] { "1", "2" } ));

                // ケイパビリティが正しく書き込まれている
                int offset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset) == 0x8017FF);
            }

            // EnableInterrupts のデータの個数が奇数の時にパディングデータが入る
            {
                byte[] aci = MakeKcEntryAci(TestDataForKc.MakeEnableInterrupt("1"));

                // ケイパビリティが正しく書き込まれている
                int offset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset) == 0xFFC017FF);
            }

            // MiscParams のデータが正しく書き込まれている
            {
                byte[] aci = MakeKcEntryAci(TestDataForKc.MakeMiscParams("1"));

                // ケイパビリティが正しく書き込まれている
                int offset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset) == 0x5FFF);
            }

            // KernelVersion のデータが正しく書き込まれている
            {
                // XmlFile desc = CreateDefaultDescFile();
                FileManager desc = new FileManager();
                MakeDesc(ref desc, "1.3");

                XmlFile meta = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultMetaParam();
                core.AddChild(defaultElem);

                meta.AddLine(elem.GetXml());
                meta.WriteData();
                FileManager outputFile = SuccessExecute(desc, meta, "1.3");
                byte[] metaBinary = outputFile.ReadFile(MetaHeaderSize, MetaOffset);
                uint aciOffset = BitConverter.ToUInt32(metaBinary, MetaAciOffsetOffset);
                uint aciSize = BitConverter.ToUInt32(metaBinary, MetaAciSizeOffset);
                byte[] aci = outputFile.ReadFile((int)aciSize, aciOffset);

                // ケイパビリティが正しく書き込まれている
                int offset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset) == 0x9BFFF);
            }

            // HandleTableSize のデータが正しく書き込まれている
            {
                byte[] aci = MakeKcEntryAci(TestDataForKc.MakeHandleTableSize("1023"));

                // ケイパビリティが正しく書き込まれている
                int offset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset) == 0x3FF7FFF);
            }

            // MiscFlags のデータが正しく書き込まれている
            // EnableDebug: True, ForceDebug: True
            {
                byte[] aci = MakeKcEntryAci(TestDataForKc.MakeMiscFlags("true", "true"));

                // ケイパビリティが正しく書き込まれている
                int offset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset) == 0x6FFFF);
            }

            // MiscFlags のデータが正しく書き込まれている
            // EnableDebug: False, ForceDebug: False
            {
                byte[] aci = MakeKcEntryAci(TestDataForKc.MakeMiscFlags("false", "false"));

                // ケイパビリティが正しく書き込まれている
                int offset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset) == 0xFFFF);
            }

            // 複数エントリ のデータが正しく書き込まれている
            {
                byte[] aci = MakeKcEntryAci(new XmlElement[] {
                    TestDataForKc.MakeEnableInterrupt("1"),
                    TestDataForKc.MakeHandleTableSize("1023"),
                });

                // ケイパビリティが正しく書き込まれている
                int offset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset) == 0xFFC017FF);
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset + KernelCapabilityEntrySize) == 0x3FF7FFF);
            }

            // サイズが 0x10 にアライメントされていないとき、終端が 0 埋めされずに終わっている
            {
                byte[] aci = MakeKcEntryAci(TestDataForKc.MakeHandleTableSize("1023"));

                int offset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                int size = BitConverter.ToInt32(aci, AciKcSizeOffset);
                Assert.IsTrue(offset + size == aci.Length);
            }

            // カーネルケイパビリティの順が エントリーNo. の順になっている
            {
                byte[] aci = MakeKcEntryAci(new XmlElement[] {
                    // No. 16
                    TestDataForKc.MakeMiscFlags("True", null),
                    // No. 15
                    TestDataForKc.MakeHandleTableSize("1023"),
                });

                int offset = BitConverter.ToInt32(aci, AciKcOffsetOffset);

                // ケイパビリティが正しく書き込まれている
                // No.15 の値
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset) == 0x3FF7FFF);

                // No.16 の値
                Assert.IsTrue(BitConverter.ToUInt32(aci, offset + KernelCapabilityEntrySize) == 0x2FFFF);
            }
        }

        [TestMethod]
        public void TestPadding()
        {
            // 先頭のアラインを揃えるためにパディング領域が付加される。
            // 付加された領域は 0 埋めされている
            {
                Dictionary<CapabilityType, XmlElement[]> dict = new Dictionary<CapabilityType,XmlElement[]>();
                XmlElement handleTable = new XmlElement("HandleTableSize");
                handleTable.InnerText = "1023";
                dict[CapabilityType.Kernel] = new XmlElement[] { handleTable };

                XmlElement entry = new XmlElement("Entry");
                XmlElement isServer = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "True";
                XmlElement name = new XmlElement("Name");
                entry.AddChild(name);
                name.InnerText = "Test";
                dict[CapabilityType.Service] = new XmlElement[] { entry };

                byte[] aci = CreateDefaultAciFileWithMultipleElements(dict);

                // Sac
                int sacSize = name.InnerText.Length + 1;
                int sacOffset = BitConverter.ToInt32(aci, AciSacOffsetOffset);
                byte[] entryBinary = new byte[sacSize];
                Array.Copy(aci, sacOffset, entryBinary, 0, sacSize);
                Assert.IsTrue(entryBinary[0] == 0x83);
                Assert.IsTrue(System.Text.Encoding.ASCII.GetString(entryBinary, 1, sacSize - 1) == name.InnerText);

                // Kc
                int kcOffset = BitConverter.ToInt32(aci, AciKcOffsetOffset);
                Assert.IsTrue((kcOffset & (BinaryAlignSize - 1)) == 0);
                Assert.IsTrue(BitConverter.ToUInt32(aci, kcOffset) == 0x3FF7FFF);

                // Padding
                int paddingSize = (int)RoundUp((uint)sacSize, BinaryAlignSize) - sacSize;
                Assert.IsTrue(paddingSize == kcOffset - sacOffset - sacSize);
                byte[] padding = new byte[kcOffset - sacOffset - sacSize];
                foreach (var c in padding)
                {
                    Assert.IsTrue(c == 0);
                }
            }
        }


    }
}
