﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Text.RegularExpressions;
using System.Collections.Generic;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using MakeInitialProgram;
using YamlDotNet.RepresentationModel;
using TestUtility;

namespace MakeInitialProgramTest
{
    [TestClass]
    public class CapabilityTest
    {
        /// <summary>
        /// ファイルの生成、編集、削除を管理するクラス
        /// </summary>
        internal class FileCreator
        {
            private List<string> Buffer { get; set; }

            internal string FilePath { get; private set; }
            internal bool IsWritten { get; set; }

            internal FileCreator()
            {
                this.FilePath = System.IO.Path.GetRandomFileName();
                this.Buffer = new List<string>();
                IsWritten = false;
            }

            ~FileCreator()
            {
                System.IO.File.Delete(FilePath);
            }

            internal FileCreator(string filePath)
            {
                this.FilePath = filePath;
                this.Buffer = new List<string>();
            }

            internal void AddLine(string line)
            {
                Buffer.Add(line);
            }

            internal void DeleteLine(int index)
            {
                Buffer.RemoveAt(index);
            }

            internal int FileLines()
            {
                return Buffer.Count;
            }

            internal void WriteData()
            {
                Assert.IsFalse(IsWritten);
                Assert.IsFalse(System.IO.File.Exists(FilePath));
                using (System.IO.FileStream fs = System.IO.File.Create(FilePath))
                {
                    using (System.IO.StreamWriter sw = new System.IO.StreamWriter(fs))
                    {
                        foreach (var line in Buffer)
                        {
                            sw.WriteLine(line);
                        }
                    }
                }
                IsWritten = true;
            }
        }

        /* 共有パラメータ */

        public TestContext TestContext { get; set; }
        private FileCreator SysCallFile { get; set; }
        private SystemCallInfo SysCallInfo { get; set; }

        // svc_BaseId.autogen.h の中身と同じ情報を入れる
        // １つ目、２つ目、最後の要素を確認する
        // 手作業でメンテナンスする
        private string[] BaseIdNameList = { "SetHeapSize", "SetMemoryPermission", "KernelDebug" };
        private uint[] BaseIdList = { 1, 2, 60 };
        private uint NumBaseIdSysCall = 58;

        // 各関数のテスト用システムコールヘッダの内容
        private string[] SysCallNameList = { "SetHeapSize", "SetMemoryPermission", "CreatePort", "CreateNamePort" };
        private uint[] SysCallIds = { 1, 2, 71, 72 };

        private string[] PermissionNames = { "RW", "RO" };
        private string[] TypeNames = { "IO", "Static" };
        private string BaseIdFile { get; set; }
        private string ServerIdFile { get; set; }
        private string DdIdFile { get; set; }
        private string DmntIdFile { get; set; }
        private string TcbIdFile { get; set; }
        private string[] SvcHeaders { get; set; }

        private string ResourceDir { get; set; }
        private string BinaryFile { get; set; }
        private string MakeSvcVeneerWorkDir { get; set; }
        private string MakeSvcVeneerTemplate { get; set; }
        private string MakeSvcVeneerIncludeDir { get; set; }

        private enum MemoryPermission
        {
            Rw = 0,
            Ro = 1,
            None = 2
        }

        private enum MemoryType
        {
            Io = 0,
            Static = 1,
            None = 2
        }

        /* SystemCall ID のヘッダファイルを作成するメソッド群 */

        private FileCreator MakeSystemCallFile()
        {
            FileCreator sysCallFile = new FileCreator();
            sysCallFile.AddLine("#define NN_SVC_ID_SET_HEAP_SIZE 1");
            sysCallFile.AddLine("#define NN_SVC_ID_SET_MEMORY_PERMISSION 2");
            sysCallFile.AddLine("#define NN_SVC_ID_CREATE_PORT             71");
            sysCallFile.AddLine("#define NN_SVC_ID_CREATE_NAME_PORT        72");
            sysCallFile.WriteData();
            return sysCallFile;
        }

        /* 共有ユーティリティメソッド */

        private FileCreator MakeParameterFile()
        {
            FileCreator header = new FileCreator();
            header.AddLine("name            : DUMMY");
            header.AddLine("program_id      : 0x0005000C10000000");
            header.AddLine("version         : 1");
            header.AddLine("stack_size      : 0x1000");
            header.AddLine("priority        : 10");
            header.AddLine("ideal_processor : 0");
            header.AddLine("use_secure_memory : False");
            header.AddLine("capability:");
            return header;
        }

        private bool CheckSignBit(uint flag, uint n)
        {
            Assert.IsFalse(n > 32);
            uint sign = (n == 32) ? 0xFFFFFFFF : ((1U << (int)n) - 1);
            if ((flag & sign) != sign)
            {
                return false;
            }
            if (n == 32)
            {
                return true;
            }
            else
            {
                sign = 1U << (int)n;
                return (flag & sign) == 0;
            }
        }

        private int CountFlags(uint[] flags)
        {
            int count = 0;
            foreach (var flag in flags)
            {
                if (flag == uint.MaxValue)
                {
                    break;
                }
                count++;
            }
            return count;
        }

        private bool CheckCreateParameterInfo(out ParameterInfo pOut, FileCreator header, SystemCallInfo syscallInfo, bool isTestMode = false)
        {
            ParameterInfo info = null;
            try
            {
                info = new ParameterInfo(header.FilePath, syscallInfo, isTestMode);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e);
                pOut = null;
                return false;
            }
            pOut = info;
            return true;
        }

        /* EnableInterrupts 用 */

        private FileCreator MakeEnableInterrupts(uint[] interrupts)
        {
            FileCreator header = MakeParameterFile();
            string[] strInterrupts = new string [interrupts.Length];
            for (int i = 0; i < interrupts.Length; i++)
            {
                strInterrupts[i] = interrupts[i].ToString();
            }
            AddEnableInterrupts(header, strInterrupts);
            return header;
        }

        private void AddEnableInterrupts(FileCreator header, string[] interrupts)
        {
            header.AddLine("    - EnableInterrupts:");
            foreach (var interrupt in interrupts)
            {
                header.AddLine(string.Format("        - {0}", interrupt));
            }
        }

        private bool CheckInterruptBit(uint flag, int bitNum, uint interrupt)
        {
            uint field = flag & ((1U << bitNum) - 1);
            return field == interrupt;
        }

        private void CheckEnableInterruptsFlag(uint flag, uint first, uint second)
        {
            Assert.IsTrue(CheckSignBit(flag, 11));
            flag >>= 12;
            Assert.IsTrue(CheckInterruptBit(flag, 10, first));
            flag >>= 10;
            Assert.IsTrue(CheckInterruptBit(flag, 10, second));
        }

        /* EnableSystemCalls 用 */

        private FileCreator MakeEnableSystemCalls()
        {
            FileCreator header = MakeParameterFile();
            AddEnableSystemCallsHeader(header);
            return header;
        }

        private void AddEnableSystemCallsHeader(FileCreator header)
        {
            header.AddLine("    - EnableSystemCalls:");
        }

        private void AddEnableSystemCalls(FileCreator header, string[] syscalls)
        {
            foreach (var syscall in syscalls)
            {
                header.AddLine(string.Format("         - {0}", syscall));
            }
        }

        private void CheckEnableSystemCallFlag(uint flag, uint[] syscalls)
        {
            // TEST 3-19
            // 識別フラグが正しい
            Assert.IsTrue(CheckSignBit(flag, 4));

            // TEST 3-20
            // 許可したシステムコール以外は許可されていない
            flag >>= 5;
            uint index = flag >> 24;
            uint checkFlag = 0;
            flag = flag & ((1U << 24) - 1);
            foreach (var syscall in syscalls)
            {
                uint sysIndex = syscall / 24;
                uint sysShift = syscall % 24;
                Assert.IsTrue(sysIndex == index);
                Assert.IsTrue((flag & (1U << (int)sysShift)) > 0);
                checkFlag |= 1U << (int)sysShift;
            }
            Assert.IsTrue(checkFlag == flag);
        }

        /* HandleTableSize 用 */

        private FileCreator MakeHandleTableSize(string tableSize)
        {
            FileCreator header = MakeParameterFile();
            AddHandleTableSize(header, tableSize);
            header.WriteData();
            return header;
        }

        private void AddHandleTableSize(FileCreator header, string tableSize)
        {
            header.AddLine(string.Format("    - HandleTableSize: {0}", tableSize));
        }

        private void CheckHandleTableSizeFlag(uint flag, uint tableSize)
        {
            // TEST 4-6
            // 識別フラグが正しい
            Assert.IsTrue(CheckSignBit(flag, 15));

            // TEST 4-7
            // 指定した値がデータフィールドに入っている
            flag >>= 16;
            Assert.IsTrue(flag == tableSize);
        }

        /* MapMemory 用 */

        private void AddMemoryMap(FileCreator header, string addr, string size, string permission, string type)
        {
            header.AddLine("    - MemoryMap:");
            if (addr != null)
            {
                header.AddLine(string.Format("        BeginAddress: {0}", addr));
            }
            if (size != null)
            {
                header.AddLine(string.Format("        Size: {0}", size));
            }
            if (permission != null)
            {
                header.AddLine(string.Format("        Permission: {0}", permission));
            }
            if (type != null)
            {
                header.AddLine(string.Format("        Type: {0}", type));
            }
        }

        private void CheckMemoryMapFlag(uint[] flags, ulong addr, ulong size, MemoryPermission permission, MemoryType type)
        {
            Assert.IsTrue(CountFlags(flags) == 2);

            // TEST 5-5
            // 識別フラグが正しい
            CheckSignBit(flags[0], 6);
            CheckSignBit(flags[1], 6);

            uint beginFlag = flags[0] >> 7;
            uint endFlag = flags[1] >> 7;
            uint checkFlag;

            // TEST 5-8, 5-9
            // MemoryPermission の指定が反映されている
            checkFlag = (uint)permission << 24;
            Assert.IsTrue((beginFlag & checkFlag) == checkFlag);

            // TEST 5-10, 5-11
            // MemoryType の指定が反映されている
            checkFlag = (uint)type << 24;
            Assert.IsTrue((endFlag & checkFlag) == checkFlag);

            uint flagMask = ~(1U << 24);
            beginFlag &= flagMask;
            endFlag &= flagMask;

            // TEST 5-6
            // データフィールドに指定したアドレスが入っている
            if ((uint)((addr >> 12) & ((1U << 24) - 1)) != beginFlag)
            {
                Console.Error.WriteLine("addr: 0x{0:X}, begin: 0x{1:X}", addr >> 12, beginFlag);
            }
            Assert.IsTrue(((addr >> 12) & ((1U << 24) - 1)) == beginFlag);

            // TEST 5-7
            // データフィールドに指定したサイズが入っている
            Assert.IsTrue((uint)((size >> 12) & ((1U << 20) - 1)) == endFlag);
        }

        private void CheckIoMappingFlag(uint flag, ulong addr)
        {
            // TEST 5-21
            // 識別フラグが正しい
            Assert.IsTrue(CheckSignBit(flag, 7));

            // TEST 5-13
            // データフィールドに指定したアドレスが入っている
            flag >>= 8;
            Assert.IsTrue(flag == ((addr >> 12) & ((1U << 24) - 1)));
        }

        private void CheckSuccessMemoryMap(ulong addr, ulong size, MemoryPermission permission, MemoryType type)
        {
            FileCreator header = MakeParameterFile();
            AddMemoryMap(header, string.Format("0x{0:X}", addr), string.Format("0x{0:X}", size), PermissionNames[(int)permission], TypeNames[(int)type]);
            header.WriteData();

            ParameterInfo paramInfo;
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));
            uint[] flags = paramInfo.Capability;
            Assert.IsTrue(CountFlags(flags) == 2);
            CheckMemoryMapFlag(flags, addr, size, permission, type);
        }

        private void CheckSuccessIoMapping(ulong addr)
        {
            FileCreator header = MakeParameterFile();
            AddMemoryMap(header, string.Format("0x{0:X}", addr), "0x1000", "RW", "IO");
            header.WriteData();

            ParameterInfo paramInfo;
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));
            uint[] flags = paramInfo.Capability;
            Assert.IsTrue(CountFlags(flags) == 1);
            CheckIoMappingFlag(flags[0], addr);
        }

        private void CheckFailMemoryMap(string addr, string size, MemoryPermission permission, MemoryType type)
        {
            FileCreator header = MakeParameterFile();
            string permissionName = (permission == MemoryPermission.None) ? null : PermissionNames[(int)permission];
            string typeName = (type == MemoryType.None) ? null : TypeNames[(int)type];
            AddMemoryMap(header, addr, size, permissionName, typeName);
            header.WriteData();

            ParameterInfo paramInfo;
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));
        }

        private void CheckFailIoMapping(ulong addr)
        {
            FileCreator header = MakeParameterFile();
            AddMemoryMap(header, string.Format("0x{0:X}", addr), "0x1000", "RW", "IO");
            header.WriteData();

            ParameterInfo paramInfo;
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));
        }

        /* Flags 用 */

        private FileCreator MakeOtherFlags(string flag)
        {
            FileCreator header = MakeParameterFile();

            AddOtherFlags(header, flag);
            return header;
        }

        private void AddOtherFlags(FileCreator header, string flag)
        {
            header.AddLine(string.Format("    - Flags: {0}", flag));
        }

        private void CheckOtherFlag(uint flag, uint dataFlag)
        {
            // TEST 6-15
            // 識別フラグが正しい
            Assert.IsTrue(CheckSignBit(flag, 16));

            // TEST 6-16
            // データフィールドに指定したフラグが入っている
            flag >>= 17;
            Assert.IsTrue(flag == dataFlag);
        }

        /* 外部ファイルを利用したテスト用 */

        private string ExecuteProgram(string outputFileName, string ymlFileName, string[] svcHeaders, string additionalOption = "")
        {
            System.Diagnostics.Process process = new System.Diagnostics.Process();
            TestUtility.TestPath testPath = new TestUtility.TestPath(this.TestContext);
            string svcHeaderParam = string.Empty;

            foreach (var headerName in svcHeaders)
            {
                svcHeaderParam = svcHeaderParam + " --svc_header " + headerName;
            }

            process.StartInfo.FileName = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\MakeInitialProgram\\MakeInitialProgram.exe";
            process.StartInfo.Arguments = string.Format(
                "--utf8 -o {0} --elf {1} --param_file {2} {3} {4}",
                outputFileName, BinaryFile, ResourceDir + ymlFileName,
                svcHeaderParam, additionalOption);
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardError = true;
            process.StartInfo.StandardErrorEncoding = System.Text.Encoding.UTF8;
            process.Start();

            string errorMsg = process.StandardError.ReadToEnd();
            process.WaitForExit();

            return errorMsg;
        }

        private void CheckBinaryData(string fileName, uint[] results)
        {
            using (System.IO.FileStream fs = System.IO.File.OpenRead(fileName))
            {
                using (System.IO.BinaryReader br = new System.IO.BinaryReader(fs))
                {
                    byte[] tmp = br.ReadBytes(0x80);
                    foreach (var flag in results)
                    {
                        uint data = unchecked((uint)br.ReadInt32());
                        Assert.IsTrue(data == flag);
                    }

                    for (int i = 0; i < 32 - results.Length; i++)
                    {
                        uint data = unchecked((uint)br.ReadInt32());
                        Assert.IsTrue(data == 0xFFFFFFFF);
                    }
                }
            }
        }

        private string ExecuteMakeSvcVeneer(string svcDefFile, string additionalOption = "")
        {
            System.Diagnostics.Process process = new System.Diagnostics.Process();
            TestUtility.TestPath testPath = new TestUtility.TestPath(this.TestContext);

            process.StartInfo.FileName = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\MakeSvcVeneer.exe";
            process.StartInfo.Arguments = string.Format(
                "--def {0} --template {1} --out_root {2} --abi arm333",
                ResourceDir + svcDefFile, MakeSvcVeneerTemplate, MakeSvcVeneerWorkDir);
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardError = true;
            process.Start();

            string errorMsg = process.StandardError.ReadToEnd();
            process.WaitForExit();

            return errorMsg;
        }

        private void ConcatSvcHeader(string outputFile, string baseFile, string addFile)
        {
            string contents;
            if (System.IO.File.Exists(outputFile))
            {
                System.IO.File.Delete(outputFile);
            }
            System.IO.File.Copy(baseFile, outputFile);
            using (System.IO.FileStream fs = System.IO.File.OpenRead(addFile))
            {
                using (System.IO.StreamReader sr = new System.IO.StreamReader(fs))
                {
                    // MakeSvcVeneer が吐き出すヘッダファイルのいらない部分を除去
                    for (int i = 0; i < 17; i++)
                    {
                        sr.ReadLine();
                    }
                    contents = sr.ReadToEnd();
                }
            }

            using (System.IO.FileStream fs = System.IO.File.OpenWrite(outputFile))
            {
                // 1改行分終端戻る
                fs.Seek(-2, System.IO.SeekOrigin.End);
                using (System.IO.StreamWriter sw = new System.IO.StreamWriter(fs))
                {
                    sw.WriteLine(contents);
                }
            }
        }

        /// <summary>
        /// テストの共有データの初期化
        /// コンストラクタでは TestContext が生成されていないため、メソッドで初期化する
        /// </summary>
        public void InitTest()
        {
            if (SysCallFile != null)
            {
                // 既に初期化済み
                return;
            }

            SysCallFile = MakeSystemCallFile();

            TestUtility.TestPath testPath = new TestUtility.TestPath(this.TestContext);
            BaseIdFile = testPath.GetSigloRoot() + "\\Programs\\Chris\\Include\\nn\\svc\\svc_BaseId.autogen.h";
            ServerIdFile = testPath.GetSigloRoot() + "\\Programs\\Chris\\Include\\nn\\svc\\svc_ServerId.autogen.h";
            DdIdFile = testPath.GetSigloRoot() + "\\Programs\\Chris\\Include\\nn\\svc\\svc_DdId.autogen.h";
            DmntIdFile = testPath.GetSigloRoot() + "\\Programs\\Chris\\Include\\nn\\svc\\svc_DmntId.autogen.h";
            TcbIdFile = testPath.GetSigloRoot() + "\\Programs\\Chris\\Include\\nn\\svc\\svc_TcbId.autogen.h";
            SvcHeaders = new string[] { BaseIdFile, ServerIdFile, DdIdFile, DmntIdFile, TcbIdFile };

            SysCallInfo = new SystemCallInfo();
            SysCallInfo.Load(SysCallFile.FilePath);

            ResourceDir = testPath.GetToolTestResourceDirectory() + "\\MakeInitialProgramTestData\\";
            BinaryFile = ResourceDir + "TestBinary.axf";
            MakeSvcVeneerWorkDir = ResourceDir + "tmpMakeSvcVeneer\\";
            MakeSvcVeneerTemplate = testPath.GetSigloRoot() + "\\Programs\\Chris\\Resources\\svc\\template.txt";
            MakeSvcVeneerIncludeDir = MakeSvcVeneerWorkDir + "Include\\nn\\svc\\";
        }

        [TestMethod]
        public void TestSystemCallInfo()
        {
            InitTest();

            SystemCallInfo syscallInfo = new SystemCallInfo();

            // TEST 1-1
            // SvcVeneer が生成したヘッダファイルを読み込むことが出来る
            try
            {
                syscallInfo.Load(BaseIdFile);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e);
                Assert.IsTrue(false);
            }

            // TEST 1-2
            // ヘッダの内容と読み込んだ内容が一致するかを確認する
            // 手作業で管理しなければならない
            Assert.IsTrue(syscallInfo.GetNumInfo() == NumBaseIdSysCall);
            for (int i = 0; i < BaseIdNameList.Length; i++)
            {
                Assert.IsTrue(syscallInfo.GetSystemCallNumber(BaseIdNameList[i]) == BaseIdList[i]);
            }

            // TEST 1-3
            // 違うフォーマットのヘッダファイルを渡したときはエラーを返す
            FileCreator invalidFile = new FileCreator();
            invalidFile.AddLine("#define SetHeapSize 10");
            invalidFile.WriteData();
            bool checkResult = false;
            try
            {
                syscallInfo.Load(invalidFile.FilePath);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e);
                checkResult = true;
            }

            Assert.IsTrue(checkResult);

            // TEST 1-4
            // 指定したパスが存在しないときはエラーを返す
            string tmpFile = System.IO.Path.GetRandomFileName();
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            checkResult = false;
            try
            {
                syscallInfo.Load(tmpFile);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e);
                checkResult = true;
            }

            Assert.IsTrue(checkResult);

            // TEST 1-5
            // 同じファイルを重複して読み込むことはできない
            checkResult = false;
            try
            {
                syscallInfo.Load(BaseIdFile);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e);
                checkResult = true;
            }

            Assert.IsTrue(checkResult);

            // TEST 1-6
            // 同じシステムコール名を重複して指定することは出来ない
            invalidFile = new FileCreator();
            invalidFile.AddLine("#define NN_SVC_ID_SAME_NAME 100");
            invalidFile.AddLine("#define NN_SVC_ID_SAME_NAME 101");
            invalidFile.WriteData();
            checkResult = false;
            try
            {
                syscallInfo.Load(invalidFile.FilePath);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e);
                checkResult = true;
            }

            Assert.IsTrue(checkResult);

            // TEST 1-7
            // 同じシステムコール番号を重複して指定することは出来ない
            invalidFile = new FileCreator();
            invalidFile.AddLine("#define NN_SVC_ID_SAME_ID1 102");
            invalidFile.AddLine("#define NN_SVC_ID_SAME_ID2 102");
            invalidFile.WriteData();
            checkResult = false;
            try
            {
                syscallInfo.Load(invalidFile.FilePath);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e);
                checkResult = true;
            }

            Assert.IsTrue(checkResult);

            // TEST 1-8
            // 同じシステムコール名とシステムコール番号を重複して指定することは出来ない
            invalidFile = new FileCreator();
            invalidFile.AddLine("#define NN_SVC_ID_SAME_SVC 103");
            invalidFile.AddLine("#define NN_SVC_ID_SAME_SVC 103");
            invalidFile.WriteData();
            checkResult = false;
            try
            {
                syscallInfo.Load(invalidFile.FilePath);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e);
                checkResult = true;
            }

            Assert.IsTrue(checkResult);

            // TEST 1-9
            // 違うファイルで宣言されているシステムコール名を重複して指定することは出来ない
            invalidFile = new FileCreator();
            invalidFile.AddLine("#define NN_SVC_ID_SET_HEAP_SIZE 104");
            invalidFile.WriteData();
            checkResult = false;
            try
            {
                syscallInfo.Load(invalidFile.FilePath);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e);
                checkResult = true;
            }

            Assert.IsTrue(checkResult);

            // TEST 1-10
            // 違うファイルで宣言されているシステムコール番号を重複して指定することは出来ない
            invalidFile = new FileCreator();
            invalidFile.AddLine("#define NN_SVC_ID_SAME_ID 1");
            invalidFile.WriteData();
            checkResult = false;
            try
            {
                syscallInfo.Load(invalidFile.FilePath);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e);
                checkResult = true;
            }

            Assert.IsTrue(checkResult);

            // TEST 1-8
            // 違うファイルで宣言されているシステムコールの情報を重複して指定することはできない
            invalidFile = new FileCreator();
            invalidFile.AddLine("#define NN_SVC_ID_SET_HEAP_SIZE 1");
            invalidFile.WriteData();
            checkResult = false;
            try
            {
                syscallInfo.Load(invalidFile.FilePath);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e);
                checkResult = true;
            }

            Assert.IsTrue(checkResult);
        }

        [TestMethod]
        public void TestEnableInterrupts()
        {
            uint[] flags;
            CapabilityParameter capability = new CapabilityParameter();
            ParameterInfo paramInfo;
            FileCreator header;

            InitTest();

            header = MakeEnableInterrupts(new uint[] { 0 });
            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));
            flags = paramInfo.Capability;
            Assert.IsTrue(CountFlags(flags) == 1);

            // TEST 2-1
            // 識別フラグが正しい
            Assert.IsTrue(CheckSignBit(flags[0], 11));
            CheckEnableInterruptsFlag(flags[0], 0, 1023);

            // TEST 2-2
            // 空いているフィールドデータが 1023 で埋められている。
            for (uint i = 1; i < 1023; i++)
            {
                header = MakeEnableInterrupts(new uint[] { i });
                header.WriteData();
                Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));
                flags = paramInfo.Capability;
                Assert.IsTrue(CountFlags(flags) == 1);
                CheckEnableInterruptsFlag(flags[0], i, 1023);
            }

            // TEST 2-3 から 2-6
            // 廃止

            // TEST 2-7
            // 1つ目のデータに 10bit 以上の値を渡す
            header = MakeEnableInterrupts(new uint[] { 1024, 1, 2, 3 });
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));

            // TEST 2-8
            // 2つ目のデータに 10bit 以上の値を渡す
            header = MakeEnableInterrupts(new uint[] { 1, 1024, 2, 3 });
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));

            // TEST 2-9
            // 3つ目のデータに 10bit 以上の値を渡す
            header = MakeEnableInterrupts(new uint[] { 1, 2, 0x10000, 3 });
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));

            // TEST 2-10
            // 4つ目のデータに 10bit 以上の値を渡す
            header = MakeEnableInterrupts(new uint[] { 1, 2, 3, 0xFFFFFFFF });
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));

            // TEST 2-11
            // 数値以外のパラメータは受け付けない
            header = MakeParameterFile();
            AddEnableInterrupts(header, new string[] { "Ethernet" });
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));

            // TEST 2-12
            // 廃止

            // TEST 2-13
            // 重複して EnableInterrupts を宣言することはできない
            header = MakeParameterFile();
            AddEnableInterrupts(header, new string[] { "10" });
            AddEnableInterrupts(header, new string[] { "11" });
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));

            // TEST 2-14
            // 重複して割り込み番号を宣言することはできない
            header = MakeParameterFile();
            AddEnableInterrupts(header, new string[] { "10", "10" });
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));

            // TEST 2-15
            // 16 進数の値は受け付けない
            header = MakeParameterFile();
            AddEnableInterrupts(header, new string[] { "0xa" });
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));
        }

        [TestMethod]
        public void TestEnableSystemCalls()
        {
            FileCreator header;
            ParameterInfo paramInfo;
            uint[] flags;
            string[] noneList = { "CreateThread", "CreateProcess", "CloseHandle" };
            string[] numList = { "1", "3", "5" };

            InitTest();

            // TEST 3-1 から 3-15
            // 廃止

            // TEST 3-16
            // EnableSystemCalls は重複して登録することが出来ない
            header = MakeEnableSystemCalls();
            AddEnableSystemCalls(header, new string[] { SysCallNameList[0], SysCallNameList[1] });
            AddEnableSystemCallsHeader(header);
            AddEnableSystemCalls(header, new string[] { SysCallNameList[2], SysCallNameList[3] });
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));

            // TEST 3-17
            // 重複してシステムコールを登録することが出来ない
            header = MakeEnableSystemCalls();
            AddEnableSystemCalls(header, SysCallNameList);
            AddEnableSystemCalls(header, SysCallNameList);
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));

            // TEST 3-18
            // 数値でシステムコールを登録することは出来ない
            header = MakeEnableSystemCalls();
            AddEnableSystemCalls(header, numList);
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));

            // TEST 3-21
            // 文字列で指定したシステムコールを登録することができる
            header = MakeEnableSystemCalls();
            AddEnableSystemCalls(header, new string[] { SysCallNameList[0] });
            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));
            flags = paramInfo.Capability;
            Assert.IsTrue(CountFlags(flags) == 1);
            CheckEnableSystemCallFlag(flags[0], new uint[] { SysCallIds[0] });

            // TEST 3-22
            // 列挙されたシステムコールの番号を 24 で割った値が同じであれば、1つのデータに集約される
            header = MakeEnableSystemCalls();
            AddEnableSystemCalls(header, new string[] { SysCallNameList[0], SysCallNameList[1] });
            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));
            flags = paramInfo.Capability;
            Assert.IsTrue(CountFlags(flags) == 1);
            CheckEnableSystemCallFlag(flags[0], new uint[] { SysCallIds[0], SysCallIds[1] });

            // TEST 3-23
            // 列挙されたシステムコールの番号を 24 で割った値が違う場合は、複数のデータに分割される
            header = MakeEnableSystemCalls();
            AddEnableSystemCalls(header, new string[] { SysCallNameList[0], SysCallNameList[2] });
            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));
            flags = paramInfo.Capability;
            Assert.IsTrue(CountFlags(flags) == 2);
            CheckEnableSystemCallFlag(flags[0], new uint[] { SysCallIds[0] });
            CheckEnableSystemCallFlag(flags[1], new uint[] { SysCallIds[2] });

            // TEST 3-24
            // SystemCallInfo にないシステムコールは受け付けない
            header = MakeEnableSystemCalls();
            AddEnableSystemCalls(header, new string[] { "InvalidSystemCall" });
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));
        }

        [TestMethod]
        public void TestHandleTableSize()
        {
            FileCreator header;
            ParameterInfo paramInfo;
            uint[] flags;

            InitTest();

            // TEST 4-1
            // 10進数のデータを受け付ける
            header = MakeHandleTableSize("0");
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));
            for (uint i = 1; i < 1024; i <<= 1)
            {
                header = MakeHandleTableSize(i.ToString());
                Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));
                flags = paramInfo.Capability;
                Assert.IsTrue(CountFlags(flags) == 1);
                CheckHandleTableSizeFlag(flags[0], i);
            }

            // TEST 4-2
            // 廃止

            // TEST 4-3
            // 1024 以上の値は受け付けない
            header = MakeHandleTableSize("1024");
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));

            // TEST 4-4
            // 重複して HandleTableSize を登録することはできない
            header = new FileCreator();
            AddHandleTableSize(header, "0");
            AddHandleTableSize(header, "1");
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));

            // TEST 4-5
            // 文字列は受け付けない
            header = MakeHandleTableSize("zero");
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));

            // TEST 4-8
            // 16 進数の値は受け付けない
            header = MakeHandleTableSize("0xa");
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));
        }

        [TestMethod]
        public void TestMemoryMap()
        {
            FileCreator header;
            ParameterInfo paramInfo;
            uint count = 1;
            uint permissionIndex = 0;
            uint typeIndex = 0;

            InitTest();

            count = 1; // count = 0 からにしてしまうと、1PageIOMapping の条件を満たしてしまう。
            for (ulong i = 0x1000; i < 0x1000000000; i <<= 1, count++)
            {
                // TEST 5-1
                // 通常のマッピングデータは、36bit までアドレスを指定することが出来、データが2つ生成される
                permissionIndex = count % 2;
                typeIndex = (count / 2) % 2;
                CheckSuccessMemoryMap(i, 0x2000, (MemoryPermission)permissionIndex, (MemoryType)typeIndex);

                if (i < 0x100000000)
                {
                    // TEST 5-2
                    // 通常のマッピングデータは、32bit までサイズを指定することが出来、データが2つ生成される
                    CheckSuccessMemoryMap(0x0, i, (MemoryPermission)permissionIndex, (MemoryType)typeIndex);
                }

                count++;
            }

            // TEST 5-1
            // 境界値
            CheckSuccessMemoryMap(0xFFFFFE000, 0x2000, MemoryPermission.Ro, MemoryType.Static);

            // TEST 5-2
            // 境界値
            CheckSuccessMemoryMap(0x0, 0xFFFFF000, MemoryPermission.Rw, MemoryType.Io);

            // TEST 5-3
            // IOPageMapping の条件を満たさない限り、2つのデータが生成される
            CheckSuccessMemoryMap(0x1000, 0x1000, MemoryPermission.Ro, MemoryType.Io);

            // TEST 5-4
            // IOPageMapping の条件を満たさない限り、2つのデータが生成される
            CheckSuccessMemoryMap(0x1000, 0x1000, MemoryPermission.Rw, MemoryType.Static);

            // TEST 5-12
            // IOPageMapping のデータは、36bit までアドレスを指定することが出来、データが1つ生成される。
            for (ulong addr = 0x1000; addr < 0x1000000000; addr <<= 1)
            {
                CheckSuccessIoMapping(addr);
            }
            // 境界値
            CheckSuccessIoMapping(0xFFFFFF000);

            count = 0;
            for (uint i = 0x1; i < 0x1000; i <<= 1, count++)
            {
                permissionIndex = count % 2;
                typeIndex = (count / 2) % 2;

                // TEST 5-14
                // アドレスが4KBにアライメントされていないと失敗する
                CheckFailMemoryMap(string.Format("0x{0:X}", i), "0x2000", (MemoryPermission)permissionIndex, (MemoryType)typeIndex);

                // TEST 5-15
                // サイズが4KBにアライメントされていないと失敗する
                CheckFailMemoryMap("0x0", string.Format("0x{0:X}", i), (MemoryPermission)permissionIndex, (MemoryType)typeIndex);

                // TEST 5-30
                // IOPageMapping のアドレスが4KBにアライメントされていないと失敗する
                CheckFailIoMapping(i);
            }

            // TEST 5-16
            // Size == 0 は失敗
            CheckFailMemoryMap("0x0", "0x0", MemoryPermission.Rw, MemoryType.Io);

            // TEST 5-17
            // アドレスに10 進数の値は受け取らない
            CheckFailMemoryMap("0", "0x1000", MemoryPermission.Ro, MemoryType.Static);

            // TEST 5-18
            // サイズに10 進数の値は受け取らない
            CheckFailMemoryMap("0x0", "4096", MemoryPermission.Ro, MemoryType.Static);

            // TEST 5-19
            // Permission に "RO" と "RW" 以外の文字列は受け付けない
            header = MakeParameterFile();
            AddMemoryMap(header, "0x0", "0x2000", "RX", TypeNames[(int)MemoryType.Io]);
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));

            // TEST 5-20
            // Type に "Static" と "Io" 以外の値は受け付けない
            header = MakeParameterFile();
            AddMemoryMap(header, "0x0", "0x2000", PermissionNames[(int)MemoryPermission.Ro], "Shared");
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo));

            // TEST 5-22
            // BeginAddress が指定されていないと失敗する
            CheckFailMemoryMap(null, "0x2000", MemoryPermission.Ro, MemoryType.Static);

            // TEST 5-23
            // Size が指定されていないと失敗する
            CheckFailMemoryMap("0x0", null, MemoryPermission.Rw, MemoryType.Static);

            // TEST 5-24
            // Permission が指定されていないと失敗する
            CheckFailMemoryMap("0x0", "0x2000", MemoryPermission.None, MemoryType.Static);

            // TEST 5-25
            // Type が指定されていないと失敗する
            CheckFailMemoryMap("0x0", "0x2000", MemoryPermission.Rw, MemoryType.None);

            // TEST 5-26
            // オーバーフローが発生するアドレスとサイズの組み合わせは受け付けない
            CheckFailMemoryMap("0xFFFFFF000", "0x2000", MemoryPermission.Ro, MemoryType.Static);
            CheckFailMemoryMap("0xFFFFFF000", "0xFFFFFFFFFFFFF000", MemoryPermission.Rw, MemoryType.Io);

            // TEST 5-27
            // BeginAddress は37 bit 以上の値を受け付けない
            CheckFailMemoryMap("0x1000000000", "0x2000", MemoryPermission.Ro, MemoryType.Static);

            // TEST 5-28
            // Size は 33 bit 以上の値を受け付けない
            CheckFailMemoryMap("0x0", "0x100000000", MemoryPermission.Ro, MemoryType.Static);

            // TEST 5-29
            // IOPageMapping の BeginAddress は 37 bit 以上の値を受け付けない
            CheckFailMemoryMap("0x1000000000", "0x1000", MemoryPermission.Rw, MemoryType.Io);
        }

        [TestMethod]
        public void TestTestMode()
        {
            FileCreator header;
            uint[] flags;
            ParameterInfo paramInfo;

            InitTest();

            // TEST 6-1
            // EnableInterrupts を重複して宣言することが出来る
            header = MakeParameterFile();
            AddEnableInterrupts(header, new string[] { "10" });
            AddEnableInterrupts(header, new string[] { "11" });
            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));
            flags = paramInfo.Capability;
            Assert.IsTrue(CountFlags(flags) == 2);
            CheckEnableInterruptsFlag(flags[0], 0xa, 1023);
            CheckEnableInterruptsFlag(flags[1], 0xb, 1023);

            // TEST 6-2
            // EnableSystemCalls を重複して宣言することが出来る
            header = MakeEnableSystemCalls();
            AddEnableSystemCalls(header, new string[] { SysCallNameList[0], SysCallNameList[1] });
            AddEnableSystemCallsHeader(header);
            AddEnableSystemCalls(header, new string[] { SysCallNameList[2], SysCallNameList[3] });
            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));
            flags = paramInfo.Capability;
            Assert.IsTrue(CountFlags(flags) == 3);
            CheckEnableSystemCallFlag(flags[0], new uint[] { SysCallIds[0], SysCallIds[1] });
            CheckEnableSystemCallFlag(flags[1], new uint[] { SysCallIds[2] });
            CheckEnableSystemCallFlag(flags[2], new uint[] { SysCallIds[3] });

            // TEST 6-3
            // EnableSystemCalls に10進数の数値を指定することが出来る
            header = MakeEnableSystemCalls();
            AddEnableSystemCalls(header, new string[] { "2" });
            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));
            flags = paramInfo.Capability;
            Assert.IsTrue(CountFlags(flags) == 1);
            CheckEnableSystemCallFlag(flags[0], new uint[] { 2 });

            // TEST 6-4
            // 廃止

            // TEST 6-5
            // EnableSystemCalls は16進数の数値を受け付けない
            header = MakeEnableSystemCalls();
            AddEnableSystemCalls(header, new string[] { "0x2" });
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));

            // TEST 6-6
            // HandleTableSize を重複して宣言することが出来る
            header = MakeParameterFile();
            AddHandleTableSize(header, "1");
            AddHandleTableSize(header, "2");
            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));
            flags = paramInfo.Capability;
            Assert.IsTrue(CountFlags(flags) == 2);
            CheckHandleTableSizeFlag(flags[0], 1);
            CheckHandleTableSizeFlag(flags[1], 2);

            // TEST 6-7
            // HandleTableSize に 1024 以上の16bit 符号なし整数の値を登録することが出来る
            for (uint i = 0x400; i < 0x10000; i <<= 1)
            {
                header = MakeParameterFile();
                AddHandleTableSize(header, i.ToString());
                header.WriteData();
                Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));
                flags = paramInfo.Capability;
                Assert.IsTrue(CountFlags(flags) == 1);
                CheckHandleTableSizeFlag(flags[0], i);
            }

            // TEST 6-7
            // 境界値
            header = MakeParameterFile();
            AddHandleTableSize(header, "65535");
            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));
            flags = paramInfo.Capability;
            Assert.IsTrue(CountFlags(flags) == 1);
            CheckHandleTableSizeFlag(flags[0], 0xFFFF);

            // TEST 6-8
            // BeginAddress と Size を足し合わせた値についてのチェックがなくなる
            header = MakeParameterFile();
            AddMemoryMap(header, "0xFFFFFF000", "0x2000", PermissionNames[(int)MemoryPermission.Ro], TypeNames[(int)MemoryType.Static]);
            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));
            flags = paramInfo.Capability;
            Assert.IsTrue(CountFlags(flags) == 2);
            CheckMemoryMapFlag(flags, 0xFFFFFF000, 0x2000, MemoryPermission.Ro, MemoryType.Static);

            header = MakeParameterFile();
            AddMemoryMap(header, "0xFFFFFF000", "0xFFFFFFFFFFFFF000", PermissionNames[(int)MemoryPermission.Ro], TypeNames[(int)MemoryType.Static]);
            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));
            flags = paramInfo.Capability;
            Assert.IsTrue(CountFlags(flags) == 2);
            CheckMemoryMapFlag(flags, 0xFFFFFF000, 0xFFFFFFFFFFFFF000, MemoryPermission.Ro, MemoryType.Static);

            // TEST 6-9
            // MemoryMap の Size に 0を指定すると、1つのデータしか生成されない
            header = MakeParameterFile();
            AddMemoryMap(header, "0xFFFFF000", "0x0", PermissionNames[(int)MemoryPermission.Ro], TypeNames[(int)MemoryType.Static]);
            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));
            flags = paramInfo.Capability;
            Assert.IsTrue(CountFlags(flags) == 1);
            CheckSignBit(flags[0], 10);
            Assert.IsTrue(((flags[0] >> 7) & 0xFFFFF) == 0xFFFFF);

            // TEST 6-10
            // Padding が使える
            header = MakeParameterFile();
            header.AddLine("    - Padding");
            AddMemoryMap(header, "0xFFFFF000", "0x0", PermissionNames[(int)MemoryPermission.Ro], TypeNames[(int)MemoryType.Static]);
            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));
            flags = paramInfo.Capability;

            // TEST 6-11
            // 識別子が正しい
            CheckSignBit(flags[0], 32);
            CheckSignBit(flags[1], 10);

            // TEST 6-12
            // TESTモードでないと Padding が使えない
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, false));

            // TEST 6-13
            // Padding の指定がマップ形式になっていると失敗する
            header = MakeParameterFile();
            header.AddLine("    - Padding:");
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));

            // TEST 6-16, 6-17
            // 廃止

            // TEST 6-18
            // TEST モードでないときには使えない
            header = MakeOtherFlags("0x1");
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, false));

            // TEST 6-19
            // 廃止

            // TEST 6-20
            // EnableSystemCalls を重複して登録する場合、他の EnableSystemCalls で宣言したシステムコールの情報を登録することが出来る
            header = MakeEnableSystemCalls();
            AddEnableSystemCalls(header, SysCallNameList);
            AddEnableSystemCallsHeader(header);
            AddEnableSystemCalls(header, SysCallNameList);
            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));
            flags = paramInfo.Capability;
            Assert.IsTrue(CountFlags(flags) == 6);
            CheckEnableSystemCallFlag(flags[0], new uint[] { SysCallIds[0], SysCallIds[1] });
            CheckEnableSystemCallFlag(flags[1], new uint[] { SysCallIds[2] });
            CheckEnableSystemCallFlag(flags[2], new uint[] { SysCallIds[3] });
            CheckEnableSystemCallFlag(flags[3], new uint[] { SysCallIds[0], SysCallIds[1] });
            CheckEnableSystemCallFlag(flags[4], new uint[] { SysCallIds[2] });
            CheckEnableSystemCallFlag(flags[5], new uint[] { SysCallIds[3] });

            // TEST 6-21
            // HandleTableSize に 17bit 以上の値を指定できない
            header = MakeParameterFile();
            AddHandleTableSize(header, "0x10000");
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));

            // TEST 6-22
            // BeginAddress に 37bit 以上の値が指定できる
            header = MakeParameterFile();
            AddMemoryMap(header, "0x1000000000", "0x2000", PermissionNames[(int)MemoryPermission.Ro], TypeNames[(int)MemoryType.Static]);
            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));
            flags = paramInfo.Capability;
            Assert.IsTrue(CountFlags(flags) == 2);
            CheckMemoryMapFlag(flags, 0, 0x2000, MemoryPermission.Ro, MemoryType.Static);

            // TEST 6-23
            // Size に 32bit 以上の値が指定できる
            header = MakeParameterFile();
            AddMemoryMap(header, "0x0", "0x100000000", PermissionNames[(int)MemoryPermission.Ro], TypeNames[(int)MemoryType.Io]);
            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));
            flags = paramInfo.Capability;
            Assert.IsTrue(CountFlags(flags) == 2);
            CheckMemoryMapFlag(flags, 0x0, 0x100000000, MemoryPermission.Ro, MemoryType.Io);

            // TEST 6-24
            // Flags に 16 進数の値を指定できる
            header = MakeParameterFile();
            AddOtherFlags(header, "0x0");
            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));
            flags = paramInfo.Capability;
            CheckOtherFlag(flags[0], 0);
            for (uint i = 1; i < 0x8000; i <<= 1)
            {
                header = MakeParameterFile();
                AddOtherFlags(header, string.Format("0x{0:X}", i));
                header.WriteData();
                Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));
                flags = paramInfo.Capability;
                CheckOtherFlag(flags[0], i);
            }
            // 境界値
            header = MakeParameterFile();
            AddOtherFlags(header, "0x7FFF");
            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));
            flags = paramInfo.Capability;
            CheckOtherFlag(flags[0], 0x7FFF);

            // TEST 6-25
            // Flags は 15 ビット符号なし整数の範囲外の値は受け付けない
            header = MakeParameterFile();
            AddOtherFlags(header, "0x8000");
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));

            // TEST 6-26
            // Flags は 16 進数の値は受け付けない
            header = MakeParameterFile();
            AddOtherFlags(header, "0");
            header.WriteData();
            Assert.IsFalse(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, true));
        }

        // TEST 7
        // 廃止

        [TestMethod]
        public void TestCapability()
        {
            FileCreator header;
            ParameterInfo paramInfo;
            uint[] flags;

            InitTest();

            // TEST 8-1
            // 複数のケーパビリティを設定することが出来る
            header = MakeParameterFile();

            AddEnableInterrupts(header, new string[] { "13", "24" });

            AddEnableSystemCallsHeader(header);
            AddEnableSystemCalls(header, SysCallNameList);

            AddHandleTableSize(header, "0");

            AddMemoryMap(header, "0x8CCCC000", "0x2000", PermissionNames[(int)MemoryPermission.Ro], TypeNames[(int)MemoryType.Static]);
            AddMemoryMap(header, "0x9CCCC000", "0x1000", PermissionNames[(int)MemoryPermission.Rw], TypeNames[(int)MemoryType.Io]);

            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, false));
            flags = paramInfo.Capability;
            Assert.IsTrue(CountFlags(flags) == 8);

            CheckEnableInterruptsFlag(flags[0], 13, 24);

            CheckEnableSystemCallFlag(flags[1], new uint[] { SysCallIds[0], SysCallIds[1] });
            CheckEnableSystemCallFlag(flags[2], new uint[] { SysCallIds[2] });
            CheckEnableSystemCallFlag(flags[3], new uint[] { SysCallIds[3] });

            CheckHandleTableSizeFlag(flags[4], 0);

            CheckMemoryMapFlag(new uint[] { flags[5], flags[6] }, 0x8CCCC000, 0x2000, MemoryPermission.Ro, MemoryType.Static);
            CheckIoMappingFlag(flags[7], 0x9CCCC000);

            // TEST 8-3
            // 残りの領域がパディング情報で埋められている
            for (int i = 8; i < 32; i++)
            {
                Assert.IsTrue(flags[i] == 0xFFFFFFFF);
            }

            // TESET 8-2
            // ケーパビリティが指定されていないと F で埋め尽くされる
            header = MakeParameterFile();
            header.DeleteLine(header.FileLines() - 1);
            header.WriteData();
            Assert.IsTrue(CheckCreateParameterInfo(out paramInfo, header, SysCallInfo, false));
            flags = paramInfo.Capability;
            Assert.IsTrue(CountFlags(flags) == 0);
        }

        [TestMethod]
        public void TestSuccessToCreateBinaryFile()
        {
            InitTest();
            string tmpFile = System.IO.Path.GetRandomFileName();
            string errorMsg;

            // TEST 9-1
            // EnableInterrupts のデータに対応するバイナリデータが出力される
            errorMsg = ExecuteProgram(tmpFile, "EnableInterrupts.yml", SvcHeaders);
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            CheckBinaryData(tmpFile, new uint[] { 0x204807FF, 0x1FC1F7FF });

            // TEST 9-2
            // EnableSystemCalls のデータに対応するバイナリデータが出力される
            errorMsg = ExecuteProgram(tmpFile, "EnableSystemCalls.yml", SvcHeaders);
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            CheckBinaryData(tmpFile, new uint[] { 0x1000004F, 0x2000002F, 0x4020000F, 0x6000200F, 0x9020000F, 0xA000002F });

            // TEST 9-3
            // MemoryMap のデータに対応するバイナリデータが出力される
            errorMsg = ExecuteProgram(tmpFile, "Memorymap.yml", SvcHeaders);
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            CheckBinaryData(tmpFile, new uint[] { 0x180003F, 0x800001BF, 0x8200003F, 0x23F, 0x8280003F, 0x8000013F, 0x300003F, 0x2BF });

            // TEST 9-4
            // IO ページマッピングのデータに対応するバイナリデータが出力される
            errorMsg = ExecuteProgram(tmpFile, "IoMemoryMapping.yml", SvcHeaders);
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            CheckBinaryData(tmpFile, new uint[] { 0x300007F });

            // TEST 9-5
            // HandleTableSize のデータに対応するバイナリデータが出力される
            errorMsg = ExecuteProgram(tmpFile, "HandleTableSize.yml", SvcHeaders);
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            CheckBinaryData(tmpFile, new uint[] { 0x647FFF });

            // TEST 9-6
            // TEST 9-1 から9-5 までの宣言を1つのファイル内で同時に宣言出来る
            errorMsg = ExecuteProgram(tmpFile, "NormalYml1.yml", SvcHeaders);
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            CheckBinaryData(tmpFile, new uint[] {
                0x204807FF, 0x1FC1F7FF,
                0x1000004F, 0x2000002F, 0x4020000F, 0x6000200F, 0x9020000F, 0xA000002F,
                0x180003F, 0x800001BF, 0x8200003F, 0x23F, 0x8280003F, 0x8000013F, 0x300003F, 0x2BF,
                0x300007F,
                0x647FFF,
            });

            // TEST 9-7
            // TEST 9-6 で宣言した順番を入れ替えてもバイナリが正しく出力される
            errorMsg = ExecuteProgram(tmpFile, "NormalYml2.yml", SvcHeaders);
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            CheckBinaryData(tmpFile, new uint[] {
                0x1000004F, 0x2000002F, 0x4020000F, 0x6000200F, 0x9020000F, 0xA000002F,
                0x300003F, 0x2BF,
                0x204807FF, 0x1FC1F7FF,
                0x180003F, 0x800001BF, 0x8200003F, 0x23F,
                0x300007F,
                0x647FFF,
                0x8280003F, 0x8000013F,
            });

            // TEST 9-8
            // Test Mode のバイナリが生成できる
            errorMsg = ExecuteProgram(tmpFile, "TestMode.yml", SvcHeaders, "--testmode");
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            CheckBinaryData(tmpFile, new uint[] {
                0xFFC017FF,
                0xFFC017FF,
                0xCF,
                0x4F,
                0x4F,
                0x7FFF,
                0x17FFF,
                0xFFFFFFBF,
                0x8000013F,
                0x8080003F,
                0x8000003F,
                0x8000003F,
                0xFFFFFFFF,
                0x2FFFF,
            });

            // TEST 9-9
            // svc_header オプションがなくてもバイナリを生成できる
            errorMsg = ExecuteProgram(tmpFile, "EnableInterrupts.yml", new string[0]);
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            CheckBinaryData(tmpFile, new uint[] { 0x204807FF, 0x1FC1F7FF });

            System.IO.File.Delete(tmpFile);
        }

        [TestMethod]
        public void TestFailToCreateBinaryFile()
        {
            InitTest();
            string tmpFile = System.IO.Path.GetRandomFileName();
            string errorMsg;
            Match match;

            if (System.IO.Directory.Exists(MakeSvcVeneerWorkDir))
            {
                System.IO.Directory.Delete(MakeSvcVeneerWorkDir, true);
            }

            string sameNameHeader = MakeSvcVeneerIncludeDir + "svc_ServerId.h";
            string sameIdHeader = MakeSvcVeneerIncludeDir + "svc_DdId.h";
            string targetHeader = MakeSvcVeneerWorkDir + "svc_SameName.h";

            errorMsg = ExecuteMakeSvcVeneer("SameSvcName.list");
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);

            errorMsg = ExecuteMakeSvcVeneer("SameSvcId.list");
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);

            // TEST 10-1
            // 同じ SVC Header 内に同じシンボル名があると失敗する
            ConcatSvcHeader(targetHeader, BaseIdFile, sameNameHeader);
            errorMsg = ExecuteProgram(tmpFile, "NormalYml1.yml", new string[] { targetHeader });
            match = Regex.Match(errorMsg, @"エラー: '[^']+' 内の 'SetHeapSize' はすでに登録されています。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-2
            // 同じ SVC Header 内に同じシステムコール番号があると失敗する
            ConcatSvcHeader(targetHeader, BaseIdFile, sameIdHeader);
            errorMsg = ExecuteProgram(tmpFile, "NormalYml1.yml", new string[] { targetHeader });
            match = Regex.Match(errorMsg, @"エラー: '[^']+' 内の 'InvalidSystemCallNumber' に指定した '1' はすでに登録されています。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-3
            // 違うファイルに同じシンボル名があると失敗する
            errorMsg = ExecuteProgram(tmpFile, "NormalYml1.yml", new string[] { BaseIdFile, sameNameHeader });
            match = Regex.Match(errorMsg, @"エラー: '[^']+' 内の 'SetHeapSize' はすでに登録されています。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-4
            // 違うファイルに同じシステムコール番号があると失敗する
            errorMsg = ExecuteProgram(tmpFile, "NormalYml1.yml", new string[] { BaseIdFile, sameIdHeader });
            match = Regex.Match(errorMsg, @"エラー: '[^']+' 内の 'InvalidSystemCallNumber' に指定した '1' はすでに登録されています。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-5
            // 重複して同じヘッダファイルを読み込ませると失敗する
            errorMsg = ExecuteProgram(tmpFile, "NormalYml1.yml", new string[] { BaseIdFile, BaseIdFile });
            match = Regex.Match(errorMsg, @"エラー: '[^']+' 内の 'SetHeapSize' はすでに登録されています。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-6
            // EnableInterrupts は 10bit 以上の値を受け付けない
            errorMsg = ExecuteProgram(tmpFile, "Test10-06.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル中の'EnableInterrupts'の値に 10 ビット整数ではない値が指定されました。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-7
            // EnableInterrupts は重複した値を受け付けない
            errorMsg = ExecuteProgram(tmpFile, "Test10-07.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'EnableInterrupts'はすでに登録されています。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-8
            // EnableInterrupts は重複して宣言出来ない
            errorMsg = ExecuteProgram(tmpFile, "Test10-08.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'EnableInterrupts'はすでに登録されています。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-9
            // EnableSystemCalls は数値を受け付けない
            errorMsg = ExecuteProgram(tmpFile, "Test10-09.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'EnableSystemCalls'に未知のシステムコール '1' が指定されました。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-10
            // EnableSystemCalls は登録されたシステムコール名以外を受け付けない
            errorMsg = ExecuteProgram(tmpFile, "Test10-10.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'EnableSystemCalls'に未知のシステムコール 'InvalidSystemCall' が指定されました。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-11
            // EnableSystemCalls は重複したシステムコール名を受け付けない
            errorMsg = ExecuteProgram(tmpFile, "Test10-11.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'EnableSystemCalls'の'SetHeapSize'は既に登録されています。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-12
            // EnableSystemCalls を重複して宣言出来ない
            errorMsg = ExecuteProgram(tmpFile, "Test10-12.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'EnableSystemCalls'はすでに登録されています。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-13
            // HandleTableSize に 1024 以上の値を入れることはできない
            errorMsg = ExecuteProgram(tmpFile, "Test10-13.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル中の'HandleTableSize'の値に 1024 以上の値が指定されています。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-14
            // HandleTableSize を重複して宣言出来ない
            errorMsg = ExecuteProgram(tmpFile, "Test10-14.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'HandleTableSize'はすでに登録されています。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-15
            // MemoryMap の BeginAddress は 4KBにアライメントされていないといけない
            errorMsg = ExecuteProgram(tmpFile, "Test10-15.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル中の'BeginAddress'の値が 4KB にアライメントされていません。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-16
            // MemoryMap の Size は 4KBにアライメントされていないといけない
            errorMsg = ExecuteProgram(tmpFile, "Test10-16.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル中の'Size'の値が 4KB にアライメントされていません。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-17
            // MemoryMap の Size は 0 を受け付けない
            errorMsg = ExecuteProgram(tmpFile, "Test10-17.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'Size'に 0 が指定されました。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-18
            // MemoryMap の BeginAddress は 10進数を受け付けない
            errorMsg = ExecuteProgram(tmpFile, "Test10-18.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'BeginAddress' への数値の指定が不正です。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-19
            // MemoryMap の Size は 10進数を受け付けない
            errorMsg = ExecuteProgram(tmpFile, "Test10-19.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'Size' への数値の指定が不正です。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-20
            // MemoryMap の Permission は "RX" を受け付けない
            errorMsg = ExecuteProgram(tmpFile, "Test10-20.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'Permission'にサポート外のパラメータ'RX'が指定されました。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-21
            // MemoryMap の Type は "Normal" を受け付けない
            errorMsg = ExecuteProgram(tmpFile, "Test10-21.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'Type'にサポート外のパラメータ'Normal'が指定されました。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-22
            // MemoryMap の BeginAddress が指定されていないとエラーになる
            errorMsg = ExecuteProgram(tmpFile, "Test10-22.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'BeginAddress'が指定されていません。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-23
            // MemoryMap の Size が指定されていないとエラーになる
            errorMsg = ExecuteProgram(tmpFile, "Test10-23.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'Size'が指定されていません。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-24
            // MemoryMap の Permission が指定されていないとエラーになる
            errorMsg = ExecuteProgram(tmpFile, "Test10-24.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'Permission'が指定されていません。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-25
            // MemoryMap の Type が指定されていないとエラーになる
            errorMsg = ExecuteProgram(tmpFile, "Test10-25.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'Type'が指定されていません。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-26
            // MemoryMap の BeginAddress と Size を足し合わせた数が 37bit 以上の値になるとエラーになる
            errorMsg = ExecuteProgram(tmpFile, "Test10-26.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'MemoryMap'のマップ領域が 36 ビットで表現出来る領域を超えています。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-27
            // MemoryMap の BeginAddress は 37bit 以上の値を受け付けない
            errorMsg = ExecuteProgram(tmpFile, "Test10-27.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル中の'BeginAddress'の値に 36 ビット整数ではない値が指定されました。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-28
            // MemoryMap の Size は 33bit 以上の値を受け付けない
            errorMsg = ExecuteProgram(tmpFile, "Test10-28.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル中の'Size'の値に 32 ビット整数ではない値が指定されました。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-29
            // EnableSystemCalls に 16進数を登録することはできない
            errorMsg = ExecuteProgram(tmpFile, "Test10-29.yml", SvcHeaders, "--testmode");
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'EnableSystemCalls'に未知のシステムコール '0x10' が指定されました。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-30
            // TestMode でないと Padding が利用できない
            errorMsg = ExecuteProgram(tmpFile, "Test10-30.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の 9 行目の 記述が不正です。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-31
            // Padding がマップ形式では指定できない
            errorMsg = ExecuteProgram(tmpFile, "Test10-31.yml", SvcHeaders, "--testmode");
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内でサポート外のパラメータ'Padding'が指定されました");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-32
            // TestMode でないと、Flags が利用できない
            errorMsg = ExecuteProgram(tmpFile, "Test10-32.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内でサポート外のパラメータ'Flags'が指定されました");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-33
            // 廃止

            // TEST 10-34
            // HandleTableSize に 17bit 以上の値を指定できない
            errorMsg = ExecuteProgram(tmpFile, "Test10-34.yml", SvcHeaders, "--testmode");
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル中の'HandleTableSize'の値に 16 ビット整数ではない値が指定されました。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-35
            // Flags に 16bit 以上の値を指定できない
            errorMsg = ExecuteProgram(tmpFile, "Test10-35.yml", SvcHeaders, "--testmode");
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル中の'Flags'の値に 15 ビット整数ではない値が指定されました。\r\n");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-36
            // EnableInterrupts に 16 進数の数値を指定できない
            errorMsg = ExecuteProgram(tmpFile, "Test10-36.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'EnableInterrupts' への数値の指定が不正です。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-37
            // HandleTableSize に 16 進数の数値を指定できない
            errorMsg = ExecuteProgram(tmpFile, "Test10-37.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'HandleTableSize' への数値の指定が不正です。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            // TEST 10-38
            // svc_header を指定しないと、システムコールを許可しない
            errorMsg = ExecuteProgram(tmpFile, "Test10-38.yml", new string[0]);
            match = Regex.Match(errorMsg, @"エラー: パラメータ指定ファイル内の'EnableSystemCalls'に未知のシステムコール 'SetHeapSize' が指定されました。");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            BinaryFile = ResourceDir + "InvalidBinary.axf";
            errorMsg = ExecuteProgram(tmpFile, "EnableInterrupts.yml", SvcHeaders);
            match = Regex.Match(errorMsg, @"エラー: セグメント ""EX"" の開始アドレスが 0 になっていません。 \(0x200000\)");
            Assert.IsTrue(match.Success);
            Assert.IsFalse(System.IO.File.Exists(tmpFile));

            System.IO.Directory.Delete(MakeSvcVeneerWorkDir, true);
        }
    }
}
