﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using MakeInitialImage;
using System.IO;
using System.Linq;
using TestUtility;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using Nintendo.Foundation.IO;
using CommandUtility;

namespace MakeInitialImageTest
{
    [TestClass]
    public class UnitTest
    {
        public TestContext TestContext { get; set; }

        [TestMethod]
        public void TestByteUnitExpression()
        {
            Assert.AreEqual(1024L, ByteUnitExpression.Parse("1 KiB").Bytes);
            Assert.AreEqual("128 MiB", new ByteUnitExpression { MebiBytes = 128 }.ToString());
        }

        [TestMethod]
        public void TestParseYaml()
        {
            var config = InitialImageConfigParser.Parse(new StringReader(SimpleTestCaseOfInitialImageConfig));
            Assert.AreEqual(32L * 1024L * 1024L * 1024L, config.StorageDefinitions[0].Size.Bytes);
        }

        [TestMethod]
        public void TestReplaceVariables()
        {
            Assert.AreEqual<string>(
                "test Replaced \\${TESTVAR}",
                Utility.ReplaceVariables(
                    "test ${TESTVAR} \\${TESTVAR}",
                    new Dictionary<string, string>() { { "TESTVAR", "Replaced" } }));
        }

        [TestMethod]
        public void TestProgramArguments()
        {
            var arguments = new string[] { "-o", "output.image", "-i", "input.config", "-D", "Arg1=Value1", "-D", "Arg2=Value2" };
            MakeInitialImageArguments parameters = new MakeInitialImageArguments();
            var parser = new CommandLineParser();
            Assert.IsTrue(parser.ParseArgs<MakeInitialImageArguments>(arguments, out parameters), "ParseArg");
            Assert.AreEqual<string>("output.image", parameters.OutputPath);
            Assert.AreEqual<string>("input.config", parameters.InputPath);
            Assert.AreEqual<string>(parameters.Arguments[0], "Arg1=Value1");
            var variables = parameters.MakeVariablesDictionary();
            Assert.AreEqual<string>("Value1", variables["Arg1"]);
            Assert.AreEqual<string>("Value2", variables["Arg2"]);
        }

        [TestMethod]
        public void TestInitialImageWriter()
        {
            var testPath = new TestPath(this.TestContext);

            using (var tempFileHolder = new ScopedTemporaryFileHolder(TestContext))
            {
                var imageFile = tempFileHolder.CreateTemporaryFilePath();

                using (var imageFileStream = imageFile.OpenWrite())
                {
                    var imageWriter = new InitialImageWriter(imageFileStream);
                    for (int i = 0; i < 32; i++)
                    {
                        imageWriter.AddPartition("test", (output) =>
                        {
                            output.WriteByte(0x12);
                            output.WriteByte(0x34);
                            output.WriteByte(0x56);
                            output.WriteByte(0x78);
                        });
                    }
                    imageWriter.WritePartitions();
                }
            }
        }

        [TestMethod]
        public void TestGuidPartionTableWriter()
        {
            using (var tempFileHolder = new ScopedTemporaryFileHolder(TestContext))
            {
                var imageFile1 = tempFileHolder.CreateTemporaryFile(new byte[] { 0x01, 0x02, 0x03, 0x04 });
                var imageFile2 = tempFileHolder.CreateTemporaryFile(new byte[] { 0x05, 0x06, 0x07, 0x08 });
                var workingDirectory = tempFileHolder.CreateTemporaryDirectory();
                var outputImage = tempFileHolder.CreateTemporaryFilePath(extension: ".InitialImage");

                var config = InitialImageConfigParser.Parse(
                    new StringReader(AlphaVersionTestCaseOfInitialImageConfig),
                    new Dictionary<string, string>()
                    {
                        { "TEST_IMAGE_1", imageFile1.FullName },
                        { "TEST_IMAGE_2", imageFile2.FullName }
                    });

                using (var imageStream = outputImage.OpenWrite())
                {
                    var constractor = new InitialImageConstructor(imageStream, config, workingDirectory, MakeFatImagePath, InstallNcaPath);
                    constractor.ConstructImage();
                }

                using (var readStream = outputImage.OpenRead())
                {
                    var reader = new InitialImageReader(readStream);

                    Assert.AreEqual<int>(7, reader.InitialImageInfo.NumPartions);
                    Assert.AreEqual<long>(4 + 512, reader.InitialImageInfo.Partitions[0].Size);
                    Assert.AreEqual<string>("ProtectiveMbr", reader.InitialImageInfo.Partitions[2].Name);
                    Assert.AreEqual<long>(512, reader.InitialImageInfo.Partitions[2].Size);
                    Assert.AreEqual<string>("Snd", reader.InitialImageInfo.Partitions[6].Name);
                    Assert.AreEqual<long>(4, reader.InitialImageInfo.Partitions[6].Size);
                }
            }
        }

        private string InitialImageConfigOfFixedAddressPartition = @"
InitialImageDefinition:
  Name: DevelopmentImage
  Version: 1.0

StorageDefinitions:
  - Name: InternalStorage
    SectorSize: 512B
    Size: 32 GiB
    PartitionFormat: GPT

    Partitions:
      - Name: Fixed
        Guid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        TypeGuid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        GeneratingType: RawImage
        Size: 4 MiB
        RawImageFile: ${IMAGE1}
        AllocationType: Fixed
        Address: 0x400000
      - Name: Sequential
        Guid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        TypeGuid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        GeneratingType: RawImage
        Size: 4 MiB
        RawImageFile: ${IMAGE2}
        AllocationType: Sequential
      - Name: Skip
        Guid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        TypeGuid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        Size: 4 MiB
        AllocationType: Keep
      - Name: NextKeepPartition
        Guid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        TypeGuid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        Size: 4 MiB
        RawImageFile: ${IMAGE3}
        AllocationType: Sequential
";

        [TestMethod]
        public void TestFixedAddressPartition()
        {
            using (var tempFileHolder = new ScopedTemporaryFileHolder(TestContext))
            {
                var imageFile1 = tempFileHolder.CreateTemporaryFile(new byte[] { 0x01, 0x02, 0x03, 0x04 });
                var imageFile2 = tempFileHolder.CreateTemporaryFile(new byte[] { 0x05, 0x06, 0x07, 0x08 });
                var outputImage = tempFileHolder.CreateTemporaryFilePath(extension: ".InitialImage");

                var config = InitialImageConfigParser.Parse(
                    new StringReader(InitialImageConfigOfFixedAddressPartition),
                    new Dictionary<string, string>()
                    {
                        { "IMAGE1", imageFile1.FullName },
                        { "IMAGE2", imageFile2.FullName }
                    });

                var addressList = PartitionAddress.MakePartitionAddress(config.StorageDefinitions[0]);

                Assert.AreEqual(PartitionAddressType.Absolute, addressList[0].Type);
                Assert.AreEqual(0x400000, addressList[0].AbsoluteAddress.Address);
                Assert.AreEqual(PartitionAddressType.Absolute, addressList[1].Type);
                Assert.AreEqual(0x400000 + 4 * 1024 * 1024, addressList[1].AbsoluteAddress.Address);
                Assert.AreEqual(PartitionAddressType.Keep, addressList[2].Type);
                Assert.AreEqual(2, addressList[2].RelativeAddress.BasePartitionIndex);
                Assert.AreEqual(0, addressList[2].RelativeAddress.Address);
                Assert.AreEqual(PartitionAddressType.Relative, addressList[3].Type);
                Assert.AreEqual(2, addressList[3].RelativeAddress.BasePartitionIndex);
                Assert.AreEqual(0, addressList[3].RelativeAddress.Address);
            }
        }

        [Ignore]
        [TestMethod]
        public void TestInstallContent()
        {
            using (var tempFileHolder = new ScopedTemporaryFileHolder(TestContext))
            {
                var imageFile1 = tempFileHolder.CreateTemporaryFile(new byte[] { 0x01, 0x02, 0x03, 0x04 });
                var imageFile2 = tempFileHolder.CreateTemporaryFile(new byte[] { 0x05, 0x06, 0x07, 0x08 });
                var imageFile3 = tempFileHolder.CreateTemporaryFile(new byte[] { 0x09, 0x0a, 0x0b, 0x0c });
                var workingDirectory = tempFileHolder.CreateTemporaryDirectory();
                var outputImage = tempFileHolder.CreateTemporaryFilePath(extension: ".InitialImage");

                var config = InitialImageConfigParser.Parse(
                    new StringReader(InitialImageConfigOfInstallContentTest),
                    new Dictionary<string, string>()
                    {
                        { "NCA_1", imageFile1.FullName },
                        { "NCA_2", imageFile2.FullName },
                        { "NCA_3", imageFile3.FullName }
                    });

                using (var imageStream = outputImage.OpenWrite())
                {
                    var constractor = new InitialImageConstructor(imageStream, config, workingDirectory, MakeFatImagePath, InstallNcaPath);
                    constractor.ConstructImage();
                }

                using (var readStream = outputImage.OpenRead())
                {
                    var reader = new InitialImageReader(readStream);

                    Assert.AreEqual<int>(6, reader.InitialImageInfo.NumPartions);
                    Assert.AreEqual<long>(4 + (64 * 2), reader.InitialImageInfo.Partitions[0].Size);
                    Assert.AreEqual<string>("ProtectiveMbr", reader.InitialImageInfo.Partitions[2].Name);
                    Assert.AreEqual<long>(512, reader.InitialImageInfo.Partitions[2].Size);
                    Assert.AreEqual<string>("SparseFat", reader.InitialImageInfo.Partitions[5].Name);
                }
            }
        }

        private string InitialImageConfigOfBugFixKeepAllocationType = @"
InitialImageDefinition:
  Name: DevelopmentImage
  Version: 1.0

StorageDefinitions:
  - Name: InternalStorage
    SectorSize: 512B
    Size: 32 GiB
    PartitionFormat: GPT

    Partitions:
      - Name: Fst
        Guid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        TypeGuid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        GeneratingType: RawImage
        Size: 16 MiB
        AllocationType: Fixed
        Address: 0x2e00000
        RawImageFile: ${SomeImage}

      - Name: Snd
        Guid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        TypeGuid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        GeneratingType: RawImage
        Size: 16 MiB
        AllocationType: Fixed
        Address: 0x202e00000
        RawImageFile: ${SomeImage}

      - Name: SysPrt
        Guid: 5ce6fbed-3c0c-4986-ac49-8eb37b5010dd
        TypeGuid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        GeneratingType: Keep
        Size: 16 MiB
        AllocationType: Keep

      - Name: SysPrt2
        Guid: 5ce6fbed-3c0c-4986-ac49-8eb37b5010dd
        TypeGuid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        GeneratingType: Installed
        Size: 16 MiB
        AllocationType: Sequential
        InstallContents:
          - ${SomeImage}

";
        [Ignore]
        [TestMethod]
        public void BugFixKeepAllocationType()
        {
            using (var tempFileHolder = new ScopedTemporaryFileHolder(TestContext))
            {
                var imageFile = tempFileHolder.CreateTemporaryFile(new byte[] { 0x01, 0x02, 0x03, 0x04 });
                var workingDirectory = tempFileHolder.CreateTemporaryDirectory();
                var outputImage = tempFileHolder.CreateTemporaryFilePath(extension: ".InitialImage");

                var config = InitialImageConfigParser.Parse(
                    new StringReader(InitialImageConfigOfBugFixKeepAllocationType),
                    new Dictionary<string, string>()
                    {
                        { "SomeImage", imageFile.FullName },
                    });

                using (var imageStream = outputImage.OpenWrite())
                {
                    var constractor = new InitialImageConstructor(imageStream, config, workingDirectory, MakeFatImagePath, InstallNcaPath);
                    constractor.ConstructImage();
                }

                using (var readStream = outputImage.OpenRead())
                {
                    var reader = new InitialImageReader(readStream);
                }
            }
        }

        [TestMethod]
        public void BugFixTemporaryFileHolder()
        {
            using (var tempFileHolder = new ScopedTemporaryFileHolder(TestContext))
            {
                var directory = tempFileHolder.CreateTemporaryDirectory();
                var workingDirectory = tempFileHolder.CreateTemporaryDirectory();
                var outputImage = tempFileHolder.CreateTemporaryFilePath(extension: ".InitialImage");

                var config = InitialImageConfigParser.Parse(
                    new StringReader(InitialImageConfigUsedBaseDirectory),
                    new Dictionary<string, string>()
                    {
                        { "TEST_DIRECTORY", directory.FullName },
                    });

                using (var imageStream = outputImage.OpenWrite())
                {
                    var constractor = new InitialImageConstructor(imageStream, config, workingDirectory, MakeFatImagePath, InstallNspPath);
                    constractor.ConstructImage();
                }

                using (var readStream = outputImage.OpenRead())
                {
                    var reader = new InitialImageReader(readStream);
                }

                Assert.AreEqual(0, workingDirectory.EnumerateFiles("*", SearchOption.AllDirectories).Count());
                Assert.AreEqual(0, workingDirectory.EnumerateDirectories("*", SearchOption.AllDirectories).Count());
            }
        }

        [TestMethod]
        public void TestExplicitCommand()
        {
            using (var tempFileHolder = new ScopedTemporaryFileHolder(TestContext))
            {
                var imageFile1 = tempFileHolder.CreateTemporaryFile(new byte[] { 0x01, 0x02, 0x03, 0x04 });
                var directory = tempFileHolder.CreateTemporaryDirectory();
                var workingDirectory = tempFileHolder.CreateTemporaryDirectory();
                var outputImage = tempFileHolder.CreateTemporaryFilePath(extension: ".InitialImage");

                var config = InitialImageConfigParser.Parse(
                    new StringReader(InitialImageConfigWithExplicitCommand),
                    new Dictionary<string, string>()
                    {
                        { "TEST_DIRECTORY", directory.FullName },
                        { "TEST_BCT", imageFile1.FullName },
                    });

                using (var imageStream = outputImage.OpenWrite())
                {
                    var constractor = new InitialImageConstructor(imageStream, config, workingDirectory, MakeFatImagePath, InstallNspPath);

                    constractor.ConstructImage();

                    var parsedCommands = constractor.Parse(config, constractor.imageWriter, constractor.gptWriter);

                    Assert.AreEqual(InitialImageCommand.CommandType.WriteGpt, parsedCommands[0].Command);
                    Assert.AreEqual(InitialImageCommand.CommandType.WritePartitionImage, parsedCommands[1].Command);
                    Assert.AreEqual(InitialImageCommand.CommandType.WriteSparsePartitionImage, parsedCommands[2].Command);
                    Assert.AreEqual(InitialImageCommand.CommandType.WriteFsPartitionImage, parsedCommands[3].Command);
                    Assert.AreEqual(InitialImageCommand.CommandType.WriteFsSparsePartitionImage, parsedCommands[4].Command);
                    Assert.AreEqual(InitialImageCommand.CommandType.EnsureProtectProductionInfo, parsedCommands[5].Command);
                    Assert.AreEqual(InitialImageCommand.CommandType.UpdateSecureInfo, parsedCommands[6].Command);
                    Assert.AreEqual(InitialImageCommand.CommandType.WriteBatchedBootPartitions, parsedCommands[7].Command);
                }

                using (var readStream = outputImage.OpenRead())
                {
                    var reader = new InitialImageReader(readStream);

                    Console.WriteLine(string.Format("NumPartition: {0}", reader.InitialImageInfo.NumPartions));
                    for (int i = 0; i < reader.InitialImageInfo.NumPartions; i++)
                    {
                        var partition = reader.InitialImageInfo.Partitions[i];
                        Console.WriteLine(string.Format("Partition[{0}]: Name={1}, Address={2}, Size={3}", i, partition.Name, partition.Address, partition.Size));
                    }

                    foreach (var command in reader.Commands)
                    {
                        Console.WriteLine("Command: {0}, Args='{1}', StrArg='{2}'", command.Command.ToString(), string.Join(",", command.Arguments), command.StringArgument);
                    }
                }
            }
        }

        private string InitialImageConfigWithExplicitCommand = @"
InitialImageDefinition:
  Name: DevelopmentImage
  Version: 1.0

StorageDefinitions:
  - Name: InternalStorage
    SectorSize: 512B
    Size: 32 GiB
    PartitionFormat: GPT

    SpecialPartitions:
      - Name: BootPartition1-Bct-1st
        FsPartitionId: BootPartition1Root
        GeneratingType: RawImage
        Offset: 0x00000000
        RawImageFile: ${TEST_BCT}
        UpdateSecureInfo: True

      - Name: BootPartition1-Normal-Package1-Main
        FsPartitionId: BootPartition1Root
        GeneratingType: RawImage
        Offset: 0x00100000
        RawImageFile: ${TEST_BCT}

    Partitions:
      - Name: Fixed
        Guid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        TypeGuid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        GeneratingType: Installed
        Size: 4 MiB
        BaseDirectory: ${TEST_DIRECTORY}
        AllocationType: Fixed
        Address: 0x400000

    Commands:
      - Command: WriteGpt
      - Command: WritePartitionImage
        PartitionName: Fixed
      - Command: WriteSparsePartitionImage
        PartitionName: Fixed
      - Command: WriteFsPartitionImage
        PartitionName: Fixed
      - Command: WriteFsSparsePartitionImage
        PartitionName: Fixed
      - Command: EnsureProtectProductionInfo
      - Command: UpdateSecureInfo
        PartitionName: BootPartition1-Bct-1st
      - Command: WriteBatchedBootPartitions
        BctPartitionName: BootPartition1-Bct-1st
        BootLoaderPartitionName: BootPartition1-Normal-Package1-Main
";

        private string InitialImageConfigUsedBaseDirectory = @"
InitialImageDefinition:
  Name: DevelopmentImage
  Version: 1.0

StorageDefinitions:
  - Name: InternalStorage
    SectorSize: 512B
    Size: 32 GiB
    PartitionFormat: GPT

    Partitions:
      - Name: Fixed
        Guid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        TypeGuid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        GeneratingType: Installed
        Size: 4 MiB
        BaseDirectory: ${TEST_DIRECTORY}
        AllocationType: Fixed
        Address: 0x400000
";

        private string AlphaVersionTestCaseOfInitialImageConfig = @"
InitialImageDefinition:
  Name: DevelopmentImage
  Version: 1.0

StorageDefinitions:
  - Name: InternalStorage
    SectorSize: 512B
    Size: 32 GiB
    PartitionFormat: GPT

    Partitions:
      - Name: Fst
        Guid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        TypeGuid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        GeneratingType: RawImage
        Size: 16 MiB
        RawImageFile: ${TEST_IMAGE_1}

      - Name: Snd
        Guid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        TypeGuid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        GeneratingType: RawImage
        Size: 16 MiB
        RawImageFile: ${TEST_IMAGE_2}
";

        private string SimpleTestCaseOfInitialImageConfig = @"
InitialImageDefinition:
  Name: DevelopmentImage
  Version: 1.0

StorageDefinitions:
  - Name: InternalStorage
    SectorSize: 512B
    Size: 32 GiB
    PartitionFormat: GPT

    Partitions:
      # ブートローダー
      - Name: BootLoader
        Guid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        TypeGuid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        GeneratingType: RawImage
        Size: 16 MiB
        RawImageFile: ${NINTENDO_SDK_ROOT}/Eris/Outputs/BootLoader.bin

      # 生産工程でのみ書き込まれるデータ
      - Name: HardwareProductionPartition
        Guid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        TypeGuid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        GeneratingType: RawImage
        Size: 16 MiB
        RawImageFile: ${NINTENDO_SDK_ROOT}/Eris/Resources/InitialCAL.bin

      # 更新OSパーティション（サブシステムパーティション）
      - Name: SubSystemPartition
        Guid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        TypeGuid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        GeneratingType: Installed
        Size: 480 MiB       # 512 MiB - 32 MiB
        FileSystem: FAT32
        InstallContents:
          - ${NINTENDO_SDK_ROOT}/Eris/Outputs/Contents/SystemProcesses/FileSystem.ncm
          - ${NINTENDO_SDK_ROOT}/Eris/Outputs/Contents/SystemProcesses/ContentManager.ncm
          - ${NINTENDO_SDK_ROOT}/Eris/Outputs/Contents/Tools/ForceSystemUpdater.ncm
          - ${NINTENDO_SDK_ROOT}/Eris/Outputs/Contents/Resources/RecorveryImage.ncm
        EncryptedArchives:
          - /db/ContentMeta.db

      # 開発OSパーティション（メインシステムパーティション）
      - Name: MainSystemPartition
        Guid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        TypeGuid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        GeneratingType: Installed
        Size: 3584 MiB      # 4GiB - 512 MiB
        FileSystem: FAT32
        InstallContents:
          - ${NINTENDO_SDK_ROOT}/Eris/Outputs/Contents/SystemProcesses/FileSystem.ncm
          - ${NINTENDO_SDK_ROOT}/Eris/Outputs/Contents/SystemProcesses/ContentManager.ncm
          - ${NINTENDO_SDK_ROOT}/Eris/Outputs/Contents/Tools/ForceSystemUpdater.ncm
        EncryptedArchives:
          - /db/ContentMeta.db

      # ユーザパーティション
      - Name: UserPartition
        Guid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        TypeGuid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        GeneratingType: Clean
        Size: 28 GiB
        FileSystem: FAT32
";

        private string InitialImageConfigOfInstallContentTest = @"
InitialImageDefinition:
  Name: DevelopmentImage
  Version: 1.0

StorageDefinitions:
  - Name: InternalStorage
    SectorSize: 512B
    Size: 32 GiB
    PartitionFormat: GPT

    Partitions:
      - Name: SparseFat
        Guid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        TypeGuid: 3F2504E0-4F89-11D3-9A0C-0305E82C3301
        GeneratingType: Installed
        Size: 16 MiB
        FileSystem: FAT32
        InstallContents:
          - ${NCA_1}
          - ${NCA_2}
          - ${NCA_3}
";

        private FileInfo MakeFatImagePath
        {
            get
            {
                var testPath = new TestPath(this.TestContext);
                return new FileInfo(Path.Combine(testPath.GetSigloRoot(), @"Tools\CommandLineTools\MakeFatImage.exe"));
            }
        }

        private FileInfo InstallNcaPath
        {
            get
            {
                var testPath = new TestPath(this.TestContext);
                return new FileInfo(Path.Combine(testPath.GetSigloRoot(), @"Tools\CommandLineTools\InstallNcaOnHost.exe"));
            }
        }

        private FileInfo InstallNspPath
        {
            get
            {
                var testPath = new TestPath(this.TestContext);
                return new FileInfo(Path.Combine(testPath.GetSigloRoot(), @"Tools\CommandLineTools\InstallNspOnHost.exe"));
            }
        }
    }
}
