﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using System.Collections.Generic;
using System.IO;
using TestUtility;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace MakeDescTest
{
    [TestClass]
    public class TestAcidBinary : TestBase
    {
        private const string AcidTagName = "<Acid>";
        private const string Signature = "ACID";

        private const uint KernelCapabilityEntrySize = 4;
        private const uint AcidAlign = 0x10;

        private const uint HeaderSignatureOffset = 0x0;
        private const uint HeaderSignatureSize = 0x100;
        private const uint HeaderDecryptKeyOffset = 0x100;
        private const uint HeaderDecryptKeySize = 0x100;
        private const uint SignatureOffset = 0x200;
        private const uint SignatureAreaSizeOffset = 0x204;
        private const uint Reserved1Offset = 0x208;
        private const uint ProgramIdMinOffset = 0x210;
        private const uint ProgramIdMaxOffset = 0x218;
        private const uint FacdOffset = 0x220;
        private const uint FacdSizeOffset = 0x224;
        private const uint FacdDefaultSize = 0x2c;
        private const uint SacdOffset = 0x228;
        private const uint SacdSizeOffset = 0x22C;
        private const uint KcdOffset = 0x230;
        private const uint KcdSizeOffset = 0x234;
        private const uint Reserved2Offset = 0x238;

        private const uint HeaderSize = 0x240;
        private const uint SignatureAreaSizeBase = 0x140;

        private readonly byte[] KeyModulus = { 0xAE, 0x1B, 0x10, 0x6C, 0x77, 0x9D, 0x41, 0x65, 0x83, 0xB1, 0x46, 0x73, 0x18, 0xBC, 0x33, 0xA1, 0x7E, 0x85, 0xCD, 0x64, 0x57, 0xC9, 0x82, 0xC0, 0xFB, 0x21, 0xC0, 0x45, 0xEC, 0x8D, 0x57, 0x60, 0xF6, 0xAD, 0x65, 0x76, 0x0A, 0x25, 0xA9, 0x4D, 0x71, 0x95, 0xF7, 0xC3, 0x35, 0x51, 0x2E, 0xA6, 0xCA, 0xBF, 0x62, 0xD5, 0x50, 0xFF, 0xF0, 0x9E, 0x16, 0x25, 0x8C, 0x22, 0x27, 0x25, 0xA6, 0x80, 0xA8, 0x6A, 0x7F, 0x1A, 0xBD, 0xCE, 0xB6, 0xED, 0x95, 0xF1, 0x73, 0x45, 0x03, 0xD1, 0x78, 0x4B, 0xDB, 0x03, 0x05, 0x52, 0x02, 0x6E, 0xA7, 0xF7, 0x2D, 0x3D, 0x65, 0x1D, 0xDB, 0x61, 0x60, 0x54, 0xC3, 0x1D, 0xA2, 0x3A, 0x49, 0x22, 0x80, 0x55, 0xA3, 0x46, 0x69, 0x7F, 0x29, 0xC8, 0xA6, 0x2D, 0x43, 0x7B, 0x20, 0xCD, 0xCA, 0x9E, 0x60, 0x29, 0x96, 0xAD, 0xC5, 0xF0, 0x71, 0xFB, 0x0E, 0x5A, 0x44, 0xC7, 0xED, 0x99, 0x26, 0xCD, 0x1F, 0xCC, 0xF1, 0xC3, 0x6E, 0x09, 0x64, 0x09, 0xCA, 0x60, 0x5B, 0x7F, 0x3A, 0x1D, 0xA8, 0x9C, 0x56, 0x38, 0x63, 0x42, 0x21, 0xBF, 0xB0, 0xA6, 0x26, 0x87, 0x51, 0xCF, 0x77, 0xC1, 0x61, 0xDC, 0x62, 0x07, 0x2A, 0x23, 0x66, 0xF7, 0xED, 0xFB, 0x48, 0x81, 0xF8, 0x05, 0xAB, 0xA0, 0x02, 0x69, 0x51, 0x78, 0xDF, 0xBC, 0x60, 0x81, 0xC7, 0x6C, 0x34, 0xA4, 0x38, 0xA9, 0x26, 0x9D, 0xD0, 0xB8, 0xE4, 0x95, 0xC8, 0x7B, 0xAE, 0x83, 0x26, 0xC8, 0x75, 0x38, 0x67, 0xDE, 0x88, 0x38, 0x22, 0xDE, 0x9F, 0x8C, 0x54, 0xDB, 0x60, 0xBA, 0x65, 0x17, 0x19, 0x1C, 0x29, 0xFD, 0x6F, 0xAC, 0x4E, 0xE8, 0x97, 0x5B, 0xBC, 0x92, 0xA1, 0x76, 0x28, 0x70, 0x3E, 0x2D, 0x37, 0xD6, 0xD2, 0x36, 0x75, 0xC8, 0x68, 0xF4, 0x2A, 0x87, 0x2A, 0x39, 0x1C, 0xCD, 0xD9, 0x77 };
        private readonly byte[] KeyPublicExponent = { 0x01, 0x00, 0x01 };

        public byte[] AcidData(FileManager descFile)
        {
            XmlElement elem = null;
            using (FileStream fs = new FileStream(descFile.FilePath, FileMode.Open, FileAccess.Read))
            {
                StreamReader sr = new StreamReader(fs);
                string line = null;
                while ((line = sr.ReadLine()) != null)
                {
                    if (line.IndexOf(AcidTagName) >= 0)
                    {
                        elem = XmlElement.Convert(line);
                        break;
                    }
                }
            }

            // Base64 でエンコードされている acid のバイナリデータをデコードする
            Assert.IsTrue(elem != null);
            string base64String = elem.InnerText;
            return Convert.FromBase64String(base64String);
        }

        private uint RoundUp(uint size, uint align)
        {
            return (size + (align - 1)) & ~(align - 1);
        }

        [TestMethod]
        public void TestHeaderSignature()
        {
            // --key が指定されなかった場合、0 埋めされる
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);
                byte[] target = new byte[HeaderSignatureSize];
                Array.Copy(contents, HeaderSignatureOffset, target, 0x0, HeaderSignatureSize);
                foreach (var c in target)
                {
                    Assert.IsTrue(c == 0);
                }
            }

            // --key が指定された場合、署名される
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement needSignature = new XmlElement("NeedSignature");
                needSignature.InnerText = "True";
                elem.AddChild(needSignature);

                desc.AddLine(elem.GetXml());
                desc.WriteData();

                string keyFile = GetKeyFile();
                FileManager result = SuccessExecute(desc, null, null, "--key " + keyFile);
                byte[] contents = AcidData(result);
                byte[] tmp = new byte[0x4];
                Array.Copy(contents, SignatureAreaSizeOffset, tmp, 0x0, 0x4);
                int signatureAreaSize = BitConverter.ToInt32(tmp, 0);
                Nintendo.MakeDesc.CryptoLibrary.Rsa2048PssSha256SignCryptoDriver rsa = new Nintendo.MakeDesc.CryptoLibrary.Rsa2048PssSha256SignCryptoDriver();
                Assert.IsTrue(rsa.VerifyBlock(KeyModulus, KeyPublicExponent, contents, 0, 0x100, contents, 0x100, signatureAreaSize));
            }
        }

        [TestMethod]
        public void TestHeaderDecryptKey()
        {
            // --key が指定されなかった場合、0 埋めされる
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);
                byte[] target = new byte[HeaderSignatureSize];
                Array.Copy(contents, HeaderDecryptKeyOffset, target, 0x0, HeaderDecryptKeySize);
                foreach (var c in target)
                {
                    Assert.IsTrue(c == 0);
                }
            }

            // --key が指定された場合、Modulus (n) が入れられている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement needSignature = new XmlElement("NeedSignature");
                needSignature.InnerText = "True";
                elem.AddChild(needSignature);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                string keyFile = GetKeyFile();
                FileManager result = SuccessExecute(desc, null, null, "--key " + keyFile);
                byte[] contents = AcidData(result);
                byte[] target = new byte[HeaderSignatureSize];
                byte[] modulus;
                Array.Copy(contents, HeaderDecryptKeyOffset, target, 0x0, HeaderDecryptKeySize);
                {
                    System.Xml.XmlDocument xml = new System.Xml.XmlDocument();
                    xml.Load(result.FilePath);
                    var rootNode = xml.SelectSingleNode("Desc");
                    var rsaNode = rootNode.SelectSingleNode("RSAKeyValue");
                    var modulusNode = rsaNode.SelectSingleNode("Modulus");
                    modulus = System.Convert.FromBase64String(modulusNode.FirstChild.Value);
                }

                Assert.IsTrue(target.SequenceEqual(modulus));
            }
        }

        [TestMethod]
        public void TestSignature()
        {
            // シグネチャが正しい
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);
                byte[] target = new byte[Signature.Length];
                Array.Copy(contents, SignatureOffset, target, 0x0, Signature.Length);
                Assert.IsTrue(System.Text.Encoding.ASCII.GetString(target) == Signature);
            }
        }

        [TestMethod]
        public void TestSignatureAreaSize()
        {
            // fsd, sacd, kcd, がすべて空
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);
                byte[] target = new byte[0x4];
                Array.Copy(contents, SignatureAreaSizeOffset, target, 0x0, 0x4);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == SignatureAreaSizeBase + FacdDefaultSize);
            }

            // sacd, kcd のサイズが 0x10 のアライメントに揃っていない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entry");
                srv.AddChild(entry);

                XmlElement isServer = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "True";

                XmlElement name = new XmlElement("Name");
                entry.AddChild(name);
                name.InnerText = "Test";

                uint srvSize = (uint)isServer.InnerText.Length + 1;

                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(kcd);

                XmlElement handleTableSize = new XmlElement("HandleTableSize");
                handleTableSize.InnerText = "256";
                kcd.AddChild(handleTableSize);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);
                byte[] target = new byte[0x4];
                Array.Copy(contents, SignatureAreaSizeOffset, target, 0x0, 0x4);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == SignatureAreaSizeBase + RoundUp(FacdDefaultSize, AcidAlign) + RoundUp(srvSize, AcidAlign) + KernelCapabilityEntrySize);
            }

            // sacd のサイズが 0x10 のアライメントに揃っていない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entry");
                srv.AddChild(entry);

                XmlElement isServer = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "True";

                XmlElement name = new XmlElement("Name");
                entry.AddChild(name);
                name.InnerText = "Test";

                uint srvSize = (uint)isServer.InnerText.Length + 1;

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);
                byte[] target = new byte[0x4];
                Array.Copy(contents, SignatureAreaSizeOffset, target, 0x0, 0x4);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == SignatureAreaSizeBase + RoundUp(FacdDefaultSize, AcidAlign) + srvSize);
            }

            // kcd のサイズが 0x10 のアライメントに揃っていない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(kcd);

                XmlElement handleTableSize = new XmlElement("HandleTableSize");
                handleTableSize.InnerText = "256";
                kcd.AddChild(handleTableSize);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);
                byte[] target = new byte[0x4];
                Array.Copy(contents, SignatureAreaSizeOffset, target, 0x0, 0x4);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == SignatureAreaSizeBase + RoundUp(FacdDefaultSize, AcidAlign) + KernelCapabilityEntrySize);
            }
            // sacd, kcd のサイズが 0x10 のアライメントに揃っている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entry");
                srv.AddChild(entry);

                XmlElement isServer = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "True";

                XmlElement name = new XmlElement("Name");
                entry.AddChild(name);
                name.InnerText = "1234567";

                // 同じ要素を追加する
                srv.AddChild(entry);

                uint srvSize = (uint)(name.InnerText.Length + 1) * 2;
                Assert.IsFalse((srvSize & 0x0F) > 0);

                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(kcd);
                for (int i = 1; i <= (AcidAlign / KernelCapabilityEntrySize) * 2; i++)
                {
                    XmlElement interrupt = new XmlElement("EnableInterrupts");
                    interrupt.InnerText = i.ToString();
                    kcd.AddChild(interrupt);
                }

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);
                byte[] target = new byte[0x4];
                Array.Copy(contents, SignatureAreaSizeOffset, target, 0x0, 0x4);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == SignatureAreaSizeBase + RoundUp(FacdDefaultSize, AcidAlign) + srvSize + AcidAlign);
            }
        }

        [TestMethod]
        public void TestProgramIdMin()
        {
            // ProgramIdMin の値が入っている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement idMin = new XmlElement("ProgramIdMin");
                idMin.InnerText = "0x1";
                elem.AddChild(idMin);

                XmlElement idMax = new XmlElement("ProgramIdMax");
                idMax.InnerText = "0x2";
                elem.AddChild(idMax);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);
                byte[] target = new byte[0x8];
                Array.Copy(contents, ProgramIdMinOffset, target, 0x0, 0x8);
                Assert.IsTrue(BitConverter.ToUInt64(target, 0) == 0x1);
            }

            // Desc/ProgramIdMin が指定されなかったときは 0 が入っている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);
                byte[] target = new byte[0x8];
                Array.Copy(contents, ProgramIdMinOffset, target, 0x0, 0x8);
                Assert.IsTrue(BitConverter.ToUInt64(target, 0) == 0x0);
            }
        }

        [TestMethod]
        public void TestProgramIdMax()
        {
            // ProgramIdMax の値が入っている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement idMin = new XmlElement("ProgramIdMin");
                idMin.InnerText = "0x1";
                elem.AddChild(idMin);

                XmlElement idMax = new XmlElement("ProgramIdMax");
                idMax.InnerText = "0x2";
                elem.AddChild(idMax);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);
                byte[] target = new byte[0x8];
                Array.Copy(contents, ProgramIdMaxOffset, target, 0x0, 0x8);
                Assert.IsTrue(BitConverter.ToUInt64(target, 0) == 0x2);
            }

            // Desc/ProgramIdMax が指定されなかったときは 0xFFFFFFFFFFFFFFFF が入っている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);
                byte[] target = new byte[0x8];
                Array.Copy(contents, ProgramIdMaxOffset, target, 0x0, 0x8);
                Assert.IsTrue(BitConverter.ToUInt64(target, 0) == 0xFFFFFFFFFFFFFFFF);
            }
        }

        [TestMethod]
        public void TestFsAccessControlDescriptor()
        {
            // 空の場合
            // オフセットに 0x240 が入っている
            // サイズに 0x2c が入っている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);

                byte[] target = new byte[0x4];
                Array.Copy(contents, FacdOffset, target, 0x0, 0x4);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == HeaderSize);

                Array.Copy(contents, FacdSizeOffset, target, 0x0, 0x4);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == FacdDefaultSize);

                Assert.IsTrue(contents.Length == HeaderSize + FacdDefaultSize);
            }
        }

        [TestMethod]
        public void TestSrvAccessControlDescriptor()
        {
            // 空の場合
            // オフセットに 0x270 が入っている
            // サイズに 0x0 が入っている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);

                byte[] target = new byte[0x4];
                Array.Copy(contents, SacdOffset, target, 0x0, 0x4);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == HeaderSize + RoundUp(FacdDefaultSize, AcidAlign));

                Array.Copy(contents, SacdSizeOffset, target, 0x0, 0x4);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == 0x0);

                Assert.IsTrue(contents.Length == HeaderSize + FacdDefaultSize);
            }

            // 1 エントリ のデータが正しく書き込まれている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entry");
                srv.AddChild(entry);

                XmlElement isServer = new XmlElement("IsServer");
                isServer.InnerText = "True";
                entry.AddChild(isServer);

                XmlElement name = new XmlElement("Name");
                name.InnerText = "Test";
                entry.AddChild(name);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);

                // オフセットの値が正しい
                byte[] target = new byte[0x4];
                Array.Copy(contents, SacdOffset, target, 0x0, 0x4);
                uint offset = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(offset == HeaderSize + RoundUp(FacdDefaultSize, AcidAlign));

                // サイズの値が正しい
                Array.Copy(contents, SacdSizeOffset, target, 0x0, 0x4);
                uint size = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(size == (uint)(name.InnerText.Length + 1));

                byte[] binary = new byte[size];
                Array.Copy(contents, offset, binary, 0, size);

                // クライアント・サーバーフラグが正しい
                byte config = binary[0];
                Assert.IsTrue((config & (1 << 7)) > 0);

                // bit 3 - 6 は未使用
                Assert.IsTrue((config & (0x78)) == 0);

                // 文字列の長さが正しい
                Assert.IsTrue((config & (0x7)) == name.InnerText.Length - 1);

                // 2byte 目以降に文字列が入っている
                Assert.IsTrue(System.Text.Encoding.ASCII.GetString(binary, 1, (int)size - 1) == name.InnerText);
            }

            // 複数エントリ のデータが正しく書き込まれている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement serverEntry = new XmlElement("Entry");
                srv.AddChild(serverEntry);

                XmlElement serverIsServer = new XmlElement("IsServer");
                serverIsServer.InnerText = "True";
                serverEntry.AddChild(serverIsServer);

                XmlElement serverName = new XmlElement("Name");
                serverName.InnerText = "Server";
                serverEntry.AddChild(serverName);

                XmlElement clientEntry = new XmlElement("Entry");
                srv.AddChild(clientEntry);

                XmlElement clientIsServer = new XmlElement("IsServer");
                clientIsServer.InnerText = "False";
                clientEntry.AddChild(clientIsServer);

                XmlElement clientName = new XmlElement("Name");
                clientName.InnerText = "Clie*";
                clientEntry.AddChild(clientName);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);

                // オフセットの値が正しい
                byte[] target = new byte[0x4];
                Array.Copy(contents, SacdOffset, target, 0x0, 0x4);
                uint offset = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(offset == HeaderSize + RoundUp(FacdDefaultSize, AcidAlign));

                // サイズの値が正しい
                Array.Copy(contents, SacdSizeOffset, target, 0x0, 0x4);
                uint size = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(size == (uint)(serverName.InnerText.Length + 1 + clientName.InnerText.Length + 1));

                byte[] binary = new byte[size];
                Array.Copy(contents, offset, binary, 0, size);

                // サーバー
                {
                    // クライアント・サーバーフラグが正しい
                    byte config = binary[0];
                    Assert.IsTrue((config & (1 << 7)) > 0);

                    // bit 3 - 6 は未使用
                    Assert.IsTrue((config & (0x78)) == 0);

                    // 文字列の長さが正しい
                    Assert.IsTrue((config & (0x7)) == serverName.InnerText.Length - 1);

                    // 2byte 目以降に文字列が入っている
                    Assert.IsTrue(System.Text.Encoding.ASCII.GetString(binary, 1, (int)serverName.InnerText.Length) == serverName.InnerText);
                }

                // クライアント
                {
                    // クライアント・サーバーフラグが正しい
                    byte config = binary[serverName.InnerText.Length + 1];
                    Assert.IsTrue((config & (1 << 7)) == 0);

                    // bit 3 - 6 は未使用
                    Assert.IsTrue((config & (0x78)) == 0);

                    // 文字列の長さが正しい
                    Assert.IsTrue((config & (0x7)) == clientName.InnerText.Length - 1);

                    // 2byte 目以降に文字列が入っている
                    Assert.IsTrue(System.Text.Encoding.ASCII.GetString(binary, serverName.InnerText.Length + 2, (int)clientName.InnerText.Length) == clientName.InnerText);
                }
            }

            // アラインした領域が 0 で埋められている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entry");
                srv.AddChild(entry);

                XmlElement isServer = new XmlElement("IsServer");
                isServer.InnerText = "True";
                entry.AddChild(isServer);

                XmlElement name = new XmlElement("Name");
                name.InnerText = "Test";
                entry.AddChild(name);

                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(kcd);

                XmlElement handleTableSize = new XmlElement("HandleTableSize");
                handleTableSize.InnerText = "256";
                kcd.AddChild(handleTableSize);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);

                // オフセットの値が正しい
                byte[] target = new byte[0x4];
                Array.Copy(contents, SacdOffset, target, 0x0, 0x4);
                uint offset = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(offset == HeaderSize + RoundUp(FacdDefaultSize, AcidAlign));

                // サイズの値が正しい
                Array.Copy(contents, SacdSizeOffset, target, 0x0, 0x4);
                uint size = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(size == (uint)(name.InnerText.Length + 1));

                uint alignSize = RoundUp(size, AcidAlign) - size;
                byte[] binary = new byte[alignSize];
                Array.Copy(contents, offset + size, binary, 0, alignSize);

                foreach (var c in binary)
                {
                    Assert.IsTrue(c == 0);
                }
            }
        }

        [TestMethod]
        public void TestKernelCapabilityDescriptor()
        {
            // 空の場合
            // オフセットに 0x270 が入っている
            // サイズに 0x0 が入っている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);

                byte[] target = new byte[0x4];
                Array.Copy(contents, KcdOffset, target, 0x0, 0x4);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == HeaderSize + RoundUp(FacdDefaultSize, AcidAlign));

                Array.Copy(contents, KcdSizeOffset, target, 0x0, 0x4);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == 0x0);

                Assert.IsTrue(contents.Length == HeaderSize + FacdDefaultSize);
            }

            // ThreadInfo のデータが正しく書き込まれている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(kcd);

                XmlElement threadInfo = new XmlElement("ThreadInfo");
                kcd.AddChild(threadInfo);

                XmlElement lowestPriority = new XmlElement("LowestPriority");
                threadInfo.AddChild(lowestPriority);
                lowestPriority.InnerText = "63";

                XmlElement highestPriority = new XmlElement("HighestPriority");
                threadInfo.AddChild(highestPriority);
                highestPriority.InnerText = "24";

                XmlElement minCoreNumber = new XmlElement("MinCoreNumber");
                threadInfo.AddChild(minCoreNumber);
                minCoreNumber.InnerText = "0";

                XmlElement maxCoreNumber = new XmlElement("MaxCoreNumber");
                threadInfo.AddChild(maxCoreNumber);
                maxCoreNumber.InnerText = "1";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);

                // オフセットの値が正しい
                byte[] target = new byte[0x4];
                Array.Copy(contents, KcdOffset, target, 0x0, 0x4);
                uint offset = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(offset == HeaderSize + RoundUp(FacdDefaultSize, AcidAlign));

                // サイズの値が正しい
                Array.Copy(contents, KcdSizeOffset, target, 0x0, 0x4);
                uint size = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(size == KernelCapabilityEntrySize);

                byte[] binary = new byte[size];
                Array.Copy(contents, offset, binary, 0, size);

                // ケイパビリティが正しく書き込まれている
                uint capability = BitConverter.ToUInt32(binary, 0);
                Assert.IsTrue(capability == 0x10063F7);
            }

            // MapMemory のデータが正しく書き込まれている
            // Permission: Ro, Type: Static
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(kcd);

                XmlElement mapMemory = new XmlElement("MemoryMap");
                kcd.AddChild(mapMemory);

                XmlElement beginAddress = new XmlElement("BeginAddress");
                mapMemory.AddChild(beginAddress);
                beginAddress.InnerText = "0x100000";

                XmlElement sizeElem = new XmlElement("Size");
                mapMemory.AddChild(sizeElem);
                sizeElem.InnerText = "0x2000";

                XmlElement permission = new XmlElement("Permission");
                mapMemory.AddChild(permission);
                permission.InnerText = "Ro";

                XmlElement typeElem = new XmlElement("Type");
                mapMemory.AddChild(typeElem);
                typeElem.InnerText = "Static";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);

                // オフセットの値が正しい
                byte[] target = new byte[0x4];
                Array.Copy(contents, KcdOffset, target, 0x0, 0x4);
                uint offset = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(offset == HeaderSize + RoundUp(FacdDefaultSize, AcidAlign));

                // サイズの値が正しい
                Array.Copy(contents, KcdSizeOffset, target, 0x0, 0x4);
                uint size = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(size == KernelCapabilityEntrySize * 2);

                byte[] binary = new byte[size];
                Array.Copy(contents, offset, binary, 0, size);

                // ケイパビリティが正しく書き込まれている
                uint capability = BitConverter.ToUInt32(binary, 0);
                Assert.IsTrue(capability == 0x8000803F);

                capability = BitConverter.ToUInt32(binary, 4);
                Assert.IsTrue(capability == 0x8000013F);
            }

            // MapMemory のデータが正しく書き込まれている
            // Permission: Rw, Type: Io
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(kcd);

                XmlElement mapMemory = new XmlElement("MemoryMap");
                kcd.AddChild(mapMemory);

                XmlElement beginAddress = new XmlElement("BeginAddress");
                mapMemory.AddChild(beginAddress);
                beginAddress.InnerText = "0x100000";

                XmlElement sizeElem = new XmlElement("Size");
                mapMemory.AddChild(sizeElem);
                sizeElem.InnerText = "0x2000";

                XmlElement permission = new XmlElement("Permission");
                mapMemory.AddChild(permission);
                permission.InnerText = "Rw";

                XmlElement typeElem = new XmlElement("Type");
                mapMemory.AddChild(typeElem);
                typeElem.InnerText = "Io";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);

                // オフセットの値が正しい
                byte[] target = new byte[0x4];
                Array.Copy(contents, KcdOffset, target, 0x0, 0x4);
                uint offset = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(offset == HeaderSize + RoundUp(FacdDefaultSize, AcidAlign));

                // サイズの値が正しい
                Array.Copy(contents, KcdSizeOffset, target, 0x0, 0x4);
                uint size = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(size == KernelCapabilityEntrySize * 2);

                byte[] binary = new byte[size];
                Array.Copy(contents, offset, binary, 0, size);

                // ケイパビリティが正しく書き込まれている
                uint capability = BitConverter.ToUInt32(binary, 0);
                Assert.IsTrue(capability == 0x0000803F);

                capability = BitConverter.ToUInt32(binary, 4);
                Assert.IsTrue(capability == 0x0000013F);
            }

            // Io の MapMemory のデータが正しく書き込まれている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(kcd);

                XmlElement mapMemory = new XmlElement("MemoryMap");
                kcd.AddChild(mapMemory);

                XmlElement beginAddress = new XmlElement("BeginAddress");
                mapMemory.AddChild(beginAddress);
                beginAddress.InnerText = "0x100000";

                XmlElement sizeElem = new XmlElement("Size");
                mapMemory.AddChild(sizeElem);
                sizeElem.InnerText = "0x1000";

                XmlElement permission = new XmlElement("Permission");
                mapMemory.AddChild(permission);
                permission.InnerText = "Rw";

                XmlElement typeElem = new XmlElement("Type");
                mapMemory.AddChild(typeElem);
                typeElem.InnerText = "Io";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);

                // オフセットの値が正しい
                byte[] target = new byte[0x4];
                Array.Copy(contents, KcdOffset, target, 0x0, 0x4);
                uint offset = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(offset == HeaderSize + RoundUp(FacdDefaultSize, AcidAlign));

                // サイズの値が正しい
                Array.Copy(contents, KcdSizeOffset, target, 0x0, 0x4);
                uint size = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(size == KernelCapabilityEntrySize);

                byte[] binary = new byte[size];
                Array.Copy(contents, offset, binary, 0, size);

                // ケイパビリティが正しく書き込まれている
                uint capability = BitConverter.ToUInt32(binary, 0);
                Assert.IsTrue(capability == 0x1007F);
            }

            // EnableInterrupts のデータが正しく書き込める
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(kcd);

                XmlElement interrupt = new XmlElement("EnableInterrupts");
                interrupt.InnerText = "1";
                kcd.AddChild(interrupt);

                XmlElement interrupt2 = new XmlElement("EnableInterrupts");
                interrupt2.InnerText = "2";
                kcd.AddChild(interrupt2);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);

                // オフセットの値が正しい
                byte[] target = new byte[0x4];
                Array.Copy(contents, KcdOffset, target, 0x0, 0x4);
                uint offset = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(offset == HeaderSize + RoundUp(FacdDefaultSize, AcidAlign));

                // サイズの値が正しい
                Array.Copy(contents, KcdSizeOffset, target, 0x0, 0x4);
                uint size = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(size == KernelCapabilityEntrySize);

                byte[] binary = new byte[size];
                Array.Copy(contents, offset, binary, 0, size);

                // ケイパビリティが正しく書き込まれている
                // EnableInterrupts
                uint capability = BitConverter.ToUInt32(binary, 0);
                Assert.IsTrue(capability == 0x8017FF);
            }

            // EnableInterrupts のデータの個数が奇数の時にパディングデータが入る
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(kcd);

                XmlElement interrupt = new XmlElement("EnableInterrupts");
                interrupt.InnerText = "1";
                kcd.AddChild(interrupt);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);

                // オフセットの値が正しい
                byte[] target = new byte[0x4];
                Array.Copy(contents, KcdOffset, target, 0x0, 0x4);
                uint offset = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(offset == HeaderSize + RoundUp(FacdDefaultSize, AcidAlign));

                // サイズの値が正しい
                Array.Copy(contents, KcdSizeOffset, target, 0x0, 0x4);
                uint size = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(size == KernelCapabilityEntrySize);

                byte[] binary = new byte[size];
                Array.Copy(contents, offset, binary, 0, size);

                // ケイパビリティが正しく書き込まれている
                // EnableInterrupts
                uint capability = BitConverter.ToUInt32(binary, 0);
                Assert.IsTrue(capability == 0xFFC017FF);
            }

            // MiscParams のデータが正しく書き込まれている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(kcd);

                XmlElement miscParams = TestDataForKc.MakeMiscParams("1");
                kcd.AddChild(miscParams);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);

                // オフセットの値が正しい
                byte[] target = new byte[0x4];
                Array.Copy(contents, KcdOffset, target, 0x0, 0x4);
                uint offset = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(offset == HeaderSize + RoundUp(FacdDefaultSize, AcidAlign));

                // サイズの値が正しい
                Array.Copy(contents, KcdSizeOffset, target, 0x0, 0x4);
                uint size = BitConverter.ToUInt32(target, 0);
                Assert.AreEqual(size, KernelCapabilityEntrySize);

                byte[] binary = new byte[size];
                Array.Copy(contents, offset, binary, 0, size);

                // ケイパビリティが正しく書き込まれている
                // MiscParams
                uint capability = BitConverter.ToUInt32(binary, 0);
                Assert.IsTrue(capability == 0x5FFF);
            }

            // KernelVersion のデータが正しく書き込まれている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, "1.3");
                byte[] contents = AcidData(result);

                // オフセットの値が正しい
                byte[] target = new byte[0x4];
                Array.Copy(contents, KcdOffset, target, 0x0, 0x4);
                uint offset = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(offset == HeaderSize + RoundUp(FacdDefaultSize, AcidAlign));

                // サイズの値が正しい
                Array.Copy(contents, KcdSizeOffset, target, 0x0, 0x4);
                uint size = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(size == KernelCapabilityEntrySize);

                byte[] binary = new byte[size];
                Array.Copy(contents, offset, binary, 0, size);

                // ケイパビリティが正しく書き込まれている
                // EnableInterrupts
                uint capability = BitConverter.ToUInt32(binary, 0);
                Assert.IsTrue(capability == 0x9BFFF);
            }

            // HandleTableSize のデータが正しく書き込まれている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(kcd);

                XmlElement handleTable = new XmlElement("HandleTableSize");
                handleTable.InnerText = "1023";
                kcd.AddChild(handleTable);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);

                // オフセットの値が正しい
                byte[] target = new byte[0x4];
                Array.Copy(contents, KcdOffset, target, 0x0, 0x4);
                uint offset = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(offset == HeaderSize + RoundUp(FacdDefaultSize, AcidAlign));

                // サイズの値が正しい
                Array.Copy(contents, KcdSizeOffset, target, 0x0, 0x4);
                uint size = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(size == KernelCapabilityEntrySize);

                byte[] binary = new byte[size];
                Array.Copy(contents, offset, binary, 0, size);

                // ケイパビリティが正しく書き込まれている
                uint capability = BitConverter.ToUInt32(binary, 0);
                Assert.IsTrue(capability == 0x3FF7FFF);
            }

            // MiscFlags のデータが正しく書き込まれている
            // EnableDebug: True, ForceDebug: True
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(kcd);

                XmlElement misc = new XmlElement("MiscFlags");
                kcd.AddChild(misc);

                XmlElement enableDebug = new XmlElement("EnableDebug");
                enableDebug.InnerText = "true";
                misc.AddChild(enableDebug);

                XmlElement forceDebug = new XmlElement("ForceDebug");
                forceDebug.InnerText = "true";
                misc.AddChild(forceDebug);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);

                // オフセットの値が正しい
                byte[] target = new byte[0x4];
                Array.Copy(contents, KcdOffset, target, 0x0, 0x4);
                uint offset = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(offset == HeaderSize + RoundUp(FacdDefaultSize, AcidAlign));

                // サイズの値が正しい
                Array.Copy(contents, KcdSizeOffset, target, 0x0, 0x4);
                uint size = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(size == KernelCapabilityEntrySize);

                byte[] binary = new byte[size];
                Array.Copy(contents, offset, binary, 0, size);

                // ケイパビリティが正しく書き込まれている
                uint capability = BitConverter.ToUInt32(binary, 0);
                Assert.IsTrue(capability == 0x6FFFF);
            }

            // MiscFlags のデータが正しく書き込まれている
            // EnableDebug: False, ForceDebug: False
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(kcd);

                XmlElement misc = new XmlElement("MiscFlags");
                kcd.AddChild(misc);

                XmlElement enableDebug = new XmlElement("EnableDebug");
                enableDebug.InnerText = "false";
                misc.AddChild(enableDebug);

                XmlElement forceDebug = new XmlElement("ForceDebug");
                forceDebug.InnerText = "false";
                misc.AddChild(forceDebug);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);

                // オフセットの値が正しい
                byte[] target = new byte[0x4];
                Array.Copy(contents, KcdOffset, target, 0x0, 0x4);
                uint offset = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(offset == HeaderSize + RoundUp(FacdDefaultSize, AcidAlign));

                // サイズの値が正しい
                Array.Copy(contents, KcdSizeOffset, target, 0x0, 0x4);
                uint size = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(size == KernelCapabilityEntrySize);

                byte[] binary = new byte[size];
                Array.Copy(contents, offset, binary, 0, size);

                // ケイパビリティが正しく書き込まれている
                uint capability = BitConverter.ToUInt32(binary, 0);
                Assert.IsTrue(capability == 0xFFFF);
            }

            // 複数エントリ のデータが正しく書き込まれている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(kcd);

                XmlElement interrupt = new XmlElement("EnableInterrupts");
                interrupt.InnerText = "1";
                kcd.AddChild(interrupt);

                XmlElement handleTable = new XmlElement("HandleTableSize");
                handleTable.InnerText = "1023";
                kcd.AddChild(handleTable);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);

                // オフセットの値が正しい
                byte[] target = new byte[0x4];
                Array.Copy(contents, KcdOffset, target, 0x0, 0x4);
                uint offset = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(offset == HeaderSize + RoundUp(FacdDefaultSize, AcidAlign));

                // サイズの値が正しい
                Array.Copy(contents, KcdSizeOffset, target, 0x0, 0x4);
                uint size = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(size == KernelCapabilityEntrySize * 2);

                byte[] binary = new byte[size];
                Array.Copy(contents, offset, binary, 0, size);

                // ケイパビリティが正しく書き込まれている
                // EnableInterrupts
                uint capability = BitConverter.ToUInt32(binary, 0);
                Assert.IsTrue(capability == 0xFFC017FF);

                // HandleTableSize
                capability = BitConverter.ToUInt32(binary, 4);
                Assert.IsTrue(capability == 0x3FF7FFF);
            }

            // サイズが 0x10 にアライメントされていないとき、終端が 0 埋めされずに終わっている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(kcd);

                XmlElement handleTable = new XmlElement("HandleTableSize");
                handleTable.InnerText = "1023";
                kcd.AddChild(handleTable);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);

                // オフセットの値が正しい
                byte[] target = new byte[0x4];
                Array.Copy(contents, KcdOffset, target, 0x0, 0x4);
                uint offset = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(offset == HeaderSize + RoundUp(FacdDefaultSize, AcidAlign));

                // サイズの値が正しい
                Array.Copy(contents, KcdSizeOffset, target, 0x0, 0x4);
                uint size = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(size == KernelCapabilityEntrySize);

                Assert.IsTrue(offset + size == contents.Length);
            }

            // カーネルケイパビリティの順が エントリーNo. の順になっている
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(kcd);

                // No. 16
                XmlElement miscFlags = new XmlElement("MiscFlags");
                kcd.AddChild(miscFlags);
                XmlElement enableDebug = new XmlElement("EnableDebug");
                miscFlags.AddChild(enableDebug);
                enableDebug.InnerText = "True";

                // No. 15
                XmlElement handleTable = new XmlElement("HandleTableSize");
                handleTable.InnerText = "1023";
                kcd.AddChild(handleTable);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FileManager result = SuccessExecute(desc, null, null);
                byte[] contents = AcidData(result);

                // オフセットの値が正しい
                byte[] target = new byte[0x4];
                Array.Copy(contents, KcdOffset, target, 0x0, 0x4);
                uint offset = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(offset == HeaderSize + RoundUp(FacdDefaultSize, AcidAlign));

                // サイズの値が正しい
                Array.Copy(contents, KcdSizeOffset, target, 0x0, 0x4);
                uint size = BitConverter.ToUInt32(target, 0);
                Assert.IsTrue(size == KernelCapabilityEntrySize * 2);

                Assert.IsTrue(offset + size == contents.Length);

                target = new byte[0x8];
                Array.Copy(contents, offset, target, 0x0, 0x8);

                // No.15 の値
                Assert.IsTrue(BitConverter.ToUInt32(target, 0x0) == 0x3FF7FFF);

                // No.16 の値
                Assert.IsTrue(BitConverter.ToUInt32(target, 0x4) == 0x2FFFF);
            }
        }
    }
}
