﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using LECore.Structures;
using LECore.Structures.Core;
using LECore.Structures.Core.Command;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Reflection;

namespace LECoreTest
{
    /// <summary>
    /// 拡張ユーザー情報に関するテスト項目
    /// </summary>
    [TestClass]
    public class ExtUserDataTest
    {
        const string NameOfFloat        = "Float";
        const string NameOfInt          = "Int";
        const string NameOfString       = "String";
        const string NameOfAnmInt       = "AnmInt";
        const string NameOfAnmFloat     = "AnmFloat";
        const string NameOfAnmIntVec2   = "AnmIntVec2";
        const string NameOfAnmFloatVec2 = "AnmFloatVec2";
        const string NameOfAnmFloatVec3 = "AnmFloatVec3";
        const string NameOfAnmByteRGBA4 = "AnmByteRGBA4";

        /// <summary>
        /// 処理テーブル
        /// </summary>
        List<KeyValuePair<string, UserDataKind>> operationList = new List<KeyValuePair<string, UserDataKind>>()
        {
            { new KeyValuePair<string, UserDataKind>(NameOfFloat, UserDataKind.Float)               },
            { new KeyValuePair<string, UserDataKind>(NameOfInt, UserDataKind.Int)                   },
            { new KeyValuePair<string, UserDataKind>(NameOfString, UserDataKind.String)             },
            { new KeyValuePair<string, UserDataKind>(NameOfAnmInt, UserDataKind.AnmInt)             },
            { new KeyValuePair<string, UserDataKind>(NameOfAnmFloat, UserDataKind.AnmFloat)         },
            { new KeyValuePair<string, UserDataKind>(NameOfAnmIntVec2, UserDataKind.AnmIntVec2)     },
            { new KeyValuePair<string, UserDataKind>(NameOfAnmFloatVec2, UserDataKind.AnmFloatVec2) },
            { new KeyValuePair<string, UserDataKind>(NameOfAnmFloatVec3, UserDataKind.AnmFloatVec3) },
            { new KeyValuePair<string, UserDataKind>(NameOfAnmByteRGBA4, UserDataKind.AnmByteRGBA4) },
        };

        /// <summary>
        /// 処理テーブル（アニメーションのみ）
        /// </summary>
        List<KeyValuePair<string, UserDataKind>> animOperationList = new List<KeyValuePair<string, UserDataKind>>()
        {
            { new KeyValuePair<string, UserDataKind>(NameOfAnmInt, UserDataKind.AnmInt)             },
            { new KeyValuePair<string, UserDataKind>(NameOfAnmFloat, UserDataKind.AnmFloat)         },
            { new KeyValuePair<string, UserDataKind>(NameOfAnmIntVec2, UserDataKind.AnmIntVec2)     },
            { new KeyValuePair<string, UserDataKind>(NameOfAnmFloatVec2, UserDataKind.AnmFloatVec2) },
            { new KeyValuePair<string, UserDataKind>(NameOfAnmFloatVec3, UserDataKind.AnmFloatVec3) },
            { new KeyValuePair<string, UserDataKind>(NameOfAnmByteRGBA4, UserDataKind.AnmByteRGBA4) },
        };

        /// <summary>
        /// テストデータ（基準値）
        /// </summary>
        Dictionary<string, object> dataTable = new Dictionary<string, object>()
                {
                    { NameOfFloat,        new float[1] { 99.9f }               },
                    { NameOfInt,          new int[1] { 99 }                    },
                    { NameOfString,       "test_string"                        },
                    { NameOfAnmInt,       99                                   },
                    { NameOfAnmFloat,     99.9f                                },
                    { NameOfAnmIntVec2,   new IVec2(99, 99)                    },
                    { NameOfAnmFloatVec2, new FVec2(99.9f, 99.9f)              },
                    { NameOfAnmFloatVec3, new FVec3(99.9f, 99.9f, 99.9f)       },
                    { NameOfAnmByteRGBA4, new RGBAColor(99, 99, 99, 99)        },
                };

        /// <summary>
        /// テストデータ（編集値）
        /// </summary>
        Dictionary<string, object> editTable = new Dictionary<string, object>()
                {
                    { NameOfFloat,        new float[1] { 33.3f }               },
                    { NameOfInt,          new int[1] { 33 }                    },
                    { NameOfString,       "edit_string"                        },
                    { NameOfAnmInt,       33                                   },
                    { NameOfAnmFloat,     33.3f                                },
                    { NameOfAnmIntVec2,   new IVec2(33, 33)                    },
                    { NameOfAnmFloatVec2, new FVec2(33.3f, 33.3f)              },
                    { NameOfAnmFloatVec3, new FVec3(33.3f, 33.3f, 33.3f)       },
                    { NameOfAnmByteRGBA4, new RGBAColor(33, 33, 33, 33)        },
                };

        #region 拡張ユーザー情報の「追加」、「編集」、「削除」が行なえる

        /// <summary>
        /// 拡張ユーザー情報の「追加」、「編集」、「削除」が行なえる
        /// </summary>
        [TestMethod]
        public void ExtUserDataTest_AddEditAndRemove()
        {
            // 準備
            {
                TestUtility.InitializeTest(MethodBase.GetCurrentMethod());
            }

            // テスト
            {
                // レイアウトを新規作成
                ISubScene newSubScene = TestUtility.CreateLayout();

                // ヌルペインを追加する
                string paneName = "NullPane";
                IPane pane = TestUtility.AddNullPane(newSubScene, paneName);

                // 拡張ユーザー情報を追加する
                foreach (KeyValuePair<string, UserDataKind> list in operationList)
                {
                    TestUtility.AddExtUserData(pane, list.Key, list.Value, dataTable[list.Key]);
                }

                // 追加されたことを確認する
                Assert.AreEqual(pane.IUserDataHolder.UserDataElementSet.Length, operationList.Count);
                AreEqualsExtUserDataValue(pane, dataTable);

                // 値を編集する
                foreach (KeyValuePair<string, UserDataKind> list in operationList)
                {
                    TestUtility.ModifyExtUserData(pane, list.Key, list.Key, list.Value, editTable[list.Key]);
                }

                // 編集されたことを確認する
                AreEqualsExtUserDataValue(pane, editTable);

                // 拡張ユーザー情報を削除する
                foreach (KeyValuePair<string, UserDataKind> list in operationList)
                {
                    TestUtility.RemoveExtUserData(pane, list.Key);
                }

                // 削除されたことを確認する
                Assert.AreEqual(pane.IUserDataHolder.UserDataElementSet.Length, 0);
            }

            // 後処理
            {
                TestUtility.FinalizeTest(MethodBase.GetCurrentMethod());
            }
        }

        #endregion

        #region 拡張ユーザー情報に「アニメーションキーの追加」が行なえる

        /// <summary>
        /// 拡張ユーザー情報に「アニメーションキーの追加」が行なえる
        /// </summary>
        [TestMethod]
        public void ExtUserDataTest_AddAnimationKey()
        {
            // 準備
            {
                TestUtility.InitializeTest(MethodBase.GetCurrentMethod());
            }

            // 結合モード時の動作確認
            {
                // レイアウトを新規作成
                ISubScene newSubScene = TestUtility.CreateLayout();

                // ヌルペインを追加する
                string paneName = "NullPane";
                IPane pane = TestUtility.AddNullPane(newSubScene, paneName);

                // 拡張ユーザー情報を追加する
                foreach (KeyValuePair<string, UserDataKind> list in animOperationList)
                {
                    TestUtility.AddExtUserData(pane, list.Key, list.Value, dataTable[list.Key]);
                }

                // 数を確認する
                Assert.AreEqual(pane.IUserDataHolder.UserDataElementSet.Length, animOperationList.Count);

                // Undoする
                var cmdQueue = (newSubScene.ISubSceneCmdQueue as SubSceneCmdQueue);
                cmdQueue.Undo();

                // 数を確認する
                Assert.AreEqual(pane.IUserDataHolder.UserDataElementSet.Length, animOperationList.Count-1);

                // Redoする
                cmdQueue.Redo();

                // 数を確認する
                Assert.AreEqual(pane.IUserDataHolder.UserDataElementSet.Length, animOperationList.Count);

                // アニメーションキーを追加する
                foreach (IUserDataElement elem in pane.IUserDataHolder.UserDataElementSet)
                {
                    TestUtility.AddExtUserDataAnimKey(elem, 50, editTable[elem.Name]);
                }

                // タイムスライダーを進める
                TestUtility.SetTime(50);

                // 編集されたことを確認する
                AreEqualsExtUserDataValue(pane, editTable);

                // アニメーションキーを追加する
                foreach (IUserDataElement elem in pane.IUserDataHolder.UserDataElementSet)
                {
                    TestUtility.AddExtUserDataAnimKey(elem, 100, dataTable[elem.Name]);
                }

                // タイムスライダーを進める
                TestUtility.SetTime(100);

                // 編集されたことを確認する
                AreEqualsExtUserDataValue(pane, dataTable);

                // タイムスライダーを戻す
                TestUtility.SetTime(0);

                // レイアウトを閉じる
                TestUtility.CloseLayout(newSubScene);
            }

            // 分割モード時の動作確認
            {
                // レイアウトを新規作成
                ISubScene newSubScene = TestUtility.CreateLayout();

                // 区間タグを追加する
                const string tagName00 = "Tag_00";
                const string tagName01 = "Tag_01";
                TestUtility.AddAnimFrameSection(newSubScene, tagName00, 0, 100);
                TestUtility.AddAnimFrameSection(newSubScene, tagName00, 0, 100);

                // 分割モードに変更する
                TestUtility.ChangeAnimEditMode(newSubScene);

                // 区間タグを切り替える
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, tagName00);

                // ヌルペインを追加する
                string paneName = "NullPane";
                IPane pane = TestUtility.AddNullPane(newSubScene, paneName);

                // 拡張ユーザー情報を追加する
                foreach (KeyValuePair<string, UserDataKind> list in animOperationList)
                {
                    TestUtility.AddExtUserData(pane, list.Key, list.Value, dataTable[list.Key]);
                }

                // 数を確認する
                Assert.AreEqual(pane.IUserDataHolder.UserDataElementSet.Length, animOperationList.Count);

                // Undoする
                var cmdQueue = (newSubScene.ISubSceneCmdQueue as SubSceneCmdQueue);
                cmdQueue.Undo();

                // 数を確認する
                Assert.AreEqual(pane.IUserDataHolder.UserDataElementSet.Length, animOperationList.Count - 1);

                // Redoする
                cmdQueue.Redo();

                // 数を確認する
                Assert.AreEqual(pane.IUserDataHolder.UserDataElementSet.Length, animOperationList.Count);

                // アニメーションキーを追加する
                foreach (IUserDataElement elem in pane.IUserDataHolder.UserDataElementSet)
                {
                    TestUtility.AddExtUserDataAnimKey(elem, 50, editTable[elem.Name]);
                }

                // タイムスライダーを進める
                TestUtility.SetTime(50);

                // 編集されたことを確認する
                AreEqualsExtUserDataValue(pane, editTable);

                // 区間を切り替える
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, tagName01);

                // アニメーションキーを追加する
                foreach (IUserDataElement elem in pane.IUserDataHolder.UserDataElementSet)
                {
                    TestUtility.AddExtUserDataAnimKey(elem, 100, dataTable[elem.Name]);
                }

                // タイムスライダーを進める
                TestUtility.SetTime(100);

                // 編集されたことを確認する
                AreEqualsExtUserDataValue(pane, dataTable);

                // 区間を切り替える
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, tagName00);

                // 編集されたことを確認する
                AreEqualsExtUserDataValue(pane, editTable);

                // 区間を未選択状態にする
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, null);

                // 編集されたことを確認する
                AreEqualsExtUserDataValue(pane, dataTable);

                // タイムスライダーを戻す
                TestUtility.SetTime(0);

                // レイアウトを閉じる
                TestUtility.CloseLayout(newSubScene);
            }

            // 後処理
            {
                TestUtility.FinalizeTest(MethodBase.GetCurrentMethod());
            }
        }

        #endregion

        #region 拡張ユーザー情報の「コピー＆ペースト」が行なえる

        /// <summary>
        /// 拡張ユーザー情報の「コピー＆ペースト」が行なえる
        /// </summary>
        [TestMethod]
        public void ExtUserDataTest_CopyAndPaste()
        {
            // 準備
            {
                TestUtility.InitializeTest(MethodBase.GetCurrentMethod());
            }

            // ペインの複製
            {
                // レイアウトを新規作成
                ISubScene newSubScene = TestUtility.CreateLayout();

                // ヌルペインを追加する
                string paneName = "NullPane";
                IPane pane = TestUtility.AddNullPane(newSubScene, paneName);

                // 拡張ユーザー情報を追加する
                foreach (KeyValuePair<string, UserDataKind> list in animOperationList)
                {
                    TestUtility.AddExtUserData(pane, list.Key, list.Value, dataTable[list.Key]);
                }

                // アニメーションキーを追加する
                foreach (IUserDataElement elem in pane.IUserDataHolder.UserDataElementSet)
                {
                    TestUtility.AddExtUserDataAnimKey(elem, 50, editTable[elem.Name]);
                }

                // タイムスライダーを進める
                TestUtility.SetTime(50);

                // ペインを複製する
                IPane dupPane = TestUtility.DuplicatePane(newSubScene, pane);

                // 数を確認する
                Assert.AreEqual(dupPane.IUserDataHolder.UserDataElementSet.Length, animOperationList.Count);

                // 値を確認する
                AreEqualsExtUserDataValue(dupPane, editTable);

                // 複製したペインにアニメーションキーを追加する
                foreach (IUserDataElement elem in dupPane.IUserDataHolder.UserDataElementSet)
                {
                    TestUtility.AddExtUserDataAnimKey(elem, 100, dataTable[elem.Name]);
                }

                // タイムスライダーを進める
                TestUtility.SetTime(100);

                // 値を確認する
                AreEqualsExtUserDataValue(dupPane, dataTable);

                // タイムスライダーを戻す
                TestUtility.SetTime(0);

                // レイアウトを閉じる
                TestUtility.CloseLayout(newSubScene);
            }

            // ペインの複製(分割モード)
            {
                // レイアウトを新規作成
                ISubScene newSubScene = TestUtility.CreateLayout();

                // 区間タグを追加する
                const string tagName00 = "Tag_00";
                const string tagName01 = "Tag_01";
                TestUtility.AddAnimFrameSection(newSubScene, tagName00, 0, 100);
                TestUtility.AddAnimFrameSection(newSubScene, tagName00, 0, 100);

                // 分割モードに変更する
                TestUtility.ChangeAnimEditMode(newSubScene);

                // 区間タグを切り替える
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, tagName00);

                // ヌルペインを追加する
                string paneName = "NullPane";
                IPane pane = TestUtility.AddNullPane(newSubScene, paneName);

                // 拡張ユーザー情報を追加する
                foreach (KeyValuePair<string, UserDataKind> list in animOperationList)
                {
                    TestUtility.AddExtUserData(pane, list.Key, list.Value, dataTable[list.Key]);
                }

                // アニメーションキーを追加する
                foreach (IUserDataElement elem in pane.IUserDataHolder.UserDataElementSet)
                {
                    TestUtility.AddExtUserDataAnimKey(elem, 50, editTable[elem.Name]);
                }

                // タイムスライダーを進める
                TestUtility.SetTime(50);

                // ペインを複製する
                IPane dupPane = TestUtility.DuplicatePane(newSubScene, pane);

                // 数を確認する
                Assert.AreEqual(dupPane.IUserDataHolder.UserDataElementSet.Length, animOperationList.Count);

                // 値を確認する
                AreEqualsExtUserDataValue(dupPane, editTable);

                // 区間タグを切り替える
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, tagName01);

                // 複製したペインにアニメーションキーを追加する
                foreach (IUserDataElement elem in dupPane.IUserDataHolder.UserDataElementSet)
                {
                    TestUtility.AddExtUserDataAnimKey(elem, 100, dataTable[elem.Name]);
                }

                // タイムスライダーを進める
                TestUtility.SetTime(100);

                // 値を確認する
                AreEqualsExtUserDataValue(dupPane, dataTable);

                // 区間タグを切り替える
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, tagName00);

                // 値を確認する
                AreEqualsExtUserDataValue(dupPane, editTable);

                // 区間未選択状態にする
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, null);

                // 値を確認する
                AreEqualsExtUserDataValue(dupPane, dataTable);

                // タイムスライダーを戻す
                TestUtility.SetTime(0);

                // レイアウトを閉じる
                TestUtility.CloseLayout(newSubScene);
            }

            // パラメータ貼り付け
            {
                // レイアウトを新規作成
                ISubScene newSubScene = TestUtility.CreateLayout();

                // ヌルペインを追加する
                string paneNameA = "NullPaneA";
                IPane paneA = TestUtility.AddNullPane(newSubScene, paneNameA);
                string paneNameB = "NullPaneB";
                IPane paneB = TestUtility.AddNullPane(newSubScene, paneNameB);

                // 拡張ユーザー情報を追加する
                foreach (KeyValuePair<string, UserDataKind> list in animOperationList)
                {
                    TestUtility.AddExtUserData(paneA, list.Key, list.Value, dataTable[list.Key]);
                }

                // アニメーションキーを追加する
                foreach (IUserDataElement elem in paneA.IUserDataHolder.UserDataElementSet)
                {
                    TestUtility.AddExtUserDataAnimKey(elem, 50, editTable[elem.Name]);
                }

                // パラメータを貼り付ける
                TestUtility.PastePaneParameter(paneA, paneB);

                // 数を確認する
                Assert.AreEqual(paneB.IUserDataHolder.UserDataElementSet.Length, animOperationList.Count);

                // 編集されたことを確認する
                AreEqualsExtUserDataValue(paneB, editTable);

                // レイアウトを閉じる
                TestUtility.CloseLayout(newSubScene);
            }

            // パラメータ貼り付け（分割モード）
            {
                // レイアウトを新規作成
                ISubScene newSubScene = TestUtility.CreateLayout();

                // 区間タグを追加する
                const string tagName00 = "Tag_00";
                const string tagName01 = "Tag_01";
                TestUtility.AddAnimFrameSection(newSubScene, tagName00, 0, 100);
                TestUtility.AddAnimFrameSection(newSubScene, tagName00, 0, 100);

                // 分割モードに変更する
                TestUtility.ChangeAnimEditMode(newSubScene);

                // ヌルペインを追加する
                string paneNameA = "NullPaneA";
                IPane paneA = TestUtility.AddNullPane(newSubScene, paneNameA);
                string paneNameB = "NullPaneB";
                IPane paneB = TestUtility.AddNullPane(newSubScene, paneNameB);

                // 拡張ユーザー情報を追加する
                foreach (KeyValuePair<string, UserDataKind> list in animOperationList)
                {
                    TestUtility.AddExtUserData(paneA, list.Key, list.Value, dataTable[list.Key]);
                }
                foreach (KeyValuePair<string, UserDataKind> list in animOperationList)
                {
                    TestUtility.AddExtUserData(paneB, list.Key, list.Value, dataTable[list.Key]);
                }

                // 区間タグを切り替える
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, tagName00);

                // アニメーションキーを追加する
                foreach (IUserDataElement elem in paneA.IUserDataHolder.UserDataElementSet)
                {
                    TestUtility.AddExtUserDataAnimKey(elem, 50, editTable[elem.Name]);
                }

                // 区間タグを切り替える
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, tagName01);

                // アニメーションキーを追加する
                foreach (IUserDataElement elem in paneA.IUserDataHolder.UserDataElementSet)
                {
                    TestUtility.AddExtUserDataAnimKey(elem, 100, dataTable[elem.Name]);
                }

                // パラメータを貼り付ける
                TestUtility.PastePaneParameter(paneA, paneB);

                // 編集されたことを確認する
                AreEqualsExtUserDataValue(paneB, dataTable);

                // 区間タグを切り替える
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, tagName00);

                // 編集されたことを確認する
                AreEqualsExtUserDataValue(paneB, editTable);

                // 区間未選択状態にする
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, null);

                // 値を確認する
                AreEqualsExtUserDataValue(paneB, dataTable);

                // レイアウトを閉じる
                TestUtility.CloseLayout(newSubScene);
            }


            // 後処理
            {
                TestUtility.FinalizeTest(MethodBase.GetCurrentMethod());
            }
        }

        #endregion

        #region private method

        /// <summary>
        /// 拡張ユーザー情報の値を比較します
        /// </summary>
        private void AreEqualsExtUserDataValue(IPane pane, Dictionary<string, object> compare)
        {
            foreach (IUserDataElement elem in pane.IUserDataHolder.UserDataElementSet)
            {
                switch (elem.UserDataKind)
                {
                    case UserDataKind.String:
                    case UserDataKind.AnmInt:
                    case UserDataKind.AnmFloat:
                        {
                            Assert.AreEqual(elem.Value, compare[elem.Name]);
                        }
                        break;
                    case UserDataKind.Int:
                    case UserDataKind.AnmIntVec2:
                    case UserDataKind.AnmByteRGBA4:
                        {

                            var arrayA = UserDataElementHelper.ConvertAnmValueToArray(elem.UserDataKind, elem.Value) as IEnumerable<int>;
                            var arrayB = UserDataElementHelper.ConvertAnmValueToArray(elem.UserDataKind, compare[elem.Name]) as IEnumerable<int>;

                            Assert.IsTrue(arrayA.SequenceEqual(arrayB));
                        }
                        break;
                    case UserDataKind.Float:
                    case UserDataKind.AnmFloatVec2:
                    case UserDataKind.AnmFloatVec3:
                        {
                            var arrayA = UserDataElementHelper.ConvertAnmValueToArray(elem.UserDataKind, elem.Value) as IEnumerable<float>;
                            var arrayB = UserDataElementHelper.ConvertAnmValueToArray(elem.UserDataKind, compare[elem.Name]) as IEnumerable<float>;

                            Assert.IsTrue(arrayA.SequenceEqual(arrayB));
                        }
                        break;
                }
            }
        }

        #endregion
    }
}
