﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using LECore.Structures;
using LECore.Structures.Core.Command;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Diagnostics;
using System.Linq;
using System.Reflection;

namespace LECoreTest
{
    /// <summary>
    /// ペインに関するテスト項目
    /// </summary>
    [TestClass]
    public class PaneTest
    {
        #region 各種ペインの「新規作成」、「編集」、「削除」が行なえる、Undo/Redoができる

        /// <summary>
        /// 各種ペインの「新規作成」、「編集」、「削除」が行なえる、Undo/Redoができる
        /// </summary>
        [TestMethod]
        public void PaneTest_AddEditRemoveAndUndoRedo()
        {
            // 準備
            {
                TestUtility.InitializeTest(MethodBase.GetCurrentMethod());
            }

            // テスト
            {
                // レイアウトを新規作成
                ISubScene newSubScene = TestUtility.CreateLayout();

                // ヌルペイン
                {
                    TestAddEditUndoRedoDelete(newSubScene, (paneName) => {
                        TestUtility.AddNullPane(newSubScene, paneName);
                    });
                }

                // ピクチャペイン
                {
                    TestAddEditUndoRedoDelete(newSubScene, (paneName) => {
                        TestUtility.AddPicturePane(newSubScene, paneName, null);
                    });
                }

                // テキストボックスペインを追加
                {
                    TestAddEditUndoRedoDelete(newSubScene, (paneName) => {
                        ILEFont font = TestUtility.LoadFont(newSubScene, "sample2.ffnt");
                        TestUtility.AddTextBoxPane(newSubScene, paneName, font, "");
                    });
                }

                // ウインドウペインを追加
                {
                    TestAddEditUndoRedoDelete(newSubScene, (paneName) => {
                        TestUtility.AddWindowPane(newSubScene, paneName);
                    });
                }

                // 境界ペインを追加
                {
                    TestAddEditUndoRedoDelete(newSubScene, (paneName) => {
                        TestUtility.AddBoudingPane(newSubScene, paneName);
                    });
                }

                // キャプチャペインを追加
                {
                    TestAddEditUndoRedoDelete(newSubScene, (paneName) => {
                        TestUtility.AddCapturePane(newSubScene, paneName);
                    });
                }

                // 部品ペインを追加
                {
                    TestAddEditUndoRedoDelete(newSubScene, (paneName) => {
                        string partsName = "animation.flyt";
                        ISubScene partsSubScene = TestUtility.LoadPartsLayout(partsName); // 部品をあらかじめ読み込んでおく
                        TestUtility.AddPartsPane(newSubScene, paneName, partsName, partsSubScene);
                    });
                }
            }

            // 後処理
            {
                TestUtility.FinalizeTest(MethodBase.GetCurrentMethod());
            }
        }

        private static void TestAddEditUndoRedoDelete(ISubScene subScene, Action<string> func)
        {
            string paneName = "NewPane";

            // ヌルペインを追加
            func(paneName);

            // ペインが追加されたことを確認する
            IPane pane = subScene.IPaneArray.FirstOrDefault();
            Assert.AreEqual(subScene.IPaneArray.Count(), 1);

            // ペインを操作する
            FVec3 oldPos = pane.Trans;
            FVec3 newPos = new FVec3(99.9f, 99.9f, 99.9f);
            TestUtility.ModifyPosition(pane, newPos);

            // 操作結果を比較する
            TestUtility.IsEqualPosition(pane, newPos);

            // Undoする
            var cmdQueue = (subScene.ISubSceneCmdQueue as SubSceneCmdQueue);
            cmdQueue.Undo();

            // 操作結果を比較する
            TestUtility.IsEqualPosition(pane, oldPos);

            // Redoする
            cmdQueue.Redo();

            // 操作結果を比較する
            TestUtility.IsEqualPosition(pane, newPos);

            // ペインを削除する
            TestUtility.RemovePane(subScene, pane);

            // ペインが削除されたことを確認する
            Assert.AreEqual(subScene.IPaneArray.Count(), 0);
        }

        #endregion

        #region ペインの複製ができる

        /// <summary>
        /// ペインの複製ができる
        /// </summary>
        [TestMethod]
        public void PaneTest_Duplicate()
        {
            // 準備
            {
                TestUtility.InitializeTest(MethodBase.GetCurrentMethod());
            }

            // テスト
            {
                // 結合モード 時間＝0
                {
                    TestUtility.SetTime(0);

                    // レイアウトを読み込み
                    ISubScene layout = TestUtility.LoadLayout("allPaneAnim.flyt");

                    TestUtility.SetTime(0);

                    // 各種ペインを取得
                    IPane nullPane = layout.FindPaneByName("N_null_00");
                    IPane picturePane = layout.FindPaneByName("P_pict_00");
                    IPane textPane = layout.FindPaneByName("T_text_00");
                    IPane windowPane = layout.FindPaneByName("W_window_00");
                    IPane boundingPane = layout.FindPaneByName("B_bounding_00");
                    IPane capturePane = layout.FindPaneByName("C_capture_00");
                    IPane partsPane = layout.FindPaneByName("L_parts_00");

                    // ペインを複製
                    IPane dupNullPane = TestUtility.DuplicatePane(layout, nullPane);
                    IPane dupPicturePane = TestUtility.DuplicatePane(layout, picturePane);
                    IPane dupTextPane = TestUtility.DuplicatePane(layout, textPane);
                    IPane dupWindowPane = TestUtility.DuplicatePane(layout, windowPane);
                    IPane dupBoundingPane = TestUtility.DuplicatePane(layout, boundingPane);
                    IPane dupCapturePane = TestUtility.DuplicatePane(layout, capturePane);
                    IPane dupPartsPane = TestUtility.DuplicatePane(layout, partsPane);

                    // 複製されたペインを比較
                    TestUtility.AreEqualsPaneParameters(nullPane, dupNullPane);
                    TestUtility.AreEqualsPaneParameters(picturePane, dupPicturePane);
                    TestUtility.AreEqualsPaneParameters(textPane, dupTextPane);
                    TestUtility.AreEqualsPaneParameters(windowPane, dupWindowPane);
                    TestUtility.AreEqualsPaneParameters(boundingPane, dupBoundingPane);
                    TestUtility.AreEqualsPaneParameters(capturePane, dupCapturePane);
                    TestUtility.AreEqualsPaneParameters(partsPane, dupPartsPane);

                    // レイアウトを閉じる
                    TestUtility.CloseLayout(layout);
                }

                // 結合モード 時間≠0
                {
                    TestUtility.SetTime(0);

                    // レイアウトを読み込み
                    ISubScene layout = TestUtility.LoadLayout("allPaneAnim.flyt");

                    TestUtility.SetTime(0);

                    // 各種ペインを取得
                    IPane nullPane = layout.FindPaneByName("N_null_00");
                    IPane picturePane = layout.FindPaneByName("P_pict_00");
                    IPane textPane = layout.FindPaneByName("T_text_00");
                    IPane windowPane = layout.FindPaneByName("W_window_00");
                    IPane boundingPane = layout.FindPaneByName("B_bounding_00");
                    IPane capturePane = layout.FindPaneByName("C_capture_00");
                    IPane partsPane = layout.FindPaneByName("L_parts_00");

                    // タイムスライダーを進める
                    TestUtility.SetTime(100);

                    // ペインを複製
                    IPane dupNullPane = TestUtility.DuplicatePane(layout, nullPane);
                    IPane dupPicturePane = TestUtility.DuplicatePane(layout, picturePane);
                    IPane dupTextPane = TestUtility.DuplicatePane(layout, textPane);
                    IPane dupWindowPane = TestUtility.DuplicatePane(layout, windowPane);
                    IPane dupBoundingPane = TestUtility.DuplicatePane(layout, boundingPane);
                    IPane dupCapturePane = TestUtility.DuplicatePane(layout, capturePane);
                    IPane dupPartsPane = TestUtility.DuplicatePane(layout, partsPane);

                    // 複製されたペインを比較
                    TestUtility.AreEqualsPaneParameters(nullPane, dupNullPane);
                    TestUtility.AreEqualsPaneParameters(picturePane, dupPicturePane);
                    TestUtility.AreEqualsPaneParameters(textPane, dupTextPane);
                    TestUtility.AreEqualsPaneParameters(windowPane, dupWindowPane);
                    TestUtility.AreEqualsPaneParameters(boundingPane, dupBoundingPane);
                    TestUtility.AreEqualsPaneParameters(capturePane, dupCapturePane);
                    // (SIGLO-73477) カーブエディタでサイズが変更できてしまうため、結果が一致しない
                    //TestUtility.AreEqualsPaneParameters(partsPane, dupPartsPane);

                    // タイムスライダーを戻す
                    TestUtility.SetTime(0);

                    // レイアウトを閉じる
                    TestUtility.CloseLayout(layout);
                }

                // 分割モード 時間＝0
                {
                    // レイアウトを読み込み
                    ISubScene layout = TestUtility.LoadLayout("allPaneAnim_separated.flyt");

                    TestUtility.SetTime(0);

                    // 各種ペインを取得
                    IPane nullPane = layout.FindPaneByName("N_null_00");
                    IPane picturePane = layout.FindPaneByName("P_pict_00");
                    IPane textPane = layout.FindPaneByName("T_text_00");
                    IPane windowPane = layout.FindPaneByName("W_window_00");
                    IPane boundingPane = layout.FindPaneByName("B_bounding_00");
                    IPane capturePane = layout.FindPaneByName("C_capture_00");
                    IPane partsPane = layout.FindPaneByName("L_parts_00");

                    // ペインを複製
                    IPane dupNullPane = TestUtility.DuplicatePane(layout, nullPane);
                    IPane dupPicturePane = TestUtility.DuplicatePane(layout, picturePane);
                    IPane dupTextPane = TestUtility.DuplicatePane(layout, textPane);
                    IPane dupWindowPane = TestUtility.DuplicatePane(layout, windowPane);
                    IPane dupBoundingPane = TestUtility.DuplicatePane(layout, boundingPane);
                    IPane dupCapturePane = TestUtility.DuplicatePane(layout, capturePane);
                    IPane dupPartsPane = TestUtility.DuplicatePane(layout, partsPane);

                    // 複製されたペインを比較
                    TestUtility.AreEqualsPaneParameters(nullPane, dupNullPane);
                    TestUtility.AreEqualsPaneParameters(picturePane, dupPicturePane);
                    TestUtility.AreEqualsPaneParameters(textPane, dupTextPane);
                    TestUtility.AreEqualsPaneParameters(windowPane, dupWindowPane);
                    TestUtility.AreEqualsPaneParameters(boundingPane, dupBoundingPane);
                    TestUtility.AreEqualsPaneParameters(capturePane, dupCapturePane);
                    TestUtility.AreEqualsPaneParameters(partsPane, dupPartsPane);

                    // レイアウトを閉じる
                    TestUtility.CloseLayout(layout);
                }

                // 分割モード 時間≠0
                {
                    // タイムスライダーを戻す
                    TestUtility.SetTime(0);

                    // レイアウトを読み込み
                    ISubScene layout = TestUtility.LoadLayout("allPaneAnim_separated.flyt");

                    TestUtility.SetTime(0);

                    // 各種ペインを取得
                    IPane nullPane = layout.FindPaneByName("N_null_00");
                    IPane picturePane = layout.FindPaneByName("P_pict_00");
                    IPane textPane = layout.FindPaneByName("T_text_00");
                    IPane windowPane = layout.FindPaneByName("W_window_00");
                    IPane boundingPane = layout.FindPaneByName("B_bounding_00");
                    IPane capturePane = layout.FindPaneByName("C_capture_00");
                    IPane partsPane = layout.FindPaneByName("L_parts_00");

                    // タイムスライダーを進める
                    TestUtility.ModifyAnimFrameSectionFromName(layout, "Tag_00");

                    TestUtility.SetTime(100);

                    // ペインを複製
                    IPane dupNullPane = TestUtility.DuplicatePane(layout, nullPane);
                    IPane dupPicturePane = TestUtility.DuplicatePane(layout, picturePane);
                    IPane dupTextPane = TestUtility.DuplicatePane(layout, textPane);
                    IPane dupWindowPane = TestUtility.DuplicatePane(layout, windowPane);
                    IPane dupBoundingPane = TestUtility.DuplicatePane(layout, boundingPane);
                    IPane dupCapturePane = TestUtility.DuplicatePane(layout, capturePane);
                    IPane dupPartsPane = TestUtility.DuplicatePane(layout, partsPane);

                    // 複製されたペインを比較
                    TestUtility.AreEqualsPaneParameters(nullPane, dupNullPane);
                    TestUtility.AreEqualsPaneParameters(picturePane, dupPicturePane);
                    TestUtility.AreEqualsPaneParameters(textPane, dupTextPane);
                    TestUtility.AreEqualsPaneParameters(windowPane, dupWindowPane);
                    TestUtility.AreEqualsPaneParameters(boundingPane, dupBoundingPane);
                    TestUtility.AreEqualsPaneParameters(capturePane, dupCapturePane);
                    // (SIGLO-73477) カーブエディタでサイズが変更できてしまうため、結果が一致しない
                    //TestUtility.AreEqualsPaneParameters(partsPane, dupPartsPane);

                    // タイムスライダーを戻す
                    TestUtility.SetTime(0);

                    // レイアウトを閉じる
                    TestUtility.CloseLayout(layout);
                }
            }

            // 後処理
            {
                TestUtility.FinalizeTest(MethodBase.GetCurrentMethod());
            }
        }

        #endregion

        #region ペイン情報貼り付け制御から貼り付けができる

        /// <summary>
        /// ペイン情報貼り付け制御から貼り付けができる
        /// </summary>
        [TestMethod]
        public void PaneTest_ParameterPaste()
        {
            // 準備
            {
                TestUtility.InitializeTest(MethodBase.GetCurrentMethod());

                LECore.Util.TemporaryFileUtil.Initialize();
            }

            // 結合モード 時間＝0
            {
                // レイアウトを読み込み
                ISubScene layout = TestUtility.LoadLayout("allPaneAnim.flyt");

                // フォント、テクスチャ、部品の読み込み
                ILEFont font = TestUtility.LoadFont(layout, "sample2.ffnt");
                string partsName = "animation.flyt";
                ISubScene partsSubScene = TestUtility.LoadPartsLayout(partsName); // 部品をあらかじめ読み込んでおく

                // 各種ペインを取得
                IPane nullPane = layout.FindPaneByName("N_null_00");
                IPane picturePane = layout.FindPaneByName("P_pict_00");
                IPane textPane = layout.FindPaneByName("T_text_00");
                IPane windowPane = layout.FindPaneByName("W_window_00");
                IPane boundingPane = layout.FindPaneByName("B_bounding_00");
                IPane capturePane = layout.FindPaneByName("C_capture_00");
                IPane partsPane = layout.FindPaneByName("L_parts_00");

                // 新規ペインを作成
                IPane newNullPane = TestUtility.AddNullPane(layout, "newNullPane");
                IPane newPicturePane = TestUtility.AddPicturePane(layout, "newPicturePane", null);
                IPane newTextPane = TestUtility.AddTextBoxPane(layout, "newTextPane", font, "");
                IPane newWindowPane = TestUtility.AddWindowPane(layout, "newWindowPane");
                IPane newBoundingPane = TestUtility.AddBoudingPane(layout, "newBoundingPane");
                IPane newCapturePane = TestUtility.AddCapturePane(layout, "newCapturePane");
                IPane newPartsPane = TestUtility.AddPartsPane(layout, "newPartsPane", partsName, partsSubScene);

                // パラメータ貼り付け
                TestUtility.PastePaneParameter(nullPane, newNullPane);
                TestUtility.PastePaneParameter(picturePane, newPicturePane);
                TestUtility.PastePaneParameter(textPane, newTextPane);
                TestUtility.PastePaneParameter(windowPane, newWindowPane);
                TestUtility.PastePaneParameter(boundingPane, newBoundingPane);
                TestUtility.PastePaneParameter(capturePane, newCapturePane);
                TestUtility.PastePaneParameter(partsPane, newPartsPane);

                // ペインを比較
                TestUtility.AreEqualsPaneParameters(nullPane, newNullPane);
                TestUtility.AreEqualsPaneParameters(picturePane, newPicturePane);
                TestUtility.AreEqualsPaneParameters(textPane, newTextPane);
                TestUtility.AreEqualsPaneParameters(windowPane, newWindowPane);
                TestUtility.AreEqualsPaneParameters(boundingPane, newBoundingPane);
                TestUtility.AreEqualsPaneParameters(capturePane, newCapturePane);
                // (SIGLO-73477) カーブエディタでサイズが変更できてしまうため、結果が一致しない
                //TestUtility.AreEqualsPaneParameters(partsPane, newPartsPane);

                // レイアウトを閉じる
                TestUtility.CloseLayout(layout);
            }

            // 結合モード 時間≠0
            {
                // レイアウトを読み込み
                ISubScene layout = TestUtility.LoadLayout("allPaneAnim.flyt");

                // フォント、テクスチャ、部品の読み込み
                ILEFont font = TestUtility.LoadFont(layout, "sample2.ffnt");
                string partsName = "animation.flyt";
                ISubScene partsSubScene = TestUtility.LoadPartsLayout(partsName); // 部品をあらかじめ読み込んでおく

                // 各種ペインを取得
                IPane nullPane = layout.FindPaneByName("N_null_00");
                IPane picturePane = layout.FindPaneByName("P_pict_00");
                IPane textPane = layout.FindPaneByName("T_text_00");
                IPane windowPane = layout.FindPaneByName("W_window_00");
                IPane boundingPane = layout.FindPaneByName("B_bounding_00");
                IPane capturePane = layout.FindPaneByName("C_capture_00");
                IPane partsPane = layout.FindPaneByName("L_parts_00");

                // 新規ペインを作成
                IPane newNullPane = TestUtility.AddNullPane(layout, "newNullPane");
                IPane newPicturePane = TestUtility.AddPicturePane(layout, "newPicturePane", null);
                IPane newTextPane = TestUtility.AddTextBoxPane(layout, "newTextPane", font, "");
                IPane newWindowPane = TestUtility.AddWindowPane(layout, "newWindowPane");
                IPane newBoundingPane = TestUtility.AddBoudingPane(layout, "newBoundingPane");
                IPane newCapturePane = TestUtility.AddCapturePane(layout, "newCapturePane");
                IPane newPartsPane = TestUtility.AddPartsPane(layout, "newPartsPane", partsName, partsSubScene);

                // タイムスライダーを進める
                TestUtility.SetTime(100);

                // パラメータ貼り付け
                TestUtility.PastePaneParameter(nullPane, newNullPane);
                TestUtility.PastePaneParameter(picturePane, newPicturePane);
                TestUtility.PastePaneParameter(textPane, newTextPane);
                TestUtility.PastePaneParameter(windowPane, newWindowPane);
                TestUtility.PastePaneParameter(boundingPane, newBoundingPane);
                TestUtility.PastePaneParameter(capturePane, newCapturePane);
                TestUtility.PastePaneParameter(partsPane, newPartsPane);

                // ペインを比較
                TestUtility.AreEqualsPaneParameters(nullPane, newNullPane);
                TestUtility.AreEqualsPaneParameters(picturePane, newPicturePane);
                TestUtility.AreEqualsPaneParameters(textPane, newTextPane);
                TestUtility.AreEqualsPaneParameters(windowPane, newWindowPane);
                TestUtility.AreEqualsPaneParameters(boundingPane, newBoundingPane);
                TestUtility.AreEqualsPaneParameters(capturePane, newCapturePane);
                // (SIGLO-73477) カーブエディタでサイズが変更できてしまうため、結果が一致しない
                //TestUtility.AreEqualsPaneParameters(partsPane, newPartsPane);

                // タイムスライダーを戻す
                TestUtility.SetTime(0);

                // レイアウトを閉じる
                TestUtility.CloseLayout(layout);
            }

            // 分割モード 時間＝0
            {
                // レイアウトを読み込み
                ISubScene layout = TestUtility.LoadLayout("allPaneAnim_separated.flyt");

                // フォント、テクスチャ、部品の読み込み
                ILEFont font = TestUtility.LoadFont(layout, "sample2.ffnt");
                string partsName = "animation.flyt";
                ISubScene partsSubScene = TestUtility.LoadPartsLayout(partsName); // 部品をあらかじめ読み込んでおく

                // 各種ペインを取得
                IPane nullPane = layout.FindPaneByName("N_null_00");
                IPane picturePane = layout.FindPaneByName("P_pict_00");
                IPane textPane = layout.FindPaneByName("T_text_00");
                IPane windowPane = layout.FindPaneByName("W_window_00");
                IPane boundingPane = layout.FindPaneByName("B_bounding_00");
                IPane capturePane = layout.FindPaneByName("C_capture_00");
                IPane partsPane = layout.FindPaneByName("L_parts_00");

                // 新規ペインを作成
                IPane newNullPane = TestUtility.AddNullPane(layout, "newNullPane");
                IPane newPicturePane = TestUtility.AddPicturePane(layout, "newPicturePane", null);
                IPane newTextPane = TestUtility.AddTextBoxPane(layout, "newTextPane", font, "");
                IPane newWindowPane = TestUtility.AddWindowPane(layout, "newWindowPane");
                IPane newBoundingPane = TestUtility.AddBoudingPane(layout, "newBoundingPane");
                IPane newCapturePane = TestUtility.AddCapturePane(layout, "newCapturePane");
                IPane newPartsPane = TestUtility.AddPartsPane(layout, "newPartsPane", partsName, partsSubScene);

                // パラメータ貼り付け
                TestUtility.PastePaneParameter(nullPane, newNullPane);
                TestUtility.PastePaneParameter(picturePane, newPicturePane);
                TestUtility.PastePaneParameter(textPane, newTextPane);
                TestUtility.PastePaneParameter(windowPane, newWindowPane);
                TestUtility.PastePaneParameter(boundingPane, newBoundingPane);
                TestUtility.PastePaneParameter(capturePane, newCapturePane);
                TestUtility.PastePaneParameter(partsPane, newPartsPane);

                // ペインを比較
                TestUtility.AreEqualsPaneParameters(nullPane, newNullPane);
                TestUtility.AreEqualsPaneParameters(picturePane, newPicturePane);
                TestUtility.AreEqualsPaneParameters(textPane, newTextPane);
                TestUtility.AreEqualsPaneParameters(windowPane, newWindowPane);
                TestUtility.AreEqualsPaneParameters(boundingPane, newBoundingPane);
                TestUtility.AreEqualsPaneParameters(capturePane, newCapturePane);
                // (SIGLO-73477) カーブエディタでサイズが変更できてしまうため、結果が一致しない
                //TestUtility.AreEqualsPaneParameters(partsPane, newPartsPane);

                // レイアウトを閉じる
                TestUtility.CloseLayout(layout);
            }

            // 分割モード 時間≠0
            {
                // レイアウトを読み込み
                ISubScene layout = TestUtility.LoadLayout("allPaneAnim_separated.flyt");

                // フォント、テクスチャ、部品の読み込み
                // ファイルを読み込んだ時点で、タグが未選択状態になる。
                // レイアウトは、基本値(flyt に書かれたBaseValue)になっている。
                // 現在の時間がどんな値になっていようと値に影響がない。
                ILEFont font = TestUtility.LoadFont(layout, "sample2.ffnt");
                string partsName = "animation.flyt";
                ISubScene partsSubScene = TestUtility.LoadPartsLayout(partsName); // 部品をあらかじめ読み込んでおく

                // 各種ペインを取得
                IPane nullPane = layout.FindPaneByName("N_null_00");
                IPane picturePane = layout.FindPaneByName("P_pict_00");
                IPane textPane = layout.FindPaneByName("T_text_00");
                IPane windowPane = layout.FindPaneByName("W_window_00");
                IPane boundingPane = layout.FindPaneByName("B_bounding_00");
                IPane capturePane = layout.FindPaneByName("C_capture_00");
                IPane partsPane = layout.FindPaneByName("L_parts_00");

                // 新規ペインを作成
                IPane newNullPane = TestUtility.AddNullPane(layout, "newNullPane");
                IPane newPicturePane = TestUtility.AddPicturePane(layout, "newPicturePane", null);
                IPane newTextPane = TestUtility.AddTextBoxPane(layout, "newTextPane", font, "");
                IPane newWindowPane = TestUtility.AddWindowPane(layout, "newWindowPane");
                IPane newBoundingPane = TestUtility.AddBoudingPane(layout, "newBoundingPane");
                IPane newCapturePane = TestUtility.AddCapturePane(layout, "newCapturePane");
                IPane newPartsPane = TestUtility.AddPartsPane(layout, "newPartsPane", partsName, partsSubScene);

                // タイムスライダーを進める
                TestUtility.SetTime(100);

                // パラメータ貼り付け
                TestUtility.PastePaneParameter(nullPane, newNullPane);
                TestUtility.PastePaneParameter(picturePane, newPicturePane);
                TestUtility.PastePaneParameter(textPane, newTextPane);
                TestUtility.PastePaneParameter(windowPane, newWindowPane);
                TestUtility.PastePaneParameter(boundingPane, newBoundingPane);
                TestUtility.PastePaneParameter(capturePane, newCapturePane);
                TestUtility.PastePaneParameter(partsPane, newPartsPane);

                // ペインを比較
                TestUtility.AreEqualsPaneParameters(nullPane, newNullPane);
                TestUtility.AreEqualsPaneParameters(picturePane, newPicturePane);
                TestUtility.AreEqualsPaneParameters(textPane, newTextPane);
                TestUtility.AreEqualsPaneParameters(windowPane, newWindowPane);
                TestUtility.AreEqualsPaneParameters(boundingPane, newBoundingPane);
                TestUtility.AreEqualsPaneParameters(capturePane, newCapturePane);
                // (SIGLO-73477) カーブエディタでサイズが変更できてしまうため、結果が一致しない
                //TestUtility.AreEqualsPaneParameters(partsPane, newPartsPane);

                // タイムスライダーを戻す
                TestUtility.SetTime(0);

                // レイアウトを閉じる
                TestUtility.CloseLayout(layout);
            }

            // 後処理
            {
                TestUtility.FinalizeTest(MethodBase.GetCurrentMethod());
            }
        }

        #endregion

        #region ペインの「ロック」、「ロック解除」ができる

        /// <summary>
        /// ペインの「ロック」、「ロック解除」ができる
        /// </summary>
        [TestMethod]
        public void PaneTest_LockAndUnlock()
        {
            // 準備
            {
                TestUtility.InitializeTest(MethodBase.GetCurrentMethod());
            }

            // テスト
            {
                // レイアウトを新規作成
                ISubScene newSubScene = TestUtility.CreateLayout();

                // ペインを作成
                IPane newNullPane1 = TestUtility.AddPicturePane(newSubScene, "newNullPane1", null);
                IPane newNullPane2 = TestUtility.AddPicturePane(newSubScene, "newNullPane2", null);
                IPane newNullPane3 = TestUtility.AddPicturePane(newSubScene, "newNullPane3", null);

                // ペインをロックする
                TestUtility.LockPane(newNullPane1);
                TestUtility.LockPane(newNullPane2);
                TestUtility.LockPane(newNullPane3);

                // ロックされたペインの一覧を取得
                IPane[] paneSet = SubSceneHelper.GetLockedPaneSet(newSubScene);

                // 比較
                Assert.AreEqual(paneSet.Count(), 3);
                Assert.AreEqual(paneSet.FirstOrDefault(pane => pane.PaneName == "newNullPane1"), newNullPane1);
                Assert.AreEqual(paneSet.FirstOrDefault(pane => pane.PaneName == "newNullPane2"), newNullPane2);
                Assert.AreEqual(paneSet.FirstOrDefault(pane => pane.PaneName == "newNullPane3"), newNullPane3);

                // ペインをアンロックする
                TestUtility.UnLockPane(newNullPane1);
                TestUtility.UnLockPane(newNullPane2);
                TestUtility.UnLockPane(newNullPane3);

                // ロックされたペインの一覧を取得
                paneSet = SubSceneHelper.GetLockedPaneSet(newSubScene);

                // 比較
                Assert.AreEqual(paneSet.Count(), 0);
                Assert.AreEqual(paneSet.FirstOrDefault(pane => pane.PaneName == "newNullPane1"), null);
                Assert.AreEqual(paneSet.FirstOrDefault(pane => pane.PaneName == "newNullPane2"), null);
                Assert.AreEqual(paneSet.FirstOrDefault(pane => pane.PaneName == "newNullPane3"), null);

                // レイアウトを閉じる
                TestUtility.CloseLayout(newSubScene);
            }

            // 後処理
            {
                TestUtility.FinalizeTest(MethodBase.GetCurrentMethod());
            }
        }

        #endregion

        #region 派生部品の「作成」、「編集」、「更新」ができる

        /// <summary>
        /// 派生部品の「作成」、「編集」、「更新」ができる
        /// </summary>
        [TestMethod]
        public void PaneTest_DerivativePartsCreateEditAndUpdate()
        {
            string partsFileName = "___animation_copy___.flyt";
            string baseFullPath;

            // 準備
            {
                // 部品ルート設定前にファイルの削除、複製を行う必要がある
                // 前回のゴミが残っている場合は消す
                TestUtility.DeleteFile(partsFileName);

                // ファイルをコピーする
                baseFullPath = TestUtility.CopyFile("animation.flyt", partsFileName);

                TestUtility.InitializeTest(MethodBase.GetCurrentMethod());
            }

            // テスト
            {
                // 派生部品を作る
                ISubScene derivSubScene = TestUtility.CreateDerivativeParts(partsFileName);

                // 派生部品を保存して閉じる
                string derivFileName = "___animation_deriv___.flyt";
                string derivFullPath = TestUtility.SaveLayout(derivSubScene, derivFileName);
                TestUtility.CloseLayout(derivSubScene);

                // 基底部品を開く
                ISubScene baseSubScene = TestUtility.LoadLayout(baseFullPath);

                // 基底部品を編集する
                IPane basePane = baseSubScene.FindPaneByName("W_window_00");
                var testColor = new FloatColor(1.0f, 0.0f, 0.0f, 1.0f);
                TestUtility.ModifyMaterialColor(basePane, testColor);

                // 基底部品を保存して閉じる
                TestUtility.SaveLayoutFullPath(baseSubScene, baseFullPath);
                TestUtility.CloseLayout(baseSubScene);

                // 派生部品を開く
                derivSubScene = TestUtility.LoadLayoutFullPath(derivFullPath);

                // 派生部品の値を確認する
                IPane derivPane = derivSubScene.FindPaneByName("W_window_00");
                TestUtility.IsEqualMaterialColor(derivPane, testColor);

                // コピーしたファイルを削除
                TestUtility.DeleteFile(partsFileName);
            }

            // 後処理
            {
                TestUtility.FinalizeTest(MethodBase.GetCurrentMethod());
            }
        }

        #endregion
    }
}
