﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Threading.Tasks;

using Microsoft.VisualStudio.TestTools.UnitTesting;

using Nintendo.HtcTools.Htclow;

namespace Nintendo.HtcTools.Htclow
{
    [TestClass]
    public class ApiTest
    {
        PacketFactory m_Factory = new PacketFactory();

        #region クロース済リソースへのアクセス

        /// <summary>
        /// シャットダウン後に Connect を実行し、例外が投げられることを確認する。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(HtclowChannelException))]
        public void ConnectAfterShutdown()
        {
            var channel = new Channel() { ModuleId = 1, ChannelId = 1 };

            using (var service = new HtclowLibraryTestService())
            {
                var util = new HandshakeUtil(m_Factory, service);

                var htclowClient = service.HtclowClient;
                var htclowChannel = htclowClient.OpenChannel(channel.ModuleId, channel.ChannelId);

                util.DoHandshakeByHost(htclowChannel, channel);
                util.DoShutdownByHost(htclowChannel, channel);

                htclowChannel.Connect();
            }
        }

        /// <summary>
        /// クローズ後に Connect を実行し、例外が投げられることを確認する。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(HtclowChannelException))]
        public void ConnectAfterClose()
        {
            var channel = new Channel() { ModuleId = 1, ChannelId = 1 };

            using (var service = new HtclowLibraryTestService())
            {
                var util = new HandshakeUtil(m_Factory, service);

                var htclowClient = service.HtclowClient;
                var htclowChannel = htclowClient.OpenChannel(channel.ModuleId, channel.ChannelId);

                util.DoHandshakeByTarget(htclowChannel, channel);
                util.DoShutdownByTarget(htclowChannel, channel);

                htclowChannel.Dispose();

                htclowChannel.Connect();
            }
        }

        /// <summary>
        /// シャットダウン後に Send を実行し、例外が投げられることを確認する。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(HtclowChannelException))]
        public void SendAfterShutdown()
        {
            const int dataSize = 1;

            var channel = new Channel() { ModuleId = 2, ChannelId = 2 };

            using (var service = new HtclowLibraryTestService())
            {
                var util = new HandshakeUtil(m_Factory, service);

                var htclowClient = service.HtclowClient;
                var htclowChannel = htclowClient.OpenChannel(channel.ModuleId, channel.ChannelId);

                util.DoHandshakeByHost(htclowChannel, channel);
                util.DoShutdownByTarget(htclowChannel, channel);

                var buffer = DataUtil.MakeRandom(dataSize);
                htclowChannel.Send(buffer);
            }
        }

        /// <summary>
        /// クローズ後に Send を実行し、例外が投げられることを確認する。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(HtclowChannelException))]
        public void SendAfterClose()
        {
            const int dataSize = 16;

            var channel = new Channel() { ModuleId = 3, ChannelId = 3 };

            using (var service = new HtclowLibraryTestService())
            {
                var util = new HandshakeUtil(m_Factory, service);

                var htclowClient = service.HtclowClient;
                var htclowChannel = htclowClient.OpenChannel(channel.ModuleId, channel.ChannelId);

                util.DoHandshakeByTarget(htclowChannel, channel);
                util.DoShutdownByHost(htclowChannel, channel);

                htclowChannel.Dispose();

                var buffer = DataUtil.MakeRandom(dataSize);
                htclowChannel.Send(buffer);
            }
        }

        /// <summary>
        /// チャンネルをクローズ後に Receive を実行し、例外が投げられることを確認する。
        /// このとき、別のチャンネルを1つオープンした状態にしておく。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(HtclowChannelException))]
        public void ReceiveAfterCloseWithAnotherChannel()
        {
            const int receiveSize = 1;

            var channel1 = new Channel() { ModuleId = 1, ChannelId = 1 };
            var channel2 = new Channel() { ModuleId = 1, ChannelId = 2 };

            using (var service = new HtclowLibraryTestService())
            {
                var util = new HandshakeUtil(m_Factory, service);

                var htclowClient = service.HtclowClient;
                var htclowChannel1 = htclowClient.OpenChannel(channel1.ModuleId, channel1.ChannelId);
                var htclowChannel2 = htclowClient.OpenChannel(channel2.ModuleId, channel2.ChannelId);

                util.DoHandshakeByTarget(htclowChannel1, channel1);
                util.DoHandshakeByTarget(htclowChannel2, channel2);

                util.DoShutdownSimultaneously(htclowChannel1, channel1);

                htclowChannel1.Dispose();
                htclowChannel1.Receive(receiveSize);
            }
        }

        #endregion

        #region 多重クローズ

        /// <summary>
        /// 多重に Shutdown や Close を呼んでも例外が投げられないことを確認する
        /// </summary>
        [System.Diagnostics.CodeAnalysis.SuppressMessage("Microsoft.Usage", "CA2202")]
        [TestMethod]
        public void MultipleClose()
        {
            var channel = new Channel() { ModuleId = 0, ChannelId = 0 };

            using (var service = new HtclowLibraryTestService())
            {
                var util = new HandshakeUtil(m_Factory, service);

                var htclowClient = service.HtclowClient;
                var htclowChannel = htclowClient.OpenChannel(channel.ModuleId, channel.ChannelId);

                util.DoHandshakeByHost(htclowChannel, channel);
                util.DoShutdownSimultaneously(htclowChannel, channel);

                htclowChannel.Dispose();
                htclowChannel.Dispose();
                htclowChannel.Shutdown();
                htclowChannel.Dispose();
                htclowChannel.Shutdown();
            }
        }

        #endregion
    }
}
