﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace DebugBridgeTest
{
    using System;
    using System.Linq;
    using System.Threading;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using Nintendo.Hid;
    using Nintendo.Htcs;

    [TestClass]
    public class DebugBridgeTest
    {
        private HtcsCommunicator htcsCommunicator = new HtcsCommunicator();
        private PortMapItem portMapItem = null;
        private object syncObject = new object();
        private AutoResetEvent autoResetEvent = new AutoResetEvent(false);

        [TestInitialize]
        public void InitializeMethod()
        {
            this.htcsCommunicator.HtcsInfoUpdated += this.OnPortMapUpdated;
            this.htcsCommunicator.Start();
            this.autoResetEvent.WaitOne();
        }

        [TestCleanup]
        public void CleanupMethod()
        {
            this.htcsCommunicator.Stop();
            this.htcsCommunicator.HtcsInfoUpdated -= this.OnPortMapUpdated;
            this.autoResetEvent.Reset();
        }

        [TestMethod]
        public void ClientTestMethod()
        {
            var client = new DebugBridgeClient();
            lock (this.syncObject)
            {
                Assert.IsNotNull(this.portMapItem);
                if (this.portMapItem == null)
                {
                    return;
                }

                client.Connect(this.portMapItem);
            }

            client.GetDebugPad().PressButtons(DebugPadButton.A);
            client.GetDebugPad().PressButtons(DebugPadButton.X);
            client.GetDebugPad().PressButtons(DebugPadButton.ZL, DebugPadButton.R);
            client.GetDebugPad().PressButtons(DebugPadButton.Start);
            client.GetDebugPad().PressButtons(DebugPadButton.Select);
            client.GetDebugPad().PressButtons(DebugPadButton.Right);
            client.GetDebugPad().PressButtons(DebugPadButton.Down);
            client.Update();
            Assert.AreEqual(0, client.GetDebugPad().GetButtonsStates().Length);
            Assert.AreEqual(0, client.GetDebugPad().GetAnalogStickLStates().Length);
            Assert.AreEqual(0, client.GetDebugPad().GetAnalogStickRStates().Length);
            Thread.Sleep(2000);

            client.GetDebugPad().MonitorAnalogStickL();
            client.GetDebugPad().ReleaseButtons(DebugPadButton.A);
            client.GetDebugPad().PressButtons(DebugPadButton.B);
            client.GetDebugPad().ReleaseButtons(DebugPadButton.X);
            client.GetDebugPad().PressButtons(DebugPadButton.Y);
            client.GetDebugPad().ReleaseButtons(DebugPadButton.ZL, DebugPadButton.R);
            client.GetDebugPad().PressButtons(DebugPadButton.L, DebugPadButton.ZR);
            client.GetDebugPad().ReleaseButtons(DebugPadButton.Select);
            client.GetDebugPad().PressButtons(DebugPadButton.Left);
            client.GetDebugPad().PressButtons(DebugPadButton.Up);
            client.GetDebugPad().ReleaseButtons(DebugPadButton.Right);
            client.GetDebugPad().ReleaseButtons(DebugPadButton.Down);
            client.GetDebugPad().SetAnalogStickL(100, -100);
            client.Update();
            DebugPadAnalogStickState[] debugPadAnalogStickLStates = client.GetDebugPad().GetAnalogStickLStates();
            Assert.AreEqual(0, client.GetDebugPad().GetButtonsStates().Length);
            Assert.AreEqual(0, client.GetDebugPad().GetAnalogStickRStates().Length);
            Assert.AreEqual(1, debugPadAnalogStickLStates.Length);
            Assert.AreEqual(100, debugPadAnalogStickLStates[0].X);
            Assert.AreEqual(-100, debugPadAnalogStickLStates[0].Y);
            Thread.Sleep(2000);

            client.GetDebugPad().MonitorButtons();
            client.GetDebugPad().MonitorAnalogStickR();
            client.GetDebugPad().UnmonitorAnalogStickL();
            client.GetDebugPad().ClearButtons();
            client.GetDebugPad().ClearAnalogStickL();
            client.GetDebugPad().SetAnalogStickR(200, -200);
            client.Update();
            DebugPadButtonsState[] debugPadButtonsStates = client.GetDebugPad().GetButtonsStates();
            DebugPadAnalogStickState[] debugPadAnalogStickRStates = client.GetDebugPad().GetAnalogStickRStates();
            Assert.AreEqual(0, client.GetDebugPad().GetAnalogStickLStates().Length);
            Assert.AreEqual(1, debugPadButtonsStates.Length);
            Assert.IsFalse(debugPadButtonsStates[0].GetButton(DebugPadButton.A));
            Assert.IsTrue(debugPadButtonsStates[0].GetButton(DebugPadButton.B));
            Assert.IsFalse(debugPadButtonsStates[0].GetButton(DebugPadButton.X));
            Assert.IsTrue(debugPadButtonsStates[0].GetButton(DebugPadButton.Y));
            Assert.IsTrue(debugPadButtonsStates[0].GetButton(DebugPadButton.L));
            Assert.IsFalse(debugPadButtonsStates[0].GetButton(DebugPadButton.R));
            Assert.IsFalse(debugPadButtonsStates[0].GetButton(DebugPadButton.ZL));
            Assert.IsTrue(debugPadButtonsStates[0].GetButton(DebugPadButton.ZR));
            Assert.IsTrue(debugPadButtonsStates[0].GetButton(DebugPadButton.Start));
            Assert.IsFalse(debugPadButtonsStates[0].GetButton(DebugPadButton.Select));
            Assert.IsTrue(debugPadButtonsStates[0].GetButton(DebugPadButton.Left));
            Assert.IsTrue(debugPadButtonsStates[0].GetButton(DebugPadButton.Up));
            Assert.IsFalse(debugPadButtonsStates[0].GetButton(DebugPadButton.Right));
            Assert.IsFalse(debugPadButtonsStates[0].GetButton(DebugPadButton.Down));
            Assert.AreEqual(1, debugPadAnalogStickRStates.Length);
            Assert.AreEqual(200, debugPadAnalogStickRStates[0].X);
            Assert.AreEqual(-200, debugPadAnalogStickRStates[0].Y);

            client.Close();
        }

        private void OnPortMapUpdated(object sender, HtcsInfoUpdatedEventArgs e)
        {
            PortMapItem portMapItem = e.PortMap.FirstOrDefault(x => x.HtcsPortDescriptor.HtcsPortName == DebugBridgeClient.HtcsPortName);
            if (portMapItem != null)
            {
                lock (this.syncObject)
                {
                    this.portMapItem = portMapItem;
                }

                this.autoResetEvent.Set();
            }
        }
    }
}
