﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using Nintendo.FsAccessLogAnalysis;

namespace FsAccessLogCheckerTest
{
    internal static class FsAccessLogCheckerExecutionTestDataGeneratorStringExtension
    {
        public static string StripLeft(this string value)
        {
            return value.TrimStart(new char[] { '\r', '\n' });
        }
    }

    internal class FsAccessLogCheckerExecutionTestParam
    {
        public ulong Time { get; set; }
        public ulong Size { get; set; }
        public int Count { get; set; }
        public string Fs { get; set; }
        public string FsOption { get; set; }

        public FsAccessLogCheckerExecutionTestParam(ulong time, ulong size, int count, string fs, string fsOption)
        {
            Time = time;
            Size = size;
            Count = count;
            Fs = fs;
            FsOption = fsOption;
        }
    }

    internal class FsAccessLogCheckerExecutionTestDataGeneratorCommon
    {
        // 解析時のウィンドウ幅 (ミリ秒)
        public static readonly ulong WindowTime = 10 * 60 * 1000;

        public static readonly int HandleIndexOriginal = 0;
        public static readonly int HandleIndexSameOnDifferentHandle = 1;
        public static readonly int HandleIndexDifferentOnSameMount = 2;
        public static readonly int HandleIndexSameOnDifferentMount = 3;
        public static readonly int HandleIndexDifferentOnDifferentMount = 4;

        // アクセスするファイル ハンドルの定数
        public static readonly string[] HandleMount1 = new string[]
        {
            "0x00000000016cd678",
            "0x00000000026cd679", // 同じファイル
            "0x00000000036cd67A"  // 異なるファイル
        };

        public static readonly string[] HandleMount2 = new string[]
        {
            "0x00000000046cd67B"  // 異なる名前でのマウントで同じファイル
        };

        public static readonly string[] HandleMount3 = new string[]
        {
            "0x00000000056cd67C"  // 異なる名前でのマウントで異なるファイル
        };

        public static readonly string[] Handles = HandleMount1
            .Concat(HandleMount2)
            .Concat(HandleMount3).ToArray();

        public static readonly string[] HandlePathMount1 = new string[]
        {
            "root:/hogehoge",
            "root:/hogehoge",
            "root:/fugafuga",
        };

        public static readonly string[] HandlePathMount2 = new string[]
        {
            "root2:/hogehoge",
        };

        public static readonly string[] HandlePathMount3 = new string[]
        {
            "root3:/hogehoge",
        };

        public static string Version()
        {
            return @"
FS_ACCESS: { sdk_version: 1.0.0, spec: NX }
".StripLeft();
        }

        public static string VersionForSystem()
        {
            return @"
FS_ACCESS: { sdk_version: 3.0.0, spec: NX, for_system: true }
".StripLeft();
        }

        // バージョン出力、マウント処理、ファイルオープン処理
        public static string MountAndOpen(ulong finish, string fs, string fsOption, Func<int> handle)
        {
            var val = string.Empty;

            var fs2 = string.Empty;
            var fs2Option = string.Empty;
            if (fs == "Rom")
            {
                fs2 = "SaveData";
                fs2Option = ", userid: 0xFEDCBA98765432100123456789ABCDEF";
            }
            else
            {
                fs2 = "Rom";
            }

            var mount = "root";

            var handles = HandleMount1;
            var handlePath = HandlePathMount1;

            if (handle.Invoke() == HandleIndexSameOnDifferentMount)
            {
                mount = "root2";
                handles = HandleMount2;
                handlePath = HandlePathMount2;
            }

            handles = handles.Concat(HandleMount3).ToArray();
            handlePath = handlePath.Concat(HandlePathMount3).ToArray();

            val += string.Format(@"
FS_ACCESS: {{ start:   {0}, end:   {0}, result: 0x00000000, handle: 0x0000000000000000, function: Mount{1}, name: ""{2}"" {3} }}
FS_ACCESS: {{ start:   {0}, end:   {0}, result: 0x00000000, handle: 0x0000000000000000, function: Mount{4}, name: ""root3"" {5} }}
", finish, fs, mount, fsOption, fs2, fs2Option).StripLeft();

            int i = 0;
            foreach (var handleValue in handles)
            {
                val += string.Format(@"
FS_ACCESS: {{ start:   {0}, end:   {0}, result: 0x00000000, handle: {1}, function: OpenFile , path: ""{2}"" }}
", finish, handleValue, handlePath[i]).StripLeft();
                ++i;
            }

            return val;
        }

        public static string CloseAndUnmount(ulong finish, Func<int> handle)
        {
            var val = string.Empty;
            var mount = "root";
            var handles = HandleMount1;
            var handlePath = HandlePathMount1;

            if (handle.Invoke() == HandleIndexSameOnDifferentMount)
            {
                mount = "root2";
                handles = HandleMount2;
                handlePath = HandlePathMount2;
            }

            handles = handles.Concat(HandleMount3).ToArray();
            handlePath = handlePath.Concat(HandlePathMount3).ToArray();

            foreach (var handleValue in handles)
            {
                val += string.Format(@"
FS_ACCESS: {{ start:   {0}, end: {0}, result: 0x00000000, handle: {1}, function: CloseFile }}
", finish, handleValue).StripLeft();
            }

            val += string.Format(@"
FS_ACCESS: {{ start:   {0}, end: {0}, result: 0x00000000, handle: 0x0000000000000000, function: Unmount, name: ""{1}"" }}
FS_ACCESS: {{ start:   {0}, end: {0}, result: 0x00000000, handle: 0x0000000000000000, function: Unmount, name: ""root3"" }}
", finish, mount).StripLeft();

            return val;
        }

        public static string MountAndOpenSingle(string fs)
        {
            var val = string.Empty;

            val += string.Format(@"
FS_ACCESS: {{ start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, function: Mount{0} , name: ""root"" }}
", fs).StripLeft();

            int i = 0;
            foreach (var handleValue in HandleMount1)
            {
                val += string.Format(@"
FS_ACCESS: {{ start: 0, end: 0, result: 0x00000000, handle: {0}, function: OpenFile , path: ""{1}"" }}
", handleValue, HandlePathMount1[i]).StripLeft();
                ++i;
            }

            return val;
        }

        public static string CloseAndUnmountSingle(ulong finish)
        {
            var val = string.Empty;

            foreach (var handleValue in HandleMount1)
            {
                val += string.Format(@"
FS_ACCESS: {{ start:   {0}, end:   {0}, result: 0x00000000, handle: {1}, function: CloseFile }}
", finish, handleValue).StripLeft();
            }

            val += string.Format(@"
FS_ACCESS: {{ start:   {0}, end: {0}, result: 0x00000000, handle: 0x0000000000000000, function: Unmount, name: ""root"" }}
", finish).StripLeft();

            return val;
        }

        // ファイル読み書きデータの出力
        // path   : 出力先ファイルパス
        // params : 以下のキーを持つハッシュの配列 (順に実行される)
        //   :time  - 読み書き実行時間
        //   :size  - 読み書きサイズ
        //   :count - 読み書き回数
        // fs     : MountXXXX の XXXX の部分
        // op     : ReadFile または WriteFile
        // handle : op の対象のハンドルの識別番号 (0, 1, ...)
        // param  : 上記のハッシュとアクセス順序を受けとりファイル アクセスの offset と size を返すブロック
        public static void GenerateReadWriteMulti(
            string path,
            FsAccessLogCheckerExecutionTestParam[] testParams,
            Func<string> op,
            Func<int> handle,
            Func<FsAccessLogCheckerExecutionTestParam, int, Tuple<ulong, ulong>> param,
            Func<string> version)
        {
            ulong startOffset = 0;
            using (var file = new StreamWriter(File.OpenWrite(path)))
            {
                file.Write(version());
                foreach (var testParam in testParams)
                {
                    var time = testParam.Time;
                    var threshold = testParam.Size;
                    var count = testParam.Count;
                    var fs = testParam.Fs;
                    var fsOption = testParam.FsOption;

                    for (int i = 0; i < count; i++)
                    {
                        var start = startOffset + time / (ulong)count * (ulong)i;
                        var finish = startOffset + (i + 1 == count ? time : time / (ulong)count * ((ulong)i + 1));
                        var handleIndex = handle.Invoke();
                        var ret = param.Invoke(testParam, i);
                        var offset = ret.Item1;
                        var size = ret.Item2;

                        if (i == 0)
                        {
                            file.Write(MountAndOpen(startOffset, fs, fsOption, delegate () { return handleIndex; }));
                        }

                        file.Write(
                            string.Format("FS_ACCESS: {{ start:   {0}, end: {1}, result: 0x00000000, handle: {2}, function: {3}, offset: {4}, size: {5} }}\r\n",
                            start, finish, Handles[handleIndex], op.Invoke(), offset, size));

                        if (i == count - 1)
                        {
                            file.Write(CloseAndUnmount(startOffset + time, delegate () { return handleIndex; }));
                        }
                    }

                    startOffset += time;
                }
            }
        }

        public static void GenerateReadWriteMulti(
            string path,
            FsAccessLogCheckerExecutionTestParam[] testParams,
            Func<string> op,
            Func<int> handle,
            Func<FsAccessLogCheckerExecutionTestParam, int, Tuple<ulong, ulong>> param)
        {
            GenerateReadWriteMulti(path, testParams, op, handle, param, Version);
        }

        // ファイル読み書きデータの出力
        // path      : 出力先ファイルパス
        // time      : 読み書き実行時間
        // threshold : 読み書きサイズ
        // count     : 読み書き回数
        // fs        : MountXXXX の XXXX の部分
        // op        : ReadFile または WriteFile
        // handle : op の対象のハンドルの識別番号 (0, 1, ...)
        // param     : generate_read_write_multi のハッシュとアクセス順序を受けとりファイル アクセスの offset と size を返すブロック
        public static void GenerateReadWrite(
            string path,
            ulong time,
            ulong size,
            int count,
            string fs,
            string fsOption,
            Func<string> op,
            Func<int> handle,
            Func<FsAccessLogCheckerExecutionTestParam, int, Tuple<ulong, ulong>> param)
        {
            GenerateReadWriteMulti(
                path,
                new FsAccessLogCheckerExecutionTestParam[]
                {
                    new FsAccessLogCheckerExecutionTestParam(time, size, count, fs, fsOption)
                },
                op,
                handle,
                delegate (FsAccessLogCheckerExecutionTestParam testParam, int index)
                {
                    return param.Invoke(testParam, index);
                });
        }

        // ファイルの重ならない領域をアクセスするデータを出力
        public static Tuple<ulong, ulong> MakeDifferentAddress(ulong threshold, int count, int i)
        {
            var size = threshold / (ulong)count;
            var offset = size * (ulong)i;
            if (i == count - 1)
            {
                size += threshold - size * (ulong)count;
            }
            return new Tuple<ulong, ulong>(offset, size);
        }

        public static void GenerateDifferentAddress(
            string path,
            ulong time,
            ulong threshold,
            int count,
            string fs,
            string fsOption,
            Func<string> op,
            Func<int> handle)
        {
            GenerateReadWrite(
                path, time, threshold, count, fs, fsOption, op, handle,
                delegate (FsAccessLogCheckerExecutionTestParam testParam, int index)
                {
                    return MakeDifferentAddress(testParam.Size, testParam.Count, index);
                });
        }

        public static void GenerateDifferentAddressMulti(
            string path,
            FsAccessLogCheckerExecutionTestParam[] testParams,
            Func<string> op,
            Func<int> handle)
        {
            GenerateReadWriteMulti(
                path, testParams, op, handle,
                delegate (FsAccessLogCheckerExecutionTestParam testParam, int index)
                {
                    return MakeDifferentAddress(testParam.Size, testParam.Count, index);
                });
        }

        // ファイルの同じ領域をアクセスするデータを出力
        public static Tuple<ulong, ulong> MakeSameAddress(ulong threshold, int count, int i)
        {
            var size = threshold / (ulong)count;
            ulong offset = 0;
            if (i == count - 1)
            {
                size += threshold - size * (ulong)count;
            }
            return new Tuple<ulong, ulong>(offset, size);
        }

        public static void GenerateSameAddress(
            string path,
            ulong time,
            ulong threshold,
            int count,
            string fs,
            string fsOption,
            Func<string> op,
            Func<int> handle)
        {
            GenerateReadWrite(
                path, time, threshold, count, fs, fsOption, op, handle,
                delegate (FsAccessLogCheckerExecutionTestParam testParam, int index)
                {
                    return MakeSameAddress(testParam.Size, testParam.Count, index);
                });
        }

        public static void GenerateSameAddressMulti(
            string path,
            FsAccessLogCheckerExecutionTestParam[] testParams,
            Func<string> op,
            Func<int> handle)
        {
            GenerateReadWriteMulti(
                path, testParams, op, handle,
                delegate (FsAccessLogCheckerExecutionTestParam testParam, int index)
                {
                    return MakeSameAddress(testParam.Size, testParam.Count, index);
                });
        }

        // ファイルの一部分だけ重なる領域をアクセスするデータを出力
        public static Tuple<ulong, ulong> MakeOverlapAddress(ulong threshold, int count, int i)
        {
            var size = threshold / (ulong)count;
            ulong offset = size / 2 * (ulong)i;
            if (i == count - 1)
            {
                size += threshold - size * (ulong)count;
            }
            return new Tuple<ulong, ulong>(offset, size);
        }

        public static void GenerateOverlapAddress(
            string path,
            ulong time,
            ulong threshold,
            int count,
            string fs,
            string fsOption,
            Func<string> op,
            Func<int> handle)
        {
            GenerateReadWrite(
                path, time, threshold, count, fs, fsOption, op, handle,
                delegate (FsAccessLogCheckerExecutionTestParam testParam, int index)
                {
                    return MakeOverlapAddress(testParam.Size, testParam.Count, index);
                });
        }

        // マウントとアンマウントを繰りかえすデータを出力
        // path   : 出力先ファイルパス
        // time   : 読み書き実行時間
        // count  : 読み書き回数
        // op     : MountXXXX または Unmount
        // name   : マウント名
        public static void GenerateMountUnmount(
            string path,
            ulong time,
            int count,
            Func<Tuple<string, string>> op,
            Func<string> name)
        {
            ulong start = 0;
            ulong finish = 0;

            using (var file = new StreamWriter(File.OpenWrite(path)))
            {
                file.Write(Version());
                for (int i = 0; i < count; i++)
                {
                    start = time / (ulong)count * (ulong)i;
                    finish = (i + 1 == count ? time : time / (ulong)count * ((ulong)i + 1));
                    var handle = "0x0000000000000000";
                    var opCurrent = op.Invoke();

                    file.Write(string.Format(
                        "FS_ACCESS: {{ start:   {0}, end: {1}, result: 0x00000000, handle: {2}, function: {3}, name: \"{4}\" {5} }}\r\n",
                        start, finish, handle, opCurrent.Item1, name.Invoke(), opCurrent.Item2));
                }
            }
        }

        // マウントとアンマウントの途中にファイル読出しを行うデータを出力
        // path   : 出力先ファイルパス
        // time   : マウント/アンマウント実行時間
        // count  : マウント/アンマウント実行回数 (ペアで一回とカウント)
        // fs     : MountXXXX の XXXX の部分
        // name   : マウント名
        public static void GenerateMountUnmountWithReadDifferentAddress(
            string path,
            ulong time,
            int count,
            string fs,
            Func<string> name)
        {
            ulong start = 0;
            ulong finish = 0;

            using (var file = new StreamWriter(File.OpenWrite(path)))
            {
                file.Write(Version());
                file.Write(MountAndOpenSingle(fs));

                // ヘッダー、フッターでの Mount/Unmount を除く
                for (int i = 0; i < count - 1; i++)
                {
                    var period = time / (ulong)count / 2;
                    var handle = "0x0000000000000000";

                    // Mount と Unmount
                    start = period * 2 * (ulong)i;
                    finish = period * (2 * (ulong)i + 1);

                    file.Write(string.Format(
                        "FS_ACCESS: {{ start:   {0}, end: {1}, result: 0x00000000, handle: {2}, function: Mount{3}, name: \"{4}\" }}\r\n",
                        start, finish, handle, fs, name.Invoke()));
                    start = period * (2 * (ulong)i + 1);
                    finish = (i + 1 == count - 1 ? time : period * (2 * (ulong)i + 2));

                    file.Write(string.Format(
                        "FS_ACCESS: {{ start:   {0}, end: {1}, result: 0x00000000, handle: {2}, function: Unmount, name: \"{3}\" }}\r\n",
                        start, finish, handle, name.Invoke()));

                    // Read
                    handle = Handles[0];
                    start = finish;
                    var rwParam = MakeDifferentAddress(10 * 1024 * 1024, count, i);

                    file.Write(string.Format(
                        "FS_ACCESS: {{ start:   {0}, end: {1}, result: 0x00000000, handle: {2}, function: ReadFile, offset: {3}, size: {4} }}\r\n",
                        start, finish, handle, rwParam.Item1, rwParam.Item2));
                }

                file.Write(CloseAndUnmountSingle(finish));
            }
        }
    }

    public class OutputFileUtility
    {
        public static string ReadAllTextExceptFirstLine(string path)
        {
            StreamReader reader = new StreamReader(path);
            reader.ReadLine();
            string text = reader.ReadToEnd();
            reader.Close();
            return text;
        }
        public static string ReadFirstLine(string path)
        {
            if (!File.Exists(path))
            {
                return null;
            }
            StreamReader reader = new StreamReader(path);
            string text = reader.ReadLine();
            reader.Close();
            return text;
        }
        public static void OverwriteFirstLine(string path, string firstLine)
        {
            if (firstLine != null)
            {
                string text = ReadAllTextExceptFirstLine(path);
                StreamWriter writer = new StreamWriter(path);
                writer.WriteLine(firstLine);
                writer.Write(text);
                writer.Close();
            }
        }
    }

    [TestClass]
    public class FsAccessLogCheckerExecutionTest
    {
        public TestContext TestContext { get; set; }

        public string TestTempDirectory { get; set; }

        private string GetTestDataDirectory()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            return testPath.GetSigloRoot() + @"\Tests\Tools\Sources\Tests\FsAccessLogCheckerTest\FsAccessLogCheckerTest";
        }

        private int ExecuteProgram(string args, out string message, out string errorMessage)
        {
            Process process = new Process();
            TestUtility.TestPath testPath = new TestUtility.TestPath(this.TestContext);

#if DEBUG
            process.StartInfo.FileName = testPath.GetSigloRoot() + @"\Programs\Eris\Outputs\AnyCPU\Tools\FsAccessLogChecker\FsAccessLogChecker\Debug\FsAccessLogChecker.exe";
#else
            process.StartInfo.FileName = testPath.GetSigloRoot() + @"\Tools\FsAccessLogChecker\FsAccessLogChecker.exe";
#endif
            process.StartInfo.Arguments = args;
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardError = true;
            process.StartInfo.RedirectStandardOutput = true;
            process.Start();

            message = process.StandardOutput.ReadToEnd();
            errorMessage = process.StandardError.ReadToEnd();
            process.WaitForExit();
            return process.ExitCode;
        }

        public class TestDestFile
        {
            public TestDestFile(string path, string expectPath)
            {
                if (CheckMode)
                {
                    FilePath = path;
                }
                else
                {
                    string direcotryPath = Path.GetDirectoryName(expectPath);
                    if (!File.Exists(direcotryPath))
                    {
                        Directory.CreateDirectory(direcotryPath);
                    }
                    FilePath = expectPath;
                    FirstLine = OutputFileUtility.ReadFirstLine(expectPath);
                }
                if (File.Exists(FilePath))
                {
                    File.Delete(FilePath);
                }
            }

            public void UpdateForExpect()
            {
                if (!CheckMode)
                {
                    OutputFileUtility.OverwriteFirstLine(FilePath, FirstLine);
                }
            }
            public string FilePath { get; private set; }
            private string FirstLine { get; set; }
        }

        private void TestInDirectory(string name)
        {
            TestInDirectoryWithOption(name, string.Empty);
        }
        private void TestInDirectoryWithOption(string name, string options)
        {
            var testDataDirectory = GetTestDataDirectory() + @"\" + name;
            var testDestinationDirectory = TestTempDirectory;
            foreach (string path in Directory.EnumerateFiles(testDestinationDirectory, "test_*.txt"))
            {
                var testSourceFile = path;
                var testExpectFile = testDataDirectory + @"\" + Path.GetFileName(path).Replace(@"test_", @"expect_");
                string testDestFile = testDestinationDirectory + @"\" + Path.GetFileName(path).Replace("test_", "out_");
                TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
                testDestFile = destFile.FilePath;

                string message;
                string errorMessage;
                ExecuteProgram(testSourceFile + " -o " + testDestFile + " --quiet " + options,
                    out message, out errorMessage);
                destFile.UpdateForExpect();
                Assert.AreEqual(string.Empty, errorMessage);
                Assert.AreEqual(
                    OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                    OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile),
                    testSourceFile + " -> " + testDestFile + "\n != " + testExpectFile);
            }
        }

        public static void DeleteReadOnlyFile(string path)
        {
            try
            {
                FileInfo fileInfo = new FileInfo(path);
                fileInfo.IsReadOnly = false;
                File.Delete(path);
            }
            catch
            {
            }
        }

        [TestInitialize]
        public void TestInitialize()
        {
            FsAccessLogCheckerTestUtility.Initialize();
            TestTempDirectory = TestUtility.TestPath.CreateTemporaryDirectory("SIGLO_FSACCESS_TEST");
        }

        [TestCleanup]
        public void TestCleanup()
        {
            TestUtility.TestPath.DeleteDirectoryIfExisted(TestTempDirectory);
        }

        [TestMethod]
        public void TestExecutionNoOption()
        {
            string message;
            string errorMessage;
            Assert.AreEqual(1, ExecuteProgram(string.Empty, out message, out errorMessage));
            Assert.IsTrue(message.Contains("Options"));
            Assert.AreEqual(string.Empty, errorMessage);
        }

        [TestMethod]
        public void TestExecutionWithFile()
        {
            var testSourceFile = GetTestDataDirectory() + @"\Log1\log.txt";

            string message;
            string errorMessage;
            Assert.AreEqual(0, ExecuteProgram(testSourceFile, out message, out errorMessage));
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
        }

        [TestMethod]
        public void TestExecutionWithReadOnlyFile()
        {
            var testSourceFile = GetTestDataDirectory() + @"\Log1\log.txt";
            var testReadOnlyFile = TestTempDirectory + @"\readonly.txt";

            File.Copy(testSourceFile, testReadOnlyFile, true);
            FileInfo fileInfo = new FileInfo(testReadOnlyFile);
            fileInfo.IsReadOnly = true;

            string message;
            string errorMessage;
            Assert.AreEqual(0, ExecuteProgram(testSourceFile, out message, out errorMessage));
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);

            fileInfo.IsReadOnly = false;
            File.Delete(testReadOnlyFile);
        }

        [TestMethod]
        public void TestExecutionWithReadOnlyOutputFile()
        {
            var testSourceFile = GetTestDataDirectory() + @"\Log1\log.txt";
            var testDestFile = TestTempDirectory + @"\readonly.txt";

            var stream = File.Create(testDestFile);
            stream.Close();
            FileInfo fileInfo = new FileInfo(testDestFile);
            fileInfo.IsReadOnly = true;

            string message;
            string errorMessage;
            Assert.AreEqual(1, ExecuteProgram(testSourceFile + " -f -o " + testDestFile, out message, out errorMessage));
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.IsTrue(message.Contains(testDestFile));

            fileInfo.IsReadOnly = false;
            File.Delete(testDestFile);
        }

        [TestMethod]
        public void TestExecutionWithOpenedFile()
        {
            var testSourceFile = GetTestDataDirectory() + @"\Log1\log.txt";
            using (FileStream stream = new FileStream(testSourceFile, FileMode.Open, FileAccess.Read, FileShare.Read))
            {
                string message;
                string errorMessage;
                Assert.AreEqual(0, ExecuteProgram(testSourceFile, out message, out errorMessage));
                Assert.AreNotEqual(string.Empty, message);
                Assert.AreEqual(string.Empty, errorMessage);
            }

            using (FileStream stream = new FileStream(testSourceFile, FileMode.Open))
            {
                string message;
                string errorMessage;
                // 単純にオープンした場合は ReadWrite アクセスなので開けないはず
                Assert.AreEqual(1, ExecuteProgram(testSourceFile, out message, out errorMessage));
                Assert.AreNotEqual(string.Empty, message);
                Assert.AreEqual(string.Empty, errorMessage);
                Assert.IsTrue(message.Contains(testSourceFile));
            }
        }

        [TestMethod]
        public void TestExecutionWithOpenedOutputFile()
        {
            var testSourceFile = GetTestDataDirectory() + @"\Log1\log.txt";
            var testDestFile = TestTempDirectory + @"\out.txt";
            using (FileStream stream = new FileStream(testDestFile, FileMode.OpenOrCreate))
            {
                string message;
                string errorMessage;
                Assert.AreEqual(1, ExecuteProgram(testSourceFile + " -f -o " + testDestFile, out message, out errorMessage));
                Assert.AreNotEqual(string.Empty, message);
                Assert.AreEqual(string.Empty, errorMessage);
                Assert.IsTrue(message.Contains(testDestFile));
            }
        }

        [TestMethod]
        public void TestExecutionWithInvalidFile()
        {
            string[] testSourceFiles = new string[]
            {
                GetTestDataDirectory() + @"\Log1\log?.txt",
                GetTestDataDirectory() + new string('a', 260) + ".txt"
            };
            foreach (var testSourceFile in testSourceFiles)
            {
                string message;
                string errorMessage;
                Assert.AreEqual(1, ExecuteProgram(testSourceFile, out message, out errorMessage));
                Assert.AreNotEqual(string.Empty, message);
                Assert.AreEqual(string.Empty, errorMessage);
                Assert.IsTrue(message.Contains(testSourceFile));
            }
        }

        [TestMethod]
        public void TestExecutionWithInvalidOutputFile()
        {
            var testSourceFile = GetTestDataDirectory() + @"\Log1\log.txt";
            string[] testDestFiles = new string[]
            {
                GetTestDataDirectory() + @"\Log1\log?.txt",
                GetTestDataDirectory() + new string('a', 260) + ".txt"
            };
            foreach (var testDestFile in testDestFiles)
            {
                string message;
                string errorMessage;
                Assert.AreEqual(1, ExecuteProgram(testSourceFile + " -o " + testDestFile, out message, out errorMessage));
                Assert.AreNotEqual(string.Empty, message);
                Assert.AreEqual(string.Empty, errorMessage);
                Assert.IsTrue(message.Contains(testDestFile));
            }
        }

        [TestMethod]
        public void TestExecutionParserErrorLog()
        {
            var testSourceFile = GetTestDataDirectory() + @"\Log1\parser_error_log.txt";

            string message;
            string errorMessage;
            // 終了コードは 1
            Assert.AreEqual(1, ExecuteProgram(testSourceFile, out message, out errorMessage));
            Assert.AreNotEqual(string.Empty, message);
            // 全体結果の FAIL と Log Error の FAIL 2つが出力されていることを確認
            int index = message.IndexOf("[FAIL]");
            Assert.IsTrue(index >= 0);
            index = message.IndexOf("[FAIL]", index + 1);
            Assert.IsTrue(index >= 0);
            Assert.IsFalse(message.IndexOf("[FAIL]", index + 1) >= 0);
            Assert.IsTrue(message.IndexOf("Log Error", index) >= 0);
            Assert.AreEqual(string.Empty, errorMessage);
        }

        [TestMethod]
        public void TestExecutionWithFileAndOutputFile()
        {
            var testSourceFile = GetTestDataDirectory() + @"\Log1\log.txt";
            var testDestFile = TestTempDirectory + @"\out.txt";

            // 解析ファイル名 -o 出力ファイル
            {
                if (File.Exists(testDestFile))
                {
                    File.Delete(testDestFile);
                }

                string message;
                string errorMessage;
                Assert.AreEqual(0, ExecuteProgram(testSourceFile + " -o " + testDestFile, out message, out errorMessage));
                Assert.AreNotEqual(string.Empty, message);
                Assert.AreEqual(string.Empty, errorMessage);
                Assert.AreEqual(true, File.Exists(testDestFile));
                Assert.IsTrue(new FileInfo(testDestFile).Length > 0);
            }

            // -o 出力ファイル 解析ファイル名、も成功
            {
                if (File.Exists(testDestFile))
                {
                    File.Delete(testDestFile);
                }

                string message;
                string errorMessage;
                Assert.AreEqual(0, ExecuteProgram("-o " + testDestFile + " " + testSourceFile, out message, out errorMessage));
                Assert.AreNotEqual(string.Empty, message);
                Assert.AreEqual(string.Empty, errorMessage);
                Assert.AreEqual(true, File.Exists(testDestFile));
            }
        }

        [TestMethod]
        public void TestExecutionNoVersion()
        {
            var testSourceFile = GetTestDataDirectory() + @"\NoVersion\write.txt";
            var testExpectFile = GetTestDataDirectory() + @"\NoVersion\expect.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            // バージョンがないと失敗する
            Assert.AreEqual(1, ExecuteProgram(testSourceFile + " -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestExecutionOldVersion()
        {
            var testSourceFile = GetTestDataDirectory() + @"\OldVersion\write.txt";
            var testExpectFile = GetTestDataDirectory() + @"\OldVersion\expect.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            // バージョンが古いと失敗する
            Assert.AreEqual(1, ExecuteProgram(testSourceFile + " -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestExecutionShortTime()
        {
            var testSourceFile = GetTestDataDirectory() + @"\ShortTime\write.txt";
            var testExpectFile = GetTestDataDirectory() + @"\ShortTime\expect.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            // 時間が短いと失敗する
            Assert.AreEqual(1, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestExecutionNoMountAccess()
        {
            var testSourceFile = GetTestDataDirectory() + @"\NoMountAccess\write.txt";
            var testExpectFile = GetTestDataDirectory() + @"\NoMountAccess\expect.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            // マウントおよび OpenFile のログがないと Unknown ターゲットとなり、解析対象から外れる
            // そのため、終了コードは 0 になる
            Assert.AreEqual(0, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestExecutionMultiLogAnyFail()
        {
            var testSourceFile = GetTestDataDirectory() + @"\MultiLog\write_any_fail.txt";
            var testExpectFile = GetTestDataDirectory() + @"\MultiLog\expect_any_fail.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            // 複数のログのうち１つでも FAIL の場合は失敗する
            Assert.AreEqual(1, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestExecutionMultiLogAllPass()
        {
            var testSourceFile = GetTestDataDirectory() + @"\MultiLog\write_all_pass.txt";
            var testExpectFile = GetTestDataDirectory() + @"\MultiLog\expect_all_pass.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            // 複数のログですべて PASS の場合は警告出力だけして、成功する
            Assert.AreEqual(0, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestExecutionRewindTime()
        {
            var testSourceFile = GetTestDataDirectory() + @"\RewindTime\write.txt";
            var testExpectFile = GetTestDataDirectory() + @"\RewindTime\expect.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            // ソート対応により、巻戻りはなくなった
            Assert.AreEqual(0, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestExecutionHostAccessWarning()
        {
            var testSourceFile = GetTestDataDirectory() + @"\MountTarget\write.txt";
            Func<string, string> exec = (string option) =>
            {
                string message;
                string errorMessage;
                Assert.AreEqual(0, ExecuteProgram(testSourceFile + " --quiet " + option,
                    out message, out errorMessage));
                Assert.AreNotEqual(string.Empty, message);
                Assert.AreEqual(string.Empty, errorMessage);
                return message;
            };

            {
                string message = exec(string.Empty);
                Assert.IsTrue(message.Contains("[WARNING] Access to the HOST was detected"));
            }
        }

        [TestMethod]
        public void TestExecutionSystemAccessLogWarning()
        {
            var testSourceFile = GetTestDataDirectory() + @"\MountTarget\write.txt";
            Func<string, string> exec = (string option) =>
            {
                string message;
                string errorMessage;
                Assert.AreEqual(0, ExecuteProgram(testSourceFile + " --quiet " + option,
                    out message, out errorMessage));
                Assert.AreNotEqual(string.Empty, message);
                Assert.AreEqual(string.Empty, errorMessage);
                return message;
            };

            {
                string message = exec(string.Empty);
                Assert.IsTrue(message.Contains("[WARNING] System access log was detected,"));
                Assert.IsTrue(message.Contains("Please use NintendoSDK 3.0.0 or later"));
            }
        }

        [TestMethod]
        public void TestExecutionCheckMountTarget()
        {
            var testSourceFile = GetTestDataDirectory() + @"\MountTarget\write.txt";
            Func<string, string> exec = (string option) =>
            {
                string message;
                string errorMessage;
                Assert.AreEqual(0, ExecuteProgram(testSourceFile + " --quiet " + option,
                    out message, out errorMessage));
                Assert.AreNotEqual(string.Empty, message);
                Assert.AreEqual(string.Empty, errorMessage);
                return message;
            };

            // オプションなしの場合
            {
                string message = exec(string.Empty);
                Assert.IsFalse(Regex.IsMatch(message, @"\sUnknown"));
                Assert.IsFalse(Regex.IsMatch(message, @"\sHost"));
                foreach (FsMountTarget target in
                    Enum.GetValues(typeof(FsMountTarget)).Cast<FsMountTarget>().Except(new FsMountTarget[] { FsMountTarget.Unknown, FsMountTarget.Host, FsMountTarget.Nfp }))
                {
                    Assert.IsTrue(Regex.IsMatch(message, @"\s" + target.ToString()), target.ToString());
                }
            }

            // --also_check_host の場合
            {
                string message = exec("--also_check_host");
                Assert.IsFalse(Regex.IsMatch(message, @"\sUnknown"));
                foreach (FsMountTarget target in
                    Enum.GetValues(typeof(FsMountTarget)).Cast<FsMountTarget>().Where(target => (target != FsMountTarget.Unknown && target != FsMountTarget.Nfp)))
                {
                    Assert.IsTrue(Regex.IsMatch(message, @"\s" + target.ToString()), target.ToString());
                }
            }
        }

        [TestMethod]
        public void TestExecutionMountTarget()
        {
            string testSourceFile = GetTestDataDirectory() + @"\MountTarget\write_mount.txt";
            string testExpectFile = GetTestDataDirectory() + @"\MountTarget\expect_mount.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            Assert.AreEqual(0, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile + " --also_check_host",
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestExecutionWriteSize()
        {
            // 一度のアクセスのみ。基準を超えるデータ。
            FsAccessLogCheckerExecutionTestDataGeneratorCommon.GenerateDifferentAddress(
                TestTempDirectory + "test_single_over.txt",
                FsAccessLogCheckerExecutionTestDataGeneratorCommon.WindowTime,
                167772170,
                1,
                "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF",
                delegate () { return "WriteFile"; },
                delegate () { return FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexOriginal; });

            // 一度のアクセスのみ。基準を超えないデータ。
            FsAccessLogCheckerExecutionTestDataGeneratorCommon.GenerateDifferentAddress(
                TestTempDirectory + "test_single_not_over.txt",
                FsAccessLogCheckerExecutionTestDataGeneratorCommon.WindowTime,
                167772150,
                1,
                "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF",
                delegate () { return "WriteFile"; },
                delegate () { return FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexOriginal; });

            // 複数のアクセス。基準を超えるデータ。
            FsAccessLogCheckerExecutionTestDataGeneratorCommon.GenerateDifferentAddress(
                TestTempDirectory + "test_multi_over.txt",
                FsAccessLogCheckerExecutionTestDataGeneratorCommon.WindowTime,
                167772170,
                100,
                "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF",
                delegate () { return "WriteFile"; },
                delegate () { return FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexOriginal; });

            // 複数のアクセス。基準を超えないデータ。
            FsAccessLogCheckerExecutionTestDataGeneratorCommon.GenerateDifferentAddress(
                TestTempDirectory + "test_multi_not_over.txt",
                FsAccessLogCheckerExecutionTestDataGeneratorCommon.WindowTime,
                167772150,
                100,
                "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF",
                delegate () { return "WriteFile"; },
                delegate () { return FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexOriginal; });

            TestInDirectory("WriteSize");
        }

        [TestMethod]
        public void TestExecutionWriteCount()
        {
            // FAIL することを期待
            FsAccessLogCheckerExecutionTestDataGeneratorCommon.GenerateDifferentAddress(
                TestTempDirectory + "test_over.txt",
                FsAccessLogCheckerExecutionTestDataGeneratorCommon.WindowTime,
                167772150,
                321,
                "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF",
                delegate () { return "WriteFile"; },
                delegate () { return FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexOriginal; });

            // PASS することを期待
            FsAccessLogCheckerExecutionTestDataGeneratorCommon.GenerateDifferentAddress(
                TestTempDirectory + "test_not_over.txt",
                FsAccessLogCheckerExecutionTestDataGeneratorCommon.WindowTime,
                167772150,
                320,
                "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF",
                delegate () { return "WriteFile"; },
                delegate () { return FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexOriginal; });

            TestInDirectory("WriteCount");
        }

        [TestMethod]
        public void TestExecutionReadPerSeconds()
        {
            Action<string, long, long> generator = (string fileName, long accessStep, long time) =>
            {
                ulong handle = 0x0000000200734060;
                var testStream = new TestStream();
                testStream.StartLog();
                testStream.MountRom(0, 0, "test");
                testStream.OpenFile(0, 0, handle, "test:/test.txt");
                long count = time / accessStep + 1;
                for (long i = 0; i <= count; ++i)
                {
                    testStream.ReadFile(i * accessStep, i * accessStep + 1, handle, 0, 1024);
                }
                testStream.SetUp();
                var path = TestTempDirectory + fileName;
                testStream.DumpFile(path);
            };

            generator("test_fail.txt", 999, FsGuideline.DefaultStepLogTimeMilliseconds);
            generator("test_success.txt", 1000, FsGuideline.DefaultStepLogTimeMilliseconds);

            TestInDirectory("ReadPerSeconds");

            {
                generator("actual_success_1min.txt", 1001, 1 * 60 * 1000);

                var testExpectFile = GetTestDataDirectory() + @"\ReadPerSeconds\expect_success_1min.txt";
                string testSourceFile = TestTempDirectory + @"\actual_success_1min.txt";
                string testDestFile = TestTempDirectory + @"\out_success_1min.txt";
                TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
                testDestFile = destFile.FilePath;

                string message;
                string errorMessage;
                // time_range 指定しているので 1 が返るがチェックはパスしているはず
                Assert.AreEqual(1, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile + " --time_range 1",
                    out message, out errorMessage));
                destFile.UpdateForExpect();
                Assert.AreNotEqual(string.Empty, message);
                Assert.AreEqual(string.Empty, errorMessage);
                Assert.AreEqual(
                    OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                    OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
            }
        }

        [TestMethod]
        public void TestExecutionReadCountMultipleMax()
        {
            Action<string, long, long> generator = (string fileName, long accessStep, long time) =>
            {
                ulong handle = 0x0000000200734060;
                var testStream = new TestStream();
                testStream.StartLog();
                testStream.MountRom(0, 0, "test");
                testStream.OpenFile(0, 0, handle, "test:/test.txt");
                long count = time / accessStep + 1;
                for (long i = 0; i <= count; ++i)
                {
                    testStream.ReadFile(i * accessStep, i * accessStep + 1, handle, 0, 1024);
                    testStream.ReadFile(i * accessStep, i * accessStep + 1, handle, 1024, 1024);
                }
                testStream.SetUp();
                var path = TestTempDirectory + fileName;
                testStream.DumpFile(path);
            };

            generator("test_fail.txt", 999, FsGuideline.DefaultStepLogTimeMilliseconds);
            generator("test_success.txt", 1000, FsGuideline.DefaultStepLogTimeMilliseconds);
        }

        [TestMethod]
        public void TestExecutionReadCount()
        {
            // 以下の組み合わせのデータを生成
            //
            // ・SaveData, Rom
            // ・回数が制限内、制限外
            // ・同じ領域へのアクセス、異なる領域へのアクセス、一度だけ重なるアクセス
            //
            // 同じ領域への制限外のアクセスのみ FAIL することを期待
            // (異なるハンドルやマウントであっても)

            var Operator = "ReadFile";

            var MountType = new Tuple<string, string, string>[]
            {
                new Tuple<string, string, string>("save", "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF"),
                new Tuple<string, string, string>("rom", "Rom", string.Empty)
            };
            var CountType = new Tuple<string, int>[]
            {
                new Tuple<string, int>("limit", 600),
                new Tuple<string, int>("non_limit", 601)
            };
            var HandleType = new Tuple<string, int[]>[]
                {
                    new Tuple<string, int[]>(
                        "same",
                        new int[] {
                            FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexOriginal
                        }),
                    new Tuple<string, int[]>(
                        "same_on_different_handle",
                        new int[] {
                            FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexOriginal,
                            FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexSameOnDifferentHandle
                        }),
                    new Tuple<string, int[]>(
                        "same_on_different_mount",
                        new int[] {
                            FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexOriginal,
                            FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexSameOnDifferentMount
                        }),
                    new Tuple<string, int[]>(
                        "different_on_same_mount",
                        new int[] {
                            FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexOriginal,
                            FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexDifferentOnSameMount
                        }),
                    new Tuple<string, int[]>(
                        "different_on_different_mount",
                        new int[] {
                            FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexOriginal,
                            FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexDifferentOnDifferentMount
                        }),
                    new Tuple<string, int[]>(
                        "different_on_different_mount_param",
                        new int[] {
                            FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexOriginal,
                            FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexOriginal,
                        }),
                };
            var AddresssType = new Tuple<string, Func<ulong, int, int, Tuple<ulong, ulong>>>[]
                {
                new Tuple<string, Func<ulong, int, int, Tuple<ulong, ulong>>>(
                    "same",
                     FsAccessLogCheckerExecutionTestDataGeneratorCommon.MakeSameAddress),
                new Tuple<string, Func<ulong, int, int, Tuple<ulong, ulong>>>(
                    "different",
                     FsAccessLogCheckerExecutionTestDataGeneratorCommon.MakeDifferentAddress),
                new Tuple<string, Func<ulong, int, int, Tuple<ulong, ulong>>>(
                    "overlap",
                     FsAccessLogCheckerExecutionTestDataGeneratorCommon.MakeOverlapAddress),
                };

            foreach (var fs in MountType)
            {
                foreach (var count in CountType)
                {
                    foreach (var addr_type in AddresssType)
                    {
                        foreach (var handle_type in HandleType)
                        {
                            // 組み合わせの除外
                            if (handle_type.Item1 == "different_on_different_mount_param")
                            {
                                if (fs.Item2 != "SaveData"
                                    || addr_type.Item1 != "same")
                                {
                                    continue;
                                }
                            }

                            var file = TestTempDirectory + string.Format(
                                "test_{0}_{1}_address_{2}_{3}.txt",
                                fs.Item1, addr_type.Item1, count.Item1, handle_type.Item1);

                            // テスト パラメータ生成
                            var testParams = new List<FsAccessLogCheckerExecutionTestParam>
                            {
                                new FsAccessLogCheckerExecutionTestParam(
                                    FsAccessLogCheckerExecutionTestDataGeneratorCommon.WindowTime / (ulong)handle_type.Item2.Length,
                                    10 * 1024 * 1024 / (ulong)handle_type.Item2.Length,
                                    count.Item2 / handle_type.Item2.Length,
                                    fs.Item2, fs.Item3)
                            };
                            if (handle_type.Item2.Length == 2)
                            {
                                var fsName = fs.Item2;
                                var fsOption = fs.Item3;

                                if (handle_type.Item1 == "different_on_different_mount_param")
                                {
                                    fsOption = fsOption.Replace("0xFEDCBA98765432100123456789ABCDEF", "0xCAFECAFECAFECAFECAFECAFE");
                                }

                                testParams.Add(new FsAccessLogCheckerExecutionTestParam(
                                    FsAccessLogCheckerExecutionTestDataGeneratorCommon.WindowTime - testParams[0].Time,
                                    10 * 1024 * 1024 - testParams[0].Size,
                                    count.Item2 - testParams[0].Count,
                                    fsName, fsOption));
                            }

                            int val = -1;
                            Func<int> handle = delegate ()
                            {
                                val += 1;
                                return handle_type.Item2[val >= (count.Item2 / handle_type.Item2.Length) ? 1 : 0];
                            };

                            FsAccessLogCheckerExecutionTestDataGeneratorCommon.GenerateReadWriteMulti(
                                file,
                                testParams.ToArray(),
                                delegate () { return Operator; },
                                handle,
                                delegate (FsAccessLogCheckerExecutionTestParam testParam, int index)
                                {
                                    return addr_type.Item2.Invoke(testParam.Size, testParam.Count, index);
                                });
                        }
                    }
                }
            }

            TestInDirectory("ReadCount");
        }

        [TestMethod]
        public void TestExecutionReadCountForSystem()
        {
            var Operator = "ReadFile";

            var MountType = new Tuple<string, string, string>[]
            {
                new Tuple<string, string, string>("rom", "Rom", string.Empty)
            };
            var CountType = new Tuple<string, int>[]
            {
                new Tuple<string, int>("limit", 150),
                new Tuple<string, int>("non_limit", 151)
            };
            var HandleType = new Tuple<string, int[]>[]
            {
                new Tuple<string, int[]>(
                    "same",
                    new int[] {
                        FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexOriginal
                    }),
            };
            var AddresssType = new Tuple<string, Func<ulong, int, int, Tuple<ulong, ulong>>>[]
            {
                new Tuple<string, Func<ulong, int, int, Tuple<ulong, ulong>>>(
                    "same",
                    FsAccessLogCheckerExecutionTestDataGeneratorCommon.MakeSameAddress),
            };

            foreach (var fs in MountType)
            {
                foreach (var count in CountType)
                {
                    foreach (var addr_type in AddresssType)
                    {
                        foreach (var handle_type in HandleType)
                        {
                            var file = TestTempDirectory + string.Format(
                                "test_{0}_{1}_address_{2}_{3}.txt",
                                fs.Item1, addr_type.Item1, count.Item1, handle_type.Item1);

                            // テスト パラメータ生成
                            var testParams = new List<FsAccessLogCheckerExecutionTestParam>
                            {
                                new FsAccessLogCheckerExecutionTestParam(
                                    FsAccessLogCheckerExecutionTestDataGeneratorCommon.WindowTime / (ulong)handle_type.Item2.Length,
                                    10 * 1024 * 1024 / (ulong)handle_type.Item2.Length,
                                    count.Item2 / handle_type.Item2.Length,
                                    fs.Item2, fs.Item3)
                            };

                            int val = -1;
                            Func<int> handle = delegate ()
                            {
                                val += 1;
                                return handle_type.Item2[val >= (count.Item2 / handle_type.Item2.Length) ? 1 : 0];
                            };

                            FsAccessLogCheckerExecutionTestDataGeneratorCommon.GenerateReadWriteMulti(
                                file,
                                testParams.ToArray(),
                                delegate () { return Operator; },
                                handle,
                                delegate (FsAccessLogCheckerExecutionTestParam testParam, int index)
                                {
                                    return addr_type.Item2.Invoke(testParam.Size, testParam.Count, index);
                                },
                                delegate ()
                                {
                                    return FsAccessLogCheckerExecutionTestDataGeneratorCommon.VersionForSystem();
                                });
                        }
                    }
                }
            }
            TestInDirectory("ReadCountForSystem");
        }

        [TestMethod]
        public void TestExecutionMountAndUnmount()
        {
            // 以下の組み合わせのデータを生成
            //
            // ・SaveData, Rom
            // ・回数が制限内、制限外
            // ・アンマウントのみ、マウントのみ、交互、交互の間にファイル読み出し
            //
            // 回数のみで PASS と FAIL が変わることを期待

            var MountType = new Tuple<string, string, string>[]
            {
                new Tuple<string, string, string>("save", "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF"),
                new Tuple<string, string, string>("rom", "Rom", string.Empty)
            };
            var CountType = new Tuple<string, int>[]
            {
                new Tuple<string, int>("limit", 600),
                new Tuple<string, int>("non_limit", 601)
            };
            var AccessType = new string[]
            {
                "umount",
                "mount",
                "mount_and_umount",
                "mount_and_umount_change_param",
                "umount_after_mount",
                "mount_and_unmount_with_read"
            };

            foreach (var fs in MountType)
            {
                foreach (var count in CountType)
                {
                    foreach (var access in AccessType)
                    {
                        // テストの間引き
                        if (access == "mount_and_umount_change_param")
                        {
                            if (fs.Item2 != "SaveData")
                            {
                                continue;
                            }
                        }

                        var file = TestTempDirectory + string.Format(
                            "test_{0}_{1}_{2}.txt",
                            fs.Item1, access, count.Item1);
                        Func<Tuple<string, string>> op = null;
                        Func<string> name = null;

                        // Unmount では増えないので回数を2倍にする
                        var count_value = count.Item2;
                        if (access != "mount")
                        {
                            count_value *= 2;
                            if (access == "mount_and_umount_change_param")
                            {
                                // UserId x 2 の分
                                count_value *= 2;
                            }
                        }

                        bool execute = true;
                        int counter = -1;
                        switch (access)
                        {
                            case "umount":
                                if (fs.Item1 != "save")
                                {
                                    execute = false;
                                }
                                else
                                {
                                    op = delegate () { return new Tuple<string, string>("Umount", string.Empty); };
                                    name = delegate ()
                                    {
                                        counter += 1;
                                        return string.Format("name{0}", counter);
                                    };
                                }
                                break;

                            case "mount":
                                op = delegate () { return new Tuple<string, string>(string.Format("Mount{0}", fs.Item2), fs.Item3); };
                                name = delegate ()
                                {
                                    counter += 1;
                                    return string.Format("name{0}", counter);
                                };
                                break;

                            case "mount_and_umount":
                            case "mount_and_umount_change_param":
                            case "mount_and_unmount_with_read":
                                op = delegate ()
                                {
                                    ++counter;

                                    if ((counter % 2) == 0)
                                    {
                                        var param = fs.Item3;

                                        if (access == "mount_and_umount_change_param")
                                        {
                                            if (counter >= count_value / 2)
                                            {
                                                param = param.Replace("0xFEDCBA98765432100123456789ABCDEF", "0xCAFECAFECAFECAFECAFECAFE");
                                            }
                                        }

                                        return new Tuple<string, string>(
                                            string.Format("Mount{0}", fs.Item2),
                                            param);
                                    }
                                    else
                                    {
                                        return new Tuple<string, string>("Unmount", string.Empty);
                                    }
                                };
                                name = delegate () { return "name"; };
                                break;

                            case "umount_after_mount":
                                op = delegate ()
                                {
                                    counter += 1;
                                    if (counter < count_value / 2)
                                    {
                                        return new Tuple<string, string>(
                                            string.Format("Mount{0}", fs.Item2),
                                            fs.Item3);
                                    }
                                    else
                                    {
                                        return new Tuple<string, string>("Unmount", string.Empty);
                                    }
                                };
                                name = delegate () { return string.Format("name{0}", counter); };
                                break;
                        }
                        if (!execute)
                        {
                            continue;
                        }

                        if (access == "mount_and_unmount_with_read")
                        {
                            FsAccessLogCheckerExecutionTestDataGeneratorCommon.GenerateMountUnmountWithReadDifferentAddress(
                                file,
                                FsAccessLogCheckerExecutionTestDataGeneratorCommon.WindowTime,
                                count_value / 2,
                                fs.Item2,
                                name);
                        }
                        else
                        {
                            FsAccessLogCheckerExecutionTestDataGeneratorCommon.GenerateMountUnmount(
                                file,
                                FsAccessLogCheckerExecutionTestDataGeneratorCommon.WindowTime,
                                count_value,
                                op,
                                name);
                        }
                    }
                }
            }

            TestInDirectory("MountUnmount");
        }

        [TestMethod]
        public void TestExecutionSlideWindow()
        {
            // 始めの 10分の平均と後の10分の両方が PASS を期待
            FsAccessLogCheckerExecutionTestDataGeneratorCommon.GenerateDifferentAddressMulti(
                TestTempDirectory + "test_pass_pass.txt",
                new FsAccessLogCheckerExecutionTestParam[]
                {
                    new FsAccessLogCheckerExecutionTestParam(
                        1 * 60 * 1000, 16777215, 10,
                        "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF"),
                    new FsAccessLogCheckerExecutionTestParam(
                        FsAccessLogCheckerExecutionTestDataGeneratorCommon.WindowTime,
                        167772150, 100,
                        "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF")
                },
                delegate () { return "WriteFile"; },
                delegate () { return FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexOriginal; });

            // 始めの 10分の平均のみ PASS を期待
            FsAccessLogCheckerExecutionTestDataGeneratorCommon.GenerateDifferentAddressMulti(
                TestTempDirectory + "test_pass_fail.txt",
                new FsAccessLogCheckerExecutionTestParam[]
                {
                    new FsAccessLogCheckerExecutionTestParam(
                        FsAccessLogCheckerExecutionTestDataGeneratorCommon.WindowTime,
                        167772150, 100,
                        "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF"),
                    new FsAccessLogCheckerExecutionTestParam(
                        1 * 60 * 1000, 16777245, 10,
                        "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF")
                },
                delegate () { return "WriteFile"; },
                delegate () { return FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexOriginal; });

            // 始めの 10分の平均と後の10分の平均の両方が FAIL することを期待
            FsAccessLogCheckerExecutionTestDataGeneratorCommon.GenerateDifferentAddressMulti(
                TestTempDirectory + "test_fail_fail.txt",
                new FsAccessLogCheckerExecutionTestParam[]
                {
                    new FsAccessLogCheckerExecutionTestParam(
                        1 * 60 * 1000, 16777285, 10,
                        "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF"),
                    new FsAccessLogCheckerExecutionTestParam(
                        FsAccessLogCheckerExecutionTestDataGeneratorCommon.WindowTime,
                        167772170, 100,
                        "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF")
                },
                delegate () { return "WriteFile"; },
                delegate () { return FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexOriginal; });

            // 5分と12分のアクセスを足すと FAIL する
            {
                TestStream testStream = new TestStream();
                testStream.StartLog();
                testStream.MountSaveData(0, 0, "test", 1);
                testStream.CreateFile(0, 0, "test:/test.txt", 1048576);
                testStream.OpenFile(0, 0, 0x3a2e7f10, "test:/test.txt");
                testStream.WriteFile(5 * 60 * 1000, 5 * 60 * 1000, 0x3a2e7f10, 0, 157286400);
                testStream.WriteFile(12 * 60 * 1000, 12 * 60 * 1000, 0x3a2e7f10, 0, 157286400);
                testStream.SetUp();
                testStream.DumpFile(TestTempDirectory + "test_fail_large_slide.txt");
            }
            // 5分と15分のアクセスは別に解析される
            {
                TestStream testStream = new TestStream();
                testStream.StartLog();
                testStream.MountSaveData(0, 0, "test", 1);
                testStream.CreateFile(0, 0, "test:/test.txt", 1048576);
                testStream.OpenFile(0, 0, 0x3a2e7f10, "test:/test.txt");
                testStream.WriteFile(5 * 60 * 1000, 5 * 60 * 1000, 0x3a2e7f10, 0, 157286400);
                testStream.WriteFile(15 * 60 * 1000, 15 * 60 * 1000, 0x3a2e7f10, 0, 157286400);
                testStream.SetUp();
                testStream.DumpFile(TestTempDirectory + "test_pass_large_slide.txt");
            }
            // 2分-5分で 600 回の Read と12分の Read アクセスで PASS -> PASS する
            {
                TestStream testStream = new TestStream();
                testStream.StartLog();
                testStream.MountSaveData(0, 0, "test", 1);
                testStream.OpenFile(00, 0, 0x3a2e7f10, "test:/test.txt");
                long baseStart = 2 * 60 * 1000;
                for (int i = 0; i < 600; ++i)
                {
                    long time = (i + 1) * (3 * 60 * 1000 / 600) + baseStart;
                    testStream.ReadFile(time, time, 0x3a2e7f10, 0, 157286400);
                }
                testStream.ReadFile(12 * 60 * 1000, 12 * 60 * 1000, 0x3a2e7f10, 0, 157286400);
                testStream.SetUp();
                testStream.DumpFile(TestTempDirectory + "test_pass_large_slide_bound.txt");
            }

            TestInDirectory("SlideWindow");
        }

        [TestMethod]
        public void TestExecutionChangeTimeRange()
        {
            // 始めの 10分の平均が PASS、次の 10分 は FAIL するログを 60分平均で解析する
            FsAccessLogCheckerExecutionTestDataGeneratorCommon.GenerateDifferentAddressMulti(
                TestTempDirectory + "test_pass.txt",
                new FsAccessLogCheckerExecutionTestParam[]
                {
                    new FsAccessLogCheckerExecutionTestParam(
                        FsAccessLogCheckerExecutionTestDataGeneratorCommon.WindowTime,
                        167772150, 100,
                        "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF"),
                    new FsAccessLogCheckerExecutionTestParam(
                        1 * 60 * 1000,
                        16777245, 10,
                        "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF"),
                    new FsAccessLogCheckerExecutionTestParam(
                        (60 - 1) * 60 * 1000 - FsAccessLogCheckerExecutionTestDataGeneratorCommon.WindowTime,
                        1, 1,
                        "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF")
                },
                delegate () { return "WriteFile"; },
                delegate () { return FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexOriginal; });

            FsAccessLogCheckerExecutionTestDataGeneratorCommon.GenerateDifferentAddressMulti(
                TestTempDirectory + "test_enough_time.txt",
                new FsAccessLogCheckerExecutionTestParam[]
                {
                    new FsAccessLogCheckerExecutionTestParam(
                        FsAccessLogCheckerExecutionTestDataGeneratorCommon.WindowTime,
                        167772150, 100,
                        "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF"),
                    new FsAccessLogCheckerExecutionTestParam(
                        1 * 60 * 1000,
                        16777245, 10,
                        "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF"),
                    new FsAccessLogCheckerExecutionTestParam(
                        (60 - 1) * 60 * 1000 - FsAccessLogCheckerExecutionTestDataGeneratorCommon.WindowTime - 1,
                        1, 1,
                        "SaveData", ", userid: 0xFEDCBA98765432100123456789ABCDEF")
                },
                delegate () { return "WriteFile"; },
                delegate () { return FsAccessLogCheckerExecutionTestDataGeneratorCommon.HandleIndexOriginal; });

            TestInDirectoryWithOption("TimeRange", "--time_range 60");
        }

        [TestMethod]
        public void TestExecutionAccumulate()
        {
            Func<TestStream> setUpCommon = () =>
            {
                // 10分間に 600 回のアクセス
                TestStream testStream = new TestStream();
                testStream.StartLog();
                testStream.MountSaveData(0, 0, "test", 1);
                testStream.CreateFile(0, 0, "test:/test.txt", 1048576);
                testStream.OpenFile(0, 0, 0x3a2e7f10, "test:/test.txt");
                for (int i = 1; i < 600; ++i)
                {
                    testStream.ReadFile(i * 1000, i * 1000, 0x3a2e7f10, 0, 1);
                }
                // 600回目は境界上におく
                testStream.ReadFile(600 * 1000 - 1, 600 * 1000 - 1, 0x3a2e7f10, 0, 1);
                return testStream;
            };

            {
                TestStream testStream = setUpCommon();
                testStream.Unmount(600 * 1000, 600 * 1000, "test");
                testStream.SetUp();
                testStream.DumpFile(TestTempDirectory + "test_bound_pass.txt");
            }
            {
                TestStream testStream = setUpCommon();
                // 境界上で連続したアクセスが、正しくチェック時間間隔に含まれるかどうかをテスト
                testStream.ReadFile(600 * 1000 - 1, 600 * 1000 - 1, 0x3a2e7f10, 0, 1);
                testStream.Unmount(600 * 1000, 600 * 1000, "test");
                testStream.SetUp();
                testStream.DumpFile(TestTempDirectory + "test_bound_fail.txt");
            }

            TestInDirectory("Accumulate");
        }

        [TestMethod]
        public void TestExecutionCacheStorage()
        {
            Func<TestStream> setup = () =>
            {
                TestStream stream = new TestStream();
                stream.StartLog();
                stream.CreateCacheStorage(0, 0, 0, 0x100, 0x100);
                stream.CreateCacheStorage(0, 0, 1, 0x100, 0x100);
                stream.CreateCacheStorage(0, 0, 2, 0x100, 0x100);
                stream.GetCacheStorageSize(0, 0, 0);
                stream.GetCacheStorageSize(0, 0, 1);
                stream.GetCacheStorageSize(0, 0, 2);
                return stream;
            };

            Func<TestStream, string, int> tearDown = (TestStream stream, string fileName) =>
            {
                stream.GetCacheStorageSize(599999, 599999, 0);
                stream.GetCacheStorageSize(599999, 599999, 1);
                stream.GetCacheStorageSize(599999, 599999, 2);
                stream.DeleteCacheStorage(599999, 599999, 0);
                stream.DeleteCacheStorage(599999, 599999, 1);
                stream.DeleteCacheStorage(599999, 600000, 2);
                stream.SetUp();
                stream.DumpFile(TestTempDirectory + fileName);
                return 0;
            };

            // 3 つマウントして、同じオフセットで読み書きを行う
            {
                TestStream testStream = setup();

                testStream.MountCacheStorage(100, 100, "test0");
                testStream.MountCacheStorage(100, 100, "test1", 1);
                testStream.MountCacheStorage(100, 100, "test2", 2);
                testStream.CreateFile(100, 100, "test0:/text", 0x100);
                testStream.CreateFile(100, 100, "test1:/text", 0x100);
                testStream.CreateFile(100, 100, "test2:/text", 0x100);
                testStream.OpenFile(100, 100, 0x1, "test0:/text");
                testStream.OpenFile(100, 100, 0x2, "test1:/text");
                testStream.OpenFile(100, 100, 0x3, "test2:/text");

                for (ulong handle = 1; handle <= 3; ++handle)
                {
                    for (ulong i = 0; i < 2; ++i)
                    {
                        testStream.ReadFile(100, 100, handle, i * 1024, (i + 1) * 1024);
                        testStream.WriteFile(100, 100, handle, i * 1024, (i + 1) * 1024);
                    }
                }
                testStream.CommitSaveData(100, 100, "test0");
                testStream.CommitSaveData(100, 100, "test1");
                testStream.CommitSaveData(100, 100, "test2");
                testStream.Unmount(100, 100, "test0");
                testStream.Unmount(100, 100, "test1");
                testStream.Unmount(100, 100, "test2");

                tearDown(testStream, "test_3_storage.txt");
            }

            // マウントして読み書き、を同じインデックスで 3 回行う
            {
                TestStream testStream = setup();

                for (ulong index = 0; index < 3; ++index)
                {
                    testStream.MountCacheStorage(100, 100, "test", 0);
                    testStream.CreateFile(100, 100, "test:/text", 0x100);
                    testStream.OpenFile(100, 100, 0x1, "test:/text");
                    for (ulong i = 0; i < 2; ++i)
                    {
                        testStream.ReadFile(100, 100, 0x1, i * 1024, (i + 1) * 1024);
                        testStream.WriteFile(100, 100, 0x1, i * 1024, (i + 1) * 1024);
                    }
                    testStream.CommitSaveData(100, 100, "test");
                    testStream.Unmount(100, 100, "test");
                }
                tearDown(testStream, "test_same_index.txt");
            }

            // マウントして読み書き、を異なるインデックスで 3 回行う
            {
                TestStream testStream = setup();

                for (int index = 0; index < 3; ++index)
                {
                    testStream.MountCacheStorage(100, 100, "test", index);
                    testStream.CreateFile(100, 100, "test:/text", 0x100);
                    testStream.OpenFile(100, 100, 0x1, "test:/text");
                    for (ulong i = 0; i < 2; ++i)
                    {
                        testStream.ReadFile(100, 100, 0x1, i * 1024, (i + 1) * 1024);
                        testStream.WriteFile(100, 100, 0x1, i * 1024, (i + 1) * 1024);
                    }
                    testStream.CommitSaveData(100, 100, "test");
                    testStream.Unmount(100, 100, "test");
                }
                tearDown(testStream, "test_variable_index.txt");
            }

            // マウントして読み書き、をインデックス有り無しで行う
            {
                TestStream testStream = setup();

                for (int index = 0; index < 3; ++index)
                {
                    if (index < 2)
                    {
                        testStream.MountCacheStorage(100, 100, "test", index);
                    }
                    else
                    {
                        testStream.MountCacheStorage(100, 100, "test");
                    }
                    testStream.CreateFile(100, 100, "test:/text", 0x100);
                    testStream.OpenFile(100, 100, 0x1, "test:/text");
                    for (ulong i = 0; i < 2; ++i)
                    {
                        testStream.ReadFile(100, 100, 0x1, i * 1024, (i + 1) * 1024);
                        testStream.WriteFile(100, 100, 0x1, i * 1024, (i + 1) * 1024);
                    }
                    testStream.CommitSaveData(100, 100, "test");
                    testStream.Unmount(100, 100, "test");
                }
                tearDown(testStream, "test_no_index.txt");
            }

            // GetCacheStorageSize をインデックス無しのみで行う
            {
                TestStream stream = new TestStream();
                stream.StartLog();
                for (int i = 0; i < 100; ++i)
                {
                    stream.MountCacheStorage(i, i, "cache");
                    stream.GetCacheStorageSize(i, i);
                    stream.Unmount(i, i, "cache");
                }
                stream.MountSaveData(599999, 599999, "save", 1);
                stream.Unmount(599999, 600000, "save");
                stream.SetUp();
                stream.DumpFile(TestTempDirectory + "test_same_read_no_index.txt");
            }

            // GetCacheStorageSize をインデックス有り無しで行う
            // マウント名を表示するため、途中に MountSaveData を挟む
            {
                TestStream stream = new TestStream();
                stream.StartLog();
                long minutes = 60 * 1000;
                for (long i = 0; i < 100; ++i)
                {
                    stream.MountCacheStorage(i, i, "cache");
                    stream.GetCacheStorageSize(i, i);
                    stream.Unmount(i, i, "cache");
                }
                stream.MountSaveData(5 * minutes, 5 * minutes, "save", 1);
                stream.Unmount(5 * minutes, 5 * minutes, "save");
                for (long i = 9 * minutes; i < 9 * minutes + 100; ++i)
                {
                    stream.MountCacheStorage(i, i, "cache");
                    stream.GetCacheStorageSize(i, i);
                    stream.Unmount(i, i, "cache");
                }
                for (long i = 10 * minutes; i < 10 * minutes + 100; ++i)
                {
                    stream.MountCacheStorage(i, i, "cache");
                    stream.GetCacheStorageSize(i, i);
                    stream.Unmount(i, i, "cache");
                }
                for (long i = 10 * minutes + 200; i < 10 * minutes + 300; ++i)
                {
                    stream.MountCacheStorage(i, i, "cache", 1);
                    stream.GetCacheStorageSize(i, i, 1);
                    stream.Unmount(i, i, "cache");
                }
                stream.MountSaveData(12 * minutes, 12 * minutes, "save", 1);
                stream.Unmount(12 * minutes, 12 * minutes, "save");
                for (long i = 15 * minutes; i < 15 * minutes + 100; ++i)
                {
                    stream.MountCacheStorage(i, i, "cache", 0);
                    stream.GetCacheStorageSize(i, i, 0);
                    stream.Unmount(i, i, "cache");
                }
                for (long i = 20 * minutes; i < 20 * minutes + 100; ++i)
                {
                    stream.MountCacheStorage(i, i, "cache");
                    stream.GetCacheStorageSize(i, i);
                    stream.Unmount(i, i, "cache");
                }
                stream.SetUp();
                stream.DumpFile(TestTempDirectory + "test_same_read_with_index.txt");
            }

            TestInDirectory("CacheStorage");
        }

        [TestMethod]
        public void TestExecutionOutputStateSuccess()
        {
            // OutputState 付きアクセスログのテスト(通常のアクセスログ)
            var testSourceFile = GetTestDataDirectory() + @"\OutputState\write_success.txt";
            var testExpectFile = GetTestDataDirectory() + @"\OutputState\expect_success.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;

            Assert.AreEqual(0, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestExecutionOutputStateIncomplate()
        {
            // OutputState 付きアクセスログのテスト(失敗したアクセスログ)
            var testSourceFile = GetTestDataDirectory() + @"\OutputState\write_incomplate.txt";
            var testExpectFile = GetTestDataDirectory() + @"\OutputState\expect_incomplate.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;

            Assert.AreEqual(1, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestExecutionOutputStateContinuation()
        {
            // OutputState 付きアクセスログのテスト(失敗後に出力継続したアクセスログ)
            var testSourceFile = GetTestDataDirectory() + @"\OutputState\write_continuation.txt";
            var testExpectFile = GetTestDataDirectory() + @"\OutputState\expect_continuation.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;

            Assert.AreEqual(1, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestExecutionMountTargetTest()
        {
            // OutputState 付きアクセスログのテスト(失敗後に出力継続したアクセスログ)
            var testSourceFile = GetTestDataDirectory() + @"\MultiMount\write.txt";
            var testExpectFile = GetTestDataDirectory() + @"\MultiMount\expect.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;

            Assert.AreEqual(0, ExecuteProgram(testSourceFile + " -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestExecutionMountTargetQuietTest()
        {
            // OutputState 付きアクセスログのテスト(失敗後に出力継続したアクセスログ)
            var testSourceFile = GetTestDataDirectory() + @"\MultiMount\write.txt";
            var testExpectFile = GetTestDataDirectory() + @"\MultiMount\expect_quiet.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;

            Assert.AreEqual(0, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestExecutionSystemMountTargetTest()
        {
            // OutputState 付きアクセスログのテスト(失敗後に出力継続したアクセスログ)
            var testSourceFile = GetTestDataDirectory() + @"\MultiMount\write_system.txt";
            var testExpectFile = GetTestDataDirectory() + @"\MultiMount\expect_system.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;

            Assert.AreEqual(0, ExecuteProgram(testSourceFile + " -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestExecutionSystemMountTargetQuietTest()
        {
            // OutputState 付きアクセスログのテスト(失敗後に出力継続したアクセスログ)
            var testSourceFile = GetTestDataDirectory() + @"\MultiMount\write_system.txt";
            var testExpectFile = GetTestDataDirectory() + @"\MultiMount\expect_system_quiet.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;

            Assert.AreEqual(0, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [Ignore]
        [TestMethod]
        public void TestExecutionPerformanceCheck()
        {
            Func<int, string> createTestData = (int index) =>
            {
                string fileName = "StreamingEmulate" + index.ToString() + ".txt";
                Random random = new Random();
                var testStream = new TestStream();
                ulong handle = 0x0000000200734060;
                testStream.StartLog();
                testStream.MountRom(0, 0, "test");
                testStream.OpenFile(0, 0, handle, "test:/test.txt");
                long time = 1;
                ulong offset = 0;
                testStream.ReadFile(time, time + 1, handle, offset, 16 * 1024);
                while (time < 60 * 60 * 1000)
                {
                    long next = (long)random.Next(1, 16 * 1024 * 1000 / 44100 / 2);
                    time += next;
                    offset += (ulong)(next * 44100 * 2 / 1000);
                    testStream.ReadFile(time, time + 1, handle, offset, 16 * 1024);
                }
                testStream.SetUp();
                var path = TestTempDirectory + fileName;
                testStream.DumpFile(path);
                return path;
            };

            Action<string, Func<int, string>> test = (string name, Func<int, string> create) =>
            {
                List<double> time = new List<double>();
                int testCount = 5;
                for (int i = 0; i < testCount; ++i)
                {
                    string testSourceFile = create(i);
                    Stopwatch stopWath = new Stopwatch();
                    string message;
                    string errorMessage;
                    stopWath.Start();
                    ExecuteProgram(testSourceFile, out message, out errorMessage);
                    stopWath.Stop();
                    time.Add(stopWath.Elapsed.TotalMilliseconds);
                }
                time.Sort();
                time.RemoveAt(testCount - 1);
                time.RemoveAt(0);

                Console.WriteLine(name + ": " + time.Average().ToString());
            };

            test("PerformanceTestReadWriteSdCard",
                (int index) =>
                {
                    return GetTestDataDirectory() + @"\Performance\PerformanceTestReadWriteSdCard.ReadFile_WriteFile.txt";
                });
            test("StreamingEmulate",
                createTestData);

            Assert.IsFalse(true);
        }

        [TestMethod]
        public void TestVerboseLogDisplayOrder()
        {
            // SIGLO-73195
            // --quiet でないときのアクセスログの表示順序がおかしい問題のテスト
            {
                TestStream testStream = new TestStream();
                ulong handle = 0x0000000000123456;
                testStream.StartLog();
                testStream.MountRom(0, 0, "test");
                testStream.OpenFile(0, 0, handle, "test:/test.txt");
                for (int i = 0; i <= 5; ++i)
                {
                    testStream.ReadFile(i, i, handle, 0 * 1024, 1 * 1024);
                    testStream.ReadFile(i, i, handle, 1 * 1024, 2 * 1024);
                }
                for (int i = 600000 - 5; i <= 600000 + 5; ++i)
                {
                    testStream.ReadFile(i, i, handle, 2 * 1024, 3 * 1024);
                    testStream.ReadFile(i, i, handle, 3 * 1024, 4 * 1024);
                }
                for (int i = 1200000 - 5; i <= 1200000 + 5; ++i)
                {
                    testStream.ReadFile(i, i, handle, 4 * 1024, 5 * 1024);
                    testStream.ReadFile(i, i, handle, 5 * 1024, 6 * 1024);
                }
                testStream.SetUp();
                testStream.DumpFile(TestTempDirectory + "test_pass1.txt");

                {
                    var testExpectFile = GetTestDataDirectory() + @"\VerboseLog\expect_pass1.txt";
                    var testSourceFile = TestTempDirectory + "test_pass1.txt";
                    string testDestFile = TestTempDirectory + "out.txt";
                    TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
                    testDestFile = destFile.FilePath;

                    string message;
                    string errorMessage;

                    Assert.AreEqual(0, ExecuteProgram(testSourceFile + " --verbose -o " + testDestFile,
                        out message, out errorMessage));
                    destFile.UpdateForExpect();
                    Assert.AreNotEqual(string.Empty, message);
                    Assert.AreEqual(string.Empty, errorMessage);
                    Assert.AreEqual(
                        OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                        OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
                }
            }
            {
                TestStream testStream = new TestStream();
                ulong handle = 0x0000000000123456;
                testStream.StartLog();
                testStream.MountRom(0, 0, "test");
                testStream.OpenFile(0, 0, handle, "test:/test.txt");
                for (int i = 0; i <= 5; ++i)
                {
                    testStream.ReadFile(i, i, handle, 0 * 1024, 1 * 1024);
                    testStream.ReadFile(i, i + 1, handle, 1 * 1024, 2 * 1024);
                }
                for (int i = 600000 - 5; i <= 600000 + 5; ++i)
                {
                    testStream.ReadFile(i, i, handle, 2 * 1024, 3 * 1024);
                    testStream.ReadFile(i, i + 1, handle, 3 * 1024, 4 * 1024);
                }
                for (int i = 1200000 - 5; i <= 1200000 + 5; ++i)
                {
                    testStream.ReadFile(i, i, handle, 4 * 1024, 5 * 1024);
                    testStream.ReadFile(i, i + 1, handle, 5 * 1024, 6 * 1024);
                }
                testStream.SetUp();
                testStream.DumpFile(TestTempDirectory + "test_pass2.txt");

                {
                    var testExpectFile = GetTestDataDirectory() + @"\VerboseLog\expect_pass2.txt";
                    var testSourceFile = TestTempDirectory + "test_pass2.txt";
                    string testDestFile = TestTempDirectory + "out.txt";
                    TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
                    testDestFile = destFile.FilePath;

                    string message;
                    string errorMessage;

                    Assert.AreEqual(0, ExecuteProgram(testSourceFile + " --verbose -o " + testDestFile,
                        out message, out errorMessage));
                    destFile.UpdateForExpect();
                    Assert.AreNotEqual(string.Empty, message);
                    Assert.AreEqual(string.Empty, errorMessage);
                    Assert.AreEqual(
                        OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                        OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
                }
            }
            {
                {
                    TestStream testStream = new TestStream();
                    testStream.StartLog();

                    for (int i = 0; i <= 6; ++i)
                    {
                        testStream.MountRom(i, i, string.Format("test{0}", i));
                    }
                    for (int i = 20000 - 5; i <= 20000 + 5; ++i)
                    {
                        testStream.MountRom(i, i, string.Format("test{0}", i * 2));
                        testStream.MountRom(i, i, string.Format("test{0}", i * 2));
                    }
                    for (int i = 40000 - 5; i <= 40000 + 5; ++i)
                    {
                        testStream.MountRom(i, i, string.Format("test{0}", i * 2));
                        testStream.MountRom(i, i, string.Format("test{0}", i * 2));
                    }
                    for (int i = 60000 - 5; i <= 60000 + 5; ++i)
                    {
                        testStream.MountRom(i, i, string.Format("test{0}", i * 2));
                        testStream.MountRom(i, i, string.Format("test{0}", i * 2));
                    }
                    testStream.SetUp();
                    testStream.DumpFile(TestTempDirectory + "test_failed.txt");

                    {
                        var testExpectFile = GetTestDataDirectory() + @"\VerboseLog\expect_failed.txt";
                        var testSourceFile = TestTempDirectory + "test_failed.txt";
                        string testDestFile = TestTempDirectory + "out.txt";
                        TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
                        testDestFile = destFile.FilePath;

                        string message;
                        string errorMessage;

                        Assert.AreEqual(1, ExecuteProgram(testSourceFile + " --time_range 1 --verbose -o " + testDestFile,
                            out message, out errorMessage));
                        destFile.UpdateForExpect();
                        Assert.AreNotEqual(string.Empty, message);
                        Assert.AreEqual(string.Empty, errorMessage);
                        Assert.AreEqual(
                            OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                            OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
                    }
                }
            }
        }

        [TestMethod]
        public void TestSameAddressRead()
        {
            // SIGLO-73008
            // Path の無い Bcat アクセスと Path のある Bcat アクセスの
            // 両方で SameAddressRead が発生すると、例外が発生する
            {
                TestStream testStream = new TestStream();
                testStream.StartLog();
                for (int i = 0; i <= 600000; i += 3000)
                {
                    if (i > 300000)
                    {
                        testStream.GetSaveDataSize(i, i, 0x87654321);
                    }
                    testStream.MountSaveData(i, i, "test", 0x12345678);
                    testStream.OpenFile(i, i, 0x1234, "test:/file");
                    testStream.ReadFile(i, i, 0x1234, (ulong)i * 10, (ulong)(i + 1) * 10);
                    testStream.Unmount(i, i, "test");
                    testStream.GetSaveDataSize(i, i, 0x12345678);
                }
                testStream.SetUp();
                testStream.DumpFile(TestTempDirectory + "test_getsavedatasize.txt");
            }
            {
                TestStream testStream = new TestStream();
                testStream.StartLog();
                testStream.MountDeliveryCacheStorage(0, 0);
                for (int i = 0; i <= 600000; i += 4000)
                {
                    testStream.EnumerateDeliveryCacheDirectory(i, i);
                    testStream.DeliveryCacheDirectoryOpen(i, i, 1, "test");
                    testStream.DeliveryCacheDirectoryRead(i, i, 1);
                    testStream.DeliveryCacheDirectoryClose(i, i, 1);
                    testStream.DeliveryCacheDirectoryDestructor(i, i, 1);
                }
                testStream.SetUp();
                testStream.DumpFile(TestTempDirectory + "test_deliverycachedirectory.txt");
            }
            {
                TestStream testStream = new TestStream();
                testStream.StartLog();
                testStream.MountDeliveryCacheStorage(0, 0);
                for (int i = 0; i <= 600000; i += 2000)
                {
                    testStream.EnumerateDeliveryCacheDirectory(i, i);
                    testStream.DeliveryCacheFileOpen(i, i, 1, "test");
                    testStream.DeliveryCacheFileRead(i, i, 1, 650, 650);
                    testStream.DeliveryCacheFileClose(i, i, 1);
                    testStream.DeliveryCacheFileDestructor(i, i, 1);
                }
                testStream.SetUp();
                testStream.DumpFile(TestTempDirectory + "test_deliverycachefile.txt");
            }

            // パスの無いアクセスを複数回行う
            {
                TestStream testStream = new TestStream();
                testStream.StartLog();
                testStream.GetCacheStorageSize(0, 1, 0);
                testStream.GetCacheStorageSize(1, 2, 1);
                testStream.GetCacheStorageSize(2, 3, 0);
                testStream.GetSaveDataSize(300000, 300002, 0x012345678);
                testStream.GetSaveDataSize(300001, 300003, 0x876543210);
                testStream.GetSaveDataSize(300002, 300004, 0x012345678);
                testStream.QueryApplicationPlayStatistics(500000, 500001);
                testStream.QueryApplicationPlayStatistics(500001, 600000);
                testStream.QueryApplicationPlayStatisticsForSystem(700000, 700001);
                testStream.QueryApplicationPlayStatisticsForSystem(700001, 800000);
                testStream.SetUp();
                testStream.DumpFile(TestTempDirectory + "test_no_path_function_call.txt");
            }
            TestInDirectory("SameAddressRead");
        }

        [TestMethod]
        public void TestCharacterSetAscii()
        {
            // 日本語を含まないアクセスログに対するテスト
            var testSourceFile = GetTestDataDirectory() + @"\CharacterSet\test_ascii.txt";
            var testExpectFile = GetTestDataDirectory() + @"\CharacterSet\expect_ascii.txt";
            string testDestFile = TestTempDirectory + @"\out_ascii.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            Assert.AreEqual(0, ExecuteProgram(
                testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestCharacterSetSJIS()
        {
            // 文字コードが SJIS のアクセスログに対するテスト
            var testSourceFile = GetTestDataDirectory() + @"\CharacterSet\test_sjis.txt";
            var testExpectFile = GetTestDataDirectory() + @"\CharacterSet\expect_sjis.txt";
            string testDestFile = TestTempDirectory + @"\out_sjis.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            Assert.AreEqual(0, ExecuteProgram(
                testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestCharacterSetUTF8()
        {
            // 文字コードが BOM なし UTF-8 のアクセスログに対するテスト
            var testSourceFile = GetTestDataDirectory() + @"\CharacterSet\test_utf8.txt";
            var testExpectFile = GetTestDataDirectory() + @"\CharacterSet\expect_utf8.txt";
            string testDestFile = TestTempDirectory + @"\out_utf8.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            Assert.AreEqual(0, ExecuteProgram(
                testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestCharacterSetUTF8Bom()
        {
            // 文字コードが BOM あり UTF-8 のアクセスログに対するテスト
            var testSourceFile = GetTestDataDirectory() + @"\CharacterSet\test_utf8_bom.txt";
            var testExpectFile = GetTestDataDirectory() + @"\CharacterSet\expect_utf8_bom.txt";
            string testDestFile = TestTempDirectory + @"\out_utf8_bom.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            Assert.AreEqual(0, ExecuteProgram(
                testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestCharacterSetUTF16LE()
        {
            // 文字コードが BOM なし UTF-16LE のアクセスログに対するテスト
            var testSourceFile = GetTestDataDirectory() + @"\CharacterSet\test_utf16le.txt";
            var testExpectFile = GetTestDataDirectory() + @"\CharacterSet\expect_utf16le.txt";
            string testDestFile = TestTempDirectory + @"\out_utf16le.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            Assert.AreEqual(1, ExecuteProgram(
                testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestCharacterSetUTF16LEBom()
        {
            // 文字コードが BOM あり UTF-16LE のアクセスログに対するテスト
            var testSourceFile = GetTestDataDirectory() + @"\CharacterSet\test_utf16le_bom.txt";
            var testExpectFile = GetTestDataDirectory() + @"\CharacterSet\expect_utf16le_bom.txt";
            string testDestFile = TestTempDirectory + @"\out_utf16le_bom.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            Assert.AreEqual(0, ExecuteProgram(
                testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestCharacterSetUTF16BE()
        {
            // 文字コードが BOM なし UTF-16BE のアクセスログに対するテスト
            var testSourceFile = GetTestDataDirectory() + @"\CharacterSet\test_utf16be.txt";
            var testExpectFile = GetTestDataDirectory() + @"\CharacterSet\expect_utf16be.txt";
            string testDestFile = TestTempDirectory + @"\out_utf16be.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            Assert.AreEqual(1, ExecuteProgram(
                testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestCharacterSetUTF16BEBom()
        {
            // 文字コードが BOM あり UTF-16BE のアクセスログに対するテスト
            var testSourceFile = GetTestDataDirectory() + @"\CharacterSet\test_utf16be_bom.txt";
            var testExpectFile = GetTestDataDirectory() + @"\CharacterSet\expect_utf16be_bom.txt";
            string testDestFile = TestTempDirectory + @"\out_utf16be_bom.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            Assert.AreEqual(0, ExecuteProgram(
                testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreNotEqual(string.Empty, message);
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestWarningRatio()
        {
            int baseTime = 0;
            Action<TestStream, int, int, ulong, int[]> generator = (TestStream stream, int mountCount, int writeCount, ulong writeSize, int[] sameReadCounts) =>
            {
                ulong fileId = 1;
                stream.MountSaveData(baseTime, baseTime, "test", 0x1);
                foreach (int sameReadCount in sameReadCounts)
                {
                    stream.OpenFile(baseTime, baseTime, fileId, string.Format("test:/file{0}", fileId));
                    for (int i = 0; i < sameReadCount; ++i)
                    {
                        stream.ReadFile(baseTime, baseTime, fileId, 10, 10);
                        ++baseTime;
                    }
                    ++fileId;
                }
                stream.OpenFile(baseTime, baseTime, fileId, string.Format("test:/file{0}", fileId));
                if (writeCount > 0)
                {
                    stream.WriteFile(baseTime, baseTime, fileId, 10, writeSize - (ulong)(writeCount - 1) * 1024);
                    for (int i = 0; i < writeCount - 1; ++i)
                    {
                        stream.WriteFile(baseTime, baseTime, fileId, 10, 1024);
                        ++baseTime;
                    }
                }
                stream.Unmount(baseTime, baseTime, "test");
                for (int i = 0; i < mountCount - 1; ++i)
                {
                    stream.MountSaveData(baseTime, baseTime, "test", 0x1);
                    stream.Unmount(baseTime, baseTime, "test");
                    ++baseTime;
                }
            };

            // testStreams[0] では Bis のみマウント
            TestStream testStream = new TestStream();
            testStream.StartLog();
            ulong MegabytePerMin = 10 * 1024 * 1024;

            // 0～600 s では、エラー閾値の 75 %  より僅かに下あたり
            baseTime = 0;
            generator(testStream, 449, 239, 12 * MegabytePerMin - 1, new int[] { 300, 449 });

            // 600～1200 s では、エラー閾値の 75 % ちょうど
            baseTime = 10 * 60 * 1000;
            generator(testStream, 450, 240, 12 * MegabytePerMin, new int[] { 450, 400, 301 });

            // 1200～1800 s では、エラー閾値の 75 % より僅かに上あたり
            baseTime = 2 * 10 * 60 * 1000;
            generator(testStream, 451, 241, 12 * MegabytePerMin + 1, new int[] { 451, 449, 450 });

            // MountRom を挟む
            ulong fileHandle = 99;
            testStream.MountRom(baseTime, baseTime, "test");
            testStream.OpenFile(baseTime, baseTime, fileHandle, string.Format("test:/file{0}", fileHandle));
            for (int i = 0; i < 451; ++i)
            {
                testStream.ReadFile(baseTime, baseTime, fileHandle, 10, 10);
                ++baseTime;
            }
            ++fileHandle;
            testStream.Unmount(baseTime, baseTime, "test");
            for (int i = 0; i < 450; ++i)
            {
                testStream.MountRom(baseTime, baseTime, "test");
                testStream.Unmount(baseTime, baseTime, "test");
                ++baseTime;
            }

            // 1800～2400 s では、エラー閾値ぴったり
            baseTime = 3 * 10 * 60 * 1000;
            generator(testStream, 600, 320, 16 * MegabytePerMin, new int[] { 600, 600, 450 });

            // 2400～3000 s では、warning_ratio が相当低くないと WARNING にならない程度
            baseTime = 4 * 10 * 60 * 1000;
            generator(testStream, 1, 1, 4096, new int[] { 1, 2 });

            // 3600～4200 s では、マウント 1 回のみ
            baseTime = 6 * 10 * 60 * 1000;
            generator(testStream, 1, 0, 0, new int[] { });

            string testSourceFile = string.Format(@"{0}test_warn_ratio_write.txt", TestTempDirectory);
            testStream.SetUp();
            testStream.DumpFile(testSourceFile);
            foreach (int ratio in new int[] { 0, 50, 75, 99, 100 })
            {
                string testDestFile = string.Format(@"{0}test_warn_ratio_output_{1}.txt", TestTempDirectory, ratio);
                string testExpectFile = string.Format(@"{0}\WarnRatio\expect_{1}.txt", GetTestDataDirectory(), ratio);
                string message;
                string errorMessage;

                TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
                testDestFile = destFile.FilePath;
                Assert.AreEqual(
                    0,
                    ExecuteProgram(
                        string.Format("{0} -o {1}  --warning_ratio {2} --quiet", testSourceFile, testDestFile, ratio),
                        out message, out errorMessage));
                destFile.UpdateForExpect();
                Assert.AreNotEqual(string.Empty, message);
                Assert.AreEqual(string.Empty, errorMessage);
                Assert.AreEqual(
                    OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                    OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile),
                    testSourceFile + " -> " + testDestFile + "\n != " + testExpectFile);
            }
        }

        [TestMethod]
        public void TestWarningRatioWithFail()
        {
            int baseTime = 0;
            Action<TestStream, int[], int, ulong, int[]> generator = (TestStream stream, int[] mountCounts, int writeCount, ulong writeSize, int[] sameReadCounts) =>
            {
                ulong fileId = 1;
                stream.MountBis(baseTime, baseTime, "test", FsFunction.BisPartitionId.User);
                foreach (int sameReadCount in sameReadCounts)
                {
                    stream.OpenFile(baseTime, baseTime, fileId, string.Format("test:/file{0}", fileId));
                    for (int i = 0; i < sameReadCount; ++i)
                    {
                        stream.ReadFile(baseTime, baseTime, fileId, 10, 10);
                        ++baseTime;
                    }
                    ++fileId;
                }
                stream.OpenFile(baseTime, baseTime, fileId, string.Format("test:/file{0}", fileId));
                stream.WriteFile(baseTime, baseTime, fileId, 10, writeSize - (ulong)(writeCount - 1) * 1024);
                for (int i = 0; i < writeCount - 1; ++i)
                {
                    stream.WriteFile(baseTime, baseTime, fileId, 10, 1024);
                    ++baseTime;
                }
                stream.Unmount(baseTime, baseTime, "test");
                for (int i = 0; i < mountCounts[0] - 1; ++i)
                {
                    stream.MountBis(baseTime, baseTime, "test", FsFunction.BisPartitionId.User);
                    stream.Unmount(baseTime, baseTime, "test");
                    ++baseTime;
                }

                if (mountCounts.Length > 1)
                {
                    for (int i = 0; i < mountCounts[1]; ++i)
                    {
                        stream.MountSaveData(baseTime, baseTime, "test", 0x1);
                        stream.Unmount(baseTime, baseTime, "test");
                        ++baseTime;
                    }
                }
                if (mountCounts.Length > 2)
                {
                    for (int i = 0; i < mountCounts[2]; ++i)
                    {
                        stream.MountBcatSaveData(baseTime, baseTime, "test", 0x2);
                        stream.Unmount(baseTime, baseTime, "test");
                        ++baseTime;
                    }
                }
            };

            // testStreams[0] では Bis のみマウント
            // testStreams[1] では 3 種類マウントし、全ての数値を連動させる
            // testStreams[2] では 3 種類マウントし、数値はばらつかせる
            TestStream[] testStreams = { new TestStream(), new TestStream(), new TestStream() };
            foreach (TestStream stream in testStreams)
            {
                stream.StartLog();
            }
            ulong MegabytePerMin = 1024 * 1024;

            // 0～60 s では、エラー閾値の 75 %  より僅かに下あたり
            baseTime = 0;
            generator(testStreams[0], new int[] { 44 }, 23, 12 * MegabytePerMin - 1, new int[] { 44 });
            generator(testStreams[1], new int[] { 44, 44, 44 }, 23, 12 * MegabytePerMin - 1, new int[] { 44, 44, 44 });
            generator(testStreams[2], new int[] { 61, 44, 30 }, 23, 12 * MegabytePerMin - 1, new int[] { 45, 30, 44 });

            // 60～120 s では、エラー閾値の 75 %  ちょうど
            baseTime = 60 * 1000;
            generator(testStreams[0], new int[] { 45 }, 24, 12 * MegabytePerMin, new int[] { 45 });
            generator(testStreams[1], new int[] { 45, 45, 45 }, 24, 12 * MegabytePerMin, new int[] { 45, 45, 45 });
            generator(testStreams[2], new int[] { 45, 61, 44 }, 24, 12 * MegabytePerMin, new int[] { 30, 45, 44 });

            // 120～180 s では、エラー閾値の 75 % より僅かに上あたり
            baseTime = 2 * 60 * 1000;
            generator(testStreams[0], new int[] { 46 }, 25, 12 * MegabytePerMin + 1, new int[] { 46 });
            generator(testStreams[1], new int[] { 46, 46, 46 }, 25, 12 * MegabytePerMin + 1, new int[] { 46, 46, 46 });
            generator(testStreams[2], new int[] { 46, 45, 61 }, 25, 12 * MegabytePerMin + 1, new int[] { 46, 47, 45 });

            // testStreams[2] に MountRom を挟む
            ulong fileHandle = 99;
            testStreams[2].MountRom(baseTime, baseTime, "test");
            testStreams[2].OpenFile(baseTime, baseTime, fileHandle, string.Format("test:/file{0}", fileHandle));
            for (int i = 0; i < 46; ++i)
            {
                testStreams[2].ReadFile(baseTime, baseTime, fileHandle, 10, 10);
                ++baseTime;
            }
            ++fileHandle;
            testStreams[2].Unmount(baseTime, baseTime, "test");
            for (int i = 0; i < 45; ++i)
            {
                testStreams[2].MountRom(baseTime, baseTime, "test");
                testStreams[2].Unmount(baseTime, baseTime, "test");
                ++baseTime;
            }

            // 180～240 s では、エラー閾値ピッタリ
            baseTime = 3 * 60 * 1000;
            generator(testStreams[0], new int[] { 60 }, 32, 16 * MegabytePerMin, new int[] { 60 });
            generator(testStreams[1], new int[] { 60, 60, 60 }, 32, 16 * MegabytePerMin, new int[] { 60, 60, 60 });
            generator(testStreams[2], new int[] { 45, 61, 60 }, 32, 16 * MegabytePerMin, new int[] { 61, 60, 45 });

            // 240～300 s では、エラー閾値ピッタリより僅かに大きい
            baseTime = 4 * 60 * 1000;
            generator(testStreams[0], new int[] { 61 }, 33, 16 * MegabytePerMin + 1, new int[] { 61 });
            generator(testStreams[1], new int[] { 61, 61, 61 }, 33, 16 * MegabytePerMin + 1, new int[] { 61, 61, 61 });
            generator(testStreams[2], new int[] { 61, 62, 60 }, 33, 16 * MegabytePerMin + 1, new int[] { 61, 60, 62 });

            foreach (int id in new int[] { 0, 1, 2 })
            {
                string testSourceFile = string.Format(@"{0}test_warn_ratio_write_{1}.txt", TestTempDirectory, id);
                testStreams[id].SetUp();
                testStreams[id].DumpFile(testSourceFile);
                foreach (int ratio in new int[] { 0, 50, 75, 99, 100 })
                {
                    string testDestFile = string.Format(@"{0}test_warn_ratio_output_{1}_{2}.txt", TestTempDirectory, id, ratio);
                    string testExpectFile = string.Format(@"{0}\WarnRatio\expect_{1}_{2}.txt", GetTestDataDirectory(), id, ratio);
                    string message;
                    string errorMessage;
                    TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
                    testDestFile = destFile.FilePath;
                    Assert.AreEqual(
                        1,
                        ExecuteProgram(
                            string.Format("{0} -o {1}  --warning_ratio {2} --quiet --time_range 1", testSourceFile, testDestFile, ratio),
                            out message, out errorMessage));
                    destFile.UpdateForExpect();
                    Assert.AreNotEqual(string.Empty, message);
                    Assert.AreEqual(string.Empty, errorMessage);
                    Assert.AreEqual(
                        OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                        OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile),
                        testSourceFile + " -> " + testDestFile + "\n != " + testExpectFile);
                }
            }
        }

        [TestMethod]
        public void TestFileCacheAccessLog()
        {
            var testSourceFile = GetTestDataDirectory() + @"\FileCacheAccess\write.txt";
            var testExpectFile = GetTestDataDirectory() + @"\FileCacheAccess\expect.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            Assert.AreEqual(0, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestFileCacheAccessLogNoFetched()
        {
            var testSourceFile = GetTestDataDirectory() + @"\FileCacheAccess\write_no_fetch.txt";
            var testExpectFile = GetTestDataDirectory() + @"\FileCacheAccess\expect_no_fetch.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            Assert.AreEqual(0, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestFileCacheAccessLogMix()
        {
            var testSourceFile = GetTestDataDirectory() + @"\FileCacheAccess\write_mix.txt";
            var testExpectFile = GetTestDataDirectory() + @"\FileCacheAccess\expect_mix.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            Assert.AreEqual(0, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestFileCacheAccessLogExceeded()
        {
            var testSourceFile = GetTestDataDirectory() + @"\FileCacheAccess\write_exceeded.txt";
            var testExpectFile = GetTestDataDirectory() + @"\FileCacheAccess\expect_exceeded.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            Assert.AreEqual(0, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestFileCacheAccessLogInvalidData()
        {
            var testSourceFile = GetTestDataDirectory() + @"\FileCacheAccess\write_invalidData.txt";
            var testExpectFile = GetTestDataDirectory() + @"\FileCacheAccess\expect_invalidData.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            Assert.AreEqual(1, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestFileCacheAccessLogInvalidLog()
        {
            var testSourceFile = GetTestDataDirectory() + @"\FileCacheAccess\write_invalidLog.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            string message;
            string errorMessage;
            Assert.AreEqual(1, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
        }

        [TestMethod]
        public void TestFileCacheAccessLogInvalidMix()
        {
            var testSourceFile = GetTestDataDirectory() + @"\FileCacheAccess\write_invalidMix.txt";
            var testExpectFile = GetTestDataDirectory() + @"\FileCacheAccess\expect_invalidMix.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            Assert.AreEqual(1, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestIndividualFileDataCache()
        {
            var testSourceFile = GetTestDataDirectory() + @"\FileCacheAccess\write_IndividualFileDataCache.txt";
            var testExpectFile = GetTestDataDirectory() + @"\FileCacheAccess\expect_IndividualFileDataCache.txt";
            string testDestFile = TestTempDirectory + @"\out.txt";
            TestDestFile destFile = new TestDestFile(testDestFile, testExpectFile);
            testDestFile = destFile.FilePath;

            string message;
            string errorMessage;
            Assert.AreEqual(0, ExecuteProgram(testSourceFile + " --quiet -o " + testDestFile,
                out message, out errorMessage));
            destFile.UpdateForExpect();
            Assert.AreEqual(string.Empty, errorMessage);
            Assert.AreEqual(
                OutputFileUtility.ReadAllTextExceptFirstLine(testExpectFile),
                OutputFileUtility.ReadAllTextExceptFirstLine(testDestFile));
        }

        [TestMethod]
        public void TestCheckMode()
        {
            // CheckMode の戻し忘れ防止用のテストです
            Assert.IsTrue(CheckMode);
        }

        // ゴールデンバージョンの出力を再取得する場合は、ここの値を false にする
        public static bool CheckMode
        {
            get
            {
                return true;
            }
        }
    }
}
