﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft2.h>
#include <nw/dw.h>
#include <nw/dw/dw_Controls.h>
#include <nw/dw/control/dw_FixedStackPanel.h>

#include <eftdemo_PerfomanceAnalyzer.h>

namespace nw      {
namespace eftdemo {

static const s32 LIST_COUNT = 16;

// Screen Size
f32                                         s_ScreenWidth = 0.0f;
f32                                         s_ScreenHeight = 0.0f;

// Windwos
nw::internal::dw::ControlWindow             s_PerfWindow;

// Name
nw::internal::dw::FixedListBox<LIST_COUNT>  s_PerfList;

// Panel
nw::internal::dw::FixedStackPanel<6>        s_Container[LIST_COUNT];

// EmitterSetName / EmitterName
nw::internal::dw::FixedLabel<32>            s_NameLabel[LIST_COUNT];

// CPU or GPU
nw::internal::dw::FixedLabel<32>            s_CalcTypeLabel[LIST_COUNT];

// PtclNum
nw::internal::dw::FixedLabel<32>            s_PtclNumLabel[LIST_COUNT];

// TexNum
nw::internal::dw::FixedLabel<32>            s_TextureNumLabel[LIST_COUNT];

// GpuCost
nw::internal::dw::FixedLabel<32>            s_GpuCostLabel[LIST_COUNT];

// Memory Consumption
nw::internal::dw::FixedLabel<32>            s_MemoryLabel[LIST_COUNT];

// Gpuカウンター用バッファ
u64*                                        s_GpuCounterBegin;
u64*                                        s_GpuCounterEnd;

// Gpu書き込みIdx
u32                                         s_GpuCounterIdx = 0;

nw::ut::MemoryAllocator*                    s_NwAllocator;

// EmitterSet名前
char                                        s_EmitterSetName[64];
//

//----------------------------------------
// 初期化
//----------------------------------------
void InitializePerfAnalyzer( nw::ut::MemoryAllocator* allocator, nw::internal::dw::WindowManager* manager, f32 width, f32 height )
{
    s_NwAllocator = allocator;

#if defined(NW_PLATFORM_CAFE)
    s_GpuCounterBegin = reinterpret_cast<u64*>( s_NwAllocator->Alloc( sizeof(u64) * LIST_COUNT, GX2_DEFAULT_BUFFER_ALIGNMENT ) );
    s_GpuCounterEnd   = reinterpret_cast<u64*>( s_NwAllocator->Alloc( sizeof(u64) * LIST_COUNT, GX2_DEFAULT_BUFFER_ALIGNMENT ) );
#else
    s_GpuCounterBegin = reinterpret_cast<u64*>( s_NwAllocator->Alloc( sizeof(u64) * LIST_COUNT ) );
    s_GpuCounterEnd   = reinterpret_cast<u64*>( s_NwAllocator->Alloc( sizeof(u64) * LIST_COUNT ) );
#endif

    s_ScreenWidth   = width;
    s_ScreenHeight  = height;

    s_PerfWindow.SetSize( 500.f, 128.f );
    s_PerfWindow.SetTitle( "Perf Analyzer" );
    s_PerfWindow.SetContent( &s_PerfList );
    s_PerfWindow.SetMeasurement( nw::internal::dw::MEASUREMENT_AUTO );
    s_PerfWindow.SetBackgroundColor( nw::ut::Color4f(0.2f, 0.3f, 0.3f, 1.0f) );

    manager->CreateWindow( &s_PerfWindow, s_ScreenWidth - s_PerfWindow.GetWidth(), s_ScreenHeight - s_PerfWindow.GetHeight() );

    #define TEXT_SCALE  0.8f

    for( u32 i = 0; i < LIST_COUNT; i++ )
    {
        s_NameLabel[i].SetTextScale( TEXT_SCALE );
        s_CalcTypeLabel[i].SetTextScale( TEXT_SCALE );
        s_PtclNumLabel[i].SetTextScale( TEXT_SCALE );
        s_TextureNumLabel[i].SetTextScale( TEXT_SCALE );
        s_GpuCostLabel[i].SetTextScale( TEXT_SCALE );
        s_MemoryLabel[i].SetTextScale( TEXT_SCALE );
    }

    for( u32 i = 0; i < 16; i++ )
    {
#if defined(NW_PLATFORM_CAFE)
        s_GpuCounterBegin[i] =  GX2_INVALID_COUNTER_VALUE_U64;
        s_GpuCounterEnd[i]   =  GX2_INVALID_COUNTER_VALUE_U64;
#else
        s_GpuCounterBegin[i] =  0;
        s_GpuCounterEnd[i]   =  0;
#endif
    }
#if defined(NW_PLATFORM_CAFE)
    DCFlushRange( s_GpuCounterBegin, sizeof(u64) * 16 );
    DCFlushRange( s_GpuCounterEnd,   sizeof(u64) * 16 );
#endif

    // EmitterSetNameを初期化
    memset( s_EmitterSetName, 0, sizeof(s_EmitterSetName) );
}

//----------------------------------------
// 終了処理
//----------------------------------------
void FinalizePerfAnalyzer()
{
    if ( s_GpuCounterBegin )
    {
        s_NwAllocator->Free( s_GpuCounterBegin );
    }

    if ( s_GpuCounterEnd )
    {
        s_NwAllocator->Free( s_GpuCounterEnd );
    }
}

//----------------------------------------
// GPU処理コールバックリセット
//----------------------------------------
void ResetPerfAnalyzer()
{
    s_GpuCounterIdx = 0;
}

//----------------------------------------
// GPU処理コールバック
//----------------------------------------
void PerfAnalyzerCallBack( nw::eft2::DrawEmitterProfilerArg& arg )
{
    if ( arg.emitter )
    {
#if defined(NW_PLATFORM_CAFE)
        bool skipMeasure = strcmp(s_EmitterSetName, arg.emitter->emitterSet->GetEmitterSetName() ) ? true : false;
        if (s_GpuCounterIdx >= LIST_COUNT)
        {
            skipMeasure = true;
        }

        if ( !skipMeasure )
        {
            GX2SampleBottomGPUCycle( (u64*)( &s_GpuCounterBegin[s_GpuCounterIdx] ) );
        }
#endif

        arg.system->DrawEmitter( (nw::eft2::Emitter*)arg.emitter, arg.shaderType, arg.userParam );

#if defined(NW_PLATFORM_CAFE)
        if ( !skipMeasure )
        {
            GX2SampleBottomGPUCycle( (u64*)( &s_GpuCounterEnd[s_GpuCounterIdx] ) );
        }
#endif
        s_GpuCounterIdx++;
    }
}


///----------------------------------------
// 処理
//----------------------------------------
u32 ProcessPerfAnalyzer( nw::eft2::EmitterSet* emitterSet )
{
    s_PerfList.ClearItems();
    if ( !emitterSet )
    {
        s_PerfWindow.SetIsVisible( false );
        return 0;
    }
    else
    {
        if ( !emitterSet->IsAlive() )
        {
            s_PerfWindow.SetIsVisible( false );
            return 0;
        }

        s_PerfWindow.SetIsVisible( true );

        const char* emitterSetName = emitterSet->GetEmitterSetName();

        if ( strcmp( emitterSetName, s_EmitterSetName ) == 0 )
        {
            s_PerfWindow.SetTitle( emitterSetName );
        }
        else
        {
            s_PerfWindow.SetTitle( s_EmitterSetName );
        }

        // 計測対象のemitterSet名を登録する。
        strcpy( s_EmitterSetName, emitterSetName );
    }

    char stringTemp[256];
    u32  lineCnt = 0;

    u32 numEmitter = emitterSet->GetNumEmitter();
    if ( numEmitter > 16 )
    {
        numEmitter = 16;
    }
    for ( s32 i = numEmitter; i > 0; i-- )
    {
        const nw::eft2::Emitter* emitter = emitterSet->GetAliveEmitter( i - 1 );
        if ( ( emitter == NULL ) || ( emitter->isChild ) )
        {
            continue;
        }
        // Panel
        s_Container[lineCnt].SetOrientation(nw::internal::dw::HORIZONTAL);
        s_Container[lineCnt].GetContents().ClearItems();
        s_PerfList.AddItem( &s_Container[lineCnt] );

        // Name
        nw::ut::snprintf( (char *)stringTemp, 256, 256 - 1, "%-32s|", emitter->emitterData->name );
        nw::internal::dw::Label& nameLabel = s_NameLabel[lineCnt];
        nameLabel.SetText( (char *)stringTemp );
        s_Container[lineCnt].GetContents().AddItem( &nameLabel );
        // CPU or GPU
        if ( emitter->emitterData->volume.isGpuEmitter )
        {

            if ( emitter->emitterData->emitter.calcType == nw::eft2::EFT_EMITTER_CALC_TYPE_GPU_SO )
            {
                nw::ut::snprintf( (char *)stringTemp, 256, 256 - 1, "|GPUSO|" );
            }
            else
            {
                nw::ut::snprintf( (char *)stringTemp, 256, 256 - 1, "| GPU |" );
            }
        }
        else
        {
            nw::ut::snprintf( (char *)stringTemp, 256, 256 - 1, "| CPU |" );
        }
        nw::internal::dw::Label& typeLabel = s_CalcTypeLabel[lineCnt];
        typeLabel.SetText( (char *)stringTemp );
        s_Container[lineCnt].GetContents().AddItem( &typeLabel );
        // PtclNum
        nw::ut::snprintf( (char *)stringTemp, 256, 256 - 1, "Ptcl:%5d|", ( emitter->ptclNum /*+ emitter->childEntryNum*/ ) );
        nw::internal::dw::Label& numLabel = s_PtclNumLabel[lineCnt];
        numLabel.SetText( (char *)stringTemp );
        s_Container[lineCnt].GetContents().AddItem( &numLabel );

        // Texture Num
        nw::ut::snprintf( (char *)stringTemp, 256, 256 - 1, "Tex:%2d|", emitter->emitterData->combiner.colorCombinerProcess );
        nw::internal::dw::Label& texNumLabel = s_TextureNumLabel[lineCnt];
        texNumLabel.SetText( (char *)stringTemp );
        s_Container[lineCnt].GetContents().AddItem( &texNumLabel );

       // Memory consumption
        u32 memSize = emitter->allocedFromDynamicHeapSize  +
                      emitter->streamOutPos.GetBufferSize() +
                      emitter->streamOutVec.GetBufferSize();

        nw::ut::snprintf( (char *)stringTemp, 256, 256 - 1, "Mem:%6d byte|", memSize );
        nw::internal::dw::Label& MemoryLabel = s_MemoryLabel[lineCnt];
        MemoryLabel.SetText( (char *)stringTemp );
        s_Container[lineCnt].GetContents().AddItem( &MemoryLabel );

#if EFT_IS_CAFE
        #define RLT_TO_MICROSEC 1.0/(GX2_TOP_BOTTOM_CLOCK_CYCLES / 1000000)

        // GpuCost
        nw::ut::Tick gpuCost(0);
        {
            bool ret0 = GX2GetGPUCycleReady( (u64 *)&s_GpuCounterBegin[lineCnt] );
            bool ret1 = GX2GetGPUCycleReady( (u64 *)&s_GpuCounterEnd[lineCnt] );

            if ( ret0 && ret1 )
            {
                nw::ut::Tick begin   = nw::ut::Tick( static_cast<s64>( GX2GPUTimeToCPUTime( s_GpuCounterBegin[lineCnt] ) ));
                nw::ut::Tick end     = nw::ut::Tick( static_cast<s64>( GX2GPUTimeToCPUTime( s_GpuCounterEnd[lineCnt] ) ));
                gpuCost = end - begin;
            }
        }

        u32 microSec = (u32)gpuCost.ToTimeSpan().GetMicroSeconds();
        f32 miliSec = (f32)microSec / 1000.f;
        f32 emitterMilisec =  miliSec;
        nw::ut::snprintf( (char *)stringTemp, 256, 256 - 1, "Gpu:%.2f msec(%.2f)", emitterMilisec, (emitterMilisec / 16.66) * 100 );
        nw::internal::dw::Label& gcostLabel = s_GpuCostLabel[lineCnt];
        gcostLabel.SetText( (char *)stringTemp );
        s_Container[lineCnt].GetContents().AddItem( &gcostLabel );
#endif

        lineCnt++;
    }
    s_PerfWindow.SetPosition( nw::math::Vector2( s_ScreenWidth - s_PerfWindow.GetWidth() - 10.f, s_ScreenHeight - s_PerfWindow.GetHeight() - 10.f ) );

    return lineCnt;

} // NOLINT(readability/fn_size)

} // namespace eftdemo
} // namespace nw
